/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "tests.h"
#include "shared.h"
#include "../common/common.h"
#include "../Hermes2/sound.h"
#include <math.h>

//------------------------------------------------------------------------

#define AUDIO_AMPLITUDE  25000
#define SND_STREAM_DW    0.000130899694 // rad/sample

// Offset from the reference frequency of 2000Hz
#define TEST_1100_HZ    -900
#define TEST_1200_HZ    -800
#define TEST_1300_HZ    -700
#define TEST_1500_HZ    -500
#define TEST_1900_HZ    -100
#define TEST_2100_HZ     100
#define TEST_2300_HZ     300

/* VIS_Leader()
 *
 * Generated a VIS Leader (Calibration) Signal
 */
  BOOLEAN
VIS_Leader( BOOLEAN reset, uint16_t add_noise )
{
  static int
    noise_count = 0,
    count = 0;

  if( reset )
  {
    count = 0;
    noise_count = 0;
    return(True);
  }

  // Generate random noise centered on 2000Hz
  if( add_noise && (noise_count < 100000) )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to 1900Hz of maximum +-dev
    double dev = (double)add_noise;
    sstv_status.signal_freq = (int16_t)( dev * noise );
    noise_count++;
    return( True );
  }

  // Generates a VIS Leader signal:
  // 300mS-1900Hz, 10mS-1200Hz, 300mS-1900Hz
  if( count < 14400 )
    sstv_status.signal_freq = TEST_1900_HZ;
  else if( count < 14880 )
    sstv_status.signal_freq = TEST_1200_HZ;
  else if( count < 29280 )
    sstv_status.signal_freq = TEST_1900_HZ;
  else return( False );

  count++;

  // Add noise to the simulated signal
  if( add_noise )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to the frequency of maximum +-dev
    double dev = (double)add_noise; // Hz
    sstv_status.signal_freq += (int16_t)( dev * noise );

    //// sstv_status.signal_freq  = (int16_t)( dev * noise ); // Noise only
  }

  // Generate audio samples of the current frequency
  static double w = 0;
  sstv_status.audio_sample = (int16_t)( AUDIO_AMPLITUDE * sin(w) );
  w += (double)( sstv_status.signal_freq + IQ_STREAM_REF_FREQ ) * SND_STREAM_DW;
  if( w >= M_2PI ) w -= M_2PI;

  return( True );
} // VIS_Leader()

//------------------------------------------------------------------------

/* VIS_Mode()
 *
 * Generates a sequence of tones that specify the SSTV Mode Number
 */
  BOOLEAN
VIS_Mode( BOOLEAN reset, uint8_t mode, uint16_t add_noise )
{
  static uint32_t
    noise_count = 0,
    count = 0;

  static uint8_t
    md = 255,
    vis_mode_num;

  if( reset )
  {
    noise_count = 0;
    count = 0;
    return( True );
  }

  if( md != mode )
  {
    vis_mode_num = sstv_mode_params[mode].vis_mode_num;
    md = mode;
  }

  // Generates the VIS Mode Number sequence of 8 30mS tone pulses
  if( count < 1440 )
    sstv_status.signal_freq = TEST_1200_HZ;   // 30mS-1200Hz Start bit
  else if( count < 2880 )
  {
    if( vis_mode_num & 0x01 )
      sstv_status.signal_freq = TEST_1100_HZ; // 30mS-1100Hz 1 Bit
    else
      sstv_status.signal_freq = TEST_1300_HZ; // 30mS-1300Hz 0 Bit
  }
  else if( count < 4320 )
  {
    if( vis_mode_num & 0x02 )
      sstv_status.signal_freq = TEST_1100_HZ; // 30mS-1100Hz 1 Bit
    else
      sstv_status.signal_freq = TEST_1300_HZ; // 30mS-1300Hz 0 Bit
  }
  else if( count < 5760 )
  {
    if( vis_mode_num & 0x04 )
      sstv_status.signal_freq = TEST_1100_HZ; // 30mS-1100Hz 1 Bit
    else
      sstv_status.signal_freq = TEST_1300_HZ; // 30mS-1300Hz 0 Bit
  }
  else if( count < 7200 )
  {
    if( vis_mode_num & 0x08 )
      sstv_status.signal_freq = TEST_1100_HZ; // 30mS-1100Hz 1 Bit
    else
      sstv_status.signal_freq = TEST_1300_HZ; // 30mS-1300Hz 0 Bit
  }
  else if( count < 8640 )
  {
    if( vis_mode_num & 0x10 )
      sstv_status.signal_freq = TEST_1100_HZ; // 30mS-1100Hz 1 Bit
    else
      sstv_status.signal_freq = TEST_1300_HZ; // 30mS-1300Hz 0 Bit
  }
  else if( count < 10080 )
  {
    if( vis_mode_num & 0x20 )
      sstv_status.signal_freq = TEST_1100_HZ; // 30mS-1100Hz 1 Bit
    else
      sstv_status.signal_freq = TEST_1300_HZ; // 30mS-1300Hz 0 Bit
  }
  else if( count < 11520 )
  {
    if( vis_mode_num & 0x40 )
      sstv_status.signal_freq = TEST_1100_HZ; // 30mS-1100Hz 1 Bit
    else
      sstv_status.signal_freq = TEST_1300_HZ; // 30mS-1300Hz 0 Bit
  }
  else if( count < 12960 )
  {
    if( vis_mode_num & 0x80 )
      sstv_status.signal_freq = TEST_1100_HZ; // 30mS-1100Hz 1 Bit
    else
      sstv_status.signal_freq = TEST_1300_HZ; // 30mS-1300Hz 0 Bit
  }
  else if( count < 14400 )
    sstv_status.signal_freq = TEST_1200_HZ; // 30mS-1200Hz Stop bit
  else
  {
    // Generate random noise centered on 2000Hz
    if( add_noise && (noise_count < 10000) )
    {
      // This gives a number -1.0 < n < 1.0
      double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

      // This adds random deviations to 1900Hz of maximum +-dev
      double dev = (double)add_noise;
      sstv_status.signal_freq = (int16_t)( dev * noise );
      noise_count++;
      return( True );
    }

    return( False );
  }

  count++;

  // Add noise to the simulated signal
  if( add_noise )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to the frequency of maximum +-dev
    double dev = (double)add_noise; // Hz
    sstv_status.signal_freq += (int16_t)( dev * noise );

    //// sstv_status.signal_freq  = (int16_t)( dev * noise ); // Noise only
  }

  // Generate audio samples of the current frequency
  static double w = 0;
  sstv_status.audio_sample = (int16_t)( AUDIO_AMPLITUDE * sin(w) );
  w += (double)( sstv_status.signal_freq + IQ_STREAM_REF_FREQ ) * SND_STREAM_DW;
  if( w >= M_2PI ) w -= M_2PI;

  return( True );
} // VIS_Mode()

//------------------------------------------------------------------------

/* Line_Sync()
 *
 * Generates Line Sync pulses
 */
  BOOLEAN
Line_Sync( BOOLEAN reset, uint8_t mode, uint16_t add_noise )
{
  static uint32_t
    interv_len,
    line_len,
    count = 0;

  static uint8_t md = 255;

  if( reset )
  {
    count = 0;
    md = 255;
    return(True);
  }

  if( md != mode )
  {
    line_len   = (uint32_t)( sstv_mode_params[mode].line_time * SND_DSP_RATE );
    interv_len = line_len - (uint32_t)( sstv_mode_params[mode].line_sync * SND_DSP_RATE );
    md = mode;
  }

  if( count < interv_len ) // Generate random noise centered on 2000Hz
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to 2000Hz of max +-dev
    double dev = 300.0; // Hz
    sstv_status.signal_freq = (int16_t)( dev * noise );
  }
  else if( count < line_len )
    sstv_status.signal_freq = TEST_1200_HZ;
  else return( False );

  count++;

  // Add noise to the simulated signal
  if( add_noise )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to the frequency of maximum +-dev
    double dev = (double)add_noise; // Hz
    sstv_status.signal_freq += (int16_t)( dev * noise );
  }

  return( True );
} // Line_Sync()

//------------------------------------------------------------------------

/* Martin_Pasokon_Test()
 *
 * Creates a simple Martin or Pasokon mode test image
 */
  void
Martin_Pasokon_Test( BOOLEAN reset, uint8_t mode, uint16_t add_noise )
{
  static uint32_t
    separator_len,  // Length of separator or porch
    sync_pulse_len, // Length of line sync pulse
    scan_length,    // Length of image pixel in discriminator samples
    samples_count;  // Counts number of frequency samples produced

  static uint8_t stage, color;

  if( reset )
  {
    scan_length    = (uint32_t)( sstv_mode_params[mode].scan_time * SND_DSP_RATE + 0.1 );
    separator_len  = (uint32_t)( sstv_mode_params[mode].separator * SND_DSP_RATE + 0.1 );
    sync_pulse_len = (uint32_t)( sstv_mode_params[mode].line_sync * SND_DSP_RATE + 0.1 );
    samples_count  = 0;
    color = 0;
    stage = 1;
  }

  switch( stage )
  {
    case 1: // Line sync
      sstv_status.signal_freq = TEST_1200_HZ;
      samples_count++;
      if( samples_count >= sync_pulse_len )
      {
        samples_count = 0;
        stage = 2;
      }
      break;

    case 2: // Sync porch
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 3;
      }
      break;

    case 3: // Green scan
      if( color == 0 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 4;
      }
      break;

    case 4: // Separator
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 5;
      }
      break;

    case 5: // Blue scan
      if( color == 1 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 6;
      }
      break;

    case 6: // Separator
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 7;
      }
      break;

    case 7: // Red scan
      if( color == 2 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        color++;
        if( color == 3 ) color = 0;
        stage = 8;
      }
      break;

    case 8: // Separator
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 1;
      }
      break;
  }

  if( add_noise )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to the frequency of maximum +-dev
    double dev = (double)add_noise; // Hz
    sstv_status.signal_freq += (int16_t)( dev * noise );
  }

  // Generate audio samples of the current frequency
  static double w = 0;
  sstv_status.audio_sample = (int16_t)( AUDIO_AMPLITUDE * sin(w) );
  w += (double)( sstv_status.signal_freq + IQ_STREAM_REF_FREQ ) * SND_STREAM_DW;
  if( w >= M_2PI ) w -= M_2PI;

} // Martin_Pasokon_Test()

//------------------------------------------------------------------------

/* SC2180_Test()
 *
 * Creates a simple SC2-180 mode test image
 */
  void
SC2180_Test( BOOLEAN reset, uint8_t mode, uint16_t add_noise )
{
  static uint32_t
    separator_len,  // Length of separator or porch
    sync_pulse_len, // Length of line sync pulse
    scan_length,    // Length of image pixel in discriminator samples
    samples_count;  // Counts number of frequency samples produced

  static uint8_t stage, color;

  if( reset )
  {
    scan_length    = (uint32_t)( sstv_mode_params[mode].scan_time * SND_DSP_RATE + 0.1 );
    separator_len  = (uint32_t)( sstv_mode_params[mode].separator * SND_DSP_RATE + 0.1 );
    sync_pulse_len = (uint32_t)( sstv_mode_params[mode].line_sync * SND_DSP_RATE + 0.1 );
    samples_count  = 0;
    color = 0;
    stage = 1;
  }

  switch( stage )
  {
    case 1: // Line sync
      sstv_status.signal_freq = TEST_1200_HZ;
      samples_count++;
      if( samples_count >= sync_pulse_len )
      {
        samples_count = 0;
        stage = 2;
      }
      break;

    case 2: // Sync porch
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 3;
      }
      break;

    case 3: // Red scan
      if( color == 0 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 4;
      }
      break;

    case 4: // Green scan
      if( color == 1 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 5;
      }
      break;

    case 5: // Blue scan
      if( color == 2 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        color++;
        if( color == 3 ) color = 0;
        stage = 1;
      }
      break;
  }

  if( add_noise )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to the frequency of maximum +-dev
    double dev = (double)add_noise; // Hz
    sstv_status.signal_freq += (int16_t)( dev * noise );
  }

  // Generate audio samples of the current frequency
  static double w = 0;
  sstv_status.audio_sample = (int16_t)( AUDIO_AMPLITUDE * sin(w) );
  w += (double)( sstv_status.signal_freq + IQ_STREAM_REF_FREQ ) * SND_STREAM_DW;
  if( w >= M_2PI ) w -= M_2PI;

} // SC2180_Test()

//------------------------------------------------------------------------

/* Scottie_Test()
 *
 * Creates a simple Martin mode test image
 */
  void
Scottie_Test( BOOLEAN reset, uint8_t mode, uint16_t add_noise )
{
  static uint32_t
    separator_len,  // Length of separator or porch
    sync_pulse_len, // Length of line sync pulse
    scan_length,    // Length of image pixel in discriminator samples
    samples_count;  // Counts number of frequency samples produced

  static uint8_t stage, color;

  if( reset )
  {
    scan_length    = (uint32_t)( sstv_mode_params[mode].scan_time * SND_DSP_RATE + 0.1 );
    separator_len  = (uint32_t)( sstv_mode_params[mode].separator * SND_DSP_RATE + 0.1 );
    sync_pulse_len = (uint32_t)( sstv_mode_params[mode].line_sync * SND_DSP_RATE + 0.1 );
    samples_count  = 0;
    color = 0;
    stage = 1;
  }

  switch( stage )
  {
    case 1: // Starting Line sync (one off)
      sstv_status.signal_freq = TEST_1200_HZ;
      samples_count++;
      if( samples_count >= sync_pulse_len )
      {
        samples_count = 0;
        stage = 2;
      }
      break;

    case 2: // Separator
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 3;
      }
      break;

    case 3: // Green scan
      if( color == 0 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 4;
      }
      break;

    case 4: // Separator
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 5;
      }
      break;

    case 5: // Blue scan
      if( color == 1 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 6;
      }
      break;

    case 6: // Line sync
      sstv_status.signal_freq = TEST_1200_HZ;
      samples_count++;
      if( samples_count >= sync_pulse_len )
      {
        samples_count = 0;
        stage = 7;
      }
      break;

    case 7: // Sync porch
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 8;
      }
      break;

    case 8: // Red scan
      if( color == 2 ) sstv_status.signal_freq = TEST_2300_HZ;
      else sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        color++;
        if( color == 3 ) color = 0;
        stage = 2;
      }
      break;
  }

  if( add_noise )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to the frequency of maximum +-dev
    double dev = (double)add_noise; // Hz
    sstv_status.signal_freq += (int16_t)( dev * noise );
  }

  // Generate audio samples of the current frequency
  static double w = 0;
  sstv_status.audio_sample = (int16_t)( AUDIO_AMPLITUDE * sin(w) );
  w += (double)( sstv_status.signal_freq + IQ_STREAM_REF_FREQ ) * SND_STREAM_DW;
  if( w >= M_2PI ) w -= M_2PI;

} // Scottie_Test()

//------------------------------------------------------------------------

/* PD_Test()
 *
 * Creates a simple PD mode test image
 */
  void
PD_Test( BOOLEAN reset, uint8_t mode, uint16_t add_noise )
{
  static uint32_t
    separator_len,  // Length of separator or porch
    sync_pulse_len, // Length of line sync pulse
    scan_length,    // Length of image pixel in discriminator samples
    samples_count;  // Counts number of frequency samples produced

  static uint8_t stage, color;

  static double R,G,B,Y,RY,BY;

  if( reset )
  {
    scan_length    = (uint32_t)( sstv_mode_params[mode].scan_time * SND_DSP_RATE + 0.1 );
    separator_len  = (uint32_t)( sstv_mode_params[mode].separator * SND_DSP_RATE + 0.1 );
    sync_pulse_len = (uint32_t)( sstv_mode_params[mode].line_sync * SND_DSP_RATE + 0.1 );
    samples_count  = 0;
    color = 0;
    stage = 1;
  }

  switch( stage )
  {
    case 1: // Line sync
      sstv_status.signal_freq = TEST_1200_HZ;
      samples_count++;
      if( samples_count >= sync_pulse_len )
      {
        samples_count = 0;
        stage = 2;
      }
      break;

    case 2: // Sync porch
      sstv_status.signal_freq = TEST_1500_HZ;
      samples_count++;
      if( samples_count >= separator_len )
      {
        samples_count = 0;
        stage = 3;
      }
      break;

    case 3: // Y0 scan
      if( color == 0 )
      {
        R = 255;
        G = 0;
        B = 0;
      }
      else if( color == 1 )
      {
        R = 0;
        G = 255;
        B = 0;
      }
      else if( color == 2 )
      {
        R = 0;
        G = 0;
        B = 255;
      }

      Y = 16.0 + 0.25677 * R + 0.50410 * G + 0.09790 * B;
      sstv_status.signal_freq = (int16_t)(3.137 * Y) TEST_1500_HZ;

      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 4;
      }
      break;

    case 4: // R-Y scan
      if( color == 0 )
      {
        R = 255;
        G = 0;
        B = 0;
      }
      else if( color == 1 )
      {
        R = 0;
        G = 255;
        B = 0;
      }
      else if( color == 2 )
      {
        R = 0;
        G = 0;
        B = 255;
      }

      RY = 128.0 + 0.439186734 * R - 0.36776 * G - 0.071421 * B;
      sstv_status.signal_freq = (int16_t)(3.137 * RY) TEST_1500_HZ;

      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 5;
      }
      break;

    case 5: // B-Y scan
      if( color == 0 )
      {
        R = 255;
        G = 0;
        B = 0;
      }
      else if( color == 1 )
      {
        R = 0;
        G = 255;
        B = 0;
      }
      else if( color == 2 )
      {
        R = 0;
        G = 0;
        B = 255;
      }

      BY = 128.0 - 0.14821 * R - 0.29097 * G + 0.439186734 * B;
      sstv_status.signal_freq = (int16_t)(3.137 * BY) TEST_1500_HZ;

      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        stage = 6;
      }
      break;

    case 6: // Y1 scan
      if( color == 0 )
      {
        R = 255;
        G = 0;
        B = 0;
      }
      else if( color == 1 )
      {
        R = 0;
        G = 255;
        B = 0;
      }
      else if( color == 2 )
      {
        R = 0;
        G = 0;
        B = 255;
      }

      Y = 16.0 + 0.25677 * R + 0.50410 * G + 0.09790 * B;
      sstv_status.signal_freq = (int16_t)(3.137 * Y) TEST_1500_HZ;

      samples_count++;
      if( samples_count >= scan_length )
      {
        samples_count = 0;
        color += 1;
        if( color >= 3 ) color -= 3;
        stage = 1;
      }
      break;
  }

  if( add_noise )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to the frequency of maximum +-dev
    double dev = (double)add_noise; // Hz
    sstv_status.signal_freq += (int16_t)( dev * noise );
  }

  // Generate audio samples of the current frequency
  static double w = 0;
  sstv_status.audio_sample = (int16_t)( AUDIO_AMPLITUDE * sin(w) );
  w += (double)( sstv_status.signal_freq + IQ_STREAM_REF_FREQ ) * SND_STREAM_DW;
  if( w >= M_2PI ) w -= M_2PI;

} // PD_Test()

//------------------------------------------------------------------------

/* Test_Image()
 *
 * Creates a test image in the SSTV Image pixbuf
 */
  void
Test_Image( void )
{
  guchar *pix = sstv_rx_image.pixels;
  uint16_t h, v;

  for( v = 0; v < sstv_rx_image.height; v++ )
  {
    for( h = 0; h < sstv_rx_image.width; h++ )
    {
      pix[0] = v & 0xff;
      pix[1] = h & 0xff;
      pix[2] = 0xff;
      pix   += sstv_rx_image.n_channels;
    }

    ////pix += sstv_image.rowstride;
  }

  Queue_Draw( sstv_gui.sstv_rx_image_drawingarea );

} // Test_Image()

//------------------------------------------------------------------------

/* Test_FSK_ID()
 *
 * Create an FSK-ID stream with 5B4AZ as the callsign.
 * Text begins with a 20 2A preamble and ends with 01.
 * Bytes are 6-bit LSB first, 1 = 1900 Hz 0 = 2100 Hz.
 * Baud rate is 45.45 Bd = 1056 samples at 48000 Hz.
 */
  void
Test_FSK_ID( BOOLEAN reset, uint16_t add_noise )
{
  static uint32_t
    mesg_idx,      // Index to current message character
    shift,         // Bitwise shift index
    samples_count, // Counts number of frequency samples produced
    noise_count;

  static const uint8_t mesg[8] =
  { 0x20, 0x2A, '5'-0x20, 'B'-0x20, '4'-0x20, 'A'-0x20, 'Z'-0x20, 0x01 };

  if( reset )
  {
    samples_count = 0;
    mesg_idx = 0;
    noise_count = 0;
    shift = 0;
    return;
  }

  // Generate random noise for 10 FSKID chars
  if( add_noise && (noise_count < 63360) )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to 2000Hz of maximum +-dev
    double dev = (double)add_noise;
    sstv_status.signal_freq = (int16_t)( dev * noise );
    noise_count++;
    return;
  }

  // Transmit FSKID characters
  if( samples_count >= 1056 ) // One bit transmitted
  {
    samples_count = 0;
    shift++;
    if( shift >= 6 ) // One character transmitted
    {
      shift = 0;
      mesg_idx++;
      if( mesg_idx >= 8 ) // Message transmitted
      {
        mesg_idx = 0;
        samples_count = 0;
        noise_count = 0;
        shift = 0;
        puts("------\n");
        //////exit(0);
        return;
      }
    }
  }

  // 1 = 1900Hz  0 = 2100Hz LSB first
  if( (mesg[mesg_idx] >> shift) & 0x01 )
    sstv_status.signal_freq = TEST_1900_HZ;
  else
    sstv_status.signal_freq = TEST_2100_HZ;

  samples_count++;

  if( add_noise )
  {
    // This gives a number -1.0 < n < 1.0
    double noise = ( 2.0 * (double)(rand()) - RAND_MAX ) / RAND_MAX;

    // This adds random deviations to the frequency of maximum +-dev
    double dev = (double)add_noise; // Hz
    sstv_status.signal_freq += (int16_t)( dev * noise );
  }

  // Generate audio samples of the current frequency
  static double w = 0;
  sstv_status.audio_sample = (int16_t)( AUDIO_AMPLITUDE * sin(w) );
  w += (double)( sstv_status.signal_freq + IQ_STREAM_REF_FREQ ) * SND_STREAM_DW;
  if( w >= M_2PI ) w -= M_2PI;

} // Test_FSK_ID()

//------------------------------------------------------------------------

