/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef SSTV_SHARED_H
#define SSTV_SHARED_H   1

#include "common.h"
#include "../common/ifft.h"
#include "../common/common.h"
#include "../common/guest_utils.h"
#include <gtk/gtk.h>

// Size in bits of the FSKID data bytes
#define FSKID_BYTE_LENGTH       6

// Runtime config data
#define CALLSIGN_SIZE    24
#define FONT_FACE_SIZE   64
typedef struct _SSTV_RC
{
  // SSTV working directory
  char sstv_dir[FILE_NAME_SIZE];

  // Station Callsign
  char callsign[CALLSIGN_SIZE];

  // Tx Image file name
  gchar tx_image_file[FILE_NAME_SIZE];

  // Size of brush used to draw on Tx Image
  double tx_brush_size;

  // Name of Tx Font Face
  char tx_font_face[FONT_FACE_SIZE];

  // Tx Image Font and Brush Color
  GdkRGBA tx_brush_font_color;

  BOOLEAN
    vis_leader_icon_yes,
    vis_mode_icon_yes,
    sync_det_icon_yes,
    sync_match_icon_yes;

} sstv_rc_t;

// Runtime config data
extern sstv_rc_t sstv_rc;

// SSTV Mode parameters
typedef struct _SSTV_MODE_PARAMS
{
  gchar    *mode_name;      // SSTV Mode name
  uint8_t   mode_index;     // SSTV Mode index, used in search of Mode
  double    image_time;     // Total image transmission time, not including CAL/VIS
  double    line_time;      // Total line transmission time, including sync pulse
  double    scan_time;      // Color scan time (duration of color-only transmission)
  uint16_t  pix_per_line;   // Pixels per Line (per color scan)
  uint16_t  num_of_lines;   // Number of lines in image, includes 16 title lines
  uint8_t   vis_mode_num;   // SSTV VIS Mode Code number
  double    line_sync;      // Length of line sync pulse in mSec
  double    separator;      // Length of sync porch and color scan separators
  uint8_t   scale;          // Scale factor to resize final image
} sstv_mode_params_t;

// SSTV Modes parameters
extern sstv_mode_params_t sstv_mode_params[];

// SSTV program status (values of various parameters and results)
#define TX_IMAGE_NAME_SIZE   8

typedef struct _SSTV_STATUS
{
  // Instantaneous FM SSTV signal frequency
  int16_t signal_freq;

  double
    sync_det_level,   // Output of Goertzel detector of 1200Hz Sync pulse
    line_sync_period, // Interval between line sync pulses, in number of freq samples
    detector_sens;    // Sensitivity of various detectors of SSTV signals

  uint8_t
    sstv_action,     // What action the SSTV sequencer should enter
    sstv_mode_index, // Index to sstv_mode_params structure
    sstv_mode_num;   // SSTV Mode Identification Number with Parity, as in VIS

  int32_t
    sstv_image_index, // Current SSTV image index to frequency buffer
    sstv_image_start, // Index to frequency buffer where where SSTV image starts
    line_sync_index,  // Index to frequency buffer where line sync was detected
    scottie_offset;   // Offset between start of line and line sync in Scottie mode

  /* Error in the frequency of VIS 1900Hz Leader pulses, used to
   * correct the frequencies detected by the FM discriminator */
  int16_t freq_calibration;

  // Audio sample from the demodulator or test function
  int16_t audio_sample;

  // File name of image to transmit
  char tx_image_name[TX_IMAGE_NAME_SIZE];

  BOOLEAN  // Initialization flags
    receiver_init,        // SSTV is in receiver initialization
    freq_discrim_init,    // Initialize I/Q Frequency Discriminator
    tone_detectors_init,  // Initialize Goertzel Tone detectors
    vis_leader_det_init,  // Initialize VIS Leader Detector
    vis_mode_det_init,    // Initialize VIS Mode Bits Detector
    line_sync_det_init,   // Initialize Line Sync Detector
    line_sync_proc_init,  // Initialize Line Sync Processor
    image_decoder_init,   // Initialize SSTV Image Decoder
    fskid_decoder_init;   // Initialize FSKID Decoder

  BOOLEAN  // Various signals of the SSTV format detected flags
    vis_leader_detected,  // VIS Leader (calibration) detected
    line_sync_detected,   // Line Sync Pulse detected
    sstv_mode_detected,   // SSTV Mde detectedcted, by line sync rocessor
    new_period_detected;  // SSTV Image new line period detected

  BOOLEAN  // Various control flags
    detect_vis_leader,    // Enable detection of VIS Leader
    detect_vis_mode,      // Enable detection of VIS Mode Number
    image_decoder_enable, // Enable the SSTV image decoder function
    auto_save_enable,     // Enable auto save of images at completion
    cw_id_enable,         // Enable CW ID transmission
    fsk_id_enable,        // Enable FSK ID treansmission
    auto_listen_enable,   // Enable auto save of images at completion
    receive_active,       // Image reception active
    transmit_active,      // Image transmission active
    sstv_mode_auto,       // SSTV Mode is set for Auto detection
    save_image,           // Finalize and save image on user signal
    new_image,            // Finalize image and prepare for new image
    tx_image_draw,        // Enable drawing on the SSTV Tx Image
    tx_image_type,        // Enable typing on the SSTV Tx Image
    capitalize_font,      // Enable capitalization of Tx image font
    image_pixbuf_ready;   // SSTV Image pixbuf configured ready to use

} sstv_status_t;

// SSTV program status
extern sstv_status_t sstv_status;

// Sstv image display data
typedef struct _SSTV_RX_IMAGE
{
  // Pixel buffer for display
  guchar    *pixels;
  GdkPixbuf *pixbuf;
  gint rowstride, n_channels;

  uint16_t width, height;
  uint16_t line_count;

} sstv_rx_image_t;

extern sstv_rx_image_t sstv_rx_image;

// Sstv GUI data
typedef struct _SSTV_GUI
{
  GtkWidget
    *window,                    // SSTV's top window
    *rx_image_window,           // SSTV's Rx Image window
    *tx_image_window,           // SSTV's Tx Image window
    *gallery_window,            // SSTV's Icon View window
    *popup_menu,                // Popup main menu
    *scope_drawingarea,         // Signal scope widget
    *spectrum_drawingarea,      // Signal spectrum widget
    *sstv_rx_image_drawingarea, // Drawingarea to display SSTV image
    *sstv_tx_image,             // GtkImage to display SSTV Tx image
    *sstv_tx_image_eventbox,    // Eventbox for above
    *sstv_tx_brush_size,        // SSTV Tx Font Size spinbutton
    *sstv_icon_view,            // GtkIconView to display images for selection
    *sstv_mode_label,           // Menu button label
    *rx_status_label,           // Receiver Status label
    *df_label,                  // Frequency Calibration label
    *line_sync_drawingarea,     // Line Sync (1200Hz) gauge
    *sstv_vis_leader_image,     // VIS Leader detected LED image
    *sstv_vis_mode_image,       // VIS Mode detected LED image
    *sstv_sync_detect_image,    // Line Sync detected LED image
    *sstv_sync_match_image,     // Line Sync decoded (mode recognised) LED image
    *xmit_status_label,         // Label for the Transmit Button
    *rcve_status_label;         // Label for the Receive Button

  cairo_surface_t *cairo_surface; // Cairo surface for drawing and text on Tx Image

  // Widgets (text entries and buttons) of the QSO Record Frame
  GtkWidget
    *callsign,
    *rst_out,
    *rst_in,
    *op_name,
    *band,
    *locator,
    *qth_name,
    *sstv_new_record,
    *sstv_save_record,
    *sstv_clear_fields;

  // Gtk builders for some above that need to be global
  GtkBuilder
    *popup_menu_builder,
    *window_builder,
    *tx_window_builder;

} sstv_gui_t;

extern sstv_gui_t sstv_gui;

// Pixbuffer for waterfall
extern pixbuffer_t sstv_wfall;

// Signal scope size
extern uint16_t
  sstv_scope_width,
  sstv_scope_height;

// What action the SSTV decoder should enter
//// extern uint8_t sstv_action;

// IFFT data for spectrum display
extern ifft_data_t sstv_ifft_data;

// SSTV Image Decoder function pointer
extern void (* Sstv_Image_Decoder) (void);

#endif

