/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "interface.h"
#include "display.h"
#include "shared.h"
#include "tests.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../Hermes2/sound.h"
#include "../common/utils.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

//------------------------------------------------------------------

#define MODE_MENU_IDS \
  "sstv_menu", \
  NULL

#define RX_IMAGE_WINDOW_IDS \
  "sstv_rx_image_window", \
  NULL

#define TX_IMAGE_WINDOW_IDS \
  "sstv_tx_image_window", \
  "brush_size_adjustment", \
  NULL

#define ICON_VIEW_IDS \
  "sstv_gallery_window", \
  "sstv_gallery", \
  NULL

#define SSTV_WINDOW_IDS \
  "sstv_window", \
  "sstv_sensitivity_adjustment", \
  NULL

//------------------------------------------------------------------

static gchar sstv_glade[FILE_NAME_SIZE];

/* Create_Sstv_Window()
 *
 * Creates the main SSTV window
 */
  GtkWidget *
Create_Sstv_Window( GtkBuilder **builder )
{
  gchar *object_ids[] = { SSTV_WINDOW_IDS };

  // Setup file path to SSTV working directory
  Strlcpy( sstv_rc.sstv_dir, getenv("HOME"),   sizeof(sstv_rc.sstv_dir) );
  Strlcat( sstv_rc.sstv_dir, "/.hermes2/sstv/", sizeof(sstv_rc.sstv_dir) );
  Strlcpy( sstv_glade, sstv_rc.sstv_dir, sizeof(sstv_glade) ) ;
  Strlcat( sstv_glade, "sstv.glade",     sizeof(sstv_glade) ) ;

  Gtk_Builder( builder, sstv_glade, object_ids );
  sstv_gui.window = Builder_Get_Object( *builder, "sstv_window" );
  gtk_widget_show( sstv_gui.window );

  // Initialize window widget pointers to Null
  sstv_gui.rx_image_window = NULL;
  sstv_gui.tx_image_window = NULL;
  sstv_gui.gallery_window  = NULL;

  // Open SSTV Rx Image window
  sstv_gui.rx_image_window = Create_Sstv_Rx_Window();

  // Open SSTV Tx Image window
  sstv_gui.tx_image_window = Create_Sstv_Tx_Window();

  // Get scope widgets
  sstv_gui.scope_drawingarea =
    Builder_Get_Object( *builder, "sstv_scope_drawingarea" );
  sstv_gui.spectrum_drawingarea =
    Builder_Get_Object( *builder, "sstv_spectrum_drawingarea" );

  // Get SSTV Modes label
  sstv_gui.sstv_mode_label =
    Builder_Get_Object( *builder, "sstv_mode_label" );

  // Get SSTV FSKID label
  sstv_gui.callsign =
    Builder_Get_Object( *builder, "callsign" );

  // Get SSTV Rx Status label
  sstv_gui.rx_status_label =
    Builder_Get_Object( *builder, "rx_status_label" );

  // Get SSTV Freq Calibration label
  sstv_gui.df_label =
    Builder_Get_Object( *builder, "df_label" );

  // Get status indicator "LED" images
  sstv_gui.sstv_vis_leader_image =
    Builder_Get_Object( *builder, "sstv_vis_leader_image" );
  sstv_gui.sstv_vis_mode_image =
    Builder_Get_Object( *builder, "sstv_vis_mode_image" );
  sstv_gui.sstv_sync_detect_image =
    Builder_Get_Object( *builder, "sstv_sync_detect_image" );
  sstv_gui.sstv_sync_match_image =
    Builder_Get_Object( *builder, "sstv_sync_match_image" );

  // Set labels to Xmit and Rcve buttons
  GtkWidget *button =
    Builder_Get_Object( *builder, "sstv_transmit_button" );
  sstv_gui.xmit_status_label = gtk_label_new( _("TRANSMIT OFF") );
  gtk_widget_show( sstv_gui.xmit_status_label );
  gtk_container_add( GTK_CONTAINER(button), sstv_gui.xmit_status_label );
  g_object_set_data_full( G_OBJECT(sstv_gui.window), "sstv_xmit_status",
      g_object_ref(sstv_gui.xmit_status_label), (GDestroyNotify)g_object_unref );

  button = Builder_Get_Object( *builder, "sstv_receive_button" );
  sstv_gui.rcve_status_label = gtk_label_new( _("RECEIVE OFF") );
  gtk_widget_show( sstv_gui.rcve_status_label );
  gtk_container_add( GTK_CONTAINER(button), sstv_gui.rcve_status_label );
  g_object_set_data_full( G_OBJECT(sstv_gui.window), "sstv_rcve_status",
      g_object_ref( sstv_gui.rcve_status_label ), (GDestroyNotify)g_object_unref );

  // Get sizes of displays
  GtkAllocation alloc;
  gtk_widget_get_allocation( sstv_gui.scope_drawingarea, &alloc );
  sstv_scope_width  = (uint16_t)alloc.width;
  sstv_scope_height = (uint16_t)alloc.height;
  gtk_widget_get_allocation( sstv_gui.spectrum_drawingarea, &alloc );
  Sstv_Spectrum_Size_Allocate( (uint16_t)alloc.width, (uint16_t)alloc.height );

  // Default status settings
  sstv_status.sstv_mode_auto  = True;
  sstv_status.save_image      = True;
  sstv_status.new_image       = False;
  sstv_status.receive_active  = False;
  sstv_status.transmit_active = False;
  sstv_status.cw_id_enable    = False;
  sstv_status.fsk_id_enable   = True;

  /* Set up the default mode and the FM discriminator buffer.
   * Also set up the Image drawing area and the Image pixbuf */
  New_Sstv_Mode( DEFAULT_MODE, True );

  // Initialize DUC Send semaphore
  if( !Init_Semaphore(&duc_send_semaphore, True) )
    return( NULL );

  // Null Transmit buffers
  xmit_buffer.xmit_buf_i = NULL;
  xmit_buffer.xmit_buf_q = NULL;

  // Load runtime config file, abort on error
  g_idle_add( Sstv_Read_Config, NULL );

  return( sstv_gui.window );
} // Create_Sstv_Window()

//------------------------------------------------------------------

/* Create_Sstv_Rx_Window()
 *
 * Creates the Rx SSTV Image window
 */
  GtkWidget *
Create_Sstv_Rx_Window( void )
{
  GtkBuilder *builder;
  gchar *object_ids[] = { RX_IMAGE_WINDOW_IDS };
  Gtk_Builder( &builder, sstv_glade, object_ids );
  GtkWidget *window = Builder_Get_Object( builder, "sstv_rx_image_window" );
  gtk_widget_show( window );

  // Get image drawingarea
  sstv_gui.sstv_rx_image_drawingarea =
    Builder_Get_Object( builder, "sstv_rx_image_drawingarea" );

  g_object_unref( builder );

  return( window );
} // Create_Sstv_Rx_Window()

//------------------------------------------------------------------

/* Create_Sstv_Tx_Window()
 *
 * Creates the Tx SSTV Image window
 */
  GtkWidget *
Create_Sstv_Tx_Window( void )
{
  gchar *object_ids[] = { TX_IMAGE_WINDOW_IDS };
  Gtk_Builder( &sstv_gui.tx_window_builder, sstv_glade, object_ids );
  GtkWidget *window =
    Builder_Get_Object( sstv_gui.tx_window_builder, "sstv_tx_image_window" );
  gtk_widget_show( window );

  // Get Tx image eventbox widget
  sstv_gui.sstv_tx_image_eventbox =
    Builder_Get_Object( sstv_gui.tx_window_builder, "sstv_tx_image_eventbox" );

  // Get Font Size spinbutton
  sstv_gui.sstv_tx_brush_size =
    Builder_Get_Object( sstv_gui.tx_window_builder, "sstv_tx_brush_size" );

  // Enable typing on Tx image
  sstv_status.tx_image_type   = True;
  sstv_status.capitalize_font = True;
  sstv_gui.cairo_surface      = NULL;
  sstv_gui.sstv_tx_image      = NULL;

  return( window );
} // Create_Sstv_Tx_Window()

//------------------------------------------------------------------

/* Create_Sstv_Icon_View()
 *
 * Creates the SSTV Icon View window
 */
  GtkWidget *
Create_Sstv_Icon_View( void )
{
  GtkBuilder *builder;
  gchar *object_ids[] = { ICON_VIEW_IDS };
  Gtk_Builder( &builder, sstv_glade, object_ids );
  GtkWidget *window = Builder_Get_Object( builder, "sstv_gallery_window" );
  gtk_widget_show( window );

  // Get image widget
  sstv_gui.sstv_icon_view = Builder_Get_Object( builder, "sstv_icon_view" );

  g_object_unref( builder );

  return( window );
} // Create_Sstv_Icon_View()

//------------------------------------------------------------------

/* Create_Sstv_Mode_Menu()
 *
 * Creates the Sstv popup menu
 */

  GtkWidget *
Create_Sstv_Mode_Menu( GtkBuilder **builder )
{
  gchar *object_ids[] = { MODE_MENU_IDS };
  Gtk_Builder( builder, sstv_glade, object_ids );
  GtkWidget *menu = Builder_Get_Object( *builder, "sstv_menu" );
  return( menu );
} // Create_Sstv_Mode_Menu()

//------------------------------------------------------------------

