/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "encode.h"
#include "callback_func.h"
#include "common.h"
#include "../Hermes2/callback_func.h"
#include <math.h>

/* Length in TX DUC samples of the VIS sequence (Raised Cos+Leader+FSK Code):
 * (5ms + 300ms + 10ms + 300ms + 10 * 30 ms) * DUC_SAMPLE_RATE (192000 S/s) */
#define VIS_LENGTH    175680

// VIS Leader length: 300ms * 192000 S/s
#define ENC_VIS_LEADER_LENGTH    57600

// VIS Break length: 10ms * 192000 S/s
#define ENC_VIS_BREAK_LENGTH    1920

// VIS Mode Bit length: 30ms * 192000 S/s
#define VIS_MODE_BIT_LENGTH     5760

// Length in DUC samples of Raised Cosine tone edges (5ms * 192kS/s)
#define RAISED_COS_LENGTH    960

// Phase angle of the SSTV sub-carrier signal
static double Phi = 0.0;

// Index to I and Q DUC buffers
static uint32_t xmit_buf_idx = 0;

// Size of Transmit buffers
static uint32_t xmit_buf_len;

//------------------------------------------------------------------------

/* Encode_VIS()
 *
 * Prepares the VIS sequence Leader-Break-Leader-Mode Train (FSK)
 */
  void
Encode_VIS( void )
{
  // Delta Phi for a given tone frequency
  double delta_phi = 0.0;


  // Aloocate Transmit buffers
  xmit_buf_len = VIS_LENGTH;
  Realloc_Xmit_Buffers( xmit_buf_len );

  // Delta Phi for 1900HZ per DUC sample
  delta_phi = M_2PI * VIS_1900HZ_REF / DUC_SAMPLE_RATE;
  xmit_buf_idx = 0;

  // *** Fill the Tx buffer with the VIS Leader tones *** //

  // Create a 5ms raised cosine rising edge of Leader tone
  double rcos_dphi = M_PI_2 / RAISED_COS_LENGTH; // Delta phi of raised cosine
  for( uint32_t idx = 0; idx < RAISED_COS_LENGTH; idx++ )
  {
    double rcos = DUC_SAMPLE_MAX * sin( rcos_dphi * (double)idx );
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( rcos * sin(Phi) );
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( rcos * cos(Phi) );
    xmit_buf_idx++;
    Phi += delta_phi;
    CLAMP_2PI( Phi );
  }

  // Create 300ms of 1900Hz Leader tone
  for( uint32_t idx = 0; idx < ENC_VIS_LEADER_LENGTH; idx++ )
  {
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(Phi) );
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(Phi) );
    xmit_buf_idx++;
    Phi += delta_phi;
    CLAMP_2PI( Phi );
  }

  // Delta Phi for 1200HZ per DUC sample
  delta_phi = M_2PI * VIS_1200HZ_REF / DUC_SAMPLE_RATE;

  // Create 10ms of 1200Hz Break tone
  for( uint32_t idx = 0; idx < ENC_VIS_BREAK_LENGTH; idx++ )
  {
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(Phi) );
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(Phi) );
    xmit_buf_idx++;
    Phi += delta_phi;
    CLAMP_2PI( Phi );
  }

  // Delta Phi for 1900HZ per DUC sample
  delta_phi = M_2PI * VIS_1900HZ_REF / DUC_SAMPLE_RATE;

  // Create 300ms of 1900Hz Leader tone
  for( uint32_t idx = 0; idx < ENC_VIS_LEADER_LENGTH; idx++ )
  {
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(Phi) );
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(Phi) );
    xmit_buf_idx++;
    Phi += delta_phi;
    CLAMP_2PI( Phi );
  }

  // *** Fill the Tx buffer with the VIS Mode Bits ** //

  // Delta Phi for 1200HZ per DUC sample
  delta_phi = M_2PI * VIS_1200HZ_REF / DUC_SAMPLE_RATE;

  // Create a 30ms 1200HZ Start bit
  for( uint32_t idx = 0; idx < VIS_MODE_BIT_LENGTH; idx++ )
  {
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(Phi) );
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(Phi) );
    xmit_buf_idx++;
    Phi += delta_phi;
    CLAMP_2PI( Phi );
  }

  // *** Transmit 7 VIS Mode data bits + one Even parity bit *** //

  // The parity bit is included in the vis_mode_num SSTV Mode parameter
  uint8_t mode = sstv_mode_params[sstv_status.sstv_mode_index].vis_mode_num;
  for( uint8_t bit = 0; bit < NUM_MODE_BITS; bit++ )
  {
    // Delta Phi per DUC sample for 1100HZ or 1300Hz
    if( (mode >> bit) & 0x01 )
      delta_phi = M_2PI * VIS_BIT1_FREQ / DUC_SAMPLE_RATE;
    else
      delta_phi = M_2PI * VIS_BIT0_FREQ / DUC_SAMPLE_RATE;

    for( uint32_t idx = 0; idx < VIS_MODE_BIT_LENGTH; idx++ )
    {
      xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(Phi) );
      xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(Phi) );
      xmit_buf_idx++;
      Phi += delta_phi;
      CLAMP_2PI( Phi );
    }
  }

  // Delta Phi for 1200HZ per DUC sample
  delta_phi = M_2PI * VIS_1200HZ_REF / DUC_SAMPLE_RATE;

  // Create a 30ms 1200HZ Stop Bit
  uint32_t length = VIS_MODE_BIT_LENGTH;
  for( uint32_t idx = 0; idx < length; idx++ )
  {
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(Phi) );
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(Phi) );
    xmit_buf_idx++;
    Phi += delta_phi;
    CLAMP_2PI( Phi );
  }

  // *** Write buffer to DUC, abort on error *** //
  xmit_buffer.xmit_buf_len = xmit_buf_len;
  sem_wait( &duc_send_semaphore );

} // Encode_VIS()

//------------------------------------------------------------------------

// Length of FSKID bit in DUC samples (192kS/s / 45.45Bd)
#define TX_FSKID_BIT_LENGTH    4224

/* Encode_FSKID()
 *
 * Prepares the FSK ID signal (Callsign in FSK train)
 */
  void
Encode_FSKID( void )
{
  // Delta Phi for a given tone frequency
  double delta_phi = 0.0;

  // Complete FSKID string, 0x20, 0x2A, callsign, 0x01
  char fskid[CALLSIGN_SIZE + 3];


  // Insert FSKID preamble (0x20 0x2A)
  fskid[0] = 0x20;
  fskid[1] = 0x2A;

  // Length of Callsign string
  uint8_t len = (uint8_t)strlen( sstv_rc.callsign );

  // Re-allocate Transmit buffers += Call length + Pre- + Post-amble + Falling edge
  xmit_buf_len += (len + 3) * FSKID_BYTE_LENGTH * TX_FSKID_BIT_LENGTH + RAISED_COS_LENGTH;
  Realloc_Xmit_Buffers( xmit_buf_len );

  // Insert callsign subtracting 0x20 because FSKID is 6-bit characters
  for( uint8_t idx = 0; idx < len; idx++ )
    fskid[idx + 2] = sstv_rc.callsign[idx] - 0x20;

  // Terminate FSKID string with postamble (0x01)
  fskid[len + 2] = 0x2A;

  // Total characters to transmit = callsign length + pre- and post-amble
  len += 3;

  // Fill buffer with FSKID tone samples for each FSKID byte
  xmit_buf_idx = 0;
  for( uint8_t cnt = 0; cnt < len; cnt++ )
  {
    // Fill buffer with FSKID tone samples for each FSKID byte's bit
    for( uint8_t bit = 0; bit < FSKID_BYTE_LENGTH; bit++ )
    {
      // Delta Phi according to 0 or 1 bit
      if( (fskid[cnt] >> bit) & 0x01 )
        delta_phi = M_2PI * FSKID_1900_HZ / DUC_SAMPLE_RATE;
      else
        delta_phi = M_2PI * FSKID_2100_HZ / DUC_SAMPLE_RATE;

      // Fill buffer with samples over the length of a bit
      for( uint16_t smp = 0; smp < TX_FSKID_BIT_LENGTH; smp++ )
      {
        xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(Phi) );
        xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(Phi) );
        xmit_buf_idx++;
        Phi += delta_phi;
        CLAMP_2PI( Phi );
      } // for( uint16_t smp = 0; smp < FSKID_BIT_SIZE; smp++ )
    } // for( uint8_t bit = 0; bit < FSKID_BYTE_LENGTH; bit++ )
  } // for( uint8_t cnt = 0; cnt < len; cnt++ )

  // Create a 5ms raised cosine falling edge
  double rcos_dphi = M_PI_2 / RAISED_COS_LENGTH; // Delta phi of raised cosine
  for( uint16_t idx = 0; idx < RAISED_COS_LENGTH; idx++ )
  {
    double rcos = DUC_SAMPLE_MAX * sin( M_PI_2 - rcos_dphi * (double)idx );
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)( rcos * sin(Phi) );
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)( rcos * cos(Phi) );
    xmit_buf_idx++;
    Phi += delta_phi;
    if( Phi >= M_2PI ) Phi -= M_2PI;
  }

  // *** Write buffer to DUC, abort on error *** //
  xmit_buffer.xmit_buf_len = xmit_buf_len;
  sem_wait( &duc_send_semaphore );

  // if( !xmit_buffer.status ) FIXME handle errors

} // Encode_FSKID()

//------------------------------------------------------------------------

  BOOLEAN
Encode_Image( void )
{

  return( True );
}

//------------------------------------------------------------------------

