/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/ifft.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../Hermes2/display.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/sound.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

// IFFT parameter definitions
#define SSTV_IFFT_UPPER_FREQ       2666 // Frequency range of IFFT
#define SSTV_SPECTRUM_UPPER_FREQ   2400 // Frequency at upper end of FFT display
#define SSTV_SPECTRUM_LOWER_FREQ   1067 // Frequency at lower end of FFT display
#define SSTV_SPECTRUM_START         410 // The IFFT bin with lowest spectrum frequency

//----------------------------------------------------------------------

// Points to plot
static GdkPoint *points = NULL;

//----------------------------------------------------------------------

/* Sstv_Spectrum_Size_Allocate()
 *
 * Handles the size_allocate callback on spectrum drawingarea
 */
  void
Sstv_Spectrum_Size_Allocate( uint16_t width, uint16_t height )
{
  // Destroy existing pixbuff
  if( sstv_wfall.pixbuf != NULL )
  {
    g_object_unref( G_OBJECT(sstv_wfall.pixbuf) );
    sstv_wfall.pixbuf = NULL;
  }

  // Create waterfall pixbuf
  sstv_wfall.pixbuf = gdk_pixbuf_new(
      GDK_COLORSPACE_RGB, FALSE, 8, width, height );
  if( sstv_wfall.pixbuf == NULL ) return;

  sstv_wfall.pixels = gdk_pixbuf_get_pixels( sstv_wfall.pixbuf );
  sstv_wfall.width  = (uint16_t)gdk_pixbuf_get_width ( sstv_wfall.pixbuf );
  sstv_wfall.height = (uint16_t)gdk_pixbuf_get_height( sstv_wfall.pixbuf );
  sstv_wfall.rowstride  = (uint16_t)gdk_pixbuf_get_rowstride( sstv_wfall.pixbuf );
  sstv_wfall.n_channels = (uint16_t)gdk_pixbuf_get_n_channels( sstv_wfall.pixbuf );
  gdk_pixbuf_fill( sstv_wfall.pixbuf, 0 );

  // Calculate ifft stride for the required frequency range of IFFT
  sstv_ifft_data.ifft_stride = SND_DSP_RATE / SSTV_IFFT_UPPER_FREQ / 2;

  /* Initialize Ifft and config. ifft size is twice
   * the waterfall size because we only need to
   * display the upper half, from 1200 to 2400 Hz */
  sstv_ifft_data.ifft_width = (uint16_t)sstv_wfall.width * 2;
  Initialize_iFFT( &sstv_ifft_data );

} // Sstv_Spectrum_Size_Allocate()

//------------------------------------------------------------------------

/* Sstv_Display_Waterfall()
 *
 * Displays audio spectrum as "waterfall"
 */
  void
Sstv_Display_Waterfall( void )
{
  uint16_t
    idh, idv,  // Index to horiz. and vert. position in warterfall
    idf, idb,  // Indices to ifft arrays
    temp;

  // Constants needed to draw white lines in waterfall
  static uint16_t
    white_line    = 0,
    black_line    = 0,
    sync_line     = 0,
    vis_1100_line = 0,
    vis_1300_line = 0,
    tty_1900_line = 0,
    tty_2100_line = 0;

  // Pointer to current pixel
  static guchar *pix;

  static BOOLEAN init = True;

  // Constants needed to draw lines in waterfall
  if( init )
  {
    init = False;

    uint16_t delta_f;
    delta_f = SSTV_WHITE_FREQ - SSTV_SPECTRUM_LOWER_FREQ;
    white_line =
      ( delta_f * (uint16_t)sstv_wfall.width ) /
      ( SSTV_SPECTRUM_UPPER_FREQ - SSTV_SPECTRUM_LOWER_FREQ );

    delta_f = SSTV_BLACK_FREQ - SSTV_SPECTRUM_LOWER_FREQ;
    black_line =
      ( delta_f * (uint16_t)sstv_wfall.width ) /
      ( SSTV_SPECTRUM_UPPER_FREQ - SSTV_SPECTRUM_LOWER_FREQ );

    delta_f = SSTV_SYNC_FREQ - SSTV_SPECTRUM_LOWER_FREQ;
    sync_line =
      ( delta_f * (uint16_t)sstv_wfall.width ) /
      ( SSTV_SPECTRUM_UPPER_FREQ - SSTV_SPECTRUM_LOWER_FREQ );

    delta_f = VIS_BIT1_FREQ - SSTV_SPECTRUM_LOWER_FREQ;
    vis_1100_line =
      ( delta_f * (uint16_t)sstv_wfall.width ) /
      ( SSTV_SPECTRUM_UPPER_FREQ - SSTV_SPECTRUM_LOWER_FREQ );

    delta_f = VIS_BIT0_FREQ - SSTV_SPECTRUM_LOWER_FREQ;
    vis_1300_line =
      ( delta_f * (uint16_t)sstv_wfall.width ) /
      ( SSTV_SPECTRUM_UPPER_FREQ - SSTV_SPECTRUM_LOWER_FREQ );

    delta_f = FSKID_2100_HZ - SSTV_SPECTRUM_LOWER_FREQ;
    tty_2100_line =
      ( delta_f * (uint16_t)sstv_wfall.width ) /
      ( SSTV_SPECTRUM_UPPER_FREQ - SSTV_SPECTRUM_LOWER_FREQ );

    delta_f = FSKID_1900_HZ - SSTV_SPECTRUM_LOWER_FREQ;
    tty_1900_line =
      ( delta_f * (uint16_t)sstv_wfall.width ) /
      ( SSTV_SPECTRUM_UPPER_FREQ - SSTV_SPECTRUM_LOWER_FREQ );
  }

  // Draw vertical lines in waterfall at detector's frequencies
  pix  = sstv_wfall.pixels + sstv_wfall.rowstride + sstv_wfall.n_channels;
  pix += sstv_wfall.n_channels * white_line;
  pix[0] = 0xff; pix[1] = 0xff; pix[2] = 0xff;

  pix  = sstv_wfall.pixels + sstv_wfall.rowstride + sstv_wfall.n_channels;
  pix += sstv_wfall.n_channels * black_line;
  pix[0] = 0xff; pix[1] = 0xff; pix[2] = 0xff;

  pix  = sstv_wfall.pixels + sstv_wfall.rowstride + sstv_wfall.n_channels;
  pix += sstv_wfall.n_channels * sync_line;
  pix[0] = 0xff; pix[1] = 0xff; pix[2] = 0;

  pix  = sstv_wfall.pixels + sstv_wfall.rowstride + sstv_wfall.n_channels;
  pix += sstv_wfall.n_channels * vis_1100_line;
  pix[0] = 0xff; pix[1] = 0; pix[2] = 0xff;

  pix  = sstv_wfall.pixels + sstv_wfall.rowstride + sstv_wfall.n_channels;
  pix += sstv_wfall.n_channels * vis_1300_line;
  pix[0] = 0xff; pix[1] = 0; pix[2] = 0xff;

  pix  = sstv_wfall.pixels + sstv_wfall.rowstride + sstv_wfall.n_channels;
  pix += sstv_wfall.n_channels * tty_1900_line;
  pix[0] = 0xff; pix[1] = 0; pix[2] = 0xff;

  pix  = sstv_wfall.pixels + sstv_wfall.rowstride + sstv_wfall.n_channels;
  pix += sstv_wfall.n_channels * tty_2100_line;
  pix[0] = 0xff; pix[1] = 0; pix[2] = 0xff;

  // Copy each line of waterfall to next one
  temp = (uint16_t)sstv_wfall.height - 2;
  for( idv = temp; idv > 0; idv-- )
  {
    pix = sstv_wfall.pixels + sstv_wfall.rowstride * idv + sstv_wfall.n_channels;
    for( idh = 0; idh < sstv_wfall.width; idh++ )
    {
      pix[0] = pix[- sstv_wfall.rowstride];
      pix[1] = pix[1 - sstv_wfall.rowstride];
      pix[2] = pix[2 - sstv_wfall.rowstride];
      pix   += sstv_wfall.n_channels;
    }
  }

  // Go to top left of pixbuf
  pix = sstv_wfall.pixels + sstv_wfall.rowstride;

  // Do the DFT on input array
  idb = 0;
  iFFT_Real( &sstv_ifft_data );
  int lim = SSTV_SPECTRUM_START + sstv_ifft_data.data_len / 2;
  for( idf = SSTV_SPECTRUM_START; idf < lim; idf += 2 )
  {
    // Calculate power of signal at each freq "bin"
    uint16_t pixel_val = iFFT_Bin_Value( &sstv_ifft_data, idf, False );
    Sstv_Display_Signal( (uint8_t)pixel_val );

    // Calculate average bin values
    sstv_ifft_data.bin_ave[idb++] = pixel_val;

    // Color code signal strength
    Colorize( pix, (uint8_t)pixel_val );
    pix += sstv_wfall.n_channels;

  } // for( idf = SSTV_SPECTRUM_START; idf < lim; idf += 2 )

  // Reset function
  iFFT_Bin_Value( &sstv_ifft_data, 0, True );

  // At last draw waterfall
  Queue_Draw( sstv_gui.spectrum_drawingarea );

} // Sstv_Display_Waterfall()

//------------------------------------------------------------------------

/* Sstv_Display_Signal()
 *
 * Updates the signal scope display
 */
  void
Sstv_Display_Signal( uint8_t plot )
{
  static uint16_t
    height = 0,     // Height of scope window
    limit,          // Limit plotting to inside of scope margin
    points_idx = 0; // Index to points array

  // Initialize on first call
  if( points == NULL )
  {
    Mem_Alloc( (void **)&points, sizeof(GdkPoint) * (size_t)sstv_scope_width );
  }

  // Initialize on parameter change
  if( height != sstv_scope_height )
  {
    height = (uint16_t)sstv_scope_height;
    limit = height - SCOPE_CLEAR;
  }

  // Save values to be plotted
  plot = (uint8_t)( ((uint16_t)plot * limit ) / 255 );
  points[points_idx].y = limit - (gint)plot;
  if( points[points_idx].y < SCOPE_CLEAR )
    points[points_idx].y = SCOPE_CLEAR;
  points[points_idx].x = points_idx;

  // Recycle buffer idx when full and plot
  points_idx++;
  if( points_idx >= sstv_scope_width )
  {
    Flag[GUEST_ENABLE_SCOPE] = True;
    Queue_Draw( sstv_gui.scope_drawingarea );
    points_idx = 0;
  } // if( points_idx == wfall_pixbuf.width )

} // Sstv_Display_Signal( void )

//------------------------------------------------------------------------

/* Sstv_Draw_Signal()
 *
 * Draws the signal detector's output
 */
  void
Sstv_Draw_Signal( cairo_t *cr )
{
  uint16_t idx;

  // Draw scope backgrounds
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_rectangle(
      cr, 0.0, 0.0,
      (double)sstv_scope_width,
      (double)sstv_scope_height );
  cairo_fill( cr );

  // Plot signal graph
  cairo_set_source_rgb( cr, SCOPE_FOREGND );

  cairo_move_to( cr,
      (double)points[0].x,
      (double)points[0].y );
  for( idx = 1; idx < sstv_scope_width; idx++ )
    cairo_line_to( cr,
        (double)points[idx].x,
        (double)points[idx].y );

  // Stroke paths
  cairo_stroke( cr );
  Flag[GUEST_ENABLE_SCOPE] = False;

} // Sstv_Draw_Signal()

//------------------------------------------------------------------------

  void
Sstv_Free_Display( void )
{
  Mem_Free( (void **)&points );
  Free_iFFT( &sstv_ifft_data );

  // Free waterfall pixbuf
  if( sstv_wfall.pixbuf ) g_object_unref( sstv_wfall.pixbuf );
  sstv_wfall.pixbuf = NULL;
}

//------------------------------------------------------------------------

/* Sstv_Set_TxRx_Labels()
 *
 * Sets up the labels in the Tx/Rx frames
 */
  void
Sstv_Set_TxRx_Labels( void )
{
  if( sstv_status.transmit_active )
  {
    gtk_label_set_markup( GTK_LABEL(sstv_gui.xmit_status_label), XMIT_ON );
  }
  else
    gtk_label_set_text( GTK_LABEL(sstv_gui.xmit_status_label),   XMIT_OFF );

  if( sstv_status.receive_active )
    gtk_label_set_markup( GTK_LABEL(sstv_gui.rcve_status_label), RECV_ON );
  else
    gtk_label_set_text( GTK_LABEL(sstv_gui.rcve_status_label),   RECV_OFF );

  while( g_main_context_iteration(NULL, FALSE) );
} // Sstv_Set_TxRx_Labels()

//------------------------------------------------------------------------

/* Sstv_Display_Stock_Image()
 *
 * Loads and displays the requested Stock Tx Image
 */
  BOOLEAN
Sstv_Display_Stock_Image( const char *image_name )
{
  // Stock directory and stock image file path
  gchar stock_dir[FILE_NAME_SIZE - 8];

  // Setup file path to stock images
  Strlcpy( stock_dir, sstv_rc.sstv_dir, sizeof(stock_dir) );
  Strlcat( stock_dir, "stock_images/",  sizeof(stock_dir) );

  // Create a file path to the selected image in icon view
  Strlcpy( sstv_rc.tx_image_file, stock_dir,  sizeof(sstv_rc.tx_image_file) );
  Strlcat( sstv_rc.tx_image_file, image_name, sizeof(sstv_rc.tx_image_file) );

  // Remove existing image from the event box
  if( sstv_gui.sstv_tx_image != NULL )
  {
    gtk_container_remove(
        GTK_CONTAINER(sstv_gui.sstv_tx_image_eventbox), sstv_gui.sstv_tx_image );
    ////gtk_widget_destroy( sstv_gui.sstv_tx_image );
  }

  // Create a new cairo surface from the stock image file
  if( sstv_gui.cairo_surface != NULL )
    cairo_surface_destroy( sstv_gui.cairo_surface );
  sstv_gui.cairo_surface = cairo_image_surface_create_from_png( sstv_rc.tx_image_file );

  // Create a new image from the Cairo surface and add to event box
  sstv_gui.sstv_tx_image = gtk_image_new_from_surface( sstv_gui.cairo_surface );
  gtk_container_add(
      GTK_CONTAINER(sstv_gui.sstv_tx_image_eventbox), sstv_gui.sstv_tx_image );
  gtk_widget_show( sstv_gui.sstv_tx_image );

  return( True );
} // Sstv_Display_Stock_Image()

//------------------------------------------------------------------------

typedef struct _IMG_PIXBUF
{
  gint width;
  gint height;
  BOOLEAN reset;
} img_pixbuf_t;

/* _Resize_Image_Pixbuf()
 *
 * Sets the size request for the SSTV Image's pixbuf as idle callback
 */
  static gboolean
_Resize_Image_Pixbuf( gpointer data )
{
  static gint w = 0, h = 0;

  if( ((img_pixbuf_t *)data)->reset )
  {
    w = 0;
    h = 0;
  }

  if( (w != ((img_pixbuf_t *)data)->width) ||
      (h != ((img_pixbuf_t *)data)->height) )
  {
    sstv_status.image_pixbuf_ready = False;
    gtk_widget_set_size_request(
        sstv_gui.sstv_rx_image_drawingarea,
        ((img_pixbuf_t *)data)->width,
        ((img_pixbuf_t *)data)->height );
    w = ((img_pixbuf_t *)data)->width;
    h = ((img_pixbuf_t *)data)->height;
  }

  return( FALSE );
} // _Resize_Image_Pixbuf()

/* Resize_Image_Pixbuf()
 *
 * Sets the size request for the SSTV Image's pixbuf
 */
  void
Resize_Image_Pixbuf( gint width, gint height, BOOLEAN reset )
{
  static img_pixbuf_t data;
  data.width  = width;
  data.height = height;
  data.reset  = reset;
  g_idle_add( _Resize_Image_Pixbuf, (gpointer)&data );
} // Resize_Image_Pixbuf()

//------------------------------------------------------------------------

