/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "detect.h"
#include "common.h"
#include "decode.h"
#include "display.h"
#include "shared.h"
#include "tests.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/ifft.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/sound.h"
#include <gtk/gtk.h>
#include <math.h>
#include <semaphore.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

//------------------------------------------------------------------------

// Indices to freq discriminator buffer to detect VIS data
#define VIS_STOPBIT_IN      0
#define VIS_STOPBIT_OUT     1440
#define VIS_STARTBIT_IN     12960
#define VIS_STARTBIT_OUT    14400
#define VIS_LEADER2_IN      14400
#define VIS_LEADER2_OUT     28800
#define VIS_BREAK_IN        28800
#define VIS_BREAK_OUT       29280
#define VIS_LEADER1_IN      29280
#define VIS_LEADER1_OUT     43680

//------------------------------------------------------------------------

/* Ring buffer for storing frequency estimates
 * produced by the FM discriminator function */
int16_t fm_discrim_buffer[FM_DISCRIM_BUF_SIZE];

// Index to new frequency entry into ring buffer
int32_t fm_discrim_buf_idx;

static int32_t
  sync_det_start_idx,  // Buffer index of start of Line Sync pulse
  vis_mode_start_idx;  // Buffer index of start of VIS Mode pulses

static int32_t
  stopbit_in_idx,   // Index to buffer to read data for Stopbit summation
  stopbit_out_idx,  // Index to buffer to read data for Stopbit substruction
  startbit_in_idx,  // Index to buffer to read data for Startbit summation
  startbit_out_idx, // Index to buffer to read data for Startbit substraction
  leader1_in_idx,   // Index to buffer to read data for Leader 1 summation
  leader1_out_idx,  // Index to buffer to read data for Leader 1 substruction
  break_in_idx,     // Index to buffer to read data for Break summation
  break_out_idx,    // Index to buffer to read data for Break substraction
  leader2_in_idx,   // Index to buffer to read data for Leader 2 summation
  leader2_out_idx;  // Index to buffer to read data for Leader 2 substruction

//------------------------------------------------------------------------

/* FM_Discriminator()
 *
 * Estimates the frequency of the incoming SSTV signal by calculating
 * the instantaneous phase of the carrier over one Sound sample duration
 * (1/48kHz). To reduce effects of noise, the difference in phase between
 * samples separated by 5 sampling intervals is used:
 * phi = atan2(i, q); dphi = phi-last_phi; freq = dphi * sample rate / 2pi / 5
 */

  BOOLEAN
FM_Discriminator( void )
{
  // Wait for sample buffers full
  static BOOLEAN wait = True;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;

  // Digimode buffer index
  static int32_t digi_buf_idx = 0;

  int16_t signal_sample;

  // Initialize if needed
  if( sstv_status.freq_discrim_init )
  {
    fm_discrim_buf_idx = 0;
    sync_det_start_idx = 0;
    digi_buf_cnt       = 0;

    // Initialize indices to FM discriminator buffer for VIS detection
    stopbit_in_idx   = VIS_STOPBIT_IN;
    stopbit_out_idx  = FM_DISCRIM_BUF_SIZE - VIS_STOPBIT_OUT;
    startbit_in_idx  = FM_DISCRIM_BUF_SIZE - VIS_STARTBIT_IN;
    startbit_out_idx = FM_DISCRIM_BUF_SIZE - VIS_STARTBIT_OUT;
    leader2_in_idx   = FM_DISCRIM_BUF_SIZE - VIS_LEADER2_IN;
    leader2_out_idx  = FM_DISCRIM_BUF_SIZE - VIS_LEADER2_OUT;
    break_in_idx     = FM_DISCRIM_BUF_SIZE - VIS_BREAK_IN;
    break_out_idx    = FM_DISCRIM_BUF_SIZE - VIS_BREAK_OUT;
    leader1_in_idx   = FM_DISCRIM_BUF_SIZE - VIS_LEADER1_IN;
    leader1_out_idx  = FM_DISCRIM_BUF_SIZE - VIS_LEADER1_OUT;

    memset( fm_discrim_buffer, 0, FM_DISCRIM_BUF_SIZE * sizeof(int16_t) );

    sstv_status.freq_discrim_init = False;
    return( True );
  } // if( sstv_status.freq_discrim_init )

  // Abort function if SSTV is in general initialization
  if( sstv_status.receiver_init ) return( True );

  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active ||
      (sstv_status.sstv_action == SSTV_ACTION_STOP) )
    return( False );

  /* Wait for digimode buffer to fill with audio samples from demodulator.
   * When this function is slow to respond due to processor loading, then
   * the sem_post() in Hermes2/demodulate.c is executed more than once, so
   * the wait function must be repeated according to the semaphore's value
   */
  if( wait )
  {
    sem_wait( &digimode_semaphore );
    wait = False;
  }

  // Get a audio sample from the digimodes buffer
  signal_sample = digimode_buffer[digi_buf_cnt][digi_buf_idx];


  /* Writes samples to file, for testing only
  {
    static FILE *did = NULL, *dqd = NULL;
    if( did == NULL ) did = fopen( "did.s", "w" );
    if( dqd == NULL ) dqd = fopen( "dqd.s", "w" );
    fwrite( (const void *)&demod_id_cpy[digi_buf_idx], sizeof(double), 1, did );
    fwrite( (const void *)&demod_qd_cpy[digi_buf_idx], sizeof(double), 1, dqd );
  } */

  /* Reads samples from file, for testing only
  {
    static FILE *did = NULL, *dqd = NULL;
    if( did == NULL ) did = fopen( "did.s", "r" );
    if( dqd == NULL )
    {
      dqd = fopen( "dqd.s", "r" );

      // Seek depends on where in recorded file the VIS begins
      fseek( did, 3000000, SEEK_SET );
      fseek( dqd, 3000000, SEEK_SET );
    }

    size_t ok;
    ok = fread( (void *)&demod_id_cpy[digi_buf_idx], sizeof(double), 1, did );
    ok = fread( (void *)&demod_qd_cpy[digi_buf_idx], sizeof(double), 1, dqd );
    if( !ok ) exit(-1);

   // printf("%ld\n",ftell(did));
  } */


  // Save the sample to iFFT buffer
  if( iFFT_Data(signal_sample, &sstv_ifft_data) )
    Sstv_Display_Waterfall();

  /* Ring buffer for calculated phase angles. It seems that the effects of noise
   * induced jitter on delta phi (dphi) can be reduced if it is calculated from
   * values of phi separated by a few samples (here we use 5 stage ring buffer) */
  static double phi_buf[PHI_BUF_LENGTH];
  static uint8_t buf_idx = 0;
  double phi  = atan2( demod_id_cpy[digi_buf_idx], demod_qd_cpy[digi_buf_idx] );
  double dphi = phi - phi_buf[buf_idx];
  phi_buf[buf_idx] = phi;
  buf_idx++;
  if( buf_idx >= PHI_BUF_LENGTH ) buf_idx = 0;

  // Correct for discontinuity at +-180 deg phase angle
  CLAMP_PI( dphi );

  /* Calculate signal frequency corresponding to delta phi. The ring buffer
   * of phi must be accounted for by dividing with the ring buffer length */
  sstv_status.signal_freq =
    (int16_t)( dphi * SND_DSP_RATE / M_2PI / PHI_BUF_LENGTH );

  // Generate test signals for the decoders
  //if( !VIS_Leader(False, 00) )
    //if( !VIS_Mode(False, M2, 00) )
      //Martin_Pasokon_Test( False, M2, 00 );
  //Scottie_Test( False, S2, 800 );
  //SC2180_Test( False, SC2_180, 800 );
  //PD_Test( False, PD90, 800 );
  //Test_FSK_ID( False, 800 );

  /* Enter new frequency into circular buffer and advance index.
   * The frequency calculated above must be referenced to 1900 Hz
   * and corrected by the VIS frequency calibration value */
  fm_discrim_buffer[fm_discrim_buf_idx] =
    sstv_status.signal_freq + IQ_STREAM_REF_FREQ - sstv_status.freq_calibration;
  fm_discrim_buf_idx++;
  if( fm_discrim_buf_idx >= FM_DISCRIM_BUF_SIZE )
    fm_discrim_buf_idx = 0;

  // Wait for digimode buffer to be filled
  digi_buf_idx++;
  if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
  {
    // Increment current audio buffer output index
    digi_buf_cnt++;
    if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
      digi_buf_cnt = 0;

    /* If sync between ring buffer input and output indices is lost,
     * sem_wait is not enabled, to allow the input index to catch up */
    if( digi_buf_input == digi_buf_cnt )
      wait = True;

    digi_buf_idx = 0;
  } // if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )

  return( True );
} // FM_Discriminator()

//------------------------------------------------------------------------

// Length in Sound DSP data samples of VIS Mode Bits
#define MODE_BIT_LENGTH     1440  // (48kHz * 30mS)

// Duration of VIS leader tones in Sound DSP Samples
#define DEC_VIS_LEADER_LENGTH   14400 // (48kHz * 300mS)
#define DEC_VIS_BREAK_LENGTH    480   // (48kHz * 10mS)

/* Per unit minimum tone frequency error threshold
 * for the detection of VIS Calibration Leader */
#define MODEBIT_ERR_THRESHOLD       0.50
#define VIS_LEADER_ERR_THRESHOLD    0.08
#define VIS_BREAK_ERR_THRESHOLD     0.50

//------------------------------------------------------------------------

/* VIS_Leader_Detect()
 *
 * Detects the Leader sequence of the VIS of SSTV transmissions.
 * This is normally 300mS of 1900Hz first Leader tone, 10mS of
 * 1200Hz Break tone and 300mS of 1900Hz second Leader tone.
 */
  void
VIS_Leader_Detect( void )
{
  static int32_t
    stopbit_err_sum,      // Sum of freq errors of Mode Stopbit tone
    startbit_err_sum,     // Sum of freq errors of Mode Startbit tone
    leader_tone1_err_sum, // Sum of freq errors of first 300mS 1900Hz Leader tone
    leader_break_err_sum, // Sum of freq errors of 10mS 1200Hz break in Leader
    leader_tone2_err_sum; // Sum of freq errors of second 300mS 1900Hz Leader tone

  // VIS Calibration value (freq error of VIS Leader 1900Hz pulses)
  static int16_t vis_calibration;

  static double
    modebit_hysteresis,   // Hysteresis in the Stopbit/Startbit Detected threshold
    break_hysteresis,     // Hysteresis in the Break Detected threshold
    leader_hysteresis,    // Hysteresis in the Leader Detected threshold
    break_min_err;        // Minimum frequency error in break tone pulse window

  double
    stopbit_error,        // Per unit error of frequency relative to 1200Hz
    startbit_error,       // Per unit error of frequency relative to 1200Hz
    leader1_error,        // Per unit error of frequency relative to 1900Hz
    break_error,          // Per unit error of frequency relative to 1200Hz
    leader2_error;        // Per unit error of frequency relative to 1900Hz

  // Index to discriminator buffer where min freq error occurs
  static int32_t min_error_idx;

  // Local VIS leader tone detected
  static BOOLEAN leader_detected = False;

  // Initialize if needed
  if( sstv_status.vis_leader_det_init )
  {
    break_min_err        = 100;
    modebit_hysteresis   = 0;
    break_hysteresis     = 0;
    leader_hysteresis    = 0;
    stopbit_err_sum      = 0;
    startbit_err_sum     = 0;
    leader_tone1_err_sum = 0;
    leader_break_err_sum = 0;
    leader_tone2_err_sum = 0;
    vis_calibration      = 0;

    // Set the VIS Leader Indicator "LED" OFF
    if( sstv_rc.vis_leader_icon_yes )
    {
      Set_Icon( GTK_IMAGE(
            sstv_gui.sstv_vis_leader_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      sstv_rc.vis_leader_icon_yes = False;
    }

    sstv_status.vis_leader_det_init = False;
    sstv_status.vis_leader_detected = False;
    return;
  } // if( sstv_status.vis_leader_det_init )

  /* Try to detect the presence of the VIS Leader tones: 300mS 1900Hz
   * 10mS 1200Hz and 300mS 1900Hz as well as the Start and Stop bits
   * of the VIS Mode bits train. The frequency error for the VIS signal
   * is calculated as the difference of discriminator output and the
   * reference frequency for each Leader tone.
   */

  // Detect Mode Stop bit 1200Hz tone
  stopbit_err_sum += fm_discrim_buffer[stopbit_in_idx];
  stopbit_err_sum -= fm_discrim_buffer[stopbit_out_idx];

  // Increment and wrap around indices
  stopbit_in_idx++;
  if( stopbit_in_idx >= FM_DISCRIM_BUF_SIZE )
    stopbit_in_idx = 0;
  stopbit_out_idx++;
  if( stopbit_out_idx >= FM_DISCRIM_BUF_SIZE )
    stopbit_out_idx = 0;

  // Detect Mode Stop bit 1200Hz tone
  startbit_err_sum += fm_discrim_buffer[startbit_in_idx];
  startbit_err_sum -= fm_discrim_buffer[startbit_out_idx];

  // Increment and wrap around indices
  startbit_in_idx++;
  if( startbit_in_idx >= FM_DISCRIM_BUF_SIZE )
    startbit_in_idx = 0;
  startbit_out_idx++;
  if( startbit_out_idx >= FM_DISCRIM_BUF_SIZE )
    startbit_out_idx = 0;

  // Detect second 300mS 1900Hz Leader tone
  leader_tone2_err_sum += fm_discrim_buffer[leader2_in_idx];
  leader_tone2_err_sum -= fm_discrim_buffer[leader2_out_idx];

  // Increment and wrap around indices
  leader2_in_idx++;
  if( leader2_in_idx >= FM_DISCRIM_BUF_SIZE )
    leader2_in_idx = 0;
  leader2_out_idx++;
  if( leader2_out_idx >= FM_DISCRIM_BUF_SIZE )
    leader2_out_idx = 0;

  // Detect 10mS 1200Hz Break pulse
  leader_break_err_sum += (int32_t)fm_discrim_buffer[break_in_idx];
  leader_break_err_sum -= (int32_t)fm_discrim_buffer[break_out_idx];

  // Increment and wrap around indices
  break_in_idx++;
  if( break_in_idx >= FM_DISCRIM_BUF_SIZE )
    break_in_idx = 0;
  break_out_idx++;
  if( break_out_idx >= FM_DISCRIM_BUF_SIZE )
    break_out_idx = 0;

  // Detect first 300mS 1900Hz Leader tone
  leader_tone1_err_sum += fm_discrim_buffer[leader1_in_idx];
  leader_tone1_err_sum -= fm_discrim_buffer[leader1_out_idx];

  // Increment and wrap around indices
  leader1_in_idx++;
  if( leader1_in_idx >= FM_DISCRIM_BUF_SIZE )
    leader1_in_idx = 0;
  leader1_out_idx++;
  if( leader1_out_idx >= FM_DISCRIM_BUF_SIZE )
    leader1_out_idx = 0;

  /* Do not continue with the rest of the calculations if VIS
   * Leader detection is not needed. Previous calculations are
   * needed in order to maintain the frequency summations current */
  if( !sstv_status.detect_vis_leader || sstv_status.image_decoder_enable )
    return;

  // Average value of VIS tone pulses frequency errors over length of pulse
  stopbit_error  = (double)stopbit_err_sum      / MODE_BIT_LENGTH;
  startbit_error = (double)startbit_err_sum     / MODE_BIT_LENGTH;
  leader2_error  = (double)leader_tone2_err_sum / DEC_VIS_LEADER_LENGTH;
  break_error    = (double)leader_break_err_sum / DEC_VIS_BREAK_LENGTH;
  leader1_error  = (double)leader_tone1_err_sum / DEC_VIS_LEADER_LENGTH;

  // Stop bit tone normalized error
  stopbit_error   -= VIS_1200HZ_REF;
  stopbit_error    = fabs( stopbit_error / VIS_1200HZ_REF );

  // Start bit tone normalized error
  startbit_error -= VIS_1200HZ_REF;
  startbit_error  = fabs( startbit_error / VIS_1200HZ_REF );

  // Second Leader tone normalized error
  leader2_error  -= VIS_1900HZ_REF; // Diff from 1900Hz
  vis_calibration = (int16_t)leader2_error;  // To calibrate FM discriminator
  leader2_error   = fabs( leader2_error / VIS_1900HZ_REF ); // Normalized

  // Leader Break tone normalized error
  break_error -= VIS_1200HZ_REF;
  break_error  = fabs( break_error / VIS_1200HZ_REF );

  // First Leader tone normalized error
  leader1_error   -= VIS_1900HZ_REF; // Diff from 1900Hz
  vis_calibration += (int16_t)leader1_error;  // To calibrate FM discriminator
  vis_calibration /= 2;  // Average of first and second Leader pulse freq errors
  leader1_error    = fabs( leader1_error / VIS_1900HZ_REF ); // Normalized

  /* If frequency error of VIS leader tones and break tone
   * are less than the minimum threshold, then the VIS
   * Leader sequence is considered to have been detected */
  double stopbit_ref_err  = MODEBIT_ERR_THRESHOLD    * sstv_status.detector_sens;
  double startbit_ref_err = stopbit_ref_err;
  double leader_ref_err   = VIS_LEADER_ERR_THRESHOLD * sstv_status.detector_sens;
  double break_ref_err    = VIS_BREAK_ERR_THRESHOLD  * sstv_status.detector_sens;

  if( (stopbit_error  < (stopbit_ref_err  + modebit_hysteresis)) &&
      (startbit_error < (startbit_ref_err + modebit_hysteresis)) &&
      (leader1_error  < (leader_ref_err   + leader_hysteresis))  &&
      (break_error    < (break_ref_err    + break_hysteresis))   &&
      (leader2_error  < (leader_ref_err   + leader_hysteresis)) )
  {
    // Hysteresis of 50% in the "detected" threshold
    modebit_hysteresis = stopbit_ref_err / 2;
    break_hysteresis   = break_ref_err   / 2;
    leader_hysteresis  = leader_ref_err  / 2;

    /* As SSTV signal frequency measurements come in from the
     * Discriminator buffer, we look for a minimum error in the
     * 10ms 1200Hz Break pulse frequency. This gives us the position
     * of the VIS Leader sequence's end and the Next data position
     * in the Frequency Discriminator's buffer */
    if( break_min_err >= break_error ) // VIS Break tone error falling
    {
      // Record the point where the new break error is lower than previous
      break_min_err = break_error;
      min_error_idx = break_in_idx;

      // Signal that VIS Leader may have been detected
      leader_detected = True;

      // Save VIS Frequency Calibration value
      sstv_status.freq_calibration = vis_calibration;
    }
  } // if( (leader1_error < VIS_LEADER_MIN_ERROR) && ...
  else // Error of VIS leader tones and break tone outside limits
  {
    if( leader_detected )
    {
      /* Raise VIS Leader (Calibration) Tones detected, disable
       * VIS Leader detection and point to the the beginning of
       * the VIS Mode Number sequence in the Discriminator buffer */
      sstv_status.vis_leader_detected = True;
      sstv_status.detect_vis_leader   = False;

      // No need to re-detect the VIS Mode Start bit after Leader
      vis_mode_start_idx = min_error_idx + DEC_VIS_LEADER_LENGTH + MODE_BIT_LENGTH;
      if( vis_mode_start_idx >= FM_DISCRIM_BUF_SIZE )
        vis_mode_start_idx -= FM_DISCRIM_BUF_SIZE;

      // Initialize VIS Mode Number detector
      sstv_status.detect_vis_mode   = True;
      sstv_status.vis_mode_det_init = True;

      // Set the VIS Leader Indicator "LED" ON
      if( !sstv_rc.vis_leader_icon_yes )
      {
        Set_Icon( GTK_IMAGE(
              sstv_gui.sstv_vis_leader_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
        sstv_rc.vis_leader_icon_yes = True;
      }

      // Display the Frequency Calibration value (dF)
      char df[9];
      snprintf( df, 9, "%d Hz", sstv_status.freq_calibration );
      Set_Label_Text( GTK_LABEL( sstv_gui.df_label), df );

    } // if( leader_detected )

    // Reset for next search
    modebit_hysteresis = 0;
    break_hysteresis   = 0;
    leader_hysteresis  = 0;
    break_min_err      = 100.0;
    leader_detected    = False;
  } // if( (leader1_error < VIS_LEADER_MIN_ERROR) && ...

} // VIS_Leader_Detect()

//------------------------------------------------------------------------

// Control of stages in the decoding process
enum _STAGES
{
  MODE_BITS_DECODE,
  LINE_SYNC_BYPASS,
  LINE_SYNC_DETECT
};

//------------------------------------------------------------------------

/* VIS_Mode_Detect()
 *
 * Detects and reads the VIS bit stream that encodes the SSTV Mode.
 * The stream is a series of 10 30mS pulses, (1) 1200Hz start bit,
 * (7) 1100Hz=1 or 1300Hz=0 pulses encoding 7 bits of the Mode number,
 * (1) Parity bit (1100Hz=Odd, 1300Hz=Even) and (1) 1200Hz Stop bit
 */
  void
VIS_Mode_Detect( void )
{
  // Summation of frequencies over one Mode Bit length
  static int32_t vis_mode_bit_sum;

  static uint16_t
    sync_counter,     // Count of frequency data bypassed for line sync bypass
    sync_length,      // Length of Line Sync pulse in number of frequency data
    bit_counter,      // Count of Mode bits decoded
    bit_sum_counter;  // Count of frequencies summed for Mode bit detection

  static uint8_t
    mode     = 0,  // Index to search for the SSTV mode number
    mode_num = 0,  // SSTV Mode number
    parity   = 0;  // Parity bit counter

  // Control of switch() cases that determine active decoding stage
  static uint8_t stage = MODE_BITS_DECODE;

  // Index to data from freq buffer to add to summation
  static int32_t lead_idx;


  // Initialize if needed
  if( sstv_status.vis_mode_det_init )
  {
    vis_mode_bit_sum = 0;
    bit_sum_counter  = 0;
    bit_counter      = 0;
    mode_num         = 0;
    parity           = 0;
    stage            = MODE_BITS_DECODE;
    lead_idx         = vis_mode_start_idx;

    sstv_status.vis_mode_det_init = False;
  }

  // Select active Mode Bit decoding stage
  switch( stage )
  {
    case MODE_BITS_DECODE:
      // Decode the Mode bit tones including parity bit
      for( bit_counter = 0; bit_counter < NUM_MODE_BITS; bit_counter++ )
      {
        // Summate frequency data over the length of one VIS mode bit
        for( bit_sum_counter = 0; bit_sum_counter < MODE_BIT_LENGTH; bit_sum_counter++ )
        {
          // Summate the current SSTV FM signal frequency, corrected by VIS calibration
          vis_mode_bit_sum +=
            (int32_t)(fm_discrim_buffer[lead_idx] - sstv_status.freq_calibration);
          lead_idx++;
          if( lead_idx >= FM_DISCRIM_BUF_SIZE )
            lead_idx = 0;
        }

        // Normalize bit frequency sum and refer to 1200 Hz (mid 1100-1300Hz)
        vis_mode_bit_sum /= MODE_BIT_LENGTH;
        int32_t bit_freq_refer = vis_mode_bit_sum - VIS_1200HZ_REF;

        /* Enter mode bits into SSTV mode number, including parity bit.
         * Right shift previous bit. */
        mode_num >>= 1;
        if( bit_freq_refer < 0 )  // Less than 1200Hz, Bit = 1
        {
          mode_num |= 0x80;
          parity++;
        }

        // Ready for next bit
        vis_mode_bit_sum = 0;
        bit_sum_counter  = 0;
      } // for( bit_counter = 0; bit_counter < NUM_MODE_BITS; bit_counter++ )

      // Add parity check here, abort on failure and re-enable VIS search
      if( parity & 0x01 )
      {
        bit_sum_counter  = 0;
        vis_mode_bit_sum = 0;

        // Re-enable VIS Leader detector
        sstv_status.detect_vis_leader = True;
        break;
      }

      // Find the SSTV Mode index corresponding to VIS Mode Number
      for( mode = 0; mode < SSTV_NUM_MODES - 1; mode++ )
      {
        if( mode_num == sstv_mode_params[mode].vis_mode_num )
          break;
      }

      // If no match abort mode detector
      if( mode == SSTV_NUM_MODES - 1 )
      {
        // Re-enable VIS Leader detector
        sstv_status.detect_vis_leader = True;
        break;
      }
      else if( sstv_status.sstv_mode_auto )
      {
        // Set up mode index if in AUTO mode
        sstv_status.sstv_mode_index = mode;
        sstv_status.sstv_mode_num   = mode_num;
      }
      else if( mode != sstv_status.sstv_mode_index ) // Detected mode != manual selection
      {
        // Re-enable VIS Leader detector
        sstv_status.detect_vis_leader = True;
        break;
      }

      /* Bypass Line Sync pulse, needed to gather frequency data
       * from FM discriminator to initialize the Line Sync detector */
      sync_length  = (uint16_t)( sstv_mode_params[mode].line_sync * SND_DSP_RATE );
      sync_counter = 0;
      stage        = LINE_SYNC_BYPASS;

      __attribute__((fallthrough));
    case LINE_SYNC_BYPASS:
      if( sync_counter < sync_length )
      {
        sync_counter++;
        return;
      }

      /* This will set up the Line Sync detector to initialize with frequency
       * data that are just after the stop bit: normally this would be the
       * first Line Sync pulse of the image transmission. Otherwise the case
       * below will continue till a Sync pulse is detected.
       */
      if( !sstv_status.image_decoder_enable )
      {
        sync_det_start_idx =
          vis_mode_start_idx + (NUM_MODE_BITS + 1) * MODE_BIT_LENGTH +
          (int32_t)( sstv_mode_params[mode].line_sync * SND_DSP_RATE );
        if( sync_det_start_idx >= FM_DISCRIM_BUF_SIZE )
          sync_det_start_idx -= FM_DISCRIM_BUF_SIZE;
      }
      else break;

      /* Set up new SSTV Mode if AUTO mode and Decoder not running. This
       * will also intialize the Line Sync detector to look for first pulse
       */
      if( sstv_status.sstv_mode_auto )
        New_Sstv_Mode( sstv_status.sstv_mode_index, False );
      else break;

      stage = LINE_SYNC_DETECT;
      __attribute__((fallthrough));

    case LINE_SYNC_DETECT:
      // The Sync detector runs in Sstv_Decode() in decode.c
      if( !sstv_status.line_sync_detected ) return;

      /* SSTV Image's index to freq buffer is just after the VIS
       * Stop pulse which includes the starting Line Sync pulse. */
      sstv_status.sstv_image_start = sstv_status.line_sync_index;
      sstv_status.sstv_image_index = sstv_status.sstv_image_start;

      // Enable Line Sync detect and SSTV image decoder function
      sstv_status.image_decoder_enable = True;
      if( !sstv_rc.vis_mode_icon_yes )
      {
        Set_Icon( GTK_IMAGE(
              sstv_gui.sstv_vis_mode_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
        sstv_rc.vis_mode_icon_yes = True;
      }

      // Set the Rx Status Label
      Set_Label_Text( GTK_LABEL(
            sstv_gui.rx_status_label), "Decoding SSTV Image ...." );
  } // switch( stage )

  // End VIS Mode Number search
  sstv_status.detect_vis_mode = False;

} // VIS_Mode_Detect()

//------------------------------------------------------------------------

// Per unit frequency error threshold
#define SYNC_ERR_THRESHOLD    0.15

/* Line_Sync_Detect()
 *
 * Detects the 1200Hz line synchronization pulse
 */
  void
Line_Sync_Detect( void )
{
  // Signal that possible line sync detected
  static BOOLEAN sync_detected = False;

  static int32_t
    sync_length,    // Length of sync pulses in number of samples
    sync_err_sum,   // Summation of frequencies to calculate frequency error
    lead_index,     // Leading index to FM dicriminator buffer
    trail_index;    // Trailing index to FM dicriminator buffer

  static double
    hysteresis,     // A hysteresis factor to detect minimum frequency error
    min_sync_err;   // Minimum error in the sync pulse frequencies summation

  // Index to buffer where minimum error occurs
  static int32_t min_err_idx = 0;


  // Initialize if requested
  if( sstv_status.line_sync_det_init )
  {
    // Sync pulse length in Sound DSP samples
    sync_length = (int32_t)
      ( sstv_mode_params[sstv_status.sstv_mode_index].line_sync * SND_DSP_RATE );

    // Set the leading buffer index to initialation value
    lead_index = sync_det_start_idx;

    /* Put trailing buffer index sync_len
     * steps behind the lead buffer index */
    trail_index = lead_index - sync_length;
    if( trail_index < 0 )
      trail_index += FM_DISCRIM_BUF_SIZE;

    // Prime some variables
    hysteresis   = 1.0;
    min_sync_err = 10.0;
    sync_err_sum = 0;

    /* Preload the sync pulse detector frequency summation with
     * samples over a sync pulse length to prime for detection */
    int32_t idx = trail_index;
    for( uint16_t cnt = 0; cnt < sync_length; cnt++ )
    {
      sync_err_sum += (int32_t)fm_discrim_buffer[idx];
      idx++;
      if( idx >= FM_DISCRIM_BUF_SIZE ) idx = 0;
    }

    sstv_status.line_sync_det_init = False;
    sstv_status.line_sync_detected = False;
  } // if( sstv_status.line_sync_det_init )

  // Abort function if SSTV is in general initialization
  if( sstv_status.receiver_init ) return;

  /* The loop is to take up any samples in the buffer
   * that are from the lead index to the current index */
  while( lead_index != fm_discrim_buf_idx )
  {
    // Summate frequencies over the length of sync pulse.
    sync_err_sum += (int32_t)fm_discrim_buffer[lead_index];
    sync_err_sum -= (int32_t)fm_discrim_buffer[trail_index];

    // Increment and wrap-around indices
    lead_index++;
    if( lead_index  >= FM_DISCRIM_BUF_SIZE )  lead_index = 0;
    trail_index++;
    if( trail_index >= FM_DISCRIM_BUF_SIZE ) trail_index = 0;

    // Per unit Sync pulse frequency error
    double sync_freq_err;
    sync_freq_err = (double)sync_err_sum / (double)sync_length;
    sync_freq_err = fabs( sync_freq_err - SSTV_SYNC_FREQ ) / SSTV_SYNC_FREQ;

    /* More frequencies in sync pulse range,
     * look for minimum frequency error */
    sstv_status.line_sync_detected = False;

    // Line Sync detection threshold
    double threshold = SYNC_ERR_THRESHOLD * sstv_status.detector_sens * hysteresis;
    if( sync_freq_err < threshold )
    {
      /* Delay exit from search once entering the search,
       * to allow for noise effects. Hysteresis is 50% */
      hysteresis = 1.5;

      // Frequency error decreases, save min error and index to buffer
      if( min_sync_err >= sync_freq_err )
      {
        min_sync_err  = sync_freq_err;
        min_err_idx   = lead_index;
        sync_detected = True;  // Sync detected if frequency error decreases
      }
    } // if( sync_err < threshold )
    else // Sync error outside detection threshold
    {
      // Reset for next search for minimum
      hysteresis   = 1.0;
      min_sync_err = 10.0;

      // Signal detection of Line sync and its position in buffer
      if( sync_detected )
      {
        sstv_status.line_sync_detected = True;
        sstv_status.line_sync_index    = min_err_idx;
        sync_detected = False;
        break;
      }
    } // if( sync_err < threshold )
  } // while( lead_index != fm_discrim_buf_idx )

} // Line_Sync_Detect()

//------------------------------------------------------------------------

/* The maximum normalized difference between successive Line Sync
 * Pulse periods before they can be considered correctly detected */
#define MAX_SYNC_PERIOD_DIFF  .04

// Minimum number of consecutive valid periods summed
#define MIN_PERIODS_SUMMED      3

// Minimum and maximum length of a valid line sync period, in samples
#define MIN_PERIOD_LENGTH     9000
#define MAX_PERIOD_LENGTH    52000
#define MAX_SYNC_LENGTH       1200

//------------------------------------------------------------------------

/* Process_Line_Sync()
 *
 * Estimates the Line Sync Pulse period (the time between sync pulses) by
 * averaging the difference between the indices to the Frequencies Buffer
 * where Line_Sync_Detect() has detected successive 1200Hz line sync pulses.
 * These differences must be within the upper limit defined above (4.0%).
 * Once a reliable sync pulse period is calculated, it is then used to find
 * the Mode of an incoming SSTV transmission by comparing with specifications.
 */
  void
Process_Line_Sync( void )
{
  static int32_t
    first_sync_index,  // Index of first valid sync pulse detected
    last_sync_index,   // Last Line Sync position in the freq buffer
    last_sync_period,  // Last calculated sync pulse period
    sync_period_sum,   // Summation of successive valid sync periods
    periods_summed,    // Number of valid sync periods summed
    sync_period_ave,   // Average (sum) of periods calculated from sync diffs
    periods_averaged;  // Number of sync periods averaged

  // Last sync index detected is valid
  static BOOLEAN sync_index_valid = False;

  // Difference between successive sync periods
  static double period_diff;

  // Detected SSTV mode
  static uint8_t mode;

  int32_t new_sync_period;


  // Initialize if signaled
  if( sstv_status.line_sync_proc_init )
  {
    /* If the SSTV Mode is detected by this function, prime the
     * sync period summation with the Mode's default period */
    if( sstv_status.sstv_mode_detected )
    {
      sync_period_sum = (int32_t)( sstv_mode_params[mode].line_time * SND_DSP_RATE );
      periods_summed = 1;
    }
    else
    {
      sync_period_sum = 0;
      periods_summed  = 0;
    }

    sync_index_valid = False;
    last_sync_index  = sstv_status.line_sync_index;
    last_sync_period = 0;
    sync_period_ave  = 0;
    periods_averaged = 0;

    // Set the Line Sync Indicator "LED" OFF
    if( sstv_rc.sync_det_icon_yes )
    {
      Set_Icon( GTK_IMAGE(
            sstv_gui.sstv_sync_detect_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      sstv_rc.sync_det_icon_yes = False;
    }

    sstv_status.line_sync_proc_init = False;
    return;
  }  // if( sstv_status.line_sync_proc_init )

  // Calculate new sync pulse period (in number of buffer elements)
  new_sync_period = sstv_status.line_sync_index - last_sync_index;
  if( new_sync_period < 0 ) new_sync_period += FM_DISCRIM_BUF_SIZE;

  /* Save the new line sync pulse index and return
   * if sync period is outside acceptable limits */
  if( (new_sync_period < MIN_PERIOD_LENGTH) ||
      (new_sync_period > MAX_PERIOD_LENGTH) )
  {
    last_sync_index = sstv_status.line_sync_index;
    return;
  }

  /* Calculate difference between two successive sync pulse periods.
   * If enough periods were summed, use the averaged period as reference */
  if( periods_summed > MIN_PERIODS_SUMMED )
  {
    period_diff = (double)new_sync_period - sstv_status.line_sync_period;
    period_diff = fabs( period_diff / sstv_status.line_sync_period );
  }
  else
  {
    period_diff = (double)( new_sync_period - last_sync_period );
    period_diff = fabs( period_diff / (double)new_sync_period );
  }
  last_sync_period = new_sync_period;

  // If the diff is within limits, add new period to period summation
  if( period_diff < MAX_SYNC_PERIOD_DIFF * sstv_status.detector_sens )
  {
    sync_period_sum += new_sync_period;
    periods_summed++;
    sync_index_valid = True;

    // Set the Line Sync indicator "LED" ON
    if( !sstv_rc.sync_det_icon_yes )
    {
      Set_Icon( GTK_IMAGE(
            sstv_gui.sstv_sync_detect_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
      sstv_rc.sync_det_icon_yes = True;
    }
  } // if( period_diff < MAX_SYNC_PERIOD_DIFF * sstv_status.detector_sens )
  else
  {
    /* Start new attempt to average successive periods
     * if not enough successive periods already summed */
    if( periods_summed < MIN_PERIODS_SUMMED )
    {
      sync_period_sum  = 0;
      periods_summed   = 0;
    }
    sync_index_valid = False;

    // Set the Line Sync indicator "LED" OFF
    if( sstv_rc.sync_det_icon_yes )
    {
      Set_Icon( GTK_IMAGE(
            sstv_gui.sstv_sync_detect_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
      sstv_rc.sync_det_icon_yes = False;
    }
  } // if( period_diff < MAX_SYNC_PERIOD_DIFF )

  // Save the new line sync pulse index
  last_sync_index = sstv_status.line_sync_index;

  /* Consecutive valid sync pulses detected. Signal
   * image decoders that sync period is updated */
  if( sync_index_valid &&
      first_sync_index &&
      (periods_summed > MIN_PERIODS_SUMMED) )
  {
    // Difference between first valid and current sync pulse positions
    int32_t diff = sstv_status.line_sync_index - first_sync_index;
    if( diff < 0 ) diff += FM_DISCRIM_BUF_SIZE;

    /* Ratio of this difference to current line sync period. This gives us
     * the number of sync pulses (lines) from first to current sync pulse */
    uint16_t ratio = (uint16_t)( (double)diff / sstv_status.line_sync_period + 0.5 );

    // Line sync period is ratio of diff to number of sync pulses
    if( ratio )
    {
      sync_period_ave  += diff;
      periods_averaged += ratio;

      /* To speed up convergence to the correct sync period, the difference
       * from averaged sync period to current sync period is added to the
       * averaged period. The diff is multiplied by the periods averaged */
      double dperiod   = (double)diff / (double)ratio - sstv_status.line_sync_period;
      sync_period_ave += (int32_t)( dperiod * periods_averaged );
      sstv_status.line_sync_period = (double)sync_period_ave / (double)periods_averaged;
      sstv_status.new_period_detected = True;
    }
  } // if( sync_index_valid && first_sync_index && ...

  // Consecutive valid sync pulses detected.
  if( periods_summed == MIN_PERIODS_SUMMED )
  {
    /* If the SSTV Mode detected previously, set the SSTV Image
     * Start index and Current index and enable Image Decoder */
    if( sstv_status.sstv_mode_detected )
    {
      sstv_status.sstv_image_start =
        last_sync_index - sync_period_sum - sstv_status.scottie_offset;
      if( sstv_status.sstv_image_start < 0 )
        sstv_status.sstv_image_start  += FM_DISCRIM_BUF_SIZE;
      sstv_status.sstv_image_index     = sstv_status.sstv_image_start;
      sstv_status.image_decoder_enable = True;
      sstv_status.sstv_mode_detected   = False;

      // Set the Rx Status Label
      Set_Label_Text( GTK_LABEL(
            sstv_gui.rx_status_label), "Decoding SSTV Image ...." );
    }

    first_sync_index = last_sync_index - sync_period_sum;
    if( first_sync_index < 0 ) first_sync_index += FM_DISCRIM_BUF_SIZE;

    // Set the Valid Syncs Detected Indicator "LED" ON
    if( !sstv_rc.sync_match_icon_yes )
    {
      Set_Icon( GTK_IMAGE(
            sstv_gui.sstv_sync_match_image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
      sstv_rc.sync_match_icon_yes = True;
    }
  } // if( periods_summed == MIN_PERIODS_SUMMED )

  /*** When the VIS Mode decoder succeeds, the following code    ***
   *** does not execute because sstv_status.image_decoder_enable ***
   *** flag is set TRUE by the VIS Mode decoder up in the file   ***/

  /* Look for a near match in the SSTV Mode specs for the average
   * sync period and identify the Mode if enough valid periods were
   * summed. Only if the SSTV Mode was not detected by the VIS Mode
   * detector and the Image Decoder is not running. */
  if( !sstv_status.image_decoder_enable &&
      (periods_summed == MIN_PERIODS_SUMMED) )
  {
    double line_time = (double)sync_period_sum / (double)periods_summed;
    line_time /= SND_DSP_RATE;

    // Minimum of line time diffs
    double min_diff = 1.0;

    // Mode number of nearest line time match
    uint8_t nearest = SSTV_NUM_MODES - 1;
    for( mode = 0; mode < SSTV_NUM_MODES - 1; mode++ )
    {
      // Image Line time diff, in seconds
      double line_time_diff = line_time - sstv_mode_params[mode].line_time;

      // Image Line time diff as a fraction
      line_time_diff = fabs( line_time_diff / sstv_mode_params[mode].line_time );

      // Find the best match to a mode's line time
      if( min_diff > line_time_diff )
      {
        min_diff = line_time_diff;
        nearest  = mode;
      }
    } // for( mode = 0; mode < SSTV_NUM_MODES - 1; mode++ )
    mode = nearest;

    /* If no match above or mode differes from the manually
     * selected, then continue listening for a matching mode */
    if( (mode == SSTV_NUM_MODES - 1) ||
        (!sstv_status.sstv_mode_auto &&
        (mode != sstv_status.sstv_mode_index)) )
    {
      // Start new attempt to average successive periods
      sync_period_sum = 0;
      periods_summed  = 0;
      sstv_status.sstv_mode_detected = False;

      // Set the Line Sync Indicator "LED" OFF
      if( sstv_rc.sync_match_icon_yes )
      {
        Set_Icon( GTK_IMAGE(
              sstv_gui.sstv_sync_match_image), "gtk-no", GTK_ICON_SIZE_BUTTON );
        sstv_rc.sync_match_icon_yes = False;
      }
      return;
    }
    else if( !sstv_status.sstv_mode_detected ) // New SSTV Mode detected
    {
      /* This for the Sync pulse detector, new start of search for a
       * sync pulse. It is the last sync pulse index minus the distance
       * to the first valid sync pulse index and minus a margin of
       * 1200 samples to go behind the longest possible sync pulse */
      sync_det_start_idx  = last_sync_index - sync_period_sum;
      sync_det_start_idx -= (int32_t)sstv_status.line_sync_period + MAX_SYNC_LENGTH;
      if( sync_det_start_idx < 0 )
        sync_det_start_idx += FM_DISCRIM_BUF_SIZE;

      // Set up new SSTV mode if Mode selection is in Auto
      if( sstv_status.sstv_mode_auto )
      {
        // No need to search for VIS Mode train
        sstv_status.detect_vis_leader = False;
        New_Sstv_Mode( mode, False );
      }
      else  // Re-initialize Sync Detector
      {
        sstv_status.line_sync_det_init  = True;
        sstv_status.line_sync_proc_init = True;
      }

      // SSTV Mode detected by Sync Processor
      sstv_status.sstv_mode_detected = True;

      // Set the Rx Status Label
      gchar txt[32];
      snprintf( txt, sizeof(txt), "Line Sync Matched: %s",
          sstv_mode_params[sstv_status.sstv_mode_index].mode_name );
      Set_Label_Text( GTK_LABEL(sstv_gui.rx_status_label), txt );

    } // if( (mode == SSTV_NUM_MODES - 1) || ...
  } // if( !sstv_status.sstv_status.image_decoder_enable && ... )

} // Process_Line_Sync()

//------------------------------------------------------------------------

