/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "decode.h"
#include "callback_func.h"
#include "common.h"
#include "detect.h"
#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../Hermes2/sound.h"
#include <gtk/gtk.h>
#include <math.h>

// Range of SSTV Image signal frequencies (2300Hz - 1500Hz)
#define SSTV_FREQ_RANGE  800

//------------------------------------------------------------------------

// Stages the image decoders must go through to produce an image
enum _IMAGE_DECODE_STAGES
{
  STAGE_1 = 1,
  STAGE_2,
  STAGE_3,
  STAGE_4,
  STAGE_5,
  STAGE_6,
  STAGE_7,
  STAGE_8
};

enum _COLOR_PIXEL
{
  RED_PIX = 0,
  GRN_PIX,
  BLU_PIX
};

//------------------------------------------------------------------------

/* Pixel_Value()
 *
 * Calculates the value of a color component of a pixel from
 * the frequency of the SSTV signal (1500Hz = 0, 2300Hz = 255)
 */
  static uint32_t
Pixel_Value( int16_t frequency )
{
  int32_t value = frequency - SSTV_BLACK_FREQ;
  value = ( value * 255 ) / SSTV_FREQ_RANGE;
  if( value < 0 )
    value = 0;
  else if( value > 255 )
    value = 255;
  return( (uint32_t)value );
} // Pixel_Value()

//------------------------------------------------------------------------

static double
  pixel_length,    // Length of image pixel in discriminator samples
  samples_count;   // Counts number of freq samples received from discriminator

static uint32_t
  pix_per_line,   // Number of pixels per line (width of image)
  pixels_count;   // Count of pixels that received color data from a color scan

// Current image decoder stage
static uint8_t decoder_stage;

// Pixels of the SSTV image pixbuf
static guchar *image_pixels;

//------------------------------------------------------------------------

/* Decode_RGB()
 *
 * Decodes a color component from the frequency data in the FM
 * discriminator's buffer and enters it to the specified pixel
 */
  static BOOLEAN
Decode_RGB( uint8_t pix_num, uint8_t stage_num )
{
  // The color value of a pixel in the pix buffer
  static uint32_t pixel_value;

  // Integer count of pixel values summed
  static uint8_t int_count;

  // Summate calculated pixel values over the length of a pixel
  pixel_value += Pixel_Value( fm_discrim_buffer[sstv_status.sstv_image_index] );
  sstv_status.sstv_image_index++;
  if( sstv_status.sstv_image_index >= FM_DISCRIM_BUF_SIZE )
    sstv_status.sstv_image_index = 0;
  int_count++;

  // Count the number of freq samples used in a pixel
  samples_count++;
  if( samples_count >= pixel_length )
  {
    samples_count -= pixel_length;

    // Enter average pixel value into color component
    ( image_pixels + pixels_count * (uint8_t)sstv_rx_image.n_channels )[pix_num] =
      (guchar)( pixel_value / int_count );
    pixel_value = 0;
    int_count   = 0;

    /* Count the number of pixels processed in a line
     * and if all pixels were processed, signal the Mode
     * decoder to go to the next stage of decoding */
    pixels_count++;
    if( pixels_count >= pix_per_line )
    {
      pixels_count  = 0;
      decoder_stage = stage_num; // Next stage in the switch() sequence
      return( True );    // Line completed
    }
  }

  return( False );
} // Decode_RGB()

//------------------------------------------------------------------------

/* Bypass_Sync_Pulse()
 *
 * Bypasses Sync pulse and Separator pulses in the SSTV image stream
 */
  static void
Bypass_Sync_Pulse( double length, uint8_t stage_num )
{
  sstv_status.sstv_image_index++;
  if( sstv_status.sstv_image_index >= FM_DISCRIM_BUF_SIZE )
    sstv_status.sstv_image_index = 0;
  samples_count++;
  if( samples_count >= length )
  {
    samples_count -= length;
    decoder_stage = stage_num;
  }
} // Bypass_Sync_Pulse()

//------------------------------------------------------------------------

static uint32_t
  lines_count,  // Count of image lines assembled (pixels prepared)
  num_of_lines; // Total number of image line to process

/* These counts need to be in float type as
 * slant correction requires high precision */
static double
  separator_len,   // Length of separator or porch
  sync_pulse_len,  // Length of Line Sync pulse
  sync_plus_separ; // Total length of sync and separator pulses (4)

// The SSTV Mode Index (M1, M2, ...)
static uint8_t mode;

static BOOLEAN
  save_image = False,    // Finalize image and save on Save Image activated
  new_image  = False,    // Finalize image on New Image activated
  finalize_image = True; // Finalize image when completed or requested

//------------------------------------------------------------------------

/* Init_Decoder()
 *
 * Common initializations for the SSTV Image decoders
 */
  static void
Init_Decoder( uint8_t num_separators, uint8_t num_scans )
{
  /* Calculate pixel length from the detected sync pulse
   * period by subtracting the sync pulse and separators.
   * Sync pulse and separator timings are in mSec */
  mode = sstv_status.sstv_mode_index;
  sync_plus_separ =
    sstv_mode_params[mode].line_sync +
    (double)num_separators * sstv_mode_params[mode].separator;
  sync_plus_separ *= SND_DSP_RATE;
  pixel_length  = sstv_status.line_sync_period - sync_plus_separ;
  pixel_length /=
    (double)num_scans * (double)sstv_mode_params[mode].pix_per_line;
  separator_len  = sstv_mode_params[mode].separator * SND_DSP_RATE;
  sync_pulse_len = sstv_mode_params[mode].line_sync * SND_DSP_RATE;
  pix_per_line   = sstv_mode_params[mode].pix_per_line;
  num_of_lines   = sstv_mode_params[mode].num_of_lines;
  lines_count    = 0;
  samples_count  = 0;
  pixels_count   = 0;
  finalize_image = True;
  save_image     = False;
  new_image      = False;
  image_pixels   = sstv_rx_image.pixels;

  // Clear SSTV Image pixbuf
  if( sstv_rx_image.pixbuf != NULL )
  {
    gdk_pixbuf_fill( sstv_rx_image.pixbuf, 0 );
    Queue_Draw( sstv_gui.sstv_rx_image_drawingarea );
  }

  sstv_status.image_decoder_init = False;
} // Init_Decoder()

//------------------------------------------------------------------------

/* New_Period()
 *
 * Recalculates pixel length on new detected Line Sync period
 */
  static void
New_Period( uint8_t num_scans )
{
  /* Calculate pixel length from the detected sync pulse
   * period by subtracting the sync pulse and separators */
  pixel_length  = sstv_status.line_sync_period - sync_plus_separ;
  pixel_length /= (double)num_scans * (double)pix_per_line;
  sstv_status.new_period_detected = False;
} // New_Period()

//------------------------------------------------------------------------

// Idle add function to initiate drawing of Rx image
  static void
Queue_Draw_Rx( void )
{
  // Draw the image with the new line and advance to next line
  if( sstv_status.image_decoder_enable &&
      sstv_status.image_pixbuf_ready )
    Queue_Draw( sstv_gui.sstv_rx_image_drawingarea );
} // Queue_Draw_Rx()

//------------------------------------------------------------------------

/* Finalize_Image()
 *
 * Sets up the decoder so that it re-decodes the image starting from
 * the earliest detected start of the image and averaged sync period
 */
  static BOOLEAN
Finalize_Image( uint8_t stage_num )
{
  /* Finalize image if Save Image or New Image was activated.
   * Cheat the code below by making num_of_lines = lines_count */
  if( sstv_status.save_image )
  {
    save_image = True;
    sstv_status.save_image = False;
    num_of_lines = lines_count;
  }
  else if( sstv_status.new_image )
  {
    new_image = True;
    sstv_status.new_image = False;
    num_of_lines = lines_count;
  }

  // SSTV Image complete or signaled by user
  if( lines_count >= num_of_lines )
  {
    lines_count = 0;
    if( finalize_image )
    {
      finalize_image = False;
      samples_count  = 0;
      pixels_count   = 0;
      decoder_stage  = stage_num;
      image_pixels   = sstv_rx_image.pixels;
      sstv_status.sstv_image_index = sstv_status.sstv_image_start;
    }
    else // Draw and save finalized pixbuf unless New Image was activated
    {
      Queue_Draw_Rx();
      if( (sstv_status.auto_save_enable && !new_image) || save_image )
      {
        Sstv_Save_Pixbuf(
            sstv_rx_image.pixbuf, sstv_mode_params[mode].mode_name, "png" );
      }
      else
      {
        // Set the Rx Status Label
        Set_Label_Text( GTK_LABEL(
              sstv_gui.rx_status_label), "SSTV Image Decoding Ended" );
      }

      /* Re-enable listening for new image if selected by checkbutton
       * or signaled by user, else just stop the decoder. Manual
       * enabling listening for new image is useful during SSTV QSO's */
      if( sstv_status.auto_listen_enable || new_image )
        Sstv_New_Image();
      else
        sstv_status.image_decoder_enable = False;

      save_image = False;
      new_image  = False;

      return( True );
    } // else of if( finalize_image )
  } //if( lines_count >= num_of_lines )

  return( False );
} // Finalize_Image()

//------------------------------------------------------------------------

/* Decode_Martin_Mode()
 *
 * Decodes SSTV images of the Martin M1 and M2 mode format
 */
  void
Decode_Martin_Mode( void )
{
  // Initialize when signaled
  if( sstv_status.image_decoder_init )
  {
    // 4 separators, 3 color scans
    Init_Decoder( 4, 3 );

    /* We start from stage 2 (sync porch) because the VIS Mode Detector bypasses
     * the Sync pulse if the Mode pulse train is detected. If the VIS Mode
     * Detector fails, same is done by the Sync Pulse processing function */
    decoder_stage = STAGE_2;
  } // if( sstv_status.image_decoder_init )

  // Recalculate pixel length on new period
  if( sstv_status.new_period_detected )
    New_Period( 3 );

  /* The Line Sync detector in the VIS Mode detector function looks
   * ahead into the FM discriminator's buffer while searching for
   * the Line Sync. So it is necessary to use up any frequency samples
   * from the VIS Line Sync to the current discriminator buffer position */
  do
  {
    // Stages to go through to decode a line of an image
    switch( decoder_stage )
    {
      case STAGE_1: // Bypass the Line Sync pulse period if sync not detected
        Bypass_Sync_Pulse( sync_pulse_len, STAGE_2 );
        break;

      case STAGE_2:  // Bypass the Sync Pulse porch period
        Bypass_Sync_Pulse( separator_len, STAGE_3 );
        break;

      case STAGE_3:  // Decode the first color scan (GREEN)
        Decode_RGB( GRN_PIX, STAGE_4 );
        break;

      case STAGE_4:  // Bypass the Green color scan separator
        Bypass_Sync_Pulse( separator_len, STAGE_5 );
        break;

      case STAGE_5:  // Decode the second color scan (BLUE)
        Decode_RGB( BLU_PIX, STAGE_6 );
        break;

      case STAGE_6:  // Bypass the Blue color scan separator
        Bypass_Sync_Pulse( separator_len, STAGE_7 );
        break;

      case STAGE_7:  // Decode the third color scan (RED)
        if( Decode_RGB(RED_PIX, STAGE_8) )
        {
          // Advance to the next line
          lines_count++;
          image_pixels =
            sstv_rx_image.pixels + lines_count * (uint32_t)sstv_rx_image.rowstride;
          Queue_Draw_Rx();
        }
        break;

      case STAGE_8:  // Bypass the Red color scan separator
        Bypass_Sync_Pulse( separator_len, STAGE_1 );
        break;

    } // switch( decoder_stage )

    // Finalize image on completion or user request
    if( Finalize_Image(STAGE_2) ) break;

  } // do
  while( sstv_status.sstv_image_index != fm_discrim_buf_idx );

  return;
} // Decode_Martin_Mode()

//------------------------------------------------------------------------

/* Decode_Scottie_Mode()
 *
 * Decodes SSTV images of the Martin M1 and M2 mode format
 */
  void
Decode_Scottie_Mode( void )
{
  // Initialize when signalled
  if( sstv_status.image_decoder_init )
  {
    // 3 separators, 3 color scans
    Init_Decoder( 3, 3 );

    /* We start from stage 1 (sync porch) because the VIS Mode Detector bypasses
     * the Sync pulse if the Mode pulse train is detected. If the VIS Mode
     * Detector fails, same is done by the Sync Pulse processing function */
    decoder_stage = STAGE_1;
  } //if( sstv_status.image_decoder_init )

  // Recalculate pixel length on new period
  if( sstv_status.new_period_detected )
    New_Period( 3 );

  /* The Line Sync detector in the VIS Mode detector function looks
   * ahead into the FM discriminator's buffer while searching for
   * the Line Sync. So it is necessary to use up any frequency samples
   * from the VIS Line Sync to the current discriminator buffer position */
  do
  {
    // Stages to go through to decode a line of an image
    switch( decoder_stage )
    {
      case STAGE_1:  // Bypass the Red color scan separator
        Bypass_Sync_Pulse( separator_len, STAGE_2 );
        break;

      case STAGE_2:  // Decode the first color scan (GREEN)
        Decode_RGB( GRN_PIX, STAGE_3 );
        break;

      case STAGE_3:  // Bypass the Green color scan separator
        Bypass_Sync_Pulse( separator_len, STAGE_4 );
        break;

      case STAGE_4:  // Decode the second color scan (BLUE)
        Decode_RGB( BLU_PIX, STAGE_5 );
        break;

      case STAGE_5: // Bypass the Line Sync pulse period if sync not detected
        Bypass_Sync_Pulse( sync_pulse_len, STAGE_6 );
        break;

      case STAGE_6:  // Bypass the Sync Pulse porch period
        Bypass_Sync_Pulse( separator_len, STAGE_7 );
        break;

      case STAGE_7:  // Decode the third color scan (RED)
        if( Decode_RGB(RED_PIX, STAGE_1) )
        {
          // Advance to next line
          lines_count++;
          image_pixels = sstv_rx_image.pixels + (uint8_t)lines_count * sstv_rx_image.rowstride;
          Queue_Draw_Rx();
        }
        break;

    } // switch( decoder_stage )

    // Finalize image on completion or user request
    if( Finalize_Image(STAGE_1) ) break;

  } // do
  while( sstv_status.sstv_image_index != fm_discrim_buf_idx );

  return;
} // Decode_Scottie_Mode()

//------------------------------------------------------------------------

/* Decode_SC2180_Mode()
 *
 * Decodes SSTV images of the Wrasse SC2-180 mode format
 */
  void
Decode_SC2180_Mode( void )
{
  // Initialize when signaled
  if( sstv_status.image_decoder_init )
  {
    // 1 separators, 3 color scans
    Init_Decoder( 1, 3 );

    /* We start from stage 2 (sync porch) because the VIS Mode Detector bypasses
     * the Sync pulse if the Mode pulse train is detected. If the VIS Mode
     * Detector fails, same is done by the Sync Pulse processing function */
    decoder_stage = STAGE_2;
  } // if( sstv_status.image_decoder_init )

  // Recalculate pixel length on new period
  if( sstv_status.new_period_detected )
    New_Period( 3 );

  /* The Line Sync detector in the VIS Mode detector function looks
   * ahead into the FM discriminator's buffer while searching for
   * the Line Sync. So it is necessary to use up any frequency samples
   * from the VIS Line Sync to the current discriminator buffer position */
  do
  {
    // Stages to go through to decode a line of an image
    switch( decoder_stage )
    {
      case STAGE_1: // Bypass the Line Sync pulse period if sync not detected
        Bypass_Sync_Pulse( sync_pulse_len, STAGE_2 );
        break;

      case STAGE_2:  // Bypass the Sync Pulse porch period
        Bypass_Sync_Pulse( separator_len, STAGE_3 );
        break;

      case STAGE_3:  // Decode the first color scan (RED)
        Decode_RGB( RED_PIX, STAGE_4 );
        break;

      case STAGE_4:  // Decode the second color scan (GREEN)
        Decode_RGB( GRN_PIX, STAGE_5 );
        break;

      case STAGE_5:  // Decode the third color scan (BLUE)
        if( Decode_RGB(BLU_PIX, STAGE_1) )
        {
          // Advance to the next line
          lines_count++;
          image_pixels = sstv_rx_image.pixels + lines_count * (uint32_t)sstv_rx_image.rowstride;
          Queue_Draw_Rx();
        }
        break;

    } // switch( decoder_stage )

    // Finalize image on completion or user request
    if( Finalize_Image(STAGE_2) ) break;

  } // do
  while( sstv_status.sstv_image_index != fm_discrim_buf_idx );

  return;
} // Decode_SC2180_Mode()

//------------------------------------------------------------------------

/* Decode_Pasokon_Mode()
 *
 * Decodes SSTV images of the Pasokon P3, P5 and P7 mode format
 */
  void
Decode_Pasokon_Mode( void )
{
  // Initialize when signaled
  if( sstv_status.image_decoder_init )
  {
    // 4 separators, 3 color scans
    Init_Decoder( 4, 3 );

    /* We start from stage 2 (sync porch) because the VIS Mode Detector bypasses
     * the Sync pulse if the Mode pulse train is detected. If the VIS Mode
     * Detector fails, same is done by the Sync Pulse processing function */
    decoder_stage = STAGE_2;
  } // if( sstv_status.image_decoder_init )

  // Recalculate pixel length on new period
  if( sstv_status.new_period_detected )
    New_Period( 3 );

  /* The Line Sync detector in the VIS Mode detector function looks
   * ahead into the FM discriminator's buffer while searching for
   * the Line Sync. So it is necessary to use up any frequency samples
   * from the VIS Line Sync to the current discriminator buffer position */
  do
  {
    // Stages to go through to decode a line of an image
    switch( decoder_stage )
    {
      case STAGE_1: // Bypass the Line Sync pulse period if sync not detected
        Bypass_Sync_Pulse( sync_pulse_len, STAGE_2 );
        break;

      case STAGE_2:  // Bypass the Sync Pulse porch period
        Bypass_Sync_Pulse( separator_len, STAGE_3 );
        break;

      case STAGE_3:  // Decode the first color scan (GREEN)
        Decode_RGB( RED_PIX, STAGE_4 );
        break;

      case STAGE_4:  // Bypass the Green color scan separator
        Bypass_Sync_Pulse( separator_len, STAGE_5 );
        break;

      case STAGE_5:  // Decode the second color scan (BLUE)
        Decode_RGB( GRN_PIX, STAGE_6 );
        break;

      case STAGE_6:  // Bypass the Blue color scan separator
        Bypass_Sync_Pulse( separator_len, STAGE_7 );
        break;

      case STAGE_7:  // Decode the third color scan (RED)
        if( Decode_RGB(BLU_PIX, STAGE_8) )
        {
          // Advance to the next line
          lines_count++;
          image_pixels = sstv_rx_image.pixels + lines_count * (uint32_t)sstv_rx_image.rowstride;
          Queue_Draw_Rx();
        }
        break;

      case STAGE_8:  // Bypass the Red color scan separator
        Bypass_Sync_Pulse( separator_len, STAGE_1 );
        break;

    } // switch( decoder_stage )

    // Finalize image on completion or user request
    if( Finalize_Image(STAGE_2) ) break;

  } // do
  while( sstv_status.sstv_image_index != fm_discrim_buf_idx );

  return;
} // Decode_Pasokon_Mode()

//------------------------------------------------------------------------

// Indices to YUV buffer (Y0, R-Y, B-Y, Y1)
enum _YUV_BUF
{
  Y_0 = 0,
  R_Y,
  B_Y,
  Y_1
};

//------------------------------------------------------------------------

/* Decode_YUV()
 *
 * Decodes a color component from the frequency data in the FM
 * discriminator's buffer and enters it to the specified buffer
 */
  static BOOLEAN
Decode_YUV( uint8_t *yuv_buf, uint8_t stage_num )
{
  // The color value of a pixel in the pix buffer
  static uint32_t pixel_value;

  // Integer count of pixel values summed
  static uint8_t int_count;

  // Summate calculated pixel values over the length of a pixel
  pixel_value += Pixel_Value( fm_discrim_buffer[sstv_status.sstv_image_index] );
  sstv_status.sstv_image_index++;
  if( sstv_status.sstv_image_index >= FM_DISCRIM_BUF_SIZE )
    sstv_status.sstv_image_index = 0;
  int_count++;

  // Count the number of freq samples used in a pixel
  samples_count++;
  if( samples_count >= pixel_length )
  {
    samples_count -= pixel_length;

    // Enter average pixel value into color component
    yuv_buf[ pixels_count ] = (uint8_t)( pixel_value / int_count );
    pixel_value = 0;
    int_count   = 0;

    /* Count the number of pixels processed in a line
     * and if all pixels were processed, signal the Mode
     * decoder to go to the next stage of decoding */
    pixels_count++;
    if( pixels_count >= pix_per_line )
    {
      pixels_count  = 0;
      decoder_stage = stage_num; // Next stage in the switch() sequence
      return( True );  // Line completed
    }
  }

  return( False );
} // Decode_YUV()

//------------------------------------------------------------------------

/* YUV_to_RGB()
 *
 * Calculates the RGB values for a pixel from the YUV buffer
 */
  static void
YUV_to_RGB( const uint8_t YUV_Buffer[4][640], guchar *pix )
{
  guchar
    *pix0 = pix, // First (even) line
    *pix1 = pix + sstv_rx_image.rowstride;  // Second (odd) line

  for( uint16_t idx = 0; idx < pix_per_line; idx++ )
  {
    double R, G, B, Y0, Y1, RY, BY;

    // Y value of pixels of first (even) line
    Y0 = 1.164308 * ( YUV_Buffer[Y_0][idx] - 16 );

    // Y value of pixels of second (odd) line
    Y1 = 1.164308 * ( YUV_Buffer[Y_1][idx] - 16 );

    // R-Y and B-Y values averaged over two lines
    RY = YUV_Buffer[R_Y][idx] - 128;
    BY = YUV_Buffer[B_Y][idx] - 128;

    // Calculate and clamp pixel color values for first line
    R = Y0 + 1.595925 * RY;
    G = Y0 - 0.391737 * BY - 0.812917 * RY;
    B = Y0 + 2.017101 * BY;
    dClamp( &R, 0.0, 255.0 );
    dClamp( &G, 0.0, 255.0 );
    dClamp( &B, 0.0, 255.0 );

    // Even line (first line) pixels
    pix0[RED_PIX] = (guchar)R;
    pix0[GRN_PIX] = (guchar)G;
    pix0[BLU_PIX] = (guchar)B;

    // Next pixel in first line
    pix0 += sstv_rx_image.n_channels;

    // Calculate and clamp pixel color values for second line
    R = Y1 + 1.595925 * RY;
    G = Y1 - 0.391737 * BY - 0.812917 * RY;
    B = Y1 + 2.017101 * BY;
    dClamp( &R, 0.0, 255.0 );
    dClamp( &G, 0.0, 255.0 );
    dClamp( &B, 0.0, 255.0 );

    // Odd line (second line) pixels
    pix1[RED_PIX] = (guchar)R;
    pix1[GRN_PIX] = (guchar)G;
    pix1[BLU_PIX] = (guchar)B;

    // Next pixel in second line
    pix1 += sstv_rx_image.n_channels;

  } // for( uint16_t idx = 0; idx < pix_per_line; idx++ )

} // YUV_to_RGB()

//------------------------------------------------------------------------

/* Decode_PD_Mode()
 *
 * Decodes SSTV images of the PD mode format
 */
  void
Decode_PD_Mode( void )
{
  /* Buffer for one scan line of YUV color
   * data: Y0, R-Y, B-Y, Y1 (2 image lines) */
  static uint8_t YUV_Buffer[4][640];

  // Initialize when signaled
  if( sstv_status.image_decoder_init )
  {
    // 1 separators, 4 color scans
    Init_Decoder( 1, 4 );

    /* We start from stage 2 (sync porch) because the VIS Mode Detector bypasses
     * the Sync pulse if the Mode pulse train is detected. If the VIS Mode
     * Detector fails, same is done by the Sync Pulse processing function */
    decoder_stage = STAGE_2;
  } // if( sstv_status.image_decoder_init )

  // Recalculate pixel length on new period
  if( sstv_status.new_period_detected )
    New_Period( 4 );

  /* The Line Sync detector in the VIS Mode detector function looks
   * ahead into the FM discriminator's buffer while searching for
   * the Line Sync. So it is necessary to use up any frequency samples
   * from the VIS Line Sync to the current discriminator buffer position */
  do
  {
    // Stages to go through to decode a line of an image
    switch( decoder_stage )
    {
      case STAGE_1: // Bypass the Line Sync pulse period if sync not detected
        Bypass_Sync_Pulse( sync_pulse_len, STAGE_2 );
        break;

      case STAGE_2:  // Bypass the Sync Pulse porch period
        Bypass_Sync_Pulse( separator_len, STAGE_3 );
        break;

      case STAGE_3:  // Decode the first color scan (Y0)
        Decode_YUV( YUV_Buffer[Y_0], STAGE_4 );
        break;

      case STAGE_4:  // Decode the second color scan (R-Y)
        Decode_YUV( YUV_Buffer[R_Y], STAGE_5 );
        break;

      case STAGE_5:  // Decode the third color scan (B-Y)
        Decode_YUV( YUV_Buffer[B_Y], STAGE_6 );
        break;

      case STAGE_6:  // Decode the fourth color scan (Y1)
                     // Decode two image lines from the YUV color buffer
        if( Decode_YUV(YUV_Buffer[Y_1], STAGE_1) )
        {
          // Converts one PD mode YUV line to two image RGB lines
          YUV_to_RGB( YUV_Buffer, image_pixels );

          // Advance to the next line
          lines_count += 2;
          image_pixels = sstv_rx_image.pixels +
            lines_count * (uint32_t)sstv_rx_image.rowstride;
          Queue_Draw_Rx();
        }
        break;

    } // switch( decoder_stage )

    // Finalize image on completion or user request
    if( Finalize_Image(STAGE_2) ) break;

  } // do
  while( sstv_status.sstv_image_index != fm_discrim_buf_idx );

  return;
} // Decode_PD_Mode()

//------------------------------------------------------------------------

// Length of an FSKID bit in samples: 48000S/s / 45.45Bd
#define RX_FSKID_BIT_LENGTH     1056

/* Length of FSKID header in bits (000001010101) */
#define FSKID_HEADER_LENGTH     8

// Detection threshold of FSKID header detector
#define FSKID_ERR_THRESHOLD     0.2

// Length of FSKID callsing string
#define FSKID_TEXT_LENGTH       15

// Mid frequency between 1900 and 2100 Hz used in FSKID
#define FSKID_2000_HZ           2000

//------------------------------------------------------------------------

/* Decode_FSKID()
 *
 * Decodes the FSK Identity stream at the end of SSTV image transmission.
 * The stream is at 45.45 Baud FSK with 1 = 1900Hz and 0 = 2100Hz. Data
 * is 6-bit bytes, LSB first, text follows a 202A header and ends with
 * a 01 tail. Adding 0x20 to data bytes makes them ASCII characters.
 */
  void
Decode_FSKID( void )
{
  // The header (202A) bit-reversed: 000001010101
  static uint16_t header = 0x055;

  static int32_t
    lead_idx,   // Incoming frequency samples index
    trail_idx,  // Outgoing frequency samples index
    fskid_idx;  // Buffer index where FSKID text begins

  static uint16_t sample_cnt = 0;  // Count of frequency samples
  static uint8_t  bit_cnt    = 0;  // Count of decoded character bits
  static uint8_t  char_cnt   = 0;  // Count of characters decoded

  // Summation of frequencies over length of header bits
  static int32_t header_sum[FSKID_HEADER_LENGTH];

  static double
    fskid_sum,      // Frequency error summation to detect FSKID bits
    hysteresis,     // A hysteresis added to the header detection reference
    error_sum_min;  // Minimum of header error summation

  // Current stage of the decoding process
  static uint8_t fskid_stage;

  // Header string 202A detected flag
  static BOOLEAN header_detect;

  // FSKID text string (callsign)
  static gchar fskid_char[FSKID_TEXT_LENGTH];


  // Initialize on first call
  if( sstv_status.fskid_decoder_init )
  {
    // Follows new frequency samples in buffer
    lead_idx = 0;

    // One fskid bit length behind the lead index
    trail_idx = FM_DISCRIM_BUF_SIZE - RX_FSKID_BIT_LENGTH;

    // Clear variables
    header_detect = False;
    error_sum_min = 10.0;
    hysteresis    = 0.0;
    fskid_sum     = 0;
    sample_cnt    = 0;
    bit_cnt       = 0;
    char_cnt      = 0;

    // Clear FSKID Header sums
    memset( header_sum, 0, FSKID_HEADER_LENGTH * sizeof(int32_t) );

    // Clear FSKID text string
    memset( fskid_char, '\0', FSKID_TEXT_LENGTH );

    // Listen for FSKID header (202A)
    fskid_stage = STAGE_1;

    sstv_status.fskid_decoder_init = False;
    return;
  } // if( sstv_status.fskid_decoder_init )

  switch( fskid_stage )
  {
    case STAGE_1: // Listen for 202A header stream
      while( lead_idx != fm_discrim_buf_idx )
      {
        /* Summate frequency samples refered to 1 or 0 bit
         * frequencies over length of FSKID header (202A).
         * Header sum should minimize at end of 202A sequence.
         * The code below is a sliding window summation of
         * deviations from the expected frequency for 1 or 0 */

        /* If the header error sum drops below the threshold value, look
         * for a minimum and record the buffer index where it was reached */
        double header_det_ref = FSKID_ERR_THRESHOLD * sstv_status.detector_sens;

        // Individual bits ON indicate that frequency error is above threshold
        uint16_t error_flags = 1;
        error_flags <<= FSKID_HEADER_LENGTH;
        error_flags  -= 1;

        //printf("\n");

        // Sum of frequency deviations for the used bits in the header
        int32_t error_sum = 0;
        for( uint8_t cnt = 0; cnt < FSKID_HEADER_LENGTH; cnt++ )
        {
          // Points to frequency sample to add to summation
          int32_t buf_in_idx = lead_idx - cnt * RX_FSKID_BIT_LENGTH;

          // Points to sample to subtract from summation (1 bit length behind above)
          int32_t buf_out_idx = trail_idx - cnt * RX_FSKID_BIT_LENGTH;

          // Keep indices in range
          if( buf_in_idx  < 0 ) buf_in_idx  += FM_DISCRIM_BUF_SIZE;
          if( buf_out_idx < 0 ) buf_out_idx += FM_DISCRIM_BUF_SIZE;

          /* Summate frequencies over length of header bit and calculate
           * average frequency over length of header bit. Then calculate
           * normalized frequency error for each header bit */
          double freq_ave, header_err, freq_diff_0, freq_diff_1;
          header_sum[cnt] += fm_discrim_buffer[buf_in_idx];
          header_sum[cnt] -= fm_discrim_buffer[buf_out_idx];
          freq_ave         = (double)header_sum[cnt] / RX_FSKID_BIT_LENGTH;
          freq_diff_0      = fabs( freq_ave - FSKID_2100_HZ );
          freq_diff_1      = fabs( freq_ave - FSKID_1900_HZ );

          //printf("%3d-%6.1f", cnt, freq_ave);

          if( (header >> cnt) & 1 ) // 1-bit = 1900 Hz
          {
            header_err = freq_diff_1 / FSKID_2000_HZ;
            if( (header_err < header_det_ref + hysteresis) && (freq_diff_1 < freq_diff_0) )
              error_flags &= ~( 1 << cnt );
            error_sum += (int32_t)freq_diff_1;
          } // if( (header >> cnt) & 1 )  1-bit = 1900 Hz
          else // 0-bit = 2100 Hz
          {
            header_err = freq_diff_0 / FSKID_2000_HZ;
            if( (header_err < header_det_ref + hysteresis) && (freq_diff_0 < freq_diff_1) )
              error_flags &= ~( 1 << cnt );
            error_sum += (int32_t)freq_diff_0;
          } // else  0-bit = 2100 Hz

          //printf("%3d-%6.4f", cnt, header_err);

        } // for( uint8_t cnt = 0; cnt < FSKID_HEADER_LENGTH; cnt++ )

        // Advance and reset indices
        lead_idx++;
        if( lead_idx  >= FM_DISCRIM_BUF_SIZE )  lead_idx = 0;
        trail_idx++;
        if( trail_idx >= FM_DISCRIM_BUF_SIZE ) trail_idx = 0;

        // Normalized frequency error refered to mid frequency
        double header_det_err = (double)error_sum / FSKID_2000_HZ;
        if( !error_flags )
        {
          //puts("XXX");
          // Hysteresis in detection is 50%
          hysteresis = header_det_ref / 2;

          // Look for a minimum of header error sum and record buffer index
          if( error_sum_min >= header_det_err )
          {
            fskid_idx     = lead_idx;
            error_sum_min = header_det_err;
            header_detect = True;
          }
        } // if( !error_flags )
        else if( header_detect ) // Go to FSKID callsign decode
        {
          header_detect = False;
          memset( fskid_char, '\0', FSKID_TEXT_LENGTH );
          fskid_stage = STAGE_2;
        }
      } // case STAGE_1:
      break;

    case STAGE_2: // Decode FSKID text (callsign). Count frequency samples for one FSKID bit
      while( fskid_idx != fm_discrim_buf_idx )
      {
        sample_cnt++;
        if( sample_cnt < RX_FSKID_BIT_LENGTH )
        {
          // Summate frequencies for bit 1 (1900Hz) and bit 0 (2100Hz)
          fskid_sum += (double)fm_discrim_buffer[fskid_idx];
        }
        else // One character bit summated
        {
          // FSKID frequency error for 0-bit and 1-bit
          int32_t fskid_err0, fskid_err1;

          sample_cnt = 0;

          // Refer frequency sums to the bit length and 1 or 0 frequency
          fskid_sum /= RX_FSKID_BIT_LENGTH;
          fskid_err0 = (int32_t)( fabs(fskid_sum - FSKID_2100_HZ) );
          fskid_err1 = (int32_t)( fabs(fskid_sum - FSKID_1900_HZ) );

          // Enter and count bits in FSKID character
          fskid_char[char_cnt] >>= 1;     // Left shift clears new bit to 0
          if( fskid_err1 < fskid_err0 )   // Bit 1 frequency lower
            fskid_char[char_cnt] |= 0x20; // Set the new bit to 1
          fskid_sum = 0;

          // One 6-bit FSKID character completed
          bit_cnt++;
          if( bit_cnt >= FSKID_BYTE_LENGTH )
          {
            // Makes it an ASCII character
            fskid_char[char_cnt] += 0x20;

            /* Allow only alpha-numerics and '/', else reset
             * and continue listening for a header sequence */
            // Not capital letters
            BOOLEAN test1 = (fskid_char[char_cnt] < 'A') || (fskid_char[char_cnt] > 'Z');
            // Not numbers
            BOOLEAN test2 = (fskid_char[char_cnt] < '0') || (fskid_char[char_cnt] > '9');
            // Not / or end of FSKID
            BOOLEAN test3 = (fskid_char[char_cnt] != '/') && (fskid_char[char_cnt] != 0x21);
            // First character is '!' or End of FSKID
            BOOLEAN test4 = (fskid_char[char_cnt] == 0x21) && (char_cnt == 0);

            // Reset and continue listening for a header sequence
            if( (test1 && test2 && test3) || test4 )
            {
              hysteresis    = 0.0;
              error_sum_min = 10.0;
              char_cnt = 0;
              bit_cnt  = 0;
              memset( fskid_char, '\0', FSKID_TEXT_LENGTH );
              fskid_stage = STAGE_1;
              break;
            }

            // If FSKID character is 21 (0x01 + 0x20), its end of string so reset
            if( (fskid_char[char_cnt] == 0x21) || (char_cnt == FSKID_TEXT_LENGTH - 1) )
            {
              // Terminate string and display in FSKID label
              fskid_char[char_cnt] = '\0';
              Set_Entry_Text( GTK_ENTRY(sstv_gui.callsign), fskid_char );

              // Set the Rx Status Label
              gchar txt[32];
              snprintf( txt, sizeof(txt), "FSKID Detected: %s", fskid_char );
              Set_Label_Text( GTK_LABEL(sstv_gui.rx_status_label), txt );

              // Reset to continue listening for a header
              hysteresis    = 0.0;
              error_sum_min = 10.0;
              char_cnt = 0;
              bit_cnt  = 0;
              memset( fskid_char, 0, FSKID_TEXT_LENGTH );
              fskid_stage = STAGE_1;
              break;
            } // if( (fskid_char[char_cnt] == 0x21) ||

            char_cnt++;
            bit_cnt = 0;
          } // if( bit_cnt >= FSKID_BYTE_LENGTH )
        } // else of if( sample_cnt < FSKID_BIT_LENGTH )

        // Advance and reset indices
        fskid_idx++;
        if( fskid_idx >= FM_DISCRIM_BUF_SIZE )
          fskid_idx = 0;
      } // while( fskid_idx != fm_discrim_buf_idx )

  } // switch( fskid_stage )

} // Decode_FSKID()

//------------------------------------------------------------------------

/* Sstv_Decode()
 *
 * Central function to coordinate various specific
 * functions that together decode an SSTV image
 */
  BOOLEAN
Sstv_Decode( void )
{
  // Loop continously around the FM discriminator till timer timeout
  while( Transceiver[Indices.TRx_Index]->receive_active &&
         sstv_status.sstv_action == SSTV_ACTION_DECODE )
  {
    // Detect the current SSTV FM signal frequency
    if( !FM_Discriminator() ) return( False );

    /* Try to detect VIS Leader tones if enabled
     * by sstv_status.detect_vis_leader = True */
     VIS_Leader_Detect();

    /* Try to detect the VIS Mode Number
     * if the Leader was detected above */
    if( sstv_status.detect_vis_mode )
      VIS_Mode_Detect();

    // Try to detect Line Sync pulses
    Line_Sync_Detect();

    // Process Line Sync pulses when detected
    if( sstv_status.line_sync_detected )
      Process_Line_Sync();

    // Decode FSKID callsign
    Decode_FSKID();

    /* Decode SSTV Image if the image's index to
     * the frequency buffer has been determined */
    if( sstv_status.image_decoder_enable &&
        sstv_status.image_pixbuf_ready )
      Sstv_Image_Decoder();
    else if( sstv_status.new_image )
    {
      Sstv_New_Image();
      sstv_status.new_image = False;
    }

  } // while( Transceiver[Indices.TRx_Index]->receive_active );

  return( True );
} // Sstv_Decode()

//------------------------------------------------------------------------

