/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "callback_func.h"
#include "detect.h"
#include "display.h"
#include "interface.h"
#include "operation.h"
#include "shared.h"
#include "tests.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/guest_utils.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/interface.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdlib.h>

//------------------------------------------------------------------

// All callback functions of the SSTV client

  void
on_sstv_window_destroy(
    GObject  *object,
    gpointer  user_data )
{
  MOX_Control( MOX_OFF ); // FIXME may not be needed
  sstv_status.sstv_action = SSTV_ACTION_STOP;

  if( sstv_gui.rx_image_window != NULL )
    gtk_widget_destroy( sstv_gui.rx_image_window );
  if( sstv_gui.tx_image_window != NULL )
    gtk_widget_destroy( sstv_gui.tx_image_window );
  if( sstv_gui.gallery_window != NULL )
    gtk_widget_destroy( sstv_gui.gallery_window );
  hermes2_gui.guest_window = NULL;

  Sstv_Cleanup();
}

//------------------------------------------------------------------


  gboolean
on_sstv_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data )
{
  // Save current config
  Sstv_Write_Config();
  sstv_status.sstv_action = SSTV_ACTION_STOP;
  Guest_Quit_Activate( sstv_gui.window );
  return( TRUE );
}

//------------------------------------------------------------------

  void
on_sstv_rx_image_window_destroy(
    GObject  *object,
    gpointer  user_data )
{
  sstv_status.sstv_action  = SSTV_ACTION_STOP;
  sstv_gui.rx_image_window = NULL;
}

//------------------------------------------------------------------

  gboolean
on_sstv_rx_image_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data )
{
  return( TRUE );
}

//------------------------------------------------------------------

  void
on_sstv_tx_image_window_destroy(
    GObject  *object,
    gpointer  user_data )
{
  sstv_status.sstv_action  = SSTV_ACTION_STOP;
  sstv_gui.tx_image_window = NULL;
}

//------------------------------------------------------------------

  gboolean
on_sstv_tx_image_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data )
{
  return( TRUE );
}

//------------------------------------------------------------------

  gboolean
on_sstv_tx_image_key_press(
    GtkWidget   *widget,
    GdkEventKey *event,
    gpointer     user_data )
{
  if( sstv_status.tx_image_type )
    Sstv_Tx_Image_Key_Press( event );
  return( TRUE );
}

//----------------------------------------------------------------------

  gboolean
on_sstv_tx_image_button_press(
    GtkWidget       *widget,
    const GdkEventButton  *event,
    gpointer         user_data )
{
  Sstv_Tx_Image_Button_Press( event );
  return( TRUE );
}

//----------------------------------------------------------------------

  gboolean
on_sstv_tx_image_motion_notify(
    GtkWidget       *widget,
    GdkEventMotion  *event,
    gpointer         user_data )
{
  if( sstv_status.tx_image_draw )
    Sstv_Tx_Image_Motion_Notify( widget, event );
  return( TRUE );
}

//----------------------------------------------------------------------

  void
on_sstv_tx_image_type_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    sstv_status.tx_image_type = True;
  else
    sstv_status.tx_image_type = False;
}

//----------------------------------------------------------------------

  void
on_sstv_tx_image_draw_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    sstv_status.tx_image_draw = True;
  else
    sstv_status.tx_image_draw = False;
}

//----------------------------------------------------------------------

  void
on_sstv_tx_image_gallery_clicked(
    GtkButton  *button,
    gpointer    user_data )
{
  Sstv_Tx_Image_Gallery_Clicked();
}

//----------------------------------------------------------------------

  void
on_sstv_tx_pick_color_set(
    GtkColorButton *button,
    gpointer        user_data )
{
  Sstv_Tx_Get_Font_Brush_Color( button );
}

//----------------------------------------------------------------------

  void
on_sstv_tx_font_chooser_font_activated(
    GtkFontChooser *self,
    const char     *fontname,
    gpointer        user_data)
{
  Strlcpy( sstv_rc.tx_font_face, fontname, FONT_FACE_SIZE );
}

//----------------------------------------------------------------------

  void
on_sstv_tx_font_chooser_font_set(
    GtkFontButton *widget,
    gpointer       user_data)
{
  gchar *font_face = gtk_font_chooser_get_font( GTK_FONT_CHOOSER(widget) );
  Strlcpy( sstv_rc.tx_font_face, font_face, FONT_FACE_SIZE );
  g_free( font_face );
}

//----------------------------------------------------------------------

  void
on_sstv_tx_font_capitals_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    sstv_status.capitalize_font = True;
  else
    sstv_status.capitalize_font = False;
}

//----------------------------------------------------------------------

  void
on_sstv_tx_brush_size_value_changed(
    GtkSpinButton *spinbutton,
    gpointer       user_data )
{
  sstv_rc.tx_brush_size = gtk_spin_button_get_value( spinbutton );
}

//----------------------------------------------------------------------

  void
on_sstv_gallery_window_destroy(
    GObject  *object,
    gpointer  user_data )
{
  sstv_gui.gallery_window = NULL;
}

//------------------------------------------------------------------

  gboolean
on_sstv_gallery_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data )
{
  gtk_widget_destroy( widget );
  return( TRUE );
}

//------------------------------------------------------------------

  void
on_sstv_gallery_item_activated(
    GtkIconView *iconview,
    GtkTreePath *path,
    gpointer     user_data )
{
  Sstv_Icon_View_Item_Activate( iconview, path );
}

//------------------------------------------------------------------

  void
on_sstv_scope_drawingarea_configure(
    GtkWidget       *widget,
    GdkEventConfigure *event,
    gpointer         user_data )
{
  sstv_scope_width  = (uint16_t)event->width;
  sstv_scope_height = (uint16_t)event->height;
}

//------------------------------------------------------------------

  gboolean
on_sstv_scope_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data )
{
  // Draw scope background
  if( !Flag[GUEST_ENABLE_SCOPE] )
  {
    cairo_set_source_rgb( cr, SCOPE_BACKGND );
    cairo_rectangle(
        cr, 0.0, 0.0, (double)sstv_scope_width, (double)sstv_scope_height );
    cairo_fill( cr );
  }
  else Sstv_Draw_Signal( cr );

  return( TRUE );
}

//------------------------------------------------------------------

  void
on_sstv_spectrum_drawingarea_configure(
    GtkWidget         *widget,
    const GdkEventConfigure *event,
    gpointer           user_data )
{
  Sstv_Spectrum_Size_Allocate(
      (uint16_t)event->width, (uint16_t)event->height );
}

//------------------------------------------------------------------

  gboolean
on_sstv_spectrum_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data )
{
  if( sstv_wfall.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, sstv_wfall.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
    return( TRUE );
  }
  return( FALSE );
}

//------------------------------------------------------------------

  void
on_sstv_rx_image_drawingarea_configure(
    GtkWidget         *widget,
    const GdkEventConfigure *event,
    gpointer         user_data )
{
  Sstv_Rx_Image_Drawingarea_Configure(
      (uint16_t)event->width, (uint16_t)event->height );
}

//------------------------------------------------------------------

  gboolean
on_sstv_rx_image_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data )
{
  if( sstv_rx_image.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, sstv_rx_image.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
    return( TRUE );
  }
  return( FALSE );
}

//------------------------------------------------------------------

  void
on_sstv_receive_button_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  if( !Transceiver[Indices.TRx_Index]->receive_active || ddv->transmit_on )
    return;
  Sstv_Receive_Button_Clicked();
}

//------------------------------------------------------------------

  void
on_sstv_transmit_button_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;
  Sstv_Transmit_Button_Clicked();
}

//------------------------------------------------------------------

  void
on_sstv_modes_button_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  GtkBuilder *builder;
  GtkWidget *modes_menu;
  modes_menu = Create_Sstv_Mode_Menu( &builder );
  gtk_menu_popup_at_pointer( GTK_MENU(modes_menu), NULL );
  g_object_unref( builder );
}

//------------------------------------------------------------------

  void
on_sstv_mode_menuitem_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data )
{
  Sstv_Mode_Menu_Item_Activate( menuitem, MANUAL_SELECT );
}

//------------------------------------------------------------------

  void
on_sstv_save_image_button_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  if( !sstv_status.image_decoder_enable &&
      sstv_status.image_pixbuf_ready )
    Sstv_Save_Pixbuf( sstv_rx_image.pixbuf,
        sstv_mode_params[sstv_status.sstv_mode_index].mode_name, "png" );
  else
    sstv_status.save_image = True;
}

//------------------------------------------------------------------

  void
on_sstv_listen_button_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  sstv_status.new_image = True;
}

//------------------------------------------------------------------

  void
on_sstv_auto_listen_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    sstv_status.auto_listen_enable = True;
  else
    sstv_status.auto_listen_enable = False;
}

//------------------------------------------------------------------

  void
on_sstv_autosave_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    sstv_status.auto_save_enable = True;
  else
    sstv_status.auto_save_enable = False;
}

//------------------------------------------------------------------

  void
on_sstv_tx_fsk_id_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    sstv_status.fsk_id_enable = True;
  else
    sstv_status.fsk_id_enable = False;
}

//------------------------------------------------------------------

  void
on_sstv_tx_cw_id_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    sstv_status.cw_id_enable = True;
  else
    sstv_status.cw_id_enable = False;
}

//------------------------------------------------------------------

  void
on_sstv_sensitivity_hscale_value_changed(
    GtkRange        *range,
    gpointer         user_data )
{
  sstv_status.detector_sens = (double)gtk_range_get_value( range ) / 10.0;
}

//------------------------------------------------------------------

  void
on_sstv_callsign_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Callsign_Changed( editable, &qso_record, "SSTV" );
}

//------------------------------------------------------------------------

  void
on_sstv_rst_out_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Process_RST( editable );
  Strlcpy( qso_record.dx_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(qso_record.dx_rst) );
}

//------------------------------------------------------------------------

  void
on_sstv_rst_in_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Process_RST( editable );
  Strlcpy( qso_record.my_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(qso_record.my_rst) );
}

//------------------------------------------------------------------------

  void
on_sstv_op_name_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field QSO record structure
  Strlcpy( qso_record.dx_name, buff, sizeof(qso_record.dx_name) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_name );
}

//------------------------------------------------------------------------

  void
on_sstv_qth_name_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field and QSO record structure
  Strlcpy( qso_record.dx_qth, buff, sizeof(qso_record.dx_qth) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_qth );
}

//------------------------------------------------------------------------

  void
on_sstv_locator_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Locator_Changed( editable, &qso_record, "SSTV" );
}

//------------------------------------------------------------------------

  void
on_sstv_band_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Band_Changed( editable, &qso_record, "SSTV" );
}

//------------------------------------------------------------------------

  void
on_sstv_new_record_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  New_Record( &qso_record, sstv_gui.window_builder, "SSTV" );
  Strlcpy( qso_record.mode_dir, "sstv", sizeof(qso_record.mode_dir) );
}

//------------------------------------------------------------------------

  void
on_sstv_save_record_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  // Save record if needed (and valid)
  if( Read_QSO_Record(&qso_record, sstv_gui.window_builder, "SSTV") )
    Guest_Save_Dialog( _("Save QSO Record to file?") );
  else
    Guest_Save_Dialog(
        _("QSO Record is incomplete\n"\
          "Do you still want to save?") );
}

//------------------------------------------------------------------------

  void
on_sstv_clear_fields_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  Clear_Record_Fields( True, sstv_gui.window_builder, "SSTV" );
}

//------------------------------------------------------------------------

