/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callback_func.h"
#include "callbacks.h"
#include "encode.h"
#include "decode.h"
#include "detect.h"
#include "display.h"
#include "interface.h"
#include "operation.h"
#include "shared.h"
#include "tests.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/modulate.h"
#include "../Hermes2/sound.h"
#include <cairo/cairo.h>
#include <ctype.h>
#include <pango/pangocairo.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <math.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>
#include <dirent.h>
#include <stdio.h>
#include <string.h>

//------------------------------------------------------------------

/* Stop_Receive()
 *
 * Stops and resets SSTV Receiving
 */
  static void
Stop_Receive( void )
{
  sstv_status.sstv_action          = SSTV_ACTION_STOP;
  sstv_status.image_decoder_enable = False;
  sstv_status.new_period_detected  = False;
  sstv_status.detect_vis_leader    = False;
  sstv_status.detect_vis_mode      = False;
  sstv_status.receive_active       = False;

  Flag[GUEST_DEMOD_IQ_DATA]   = False;
  Flag[GUEST_RECEIVE_MODE]    = False;

} // Stop_Receive()

//------------------------------------------------------------------

/* Start_Receive()
 *
 * Starts SSTV Receiving
 */
  static void
Start_Receive( void )
{
  // Allocate demodulator buffers
  Alloc_Demod_Buf_Copies( SOUND_OP_BUF_SIZE );

  // Initialize Digimode semaphore
  if( !Init_Semaphore(&digimode_semaphore, True) )
    return;

  // Start receiving and decoding signals
  Flag[GUEST_DEMOD_IQ_DATA]   = True;
  Flag[GUEST_RECEIVE_MODE]    = True;
  Flag[GUEST_RECEIVING] = True;

  // Reset the decoder functions and flags for a new image
  Sstv_New_Image();

  // Start SSTV Receive/Decode functions
  sstv_status.sstv_action = SSTV_ACTION_DECODE;
  if( !Pthread_Create(
        &hermes2_rc.guest_rx_thread, NULL, Sstv_Receive_Control, NULL,
        _("Failed to create Receive/Decode thread")) )
  {
    Flag[GUEST_DEMOD_IQ_DATA] = False;
    Flag[GUEST_RECEIVE_MODE]  = False;
    sstv_status.sstv_action   = SSTV_ACTION_STOP;
    Init_Semaphore( &digimode_semaphore, False );
    return;
  }

  sstv_status.receive_active = True;
} // Start_Receive()

//------------------------------------------------------------------

/* Sstv_Receive_Button_Clicked()
 *
 * Handles clicked event of Receive button
 */
  void
Sstv_Receive_Button_Clicked( void )
{
  // Receive button clicked
  if( !sstv_status.transmit_active )
  {
    // If not transmitting or receiving, activate Receive
    if( !sstv_status.receive_active)
    {
      Start_Receive();
    }
    else
    {
      // Stop Reception and set the Rx Status Label text
      Stop_Receive();
      gtk_label_set_text( GTK_LABEL(sstv_gui.rx_status_label), "Reception Stopped" );
    }

    // Set Tx/Rx button labels
    Sstv_Set_TxRx_Labels();
    return;
  }
} // Sstv_Receive_Button_Clicked()

//------------------------------------------------------------------------

/* Sstv_New_Image_Button_Clicked()
 *
 * Prepares SSTV for decoding a new image
 */
  void
Sstv_New_Image( void )
{
  sstv_status.new_image = False;

  /* Reset test functions */
  VIS_Leader( True, 0 );
  VIS_Mode( True, M2, 0 );
  Line_Sync( True, M2, 0 );
  Martin_Pasokon_Test( True, M2, 0 );
  Scottie_Test( True, S2, 0 );
  SC2180_Test( True, SC2_180, 0 );
  PD_Test( True, PD90, 0 );
  Test_FSK_ID( True, 0 );

  // Signal various functions to initialize at start
  sstv_status.receiver_init        = True;
  sstv_status.freq_discrim_init    = True;
  sstv_status.tone_detectors_init  = True;
  sstv_status.vis_leader_det_init  = True;
  sstv_status.vis_mode_det_init    = True;
  sstv_status.line_sync_det_init   = True;
  sstv_status.line_sync_proc_init  = True;
  sstv_status.image_decoder_init   = True;
  sstv_status.fskid_decoder_init   = True;

  // Initialize control flags
  sstv_status.detect_vis_leader    = True;
  sstv_status.detect_vis_mode      = False;
  sstv_status.new_period_detected  = False;
  sstv_status.image_decoder_enable = False;
  sstv_status.sstv_mode_detected   = False;
  sstv_status.save_image           = False;

  // Set the default SSTV mode if in AUTO
  if( sstv_status.sstv_mode_auto )
  {
    New_Sstv_Mode( DEFAULT_MODE, False );
  }

  // Call functions to initialize
  FM_Discriminator();
  VIS_Leader_Detect();
  VIS_Mode_Detect();
  Line_Sync_Detect();
  Process_Line_Sync();
  Decode_FSKID();
  sstv_status.receiver_init = False;

  // Clear "LED" icons
  sstv_rc.vis_leader_icon_yes = False;
  sstv_rc.vis_mode_icon_yes   = False;
  sstv_rc.sync_det_icon_yes   = False;
  sstv_rc.sync_match_icon_yes = False;

  // Set the VIS Leader Indicator "LED" OFF
  Set_Icon( GTK_IMAGE(sstv_gui.sstv_vis_leader_image), "gtk-no", GTK_ICON_SIZE_BUTTON );

  // Set the VIS Mode Indicator "LED" OFF
  Set_Icon( GTK_IMAGE(sstv_gui.sstv_vis_mode_image), "gtk-no", GTK_ICON_SIZE_BUTTON );

  // Set the Line Sync Indicator "LED" OFF
  Set_Icon( GTK_IMAGE(sstv_gui.sstv_sync_detect_image), "gtk-no", GTK_ICON_SIZE_BUTTON );

  // Set the Sync Detected Indicator "LED" OFF
  Set_Icon( GTK_IMAGE(sstv_gui.sstv_sync_match_image), "gtk-no", GTK_ICON_SIZE_BUTTON );

  // Set the Rx Status Label
  Set_Label_Text( GTK_LABEL( sstv_gui.rx_status_label), "Listening for VIS and Sync ...." );

  // Clear the dF Calibration label
  Set_Label_Text( GTK_LABEL( sstv_gui.df_label), "0 Hz" );
  sstv_status.freq_calibration = 0;

} // Sstv_New_Image()

//------------------------------------------------------------------------

/* Sstv_Initialize_Transmit()
 *
 * Initializes the Transmit mode
 */
  static void
Sstv_Initialize_Transmit( void )
{
  // Set transmit mode
  Flag[HERMES2_SEND_DUC_PACKET] = True;
  Flag[GUEST_TRANSMIT_MODE]     = True;

  // Enable Transmit if needed
  MOX_Control( MOX_ON );
  Modulator = DUC_Buffer_Transmit;
} // Sstv_Initialize_Transmit()

//------------------------------------------------------------------------

/* Sstv_Stop_Transmit()
 *
 * Stop SSTV transmission and revert to Receive
 */
  void
Sstv_Stop_Transmit( void )
{
  // Take Transceiver out of Transmit mode
  sstv_status.sstv_action = SSTV_ACTION_STOP;
  pthread_join( hermes2_rc.guest_tx_thread, NULL );
  sstv_status.transmit_active = False;
  MOX_Control( MOX_OFF );
  Flag[GUEST_TRANSMIT_MODE] = False;
  Flag[GUEST_RECEIVE_MODE]  = True;

  // Put SSTV in Receive mode
  Start_Receive();

  // Set Tx/Rx button labels
  Sstv_Set_TxRx_Labels();

} // Sstv_Stop_Transmit()

//------------------------------------------------------------------------

/* Sstv_Transmit_Button_Clicked()
 *
 * Handles clicked event of Transmit button
 */
  void
Sstv_Transmit_Button_Clicked( void )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // If Transmit button clicked
  if( !sstv_status.transmit_active )
  {
    if( !ddv->transmit_on )
    {
      // Stop Reception and set the Rx Status Label text
      Stop_Receive();
      sstv_status.transmit_active = True;

      // Set Tx/Rx Button labels
      Sstv_Set_TxRx_Labels();
      Set_Label_Text( GTK_LABEL(sstv_gui.rx_status_label), "Transmitting Started" );

      // Initialize and start Transmit thread
      Sstv_Initialize_Transmit();
      if( !Pthread_Create(
            &hermes2_rc.guest_tx_thread, NULL, Sstv_Transmit_Control, NULL,
            _("Failed to create Transmit thread")) )
      {
        Sstv_Stop_Transmit();
        sstv_status.sstv_action = SSTV_ACTION_STOP;
        return;
      }
    }
  }
  else // Terminate Transmission and start Receive
  {
    Sstv_Stop_Transmit();
  }

} // Sstv_Transmit_Button_Clicked()

//------------------------------------------------------------------------

/* Modes_Menu_Item_Activate()
 *
 * Handles the on_modes_menuitem_activate callback
 */
  void
Sstv_Mode_Menu_Item_Activate( GtkMenuItem *menuitem, uint8_t mode_num )
{
  uint8_t mode;
  const gchar *label = "";

  // If from menu callback
  if( menuitem ) label = gtk_menu_item_get_label( menuitem );

  // Not from config file
  if( mode_num == MANUAL_SELECT )
  {
    // Identify the menu item
    const gchar *menu_items[SSTV_NUM_MODES] = { SSTV_MODE_MENU_ITEMS };
    for( mode = 0; mode < SSTV_NUM_MODES; mode++ )
    {
      if( strcmp(label, menu_items[mode]) == 0 )
        break;
    }
  }
  else mode = mode_num; // From config file

  if( mode < AUTO ) // Mode is manual
  {
    // Set SSTV Modes Button label to Manual
    gtk_button_set_label( GTK_BUTTON((
            Builder_Get_Object( sstv_gui.window_builder, "sstv_modes_button"))), "Mode Manual" );
    sstv_status.sstv_mode_auto = False;

    // Set the Rx Status Label
    gchar txt[32];
    snprintf( txt, sizeof(txt), "Manual Mode Set: %s", label );
    gtk_label_set_text( GTK_LABEL(sstv_gui.rx_status_label), txt );
  }
  else // Mode is Auto
  {
    // Set SSTV Modes Button label and mode auto flag to Auto
    gtk_button_set_label( GTK_BUTTON((
            Builder_Get_Object( sstv_gui.window_builder, "sstv_modes_button"))), "Mode Auto" );
    sstv_status.sstv_mode_auto = True;
    mode = DEFAULT_MODE;

    // Set the Rx Status Label
    gtk_label_set_text( GTK_LABEL(
          sstv_gui.rx_status_label), "Mode Set to Auto" );
  }

  // Set up new manual or default mode
  if( !sstv_status.image_decoder_enable )
    New_Sstv_Mode( mode, False );

} // Modes_Menu_Item_Activate()

//----------------------------------------------------------------------

/* Sstv_Rx_Image_Drawingarea_Configure()
 *
 * Configures the SSTV image drawingarea after a resize
 */
  void
Sstv_Rx_Image_Drawingarea_Configure( uint16_t width, uint16_t height )
{
  // Destroy existing pixbuff
  if( sstv_rx_image.pixbuf != NULL )
  {
    g_object_unref( G_OBJECT(sstv_rx_image.pixbuf) );
    sstv_rx_image.pixbuf = NULL;
  }

  // Create SSTV image pixbuf
  sstv_rx_image.pixbuf = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8, width, height );
  if( sstv_rx_image.pixbuf == NULL ) return;

  sstv_rx_image.pixels = gdk_pixbuf_get_pixels( sstv_rx_image.pixbuf );
  sstv_rx_image.width  = (uint16_t)gdk_pixbuf_get_width ( sstv_rx_image.pixbuf );
  sstv_rx_image.height = (uint16_t)gdk_pixbuf_get_height( sstv_rx_image.pixbuf );
  sstv_rx_image.rowstride  = (uint16_t)gdk_pixbuf_get_rowstride( sstv_rx_image.pixbuf );
  sstv_rx_image.n_channels = (uint16_t)gdk_pixbuf_get_n_channels( sstv_rx_image.pixbuf );
  gdk_pixbuf_fill( sstv_rx_image.pixbuf, 0 );

  sstv_status.image_pixbuf_ready = True;

} // Sstv_Rx_Image_Drawingarea_Configure()

//----------------------------------------------------------------------

#define STRING_LENGTH   128

// SSTV Image co-ordinates to render key strokes by user
static gdouble text_x = 0, text_y = 0;

// Character string used to print on Tx image
static gchar char_string[STRING_LENGTH];
static uint8_t string_length = 0;

// Rrint character string veertically flag
static BOOLEAN vertical = False;

//----------------------------------------------------------------------

/* Sstv_Tx_Get_Font_Brush_Color()
 *
 * Gets the color from the Color Chooser callback
 */
  void
Sstv_Tx_Get_Font_Brush_Color( GtkColorButton *button )
{
  gtk_color_chooser_get_rgba( GTK_COLOR_CHOOSER(button), &sstv_rc.tx_brush_font_color );
} // Sstv_Tx_Get_Font_Brush_Color()

//----------------------------------------------------------------------

/* Sstv_Tx_Image_Key_Press()
 *
 * Handles button press events on the Tx Image widget
 */
  void
Sstv_Tx_Image_Key_Press( GdkEventKey *event )
{
  PangoLayout *layout;
  PangoFontDescription *font_description;

  // Backspace character
  if( event->keyval == GDK_KEY_BackSpace )
  {
    if( string_length ) string_length--;
    char_string[string_length] = '\0';

    // Re-create cairo surface from the current image file
    Sstv_Display_Stock_Image( sstv_status.tx_image_name );
  }
  else
  {
    // Don't process non-ASCII characters
    if( (event->keyval &= 0xFF) > 0x7F ) return;

    // Limit string to max length of STR_LENGTH - 1
    if( string_length < STRING_LENGTH - 2 )
    {
      // Capitalize letters if enabled
      if( sstv_status.capitalize_font )
        char_string[string_length] = (gchar)toupper( event->keyval );
      else
        char_string[string_length] = (gchar)event->keyval;

      string_length++;
      char_string[string_length] = '\0';
    }
  }

  // Create cairo concept from cairo surface
  cairo_t *cr = cairo_create( sstv_gui.cairo_surface );

  // Create a Pango font description from the font face
  font_description = pango_font_description_from_string( sstv_rc.tx_font_face );

  // Set font gravity to East if vertical text enabled
  if( vertical )
    pango_font_description_set_gravity( font_description, PANGO_GRAVITY_EAST );

  // Create a Pango layout from the Cairo context and set description
  layout = pango_cairo_create_layout( cr );
  pango_layout_set_font_description( layout, font_description );

  // Set the text string created by key strokes
  pango_layout_set_text( layout, char_string, -1 );

  // Set the font color
  cairo_set_source_rgba( cr,
      sstv_rc.tx_brush_font_color.red,
      sstv_rc.tx_brush_font_color.green,
      sstv_rc.tx_brush_font_color.blue,
      sstv_rc.tx_brush_font_color.alpha  );

  // Move to x, y co-ordinates of pointer click
  cairo_move_to( cr, text_x, text_y );

  // Rotate Cairo context for vertical text and show layout
  if( vertical ) cairo_rotate( cr, M_PI / 2 );
  pango_cairo_show_layout( cr, layout );

  // Free objects
  cairo_destroy( cr );
  g_object_unref( layout );
  pango_font_description_free( font_description );

  // Display the image in the Tx Image window
  gtk_image_set_from_surface(
      GTK_IMAGE(sstv_gui.sstv_tx_image), sstv_gui.cairo_surface );

} // Sstv_Tx_Image_Key_Press()

//----------------------------------------------------------------------

/* Sstv_Tx_Image_Button_Press()
 *
 * Handles button press events on the Tx Image widget
 */
  void
Sstv_Tx_Image_Button_Press( const GdkEventButton *event )
{
  if( sstv_status.tx_image_type )
  {
    if( (event->button == GDK_BUTTON_PRIMARY) ||
        (event->button == GDK_BUTTON_MIDDLE) )
    {
      text_x = event->x;
      text_y = event->y;
      string_length  = 0;
      char_string[0] = '\0';

      if( event->button == GDK_BUTTON_PRIMARY )
        vertical = False;
      else
        vertical = True;
    }
  }
} // Sstv_Tx_Image_Button_Press()

//----------------------------------------------------------------------

/* Sstv_Tx_Image_Motion_Notify()
 *
 * Handles the Button Motion Notify event callback on the Tx Image
 */
  void
Sstv_Tx_Image_Motion_Notify( GtkWidget *widget, GdkEventMotion *event )
{
  // Create cairo concept from cairo surface
  cairo_t *cr = cairo_create( sstv_gui.cairo_surface );
  cairo_set_source_rgba( cr,
      sstv_rc.tx_brush_font_color.red,
      sstv_rc.tx_brush_font_color.green,
      sstv_rc.tx_brush_font_color.blue,
      sstv_rc.tx_brush_font_color.alpha );
  cairo_arc( cr, event->x, event->y, sstv_rc.tx_brush_size, 0.0, M_2PI );
  cairo_fill( cr );
  cairo_destroy( cr );

  // Display the image in the Tx Image window
  gtk_image_set_from_surface(
      GTK_IMAGE(sstv_gui.sstv_tx_image), sstv_gui.cairo_surface );

} // Sstv_Tx_Image_Motion_Notify()

//----------------------------------------------------------------------

#define MAX_NUM_FILES   100

enum _ICONVIEW_COLS
{
  PIXBUF_COL = 0,
  STRING_COL
};

/* Sstv_Tx_Image_Gallery_Clicked()
 *
 * Handles button press events on the Tx Image widget
 */
  void
Sstv_Tx_Image_Gallery_Clicked( void )
{
  // List  of image files ordered by file number
  gchar **ordered_list = NULL;

  // Count of files listed by readdir()
  uint8_t file_cnt = 0;

  // Used for directory listing
  const struct dirent *dir;
  DIR *dr;

  // Used for creating a GTK list store for the icon view
  GtkTreeIter iter;
  GError *error = NULL;

  char
    stock_dir[FILE_NAME_SIZE - 8],  // Stock images directory path
    **file_list = NULL;            // List of image files from readdir()


  // Open the Icon View window if not already open
  if( sstv_gui.gallery_window != NULL )
    return;
  sstv_gui.gallery_window = Create_Sstv_Icon_View();

  // Setup file path to stock images
  Strlcpy( stock_dir, sstv_rc.sstv_dir, sizeof(stock_dir) );
  Strlcat( stock_dir, "stock_images/",  sizeof(stock_dir) );

  // Open images directory
  dr = opendir( stock_dir );
  if( dr == NULL )
  {
    perror( stock_dir );
    Error_Dialog(
        _("Failed to open stock icons directory\n"\
          "Quit SSTV and correct"), HIDE_OK );
    return;
  }

  // Read directrory listing till NULL terminator, max 100 files
  while( ((dir = readdir(dr)) != NULL) && (file_cnt < MAX_NUM_FILES) )
  {
    // List only regular (image) files
    if( dir->d_type == DT_REG )
    {
      // Check file name to be in format nn.png (00.png-99.png)
      int8_t num = (int8_t)( atoi(dir->d_name) );
      size_t len  = strlen(dir->d_name) + 1;
      if( (num < 0) || (num > 99) || (len > 7) ||
          (strncmp(&dir->d_name[3], "png", 3) != 0) )
        continue;

      // Count number of files listed
      file_cnt++;

      // Re-allocate file list pointer
      Mem_Realloc( (void **)&file_list, (size_t)file_cnt * sizeof(char *) );

      // Allocate file name pointer
      file_list[file_cnt - 1] = NULL;
      Mem_Alloc( (void **)&file_list[file_cnt - 1], sizeof(char) * len );
      Strlcpy( file_list[file_cnt - 1], dir->d_name, len );
    }
  } // while( (dir = readdir(dr)) != NULL )
  closedir( dr );

  // If no stock image file found
  if( !file_cnt )
  {
    Error_Dialog(
        _("No image files found in stock icons directory\n"\
          "Quit SSTV and correct"), HIDE_OK );
    return;
  }

  /* Order file list so that file names (00-99) are in numerical order.
   * Allocate the ordered file list */
  Mem_Alloc( (void **)&ordered_list, (size_t)file_cnt * sizeof(char *) );

  // All file names are (should be) of same length (nn.png, nn = 00-99)
  size_t len = strlen( file_list[0] ) + 1;
  for( uint8_t idx = 0; idx < file_cnt; idx++ )
    Mem_Alloc( (void **)&ordered_list[idx], sizeof(char) * len );

  // Re-order file list according to file name (00.png - 99.png)
  for( uint8_t idx = 0; idx < file_cnt; idx++ )
  {
    // Make sure index to ordered_list[] is in range
    int8_t file_num  = (int8_t)atoi( file_list[idx] );
    if( (file_num >= 0) && (file_num < file_cnt) )
      Strlcpy( ordered_list[file_num], file_list[idx], sizeof(char) * len );
  }

  // Create a model for the icon view
  GtkListStore *model = gtk_list_store_new( 2, GDK_TYPE_PIXBUF, G_TYPE_STRING );

  // Append images to Icon View list store
  for( uint8_t idx = 0; idx < file_cnt; idx++ )
  {
    // Append a new column to list store
    gtk_list_store_append( model, &iter );

    // Set the file name to the list store column
    gtk_list_store_set( model, &iter, STRING_COL, ordered_list[idx], -1 );

    // Load an image file into a pixbuf to set to list store
    char image_file[FILE_NAME_SIZE];
    Strlcpy( image_file, stock_dir, sizeof(image_file) );
    Strlcat( image_file, ordered_list[idx], sizeof(image_file) );
    GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file_at_size( image_file, 160, 120, &error );
    gtk_list_store_set( model, &iter, PIXBUF_COL, pixbuf, -1 );
    g_object_unref( pixbuf );
  } // for( uint8_t idx = 0; idx < file_cnt; idx++ )

  // Set the List store as the model for the Icon View
  gtk_icon_view_set_model( GTK_ICON_VIEW(sstv_gui.sstv_icon_view), GTK_TREE_MODEL(model) );
  gtk_icon_view_set_pixbuf_column( GTK_ICON_VIEW(sstv_gui.sstv_icon_view), PIXBUF_COL );
  gtk_icon_view_set_text_column(   GTK_ICON_VIEW(sstv_gui.sstv_icon_view), STRING_COL );

  // Free the file lists
  for( uint8_t idx = 0; idx < file_cnt; idx++ )
    Mem_Free( (void **)&file_list[idx] );
  Mem_Free( (void **)&file_list );
  for( uint8_t idx = 0; idx < file_cnt; idx++ )
    Mem_Free( (void **)&ordered_list[idx] );
  Mem_Free( (void **)&ordered_list );

  return;
} // Sstv_Tx_Image_Gallery_Clicked()

//----------------------------------------------------------------------

/* Sstv_Icon_View_Item_Activate()
 *
 * Handles the gallery_item_activated callback on SSTV Icon View
 */
  void
Sstv_Icon_View_Item_Activate( GtkIconView *iconview, GtkTreePath *path )
{
  // Iterator created from the tree path provided by the callback
  GtkTreeIter iter;

  // Tree model of the Icon View
  GtkTreeModel *tree_model;

  // Stock image file name. gchararray is "gchar *"
  gchararray str;


  // Get the Model of the icon view (actually it is a lsit store)
  tree_model = gtk_icon_view_get_model( iconview );

  // Get the tree model iteration to the activated path
  if( !gtk_tree_model_get_iter(tree_model, &iter, path) )
  {
    Error_Dialog(
        _("Failed to process Icon View item activation\n"\
          "Quit SSTV and correct"), HIDE_OK );
    return;
  }

  // Get the string item from the model, which is the file name
  gtk_tree_model_get( tree_model, &iter, STRING_COL, &str, -1 );

  // Load image file and display in Tx Image window
  Sstv_Display_Stock_Image( str );
  Strlcpy( sstv_status.tx_image_name, str, TX_IMAGE_NAME_SIZE );

  g_free( str );
  gtk_widget_destroy( sstv_gui.gallery_window );

} // Sstv_Icon_View_Item_Activate()

//----------------------------------------------------------------------

