/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "codec.h"
#include "common.h"
#include "detect.h"
#include "display.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/sound.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>

//------------------------------------------------------------------------

/* Rtty_Read_Config()
 *
 * Loads the rttyrc configuration file
 */
  gboolean
Rtty_Read_Config( gpointer data )
{
  // Buffer for Read_Line
  char line[READ_LINE_BUF_SIZE];

  // Config file pointer
  FILE *rttyrc;

  // Set text view font
  PangoFontDescription *font_desc;

  // For setting squelch level
  GtkWidget *hscale;
  GtkAdjustment *adjustment;


  // Setup file path to rttyrc and create the rtty config file name
  Strlcpy( rtty_rc.rc_fpath, getenv("HOME"), sizeof(rtty_rc.rc_fpath) );
  Strlcat( rtty_rc.rc_fpath, "/.hermes2/rtty/rtty.config", sizeof(rtty_rc.rc_fpath) );

  // Open rttyrc file
  rttyrc = fopen( rtty_rc.rc_fpath, "r" );
  if( rttyrc == NULL )
  {
    perror( rtty_rc.rc_fpath );
    Error_Dialog(
        _("Failed to open rttyrc file\n"\
          "Quit RTTY and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Baud Rate, abort if EOF
  if( Read_Line(line, rttyrc, _("Baud Rate") ) != SUCCESS )
    return( FALSE );
  rtty_rc.baud_rate = Strtod( line );

  // Duration of one Rx RTTY "bit" in SND samples
  if( rtty_rc.baud_rate == 0.0 ) rtty_rc.baud_rate = 45.5; // Coverity report (div by 0)
  double temp = SND_DSP_RATE / rtty_rc.baud_rate;
  rtty_rc.rx_rtty_bit = (uint16_t)temp;

  // Duration of one Tx RTTY "bit" in TX DAC samples
  temp = DUC_SAMPLE_RATE / rtty_rc.baud_rate;
  rtty_rc.tx_rtty_bit = (uint32_t)temp;

  // Read AFSK frequency shift, abort if EOF
  if( Read_Line(line, rttyrc, _("AFSK Frequency Shift") ) != SUCCESS )
    return( FALSE );
  rtty_rc.afsk_shift = (uint16_t)atoi( line );

  // Read Reverse AFSK flag, abort if EOF
  if( Read_Line(line, rttyrc, _("Reverse AFSK Enable Flag") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rtty_gui.popup_menu_builder, "rtty_reverse_afsk")), TRUE );
    Flag[RTTY_REVERSE_AFSK] = True;
  }
  else if( strcmp(line, "no") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rtty_gui.popup_menu_builder, "rtty_reverse_afsk")), FALSE );
    Flag[RTTY_REVERSE_AFSK] = False;
  }
  else
  {
    Close_File( &rttyrc );
    Error_Dialog(
        _("Error reading rttyrc\n"\
          "Unrecognized menu option\n"\
          "Quit xritty and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read stop bit length, abort if EOF
  if( Read_Line(line, rttyrc, _("Stop Bit Length") ) != SUCCESS )
    return( FALSE );
  rtty_rc.stop_bit = Strtod( line );

  // Read preamble tone duration (in dsec, minimum 2), abort if EOF
  if( Read_Line(line, rttyrc, _("Preamble Tone Duration") ) != SUCCESS )
    return( FALSE );
  rtty_rc.tx_tone_dur = (uint32_t)atoi( line );
  if( rtty_rc.tx_tone_dur < 2 ) rtty_rc.tx_tone_dur = 2;

  // Tone duration now in dsp samples
  rtty_rc.tx_tone_dur = ( rtty_rc.tx_tone_dur * DUC_SAMPLE_RATE ) / 10;

  // Read squelch threshold, abort if EOF
  if( Read_Line(line, rttyrc, _("Squelch Threshold") ) != SUCCESS )
    return( FALSE );
  rtty_rc.sqlch_thr = (uint8_t)atoi( line );

  // Enter squelch level to hscale
  hscale = Builder_Get_Object( rtty_gui.window_builder, "rtty_squelch_value" );
  adjustment = gtk_range_get_adjustment( GTK_RANGE(hscale) );
  gtk_adjustment_set_value( adjustment, (gdouble)rtty_rc.sqlch_thr );
  rtty_rc.sqlch_thr *= 5;

  // Read Enable Diversity flag, abort if EOF
  if( Read_Line(line, rttyrc, _("Diversity Enable Flag") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rtty_gui.popup_menu_builder, "rtty_enable_diversity")), TRUE );
    Flag[RTTY_ENABLE_DIVERSITY] = True;
  }
  else if( strcmp(line, "no") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rtty_gui.popup_menu_builder, "rtty_enable_diversity")), FALSE );
    Flag[RTTY_ENABLE_DIVERSITY] = False;
  }
  else
  {
    Close_File( &rttyrc );
    Error_Dialog(
        _("Error reading rttyrc\n"\
          "Unrecognized menu option\n"\
          "Quit xritty and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read USOS flag, abort if EOF
  if( Read_Line(line, rttyrc, _("USOS Enable Flag") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rtty_gui.popup_menu_builder, "rtty_usos")), TRUE );
    Flag[RTTY_ENABLE_USOS] = True;
  }
  else if( strcmp(line, "no") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rtty_gui.popup_menu_builder, "rtty_usos")), FALSE );
    Flag[RTTY_ENABLE_USOS] = False;
  }
  else
  {
    Close_File( &rttyrc );
    Error_Dialog(
        _("Error reading rttyrc\n"\
          "Unrecognized menu option\n"\
          "Quit xritty and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read word wrap column, abort if EOF
  if( Read_Line(line, rttyrc, _("Word Wrap Column") ) != SUCCESS )
    return( FALSE );
  rtty_rc.word_wrap = (uint8_t)atoi( line );

  // Read font string for text views, abort if EOF
  if( Read_Line(line, rttyrc, _("Text View font string") ) != SUCCESS )
    return( FALSE );
  Strlcpy( rtty_rc.font, line, sizeof(rtty_rc.font) );

  // *** Setup default main menu items ***
  // Read Record QSO enable flag, abort if EOF
  if( Read_Line(line, rttyrc, _("Record QSOs Enable") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rtty_gui.popup_menu_builder, "rtty_record_qsos")), TRUE );
    Flag[GUEST_RECORD_QSO] = True;
  }
  else if( strcmp(line, "no") == 0 )
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(rtty_gui.popup_menu_builder, "rtty_record_qsos")), FALSE );
  else
  {
    Close_File( &rttyrc );
    Error_Dialog(
        _("Error reading rttyrc\n"\
          "Unrecognized menu option\n"\
          "Quit xritty and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Station info and Macros
  if( !Read_Station_Info( rttyrc, rtty_macro,
        "rtty_label", rtty_gui.window_builder) )
    return( FALSE );

  // Read main Window's position and place it
  if( Read_Line(line, rttyrc, _("Window Position") ) != SUCCESS )
    return( FALSE );
  gint x = (gint)atoi( line );
  gint y = (gint)atoi( &line[5] );
  gtk_window_move( GTK_WINDOW(rtty_gui.window), x, y );

  // *** Do the global config variables ***
  // Set appropriate baud rate menu item
  {
    char btn[12];
    GtkWidget *item;

    snprintf( btn, sizeof(btn), "rtty_bd%d", (int)rtty_rc.baud_rate );
    btn[11] = '\0';
    item = Builder_Get_Object( rtty_gui.popup_menu_builder, btn );
    if( !item )
    {
      Close_File( &rttyrc );
      Error_Dialog(
          _("Error reading rttyrc\n"\
            "Invalid Baud rate\n"\
            "Quit xritty and correct"), HIDE_OK );
      return( FALSE );
    }
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(item), FALSE );
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(item), TRUE );
  }

  // Set appropriate AFSK menu item
  {
    char btn[15];
    GtkWidget *item;

    snprintf( btn, sizeof(btn),
        "rtty_afsk%d", (int)rtty_rc.afsk_shift );
    btn[13] = '\0';
    item = Builder_Get_Object( rtty_gui.popup_menu_builder, btn );
    if( !item )
    {
      Close_File( &rttyrc );
      Error_Dialog(
          _("Error reading rttyrc\n"\
            "Invalid AFSK shift\n"\
            "Quit xritty and correct"), HIDE_OK );
      return( FALSE );
    }
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(item), TRUE );
  }

  // Set appropriate stop bit menu item
  {
    char btn[12];
    GtkWidget *item;

    snprintf( btn, sizeof(btn),
        "rtty_stop%d", (int)(10.0 * rtty_rc.stop_bit) );
    btn[11] = '\0';
    item = Builder_Get_Object( rtty_gui.popup_menu_builder, btn );
    if( !item )
    {
      Close_File( &rttyrc );
      Error_Dialog(
          _("Error reading rttyrc\n"\
            "Invalid stop bit size\n"\
            "Quit xritty and correct"), HIDE_OK );
      return( FALSE );
    }
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(item), TRUE );
  }

  // Set Rx and Tx text view font
  font_desc = pango_font_description_from_string( rtty_rc.font );
  pango_font_description_free( font_desc );

  // Initialize configuration
  Rtty_Init_Config();
  Flag[RTTY_LTRS_SHIFT] = True;
  g_idle_add( Rtty_Set_Shift_Label, NULL );

  Close_File( &rttyrc );
  return( FALSE );
} // End of Rtty_Read_Config()

//------------------------------------------------------------------

/* Rtty_Cleanup()
 *
 * Cleans up before quitting
 */
  void
Rtty_Cleanup( void )
{
  // Save outstanding records
  if( Flag[GUEST_SAVE_RECORD] )
    Save_QSO_Record(&qso_record);

  // Close open files
  Close_File( &(qso_record.qso_record_fp) );
  Close_File( &(qso_record.adif_log_fp) );
  Close_File( &(qso_record.station_log_fp) );

  // Free Digimode semaphores
  Init_Semaphore( &digimode_semaphore, False );
  Init_Semaphore( &duc_send_semaphore, False );

  // Free buffer allocations
  Rtty_Free_Codec();
  Rtty_Free_Detect();
  Free_Xmit_Buffers();
  Free_iFFT( &rtty_ifft_data );

  // Free macro buffer pointers
  for( int idx = 0; idx < NUM_OF_LABELS; idx++ )
    Mem_Free( (void **) &rtty_macro[idx] );

  // Free waterfall pixbuf
  if( rtty_wfall.pixbuf )
    g_object_unref( rtty_wfall.pixbuf );
  rtty_wfall.pixbuf = NULL;

  // Close some GUI objects
  g_object_unref( rtty_gui.window_builder );
  rtty_gui.window = NULL;
  rtty_gui.window_builder  = NULL;
  hermes2_gui.guest_window = NULL;

  if( rtty_gui.popup_menu_builder != NULL )
  {
    g_object_unref( rtty_gui.popup_menu_builder );
    rtty_gui.popup_menu_builder = NULL;
  }

  // Clear General flags
  Flag[GUEST_RECORD_QSO]        = False;
  Flag[GUEST_RECEIVE_MODE]      = False;
  Flag[GUEST_TRANSMIT_MODE]     = False;
  Flag[GUEST_TRANSMIT_TAG]      = False;
  Flag[GUEST_TRANSMIT_MACRO]    = False;
  Flag[GUEST_TRANSMIT_KEYBD]    = False;
  Flag[GUEST_KEYBD_BUSY]        = False;
  Flag[GUEST_CAPITALIZE]        = False;
  Flag[GUEST_SAVE_RECORD]       = False;
  Flag[GUEST_QUIT]              = False;
  Flag[HERMES2_SEND_DUC_PACKET] = False;
  Flag[TRANSMIT_MORSE_MESG]     = False;
  Flag[GUEST_RECEIVING]         = False;
  Flag[GUEST_TRANSMITTING]      = False;

  // Clear RTTY flags
  Flag[RTTY_ENABLE_USOS]      = False;
  Flag[RTTY_SQUELCH_OPEN]     = False;
  Flag[RTTY_ENABLE_SIGNAL]    = False;
  Flag[RTTY_LTRS_SHIFT]       = False;
  Flag[RTTY_REVERSE_AFSK]     = False;
  Flag[RTTY_ENABLE_DIVERSITY] = False;

  // Disable the common Guest modulation mode
  if( Transceiver[Indices.TRx_Index] != NULL )
    Transceiver[Indices.TRx_Index]->guest_modulation_mode = RX_MODE_ITEMS;

} // Rtty_Cleanup( void )

//------------------------------------------------------------------------

/* Rtty_Init_Config()
 *
 * Initializes runtime config variables
 */
  void
Rtty_Init_Config( void )
{
  /*  Total number of signal samples of RTTY character
   *  is 1 start bit + data bits + stop bit */
  rtty_rc.samples_per_char =
    LEVELS_PER_BIT +
    LEVELS_PER_BIT * DATABITS_PER_CHAR +
    (uint8_t)( LEVELS_PER_BIT * rtty_rc.stop_bit + 0.5 );

  // Length of Rx RTTY elements
  double temp;
  temp = SND_DSP_RATE / rtty_rc.baud_rate;
  rtty_rc.rx_rtty_bit4    = (uint16_t)( temp / LEVELS_PER_BIT );
  rtty_rc.rx_rtty_bit     = rtty_rc.rx_rtty_bit4 * 4;
  rtty_rc.rx_rtty_2bit    = rtty_rc.rx_rtty_bit  * 2;

  // Length of Tx RTTY elements
  temp = DUC_SAMPLE_RATE / rtty_rc.baud_rate;
  rtty_rc.tx_rtty_bit     = (uint32_t)temp;
  rtty_rc.tx_stop_bit_len = (uint32_t)( temp * rtty_rc.stop_bit );

  // Allocate transmit buffers
  uint32_t siz = rtty_rc.tx_rtty_bit + rtty_rc.tx_tone_dur;
  Alloc_Xmit_Buffers( siz );

  // Tx DAC Omega increments
  double w = M_2PI / DUC_SAMPLE_RATE;
  double mark_frq, space_frq;

  // Set up Demodulator Bandwidth and Weaver frequency (BFO tone)
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  uint16_t min_bw = (uint16_t)rtty_rc.baud_rate + rtty_rc.afsk_shift;
  if( min_bw <= 250 )
    Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_RTTYN );
  else if( min_bw <= 600 )
    Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_RTTYM );
  else if( min_bw <= 1000 )
    Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_RTTYW );
  rtty_rc.tone_freq = TRx->rx_weaver_frequency;

  // Puts the tone freq in middle of waterfall
  uint16_t idx = ( 2 * rtty_ifft_data.data_len ) / rtty_wfall.width;
  if( idx )
    rtty_ifft_data.ifft_stride = SND_DSP_RATE / rtty_rc.tone_freq / idx;

  // AFSK Rx mark and space frequencies
  if( Flag[RTTY_REVERSE_AFSK] )
  {
    rtty_rc.mark_freq  = rtty_rc.tone_freq - rtty_rc.afsk_shift / 2;
    rtty_rc.space_freq = rtty_rc.tone_freq + rtty_rc.afsk_shift / 2;
    space_frq = (double)( rtty_rc.afsk_shift / 2 );
    mark_frq  = -space_frq;
  }
  else
  {
    rtty_rc.mark_freq  = rtty_rc.tone_freq + rtty_rc.afsk_shift / 2;
    rtty_rc.space_freq = rtty_rc.tone_freq - rtty_rc.afsk_shift / 2;
    mark_frq  = (double)( rtty_rc.afsk_shift / 2 );
    space_frq = -mark_frq;
  }

  // AFSK Tx mark and space frequencies
  rtty_rc.mark_dW  = mark_frq  * w;
  rtty_rc.space_dW = space_frq * w;

  Rtty_Make_Cosine_Table();

} // Rtty_Init_Config()

//------------------------------------------------------------------

