/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "operation.h"
#include "common.h"
#include "callback_func.h"
#include "codec.h"
#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../common/guest_utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/modulate.h"
#include <ctype.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdint.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

// 'NULL' character used as 'last character' flag
#define RTTY_NULL_CHAR   0x00

#define TAG_STRING_SIZE    13
#define TAG_REPLACE_SIZE   22

// Index to macros
uint8_t rtty_macro_idx;

//------------------------------------------------------------------------

/* Rtty_Initialize_Transmit()
 *
 * Initializes the Transmit mode
 */
  BOOLEAN
Rtty_Initialize_Transmit( void )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Open qso record file
  if( !Open_Record_File("rtty") )
  {
    Flag[HERMES2_SEND_DUC_PACKET] = False;
    Flag[GUEST_TRANSMIT_KEYBD]    = False;
    Flag[GUEST_TRANSMIT_MACRO]    = False;
    Flag[GUEST_TRANSMIT_MODE]     = False;
    g_idle_add( Rtty_Set_TxRx_Labels, NULL );
    MOX_Control( MOX_OFF );
    return( False );
  }

  // Set transmit mode
  Flag[RSIDTX_ACTIVE] = False;

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(rtty_gui.window_builder, "rtty_tx_textview") );

  Flag[GUEST_TRANSMIT_MODE] = True;
  g_idle_add( Rtty_Set_TxRx_Labels, NULL );

  // This is just a default since the RSID modulator produces signal samples
  // centered on the Tx weaver frequency and hence centered on the dial frequency
  TRx->tx_weaver_frequency = TRx->rx_weaver_frequency;

  // These are needed by the SSB modulator
  TRx->tx_modulation_mode  = TX_MODE_USBN;
  TRx->tx_bandwidth        = 250;

  // Enable Transmit if needed
  MOX_Control( MOX_ON );
  Flag[HERMES2_SEND_DUC_PACKET] = True;

  // Send RSID tones or Hell signals
  if( Transceiver[Indices.TRx_Index]->tx_rsid_enable )
    Next_Modulator = DUC_Buffer_Transmit;
  else
    Modulator = DUC_Buffer_Transmit;

  // Transmit carrier and reversals, abort on error
  if( !Rtty_Transmit_Preamble() )
  {
    Flag[HERMES2_SEND_DUC_PACKET] = False;
    Flag[GUEST_TRANSMIT_KEYBD]    = False;
    Flag[GUEST_TRANSMIT_MACRO]    = False;
    Flag[GUEST_TRANSMIT_MODE]     = False;
    g_idle_add( Rtty_Set_TxRx_Labels, NULL );
    Error_Dialog( _("Failed to transmit Preamble"), HIDE_OK );
    return( False );
  }

  return( True );
} // Rtty_Initialize_Transmit()

//------------------------------------------------------------------------

/* Rtty_Transmit_Keybd()
 *
 * Transmits directly from the keyboard
 */
  void *
Rtty_Transmit_Keybd( void *data )
{
  // First call of function flag
  static BOOLEAN first_call = True;


  Flag[GUEST_TRANSMITTING] = True;

  // *** Initialize if needed ***
  if( first_call )
  {
    first_call = False;
    keybd_buf.char_pointer = 0;
    keybd_buf.key_count    = 0;
    tx_print_chr.printchr  = LF;
    Queue_Character( (gpointer) &tx_print_chr );
  } // if( first_call )

  // *** Cancel Keyborad transmit if disabled by user ***
  while( !Flag[GUEST_QUIT] && Flag[GUEST_TRANSMIT_KEYBD] )
  {
    // *** Process keystrokes from user ***
    if( keybd_buf.key_count != keybd_buf.char_pointer )
    {
      // Send CR and LF on Return key
      if( keybd_buf.key_buf[keybd_buf.char_pointer] == GDK_KEY_Return )
      {
        if( !Transmit_RTTY_Character(CR) ||
            !Transmit_RTTY_Character(LF) )
          break;

        // Increment and reset key buffer pointer
        keybd_buf.char_pointer++;
        if( keybd_buf.char_pointer >= KEY_BUF_SIZE )
          keybd_buf.char_pointer = 0;

        // Don't do anything else
        continue;
      }

      // Send BS on GDK_KEY_BackSpace
      if( keybd_buf.key_buf[keybd_buf.char_pointer] == GDK_KEY_BackSpace )
        keybd_buf.key_buf[keybd_buf.char_pointer] = BS;

      // Transmit Keyboard character and record
      if( !Transmit_RTTY_Character(keybd_buf.key_buf[keybd_buf.char_pointer]) )
        break;

      // Increment and Reset key buffer pointer
      keybd_buf.char_pointer++;
      if( keybd_buf.char_pointer >= KEY_BUF_SIZE )
        keybd_buf.char_pointer = 0;

    } // if( keybd_buf.key_count != keybd_buf.char_pointer )
    else
    {
      // Transmit CW identity
      if( Flag[GUEST_TRANSMIT_ID] )
      {
        Clear_DUC();
        snprintf( hermes2_rc.morse_mesg,
            sizeof(hermes2_rc.morse_mesg), " DE %s TU", op_data.call );
        Flag[HERMES2_SEND_DUC_PACKET] = False;
        Flag[TRANSMIT_MORSE_MESG]     = True;
        Modulator = Morse_Transmit;
        if( Flag[GUEST_TRANSMIT_MODE] ) sem_wait( &duc_send_semaphore );
        Flag[HERMES2_SEND_DUC_PACKET] = True;
        break;
      }
      else if( !Transmit_RTTY_Character('v') ) // Transmit didles
        break;
    }

  } // while( Flag[GUEST_TRANSMIT_KEYBD] ... )
  first_call = True;

  // Abort on user quit
  if( Flag[GUEST_QUIT] )
  {
    Flag[GUEST_TRANSMITTING]  = False;
    Flag[GUEST_TRANSMIT_MODE] = False;
    MOX_Control( MOX_OFF );
    g_idle_add( Rtty_Set_TxRx_Labels, NULL );
    return( NULL );
  }

  /* If no Macro selected, transmit
   * postamble and switch to receive */
  if( !Flag[GUEST_TRANSMIT_MACRO] )
  {
    if( !Flag[GUEST_TRANSMIT_ID] )
      if( !Rtty_Transmit_Postamble() )
      {
        Flag[GUEST_TRANSMITTING] = False;
        return( NULL );
      }

    Flag[GUEST_TRANSMIT_KEYBD] = False;
    Flag[GUEST_TRANSMIT_MODE]  = False;
    Flag[GUEST_TRANSMIT_ID]    = False;
    Flag[GUEST_RECEIVE_MODE]   = True;
    Flag[GUEST_TRANSMITTING]   = False;
    MOX_Control( MOX_OFF );

    if( !Pthread_Create(
          &hermes2_rc.guest_rx_thread, NULL, Rtty_Receive_Mode, NULL,
          _("Failed to create Receive thread")) )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      return( NULL );
    }

    g_idle_add( Rtty_Set_TxRx_Labels, NULL );
  } //if( !Flag[GUEST_TRANSMIT_MACRO] )
  else // Create a thread to Transmit Macros
  {
    if( !Pthread_Create(
          &hermes2_rc.guest_tx_thread, NULL, Rtty_Transmit_Macro, NULL,
          _("Failed to create Transmit_Macro thread")) )
    {
      Flag[GUEST_TRANSMIT_MODE]  = False;
      Flag[GUEST_TRANSMIT_MACRO] = False;
      return( NULL );
    }
  }

  Flag[GUEST_TRANSMITTING] = False;
  return( NULL );
} // End of Transmit_Keybd()

//------------------------------------------------------------------------

/* Rtty_Transmit_Macro()
 *
 * Transmits a pre-recorded Macro
 */
  void *
Rtty_Transmit_Macro( void *data )
{
  // Text buffer marker
  static uint8_t
    macro_cnt = 0, // Count macro characters tx'd
    tag_idx   = 0; // Index for transmitting tags

  // Tags for entering fields in macros
  const char *tags[NUM_OF_TAGS] = TAG_TABLE;

  // String within a tag <>
  char tag_string[TAG_STRING_SIZE];

  // Tag replacement string
  static char tag_replace[TAG_REPLACE_SIZE];


  Flag[GUEST_TRANSMITTING] = True;

  // Stop macro if user disabled, remain in Tx mode
  // Stop macro on '~' character, remain in Tx mode
  while( !Flag[GUEST_QUIT] && Flag[GUEST_TRANSMIT_MACRO] &&
        (rtty_macro[rtty_macro_idx][macro_cnt] != '~') )
  {
    /* Stop macro at string terminator
     * or CW ID char (*), go to Rx mode */
    if( (rtty_macro[rtty_macro_idx][macro_cnt] == '\0') ||
        (rtty_macro[rtty_macro_idx][macro_cnt] == '*') )
    {
      // Flush qso record file
      if( Flag[GUEST_RECORD_QSO] )
        fflush( qso_record.qso_record_fp );

      // Transmit postamble
      if( !Rtty_Transmit_Postamble() )
      {
        fflush( qso_record.qso_record_fp );
        Flag[GUEST_TRANSMIT_MACRO] = False;
        Flag[GUEST_TRANSMIT_MODE]  = False;
        g_idle_add( Rtty_Set_TxRx_Labels, NULL );
        Flag[GUEST_TRANSMITTING] = False;
        return( NULL );
      }

      // Transmit Morse ID if '*'
      if( rtty_macro[rtty_macro_idx][macro_cnt] == '*' )
      {
        Clear_DUC();
        snprintf( hermes2_rc.morse_mesg, sizeof(hermes2_rc.morse_mesg),
            " DE %s TU", op_data.call );
        Flag[HERMES2_SEND_DUC_PACKET] = False;
        Flag[TRANSMIT_MORSE_MESG]     = True;
        Modulator = Morse_Transmit;
        if( !Flag[GUEST_QUIT] && Flag[GUEST_TRANSMIT_MODE] )
          sem_wait( &duc_send_semaphore );
        Flag[HERMES2_SEND_DUC_PACKET] = True;
      }

      // Enable receive mode at end of macro or CW ID
      macro_cnt = tag_idx = 0;
      Flag[GUEST_TRANSMIT_MACRO] = False;
      Flag[GUEST_TRANSMIT_KEYBD] = False;
      Flag[GUEST_TRANSMIT_MODE]  = False;
      Flag[GUEST_TRANSMIT_TAG]   = False;
      Flag[GUEST_TRANSMITTING]   = False;
      MOX_Control( MOX_OFF );

      // Go to receive mode if no user quit
      if( !Flag[GUEST_QUIT] )
      {
        Flag[GUEST_RECEIVE_MODE] = True;
        if( !Pthread_Create(
              &hermes2_rc.guest_rx_thread, NULL, Rtty_Receive_Mode, NULL,
              _("Failed to create Receive thread")) )
        {
          Flag[GUEST_RECEIVE_MODE] = False;
          return( NULL );
        }
      }
      return( NULL );
    } // if( (rtty_macro[macro_idx][macro_cnt] == '\0') || ...

    // If Macro begins with a ^ transmit new line
    if( rtty_macro[rtty_macro_idx][macro_cnt] == '^' )
    {
      tx_print_chr.printchr = LF;
      Queue_Character( &tx_print_chr );
      macro_cnt++;
      if( !Transmit_RTTY_Character(CR) ||
          !Transmit_RTTY_Character(LF) )
      {
        Flag[GUEST_TRANSMITTING] = False;
        return( NULL );
      }
      continue;
    }

    // *** Look for tags and substitude accordingly ***
    if( (rtty_macro[rtty_macro_idx][macro_cnt] == '<') &&
        !Flag[GUEST_TRANSMIT_TAG] )
    {
      uint16_t idx;

      // *** Copy tag string to buffer and identify ***
      macro_cnt++; // Move into tag field
      Flag[GUEST_TRANSMIT_TAG] = True;

      // Copy line to tag, max 12 chars or EOS
      idx = 0; uint16_t siz = (uint16_t)sizeof( tag_string ) - 1;
      while( (idx < siz) && (rtty_macro[rtty_macro_idx][macro_cnt] != '>') )
      {
        tag_string[idx] = rtty_macro[rtty_macro_idx][macro_cnt];
        idx++;
        macro_cnt++;
      }

      // Terminate tag buffer and advance macro buffer
      tag_string[idx] = '\0';
      macro_cnt++;

      // Identify tag and substitude accordingly
      idx = 0;
      while( idx < NUM_OF_TAGS )
      {
        if( strcmp(tag_string, tags[idx]) == 0 )
          break;
        idx++;
      }

      // Abort if tag unidentified
      if( idx == NUM_OF_TAGS )
      {
        Error_Dialog(
            _("Error reading rtty.config file\n"\
              "A tag in a macro is invalid\n"\
              "Quit rtty and correct"), HIDE_OK);
        MOX_Control( MOX_OFF );
        Flag[GUEST_TRANSMITTING] = False;
        return( NULL );
      }

      // Replace tags
      siz = sizeof( tag_replace );
      switch( idx )
      {
        case 0: // own-call
          Strlcpy( tag_replace, op_data.call, siz );
          break;

        case 1: // own-name
          Strlcpy( tag_replace, op_data.name, siz );
          break;

        case 2: // own-qth
          Strlcpy( tag_replace, op_data.qth, siz );
          break;

        case 3: // own-loc
          Strlcpy( tag_replace, op_data.loc, siz );
          break;

        case 4: // own-rst
          Strlcpy( tag_replace, qso_record.my_rst, siz );
          break;

        case 5: // rem-call
          Strlcpy( tag_replace, qso_record.dx_call, siz );
          break;

        case 6: // rem-name
          Strlcpy( tag_replace, qso_record.dx_name, siz );
          break;

        case 7: // rem-qth
          Strlcpy( tag_replace, qso_record.dx_qth, siz );
          break;

        case 8: // rem-loc
          Strlcpy( tag_replace, qso_record.dx_loc, siz );
          break;

        case 9: // rem-rst
          Strlcpy( tag_replace, qso_record.dx_rst, siz );
          break;

        case 10: // date-time
          Strlcpy( tag_replace, qso_record.date, siz );
          Strlcat ( tag_replace, " ", siz );
          Strlcat( tag_replace, qso_record.time, siz );
          break;

        case 11: // op-freq, strip leading spaces
          Strlcpy( tag_replace, qso_record.freq, siz );
          break;

        case 12: // app-version
          snprintf( tag_replace, siz, "%s", PACKAGE_STRING );
          break;

      } // switch( idx )
    } // if( (rtty_macro[macro_idx][macro_cnt] == '<') && )

    // *** Transmit tag replacement ***
    if( Flag[GUEST_TRANSMIT_TAG] )
    {
      if( tag_replace[tag_idx] != '\0' )
      {
        // Capitalize letters
        tag_replace[tag_idx] = (char)toupper( tag_replace[tag_idx] );
        tx_print_chr.printchr = (guint)tag_replace[tag_idx];
        Queue_Character( &tx_print_chr );

        if( !Transmit_RTTY_Character((guint)tag_replace[tag_idx]) )
        {
          Flag[GUEST_TRANSMITTING] = False;
          return( NULL );
        }
        tag_idx++;
      } // if( (tag_replace[tag_idx] != '\0') )
      else
      {
        Flag[GUEST_TRANSMIT_TAG] = False;
        tag_idx = 0;
      }

      continue;
    } // if( Flag[GUEST_TRANSMIT_TAG) &&

    // Capitalize letters
    rtty_macro[rtty_macro_idx][macro_cnt] =
      (char)toupper( rtty_macro[rtty_macro_idx][macro_cnt] );
    tx_print_chr.printchr = (guint)rtty_macro[rtty_macro_idx][macro_cnt];
    Queue_Character( &tx_print_chr );

    // Transmit a char from Macro
    if( !Transmit_RTTY_Character((guint)rtty_macro[rtty_macro_idx][macro_cnt]) )
    {
      Flag[GUEST_TRANSMITTING] = False;
      return( NULL );
    }
    macro_cnt++;
  } // while( Flag[GUEST_TRANSMIT_MACRO] && (rtty_macro[macro_idx][macro_cnt] != '~')

  // Clean up some
  if( Flag[GUEST_RECORD_QSO] ) fflush( qso_record.qso_record_fp );
  macro_cnt = tag_idx = 0;
  Flag[GUEST_TRANSMIT_TAG]   = False;
  Flag[GUEST_TRANSMIT_MACRO] = False;

  // Abort on user quit
  if( Flag[GUEST_QUIT] )
  {
    MOX_Control( MOX_OFF );
    Flag[GUEST_TRANSMITTING] = False;
    Flag[GUEST_TRANSMIT_MODE]  = False;
    g_idle_add( Rtty_Set_TxRx_Labels, NULL );
    return( NULL );
  }

  /* Switch to Transmit from Keyboard mode
   * if no Receive mode selected */
  if( !Flag[GUEST_RECEIVE_MODE] )
  {
    if( Transmit_RTTY_Character(RTTY_NULL_CHAR) )
    {
      Flag[GUEST_TRANSMIT_KEYBD] = True;
      if( !Pthread_Create(
            &hermes2_rc.guest_tx_thread, NULL, Rtty_Transmit_Keybd, "from_macro",
            _("Failed to create Transmit_Keybd thread")) )
      {
        Flag[GUEST_TRANSMIT_KEYBD] = False;
        Flag[GUEST_TRANSMITTING]   = False;
        MOX_Control( MOX_OFF );
      }
    }

    Flag[GUEST_TRANSMITTING] = False;
    return( NULL );
  } // if( !Flag[GUEST_RECEIVE_MODE] )
  else
  {
    if( !Rtty_Transmit_Postamble() )
    {
      Flag[GUEST_TRANSMIT_MODE]    = False;
      Flag[GUEST_TRANSMITTING] = False;
      return( NULL );
    }

    Flag[GUEST_TRANSMIT_MODE] = False;
    Flag[GUEST_RECEIVE_MODE]  = True;
    Flag[GUEST_TRANSMITTING]  = False;
    MOX_Control( MOX_OFF );

    if( !Pthread_Create(
          &hermes2_rc.guest_rx_thread, NULL, Rtty_Receive_Mode, NULL,
          _("Failed to create Receive thread")) )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      Flag[GUEST_TRANSMITTING] = False;
      return( NULL );
    }
    g_idle_add( Rtty_Set_TxRx_Labels, NULL );
  }

  Flag[GUEST_TRANSMITTING] = False;
  return( NULL );
} // Rtty_Transmit_Macro()

//------------------------------------------------------------------------

/* Rtty_Receive_Mode()
 *
 * The receive thread of the RTTY client
 */
  void *
Rtty_Receive_Mode( void *data )
{
  // First call of function flag
  static BOOLEAN first_call = True;

  // A character decoded from RTTY signals
  uint8_t dec_chr;

  // Received char count
  static uint8_t chr_cnt = 0;

  Flag[GUEST_RECEIVING] = True;

  // *** Initialize ***
  if( first_call )
  {
    // Abort if QSO record file fails to open
    if( !Open_Record_File("rtty") )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      g_idle_add( Rtty_Set_TxRx_Labels, NULL );
      first_call = True;
      Flag[GUEST_RECEIVING] = False;
      return( NULL );
    }

    // Clear Rx window
    rx_print_chr.printchr = LF;
    Queue_Character( &rx_print_chr );

    // Reset char count on line feed
    chr_cnt = 0;
    g_idle_add( Rtty_Set_TxRx_Labels, NULL );
    first_call = False;
  } // if( first_call )

  // Cancel Receive if Rx flag cleared
  while( !Flag[GUEST_QUIT] && Flag[GUEST_RECEIVE_MODE] )
  {
    // *** Print characters to Receive window & file ***
    // Read a character, abort on dsp error
    if( !Decode_RTTY_Character(&dec_chr) )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      g_idle_add( Rtty_Set_TxRx_Labels, NULL );
      break;
    }

    if( dec_chr < NO_CHARACTER )
    {
      // Print character to Rx viewer
      rx_print_chr.printchr = (guint)dec_chr;
      Queue_Character( &rx_print_chr );

      // Reset char count on line feed
      if( dec_chr == LF )
        chr_cnt = 0;
      else
      {
        // Do word wrapping
        chr_cnt++;
        if( (chr_cnt >= rtty_rc.word_wrap) &&
            (dec_chr == ' ') )
        {
          rx_print_chr.printchr = LF;
          Queue_Character( &rx_print_chr );
          chr_cnt = 0;
        }
      } // else
    } // if( dec_chr <= NO_CHARACTER )

  } // while( Flag[GUEST_RECEIVE_MODE] )

  first_call = True;
  Flag[GUEST_RECEIVING]    = False;
  Flag[GUEST_RECEIVE_MODE] = False;
  g_idle_add( Rtty_Set_TxRx_Labels, NULL );
  return( NULL );

} // Rtty_Receive_Mode()

//------------------------------------------------------------------------

