/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "common.h"
#include "shared.h"
#include "../common/ifft.h"
#include "../common/guest_utils.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <math.h>
#include <stdint.h>

//------------------------------------------------------------------------

// Receive and Transmit status indicators
#define RECV_RTTY \
  _("<span background=\"green\" foreground=\"white\"> RECEIVE RTTY </span>")

#define XMIT_RTTY \
  _("<span background=\"red\" foreground=\"white\"> TRANSMIT RTTY </span>")

#define LTRS \
  _("<span background=\"darkgreen\" foreground=\"white\"> LTRS </span>")

#define FIGS \
  _("<span background=\"darkred\" foreground=\"white\"> FIGS </span>")

#define GRAPH_OUTLINE   0.0, 0.8, 0.0
#define SQUELCH_THRHLD  0.8, 0.0, 0.0

//------------------------------------------------------------------------

/* Rtty_Display_Waterfall()
 *
 * Displays audio spectrum as "waterfall" in scope2
 */
  void
Rtty_Display_Waterfall( pixbuffer_t *wfall, ifft_data_t *ifft_data )
{
  uint16_t
    idh, idv,   // Index to hor. and vert. position in warterfall
    idb,        // Index to average bin values array
    idf;        // Index to ifft output array

  // Constants needed to draw white lines in waterfall
  uint16_t mark_line, space_line, range;

  // Pointer to current pixel
  static guchar *pix;

  // Constants needed to draw white lines in waterfall
  range = rtty_rc.tone_freq * 2;
  mark_line  = (rtty_rc.mark_freq  * wfall->width) / range;
  space_line = (rtty_rc.space_freq * wfall->width) / range;

  // Draw a vertical white line in waterfall at detector's freq.
  pix = wfall->pixels + wfall->rowstride + wfall->n_channels;
  pix += wfall->n_channels * mark_line;
  pix[0] = pix[1] = pix[2] = 0xff;
  pix = wfall->pixels + wfall->rowstride + wfall->n_channels;
  pix += wfall->n_channels * space_line;
  pix[0] = pix[1] = pix[2] = 0xff;

  // Copy each line of waterfall to next one
  uint16_t temp = wfall->height - 2;
  for( idv = temp; idv > 0; idv-- )
  {
    pix = wfall->pixels + wfall->rowstride * idv + wfall->n_channels;

    for( idh = 0; idh < wfall->width; idh++ )
    {
      *pix = *( pix - wfall->rowstride ); pix++;
      *pix = *( pix - wfall->rowstride ); pix++;
      *pix = *( pix - wfall->rowstride );
      pix += wfall->n_channels - 2;
    }
  }

  // Got to top left of pixbuf
  pix = wfall->pixels;

  // First (DC) ifft bin not used
  ifft_data->bin_ave[0] = 0;

  // Do the iFFT on input array
  iFFT_Real( ifft_data );
  idb = 0;
  for( idf = 2; idf < ifft_data->data_len; idf += 2 )
  {
    /* Calculate vector magnitude of
     * signal at each frequency ("bin") */

    // Greyscale value of pixel derived from ifft o/p
    uint8_t pixel_val = iFFT_Bin_Value( ifft_data, idf, False );

    // Calculate average bin values
    ifft_data->bin_ave[idb++] = pixel_val;

    // Color code signal strength
    Colorize( pix, pixel_val );
    pix += wfall->n_channels;

  } // for( idf = 2; idf < ifft_data->data_len; idf += 2 )

  // Draw waterfall
  Queue_Draw( wfall->canvas );

  // Reset function
  iFFT_Bin_Value( ifft_data, 0, True );

} // Rtty_Display_Waterfall()

//------------------------------------------------------------------------

/* Rtty_Display_Synchro()
 *
 * Updates the Synchro display in scope1
 */
  void
Rtty_Display_Synchro( cairo_t *cr, uint8_t sync_pos, uint8_t sync_lev )
{
  // Factors needed to draw startbit position pointer
  static double
    width  = 0.0,
    height = 0.0,
    sync_height,
    sync_width,
    sync_y,
    sync_x,
    sync_r,
    scope_center_x,
    scope_center_y;

  double ind_x, ind_y;
  double w, sin_w, cos_w;

  // Initialize on first call or change of params
  if( (height < (double)rtty_wfall.height) ||
      (height > (double)rtty_wfall.height) ||
      (width  < (double)rtty_wfall.width)  ||
      (width  > (double)rtty_wfall.width) )
  {
    height = (double)rtty_wfall.height;
    width  = (double)rtty_wfall.width;
    sync_height = height - 16.0;
    sync_width  = round( width / 12.0 );
    sync_y = height - 8.0;
    sync_x = width  - sync_width - 8.0;
    sync_r = (sync_x - 64.0 ) / 2.0;
    scope_center_x = sync_r + 32.0;
    scope_center_y = height - scope_center_x;
  }

  // Fill scope backround black
  cairo_set_source_rgb( cr, 0.0, 0.0, 0.0 );
  cairo_rectangle( cr,
      0.0, 0.0,
      (double)rtty_wfall.width,
      (double)rtty_wfall.height );
  cairo_fill( cr );

  // Calculate co-ordinates for sync position plot
  w = M_2PI * (double)sync_pos /
    (double)(rtty_rc.samples_per_char * NUM_BUFFERED_CHARS);
  sin_w = sin( w);
  cos_w = cos( w );
  ind_x = sync_r * sin_w;
  ind_y = sync_r * cos_w;

  // Draw a green circle for the synchro scope
  cairo_set_source_rgb( cr, SCOPE_FOREGND );
  cairo_arc( cr,
      scope_center_x,
      scope_center_y,
      sync_r,
      0.0, M_2PI );

  // Draw the start bit position in the circular buffer
  cairo_move_to( cr, scope_center_x, scope_center_y );
  cairo_line_to( cr,
      scope_center_x + ind_x ,
      scope_center_y - ind_y );
  cairo_stroke( cr );

  // Show the squelch level in red
  if( sync_lev < rtty_rc.sqlch_thr )
    cairo_set_source_rgb( cr, SQUELCH_THRHLD );
  else
    cairo_set_source_rgb( cr, GRAPH_OUTLINE );

  // Create bar graph of sync level
  if( (double)sync_lev > sync_height )
    sync_lev = (uint8_t)sync_height;
  cairo_rectangle( cr,
      sync_x,
      sync_y - (double)sync_lev,
      sync_width,
      (double)sync_lev );
  cairo_fill( cr );

  // Create bar graph outline of sync level
  cairo_set_source_rgb( cr, GRAPH_OUTLINE );
  cairo_rectangle( cr,
      sync_x,
      8.0,
      sync_width,
      sync_height );
  cairo_stroke( cr );

} // Rtty_Display_Synchro()

//------------------------------------------------------------------------

/* Rtty_Display_Signal()
 *
 * Updates the signal scope display
 */
  void
Rtty_Display_Signal( cairo_t *cr, GdkPoint *points )
{
  // Draw scope backgrounds
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_rectangle(
      cr, 0.0, 0.0,
      (double)rtty_wfall.width,
      (double)rtty_wfall.height );
  cairo_fill( cr );

  // Plot signal graph
  cairo_set_source_rgb( cr, SCOPE_FOREGND );
  cairo_move_to( cr,
      (double)points[0].x,
      (double)points[0].y );
  for( uint16_t idx = 1; idx < rtty_wfall.width; idx++ )
    cairo_line_to( cr,
        (double)points[idx].x,
        (double)points[idx].y );

  // Stroke paths
  cairo_stroke( cr );

} // Rtty_Display_Signal( void )

//------------------------------------------------------------------------

/* Rtty_Set_TxRx_Labels()
 *
 * Sets up the labels in the Tx/Rx frames
 */
  gboolean
Rtty_Set_TxRx_Labels( gpointer data )
{
  if( Flag[GUEST_TRANSMIT_MODE] )
    gtk_label_set_markup( GTK_LABEL(rtty_gui.xmit_status), XMIT_RTTY );
  else
    gtk_label_set_markup( GTK_LABEL(rtty_gui.xmit_status), XMIT_OFF );

  if( Flag[GUEST_RECEIVE_MODE] )
    gtk_label_set_markup( GTK_LABEL(rtty_gui.rcve_status), RECV_RTTY );
  else
    gtk_label_set_markup( GTK_LABEL(rtty_gui.rcve_status), RECV_OFF );
  return( FALSE );
} // Rtty_Set_TxRx_Labels()

//------------------------------------------------------------------------

/* Rtty_Set_Shift_Label()
 *
 * Sets LTRS or FIGS to the Shift Status label
 */
  gboolean
Rtty_Set_Shift_Label( gpointer data )
{
  if( Flag[RTTY_LTRS_SHIFT] )
    gtk_label_set_markup( GTK_LABEL(rtty_gui.shift_label), LTRS );
  else
    gtk_label_set_markup( GTK_LABEL(rtty_gui.shift_label), FIGS );
  return( FALSE );
} // Rtty_Set_Shift_Label()

//------------------------------------------------------------------------

