/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "detect.h"
#include "display.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../Hermes2/sound.h"
#include <stddef.h>
#include <math.h>

// Mark and space signal scale factor
#define RTTY_SIGNAL_SCALE    900.0

//------------------------------------------------------------------------

// Circular signal samples buffer for level detector
static int16_t *samples_buff = NULL;

static uint16_t
  rtty_bit  = 0, // For initialization
  rtty_bit4 = 0,
  mark_freq = 0; // For initialization

/* Rtty_Afsk_Detect()
 *
 * Detects the level of mark and space AFSK frequencies
 */
  uint8_t
Rtty_Afsk_Detect( uint8_t *mark_signal, uint8_t *space_signal )
{
  static uint16_t
    cnt = 0,    // Sound samples counter
    buff_idx;   // Sample buffer index

  // Variables for the Goertzel algorithm
  static double space_coeff;
  static double mark_coeff;
  double space_q1, space_q2;
  double mark_q1, mark_q2;
  double scale, sig;

  // Wait for sample buffers full
  static BOOLEAN wait = True;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;

  /* Initialize on change of mark/space frequencies */
  if( !rtty_rc.mark_freq || !rtty_rc.rx_rtty_bit )
    return( ABORT );
  if( mark_freq != rtty_rc.mark_freq )
  {
    mark_freq = rtty_rc.mark_freq;

    // Omega for the mark frequency
    double w;
    w = M_2PI * (double)rtty_rc.mark_freq / SND_DSP_RATE;
    double mark_cosw  = cos( w );
    mark_coeff = 2.0 * mark_cosw;

    // Omega for the space frequency
    w = M_2PI * (double)rtty_rc.space_freq / SND_DSP_RATE;
    double space_cosw  = cos( w );
    space_coeff = 2.0 * space_cosw;
  } // if( (mark_freq != rtty_rc.mark_freq ) || ... )

  // Initialize on change of baud rate
  if( rtty_bit != rtty_rc.rx_rtty_bit )
  {
    rtty_bit     = rtty_rc.rx_rtty_bit;
    rtty_bit4    = rtty_bit / 4;
    digi_buf_cnt = 0;

    // Allocate samples buffer for one RTTY bit, and clear
    static size_t mreq;
    mreq = sizeof(short) * (size_t)rtty_rc.rx_rtty_bit;
    Mem_Realloc( (void **) &samples_buff, mreq );
    memset( samples_buff, 0, mreq );
    buff_idx = 0;
    cnt      = 0;
  } // if( (mark_freq != rtty_rc.mark_freq ) || ... )

  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return( ABORT );

  /* Wait for digimode buffer to fill with audio samples from demodulator.
   * When this function is slow to respond due to processor loading, then
   * the sem_post() in Hermes2/detect.c is executed more than once, so the
   * wait function must be repeated according to the semaphore's value
   */
  if( wait )
  {
    int sval;
    sem_getvalue( &digimode_semaphore, &sval );
    for( uint8_t w = 0; w <= sval; w++ )
      sem_wait( &digimode_semaphore );
    wait = False;
  }

  // Save samples for a 1/4 of RTTY element (bit)
  for( ; cnt < rtty_bit4; cnt++ )
  {
    // Digimode buffer index
    static uint16_t digi_buf_idx = 0;

    // Get a audio sample from the digimodes buffer
    samples_buff[buff_idx] = digimode_buffer[digi_buf_cnt][digi_buf_idx];

    // Display waterfall when input buffer full
    if( !Flag[RTTY_ENABLE_SIGNAL] )
    {
      if( iFFT_Data(samples_buff[buff_idx], &rtty_ifft_data) )
      Rtty_Display_Waterfall( &rtty_wfall, &rtty_ifft_data );
    }

    // Increment/reset circular buffer's index
    buff_idx++;
    if( buff_idx >= rtty_rc.rx_rtty_bit )
      buff_idx = 0;

    // Wait for digimode buffer to be filled
    digi_buf_idx++;
    if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
    {
      digi_buf_idx = 0;

      // Increment current audio sample output buffer index
      digi_buf_cnt++;
      if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
        digi_buf_cnt = 0;

      if( digi_buf_cnt == digi_buf_input )
      {
        wait = True;
        return( REPEAT );
      }
    } // if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )

  } // for( idx = 0; idx < rtty_rc.rtty_bit4; idx++ )
  cnt = 0;

  /* Calculate signal level of black and white
   * tone frequencies using Goertzel algorithm */
  space_q1 = space_q2 = 0.0;
  mark_q1  = mark_q2  = 0.0;
  for( uint16_t idx = 0; idx < rtty_rc.rx_rtty_bit; idx++ )
  {
    double space_q0 =
      space_coeff * space_q1 - space_q2 + (double)samples_buff[buff_idx];
    space_q2 = space_q1;
    space_q1 = space_q0;

    double mark_q0 =
      mark_coeff * mark_q1 - mark_q2 + (double)samples_buff[buff_idx];
    mark_q2 = mark_q1;
    mark_q1 = mark_q0;

    // Increment/reset circular buffers' index
    buff_idx++;
    if( buff_idx >= rtty_rc.rx_rtty_bit )
      buff_idx = 0;

  } // for( idx = 0; idx < rtty_rc.rtty_bit; idx++ )

  // Magnitude of space tone scaled by tone freq
  scale = RTTY_SIGNAL_SCALE * rtty_rc.rx_rtty_bit;
  space_q1 /= scale;
  space_q2 /= scale;
  sig = space_q1 * space_q1 + space_q2 * space_q2 -
    space_q1 * space_q2 * space_coeff;
  if( sig > 255.0 ) sig = 255.0;
  *space_signal = (uint8_t)sig;

  // Magnitude of mark tone scaled by tone freq
  mark_q1 /= scale;
  mark_q2 /= scale;
  sig = mark_q1 * mark_q1 + mark_q2 * mark_q2 -
    mark_q1 * mark_q2 * mark_coeff;
  if( sig > 255.0 ) sig = 255.0;
  *mark_signal = (uint8_t)sig;

  return( READY );
} // End of Rtty_Afsk_Detect()

//------------------------------------------------------------------------

// Free resources
  void
Rtty_Free_Detect( void )
{
  Mem_Free((void **) &samples_buff );
  rtty_bit  = 0;
  rtty_bit4 = 0;
  mark_freq = 0;
}

//------------------------------------------------------------------------

