/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "codec.h"
#include "common.h"
#include "detect.h"
#include "display.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/modulate.h"
#include <cairo/cairo.h>
#include <math.h>
#include <semaphore.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

// Signal sliding average window size
#define SIGNAL_AVE_WINDOW   2

/* The offset between the start bit
 * detection and first RTTY data bit */
#define STARTBIT_CORRECTION     1

// Scale factor for syncro display
#define SYNCHRO_SCALE   8

// Scale factor for scope display
#define SCOPE_SCALE     2

/* Buffer size of RTTY decoder buffers. We need to
 * buffer 3 RTTY characters of total 32 signal samples
 * (1 start, 5 data, 2 stop bits) of 4 samples each */
#define DECODER_BUFF_SIZE   32 * NUM_BUFFERED_CHARS

// Definitions for RTTY to ASCII conversion
#define NUM_ITA2_CHARS      64
#define NUM_BAUDOT_CHARS    32
#define FIGS_SHIFT_IDX      27
#define LTRS_SHIFT_IDX      31

/* The characters in this table can be changed before compilation
 * if a different character code is needed. PLEASE NOTE that the
 * non-printable characters '|', '>', '<', '^', 'v', 'e', 'b' must
 * not be changed, otherwise rtty will not work properly */
#define ITA2_CODE \
{ \
  '|','E','>','A',' ','S','I','U','<','D','R','J','N','F','C','K',  \
  'T','Z','L','W','H','Y','P','Q','O','B','G','^','M','X','V','v',  \
  '|','3','>','-',' ','\'','8','7','<','e','4','b',',','!',':','(', \
  '5','+',')','2','#','6','0','1','9','?','&','^','.','/',';','v'   \
}

// Index to transmit buffer
static uint32_t xmit_buf_idx = 0;

static uint8_t
  sync_max_idx,
  sync_level   = 0,  // Sync pulse level
  startbit_idx = 0;  // Detected position of sync train in buffers

// cos lookup wavetable
static uint16_t *cosine_table = NULL;

/* The phase angle Omega for mark
 * and space signal generation */
static double omega = 0.0;

// Points to plot signals
static GdkPoint *points1 = NULL;
static GdkPoint *points2 = NULL;

enum
{
  MARK = 0,
  SPACE,
};

//------------------------------------------------------------------------

/* Decode_RTTY_Character()
 *
 * Decodes a baudot-coded character from the
 * sequence of mark/space signals in RTTY
 * signal, producing an ASCII equivalent
 */
  BOOLEAN
Decode_RTTY_Character( uint8_t *dec_char )
{
  static const uint8_t ita2_code[NUM_ITA2_CHARS] = ITA2_CODE;

  // Mark and space signal level
  uint8_t mark, space;

  /* Circular buffers for mark
   * space and average signal */
  static uint8_t
    mark_lev[DECODER_BUFF_SIZE],
    space_lev[DECODER_BUFF_SIZE];

  static int16_t
    dsp_diversity = 0,
    signal_ave[DECODER_BUFF_SIZE];

  /* Total number of signal samples per RTTY character.
   * Four samples of signal level are taken for each
   * element in an RTTY character: (1 start bit,
   * 5 data bits and 1 or 1.5 or 2 stop bits) */
  static uint8_t samples_per_stopbit;

  static uint8_t
    input_idx  = 0,   // Index to input signal samples buffers
    buffer_len = 0;   // Length in signal samples of signal buffers

  // Count of signal samples input to buffers
  static int8_t samples_cnt = 0;

  uint8_t height = (uint8_t)rtty_wfall.height;

  // Initialize on first call
  if( points1 == NULL )
    Mem_Alloc((void **) &points1, sizeof(GdkPoint) * (size_t)rtty_wfall.width);
  if( points2 == NULL )
    Mem_Alloc((void **) &points2, sizeof(GdkPoint) * (size_t)rtty_wfall.width);

  // Initialize on first call
  static double stop_bit = 0.0;
  if( (stop_bit < rtty_rc.stop_bit) ||
      (stop_bit > rtty_rc.stop_bit) )
  {
    stop_bit = rtty_rc.stop_bit;

    // Samples per stop bit to look for sync train
    samples_per_stopbit =
      (uint8_t)( LEVELS_PER_BIT * stop_bit + 0.5 );

    // Length of samples buffers, for 3 RTTY characters
    buffer_len = NUM_BUFFERED_CHARS * rtty_rc.samples_per_char;

    // Wait for 3 characters to be input to buffers
    samples_cnt = (int8_t)( -(2 * rtty_rc.samples_per_char) );

    for( int idx = 0; idx < DECODER_BUFF_SIZE; idx++ )
      signal_ave[idx] = 0;
  } // if( stop_bit != rc_data.stop_bit )

  /* Take a signal sample over 1/4 of RTTY element.
   * If REPEAT is returned, wait for Rtty_Afsk_Detect()
   * to receive signal samples from the demodulator */
  uint8_t ret = Rtty_Afsk_Detect( &mark, &space );
  if( ret == REPEAT )
  {
    ret = Rtty_Afsk_Detect( &mark, &space );
    if( ret == ABORT ) return( False );
  }
  else if( ret == ABORT ) return( False );

  // Display mark and space signals
  if( Flag[RTTY_ENABLE_SIGNAL] )
  {
    // Needed to plot signals
    static uint16_t points_idx = 0;
    uint8_t plot;

    // Save mark values to be plotted
    plot = mark / SCOPE_SCALE;
    if( plot > height ) plot = height;
    points1[points_idx].y = height - (gint)plot - 1;
    points1[points_idx].x = points_idx;

    // Save space values to be plotted
    plot = space / SCOPE_SCALE;
    if( plot > height ) plot = height;
    points2[points_idx].y = height - (gint)plot - 1;
    points2[points_idx].x = points_idx;

    // Recycle buffer idx when full and plot
    points_idx++;
    if( points_idx >= rtty_wfall.width )
    {
      Queue_Draw( rtty_wfall.canvas );
      Queue_Draw( rtty_gui.rtty_scope );
      points_idx = 0;
    }
  } // if( Flag[RTTY_ENABLE_SIGNAL] )

  // Save samples in ring buffers
  mark_lev[input_idx]  = mark;
  space_lev[input_idx] = space;

  /* Keep sliding average of RTTY signal samples.
   * Space is considered a negative signal here */
  signal_ave[input_idx] *= ( SIGNAL_AVE_WINDOW - 1 );
  signal_ave[input_idx] += (int16_t)mark - (int16_t)space - dsp_diversity;
  signal_ave[input_idx] /= SIGNAL_AVE_WINDOW;

  // Increment buffer index and cycle
  input_idx++;
  if( input_idx >= buffer_len ) input_idx = 0;

  /* For every samples_per_bit total samples read in from
   * the detector, try to decode an RTTY character using
   * the start bit detected previously. Then look for a
   * start bit, to be used with the next samples_per_bit
   * total read in. This to make sure that samples for a
   * complete RTTY character train have been read */
  if( samples_cnt >= rtty_rc.samples_per_char )
  {
    /* Variables used by the natched filter that identifies
     * the pattern of signal samples in the (mark - space)
     * buffer which best matches the pattern of the sync train */
    uint8_t
      sync_filter_sum_idx, // Index used by matched filter to sum sync train
      sync_filter_cnt;     // Counts the number of buffer locations used above

    int32_t sync_filter_max; // Maximum signal level found by sync train filter

    // Look for a start bit in the samples train
    /* Find the highest average signal level summed over the
     * sync train length. The start bit is negated as its space */
    sync_filter_max = -10000;
    for( sync_filter_cnt = 0;
        sync_filter_cnt < rtty_rc.samples_per_char;
        sync_filter_cnt++ )
    {
      int32_t sync_filter_lev;  // Sync train level sumed up by matched filter
      uint8_t sync_filter_idx;  // Index to samples buffer for sync train filter

      // Index to start of search for sync train
      static uint8_t search_idx = 0;

      // Setup variables for the sync train filter
      sync_filter_lev = 0;
      sync_filter_idx = search_idx;

      // Summate signal averages over the length of stop bit
      for( sync_filter_sum_idx = 0;
          sync_filter_sum_idx < samples_per_stopbit;
          sync_filter_sum_idx++ )
      {
        if( sync_filter_idx == startbit_idx )
          sync_filter_lev -= 10000;
        else
          sync_filter_lev += signal_ave[sync_filter_idx];
        sync_filter_idx++;
        if( sync_filter_idx >= buffer_len ) sync_filter_idx = 0;
      }

      // Summate signal averages over the length of start bit
      for( sync_filter_sum_idx = 0;
          sync_filter_sum_idx < LEVELS_PER_BIT;
          sync_filter_sum_idx++ )
      {
        if( sync_filter_idx == startbit_idx )
          sync_filter_lev -= 10000;
        else
          sync_filter_lev -= signal_ave[sync_filter_idx];
        sync_filter_idx++;
        if( sync_filter_idx >= buffer_len ) sync_filter_idx = 0;
      }

      // Save max summation value and index
      if( sync_filter_max < sync_filter_lev )
      {
        sync_filter_max = sync_filter_lev;
        sync_max_idx    = sync_filter_idx;
      }

      // Reset search index to next signal sample average
      search_idx++;
      if( search_idx >= buffer_len ) search_idx = 0;
    } // sync_filter_cnt = 0; sync_filter_cnt < rc_data.samples_per_char; ...
    startbit_idx = sync_max_idx;

    /* Look ahead over the samples of a character to find
     * the unbalance between mark and space signals due
     * to selective fading of mark and space frequencies */
    if( Flag[RTTY_ENABLE_DIVERSITY] )
    {
      uint8_t
        look_ahead_idx,
        look_ahead_cnt,
        look_ahead_mark_max,
        look_ahead_space_max;

      look_ahead_mark_max  = 0;
      look_ahead_space_max = 0;
      look_ahead_idx = startbit_idx + STARTBIT_CORRECTION;
      if( look_ahead_idx >= buffer_len )
        look_ahead_idx = 0;

      for( look_ahead_cnt = 0;
          look_ahead_cnt < rtty_rc.samples_per_char;
          look_ahead_cnt++ )
      {
        // Find the max values of mark and space samples
        if( look_ahead_mark_max < mark_lev[look_ahead_idx] )
          look_ahead_mark_max = mark_lev[look_ahead_idx];
        if( look_ahead_space_max < space_lev[look_ahead_idx] )
          look_ahead_space_max = space_lev[look_ahead_idx];

        look_ahead_idx++;
        if( look_ahead_idx >= buffer_len )
          look_ahead_idx = 0;
      }
      dsp_diversity = ( look_ahead_mark_max - look_ahead_space_max ) / 4;
    }
    else dsp_diversity = 0;

    /* Variables used by the natched filter that identifies
     * the RTTY character code which best matches the pattern
     * of signal samples in the mark/space signal buffers */
    int16_t char_filter_max; // The highest signal level found by the filter

    uint8_t
      char_filter_max_idx,  // ITA2 character index of highest signal sum
      char_filter_ita2_idx, // Buffers index used by matched filters
      char_filter_sum_idx;  //Count number of RTTY data bits used by filter

    // Saves the last RTTY character received
    static uint8_t last_char = 0;

    /* Find which matching filter produces a maximum
     * signal level summation. This is done when the
     * circular buffers have signal level samples for
     * a complete RTTY character, after the position
     * of the start bit */
    char_filter_max = -10000;
    char_filter_max_idx = 0;
    for( char_filter_ita2_idx = 0;
        char_filter_ita2_idx < NUM_BAUDOT_CHARS;
        char_filter_ita2_idx++ )
    {
      uint8_t char_filter_idx;  // Buffers index for RTTY character matched filter
      int16_t char_filter_lev;  // Signal level produced by matched filters

      /* Start summation of levels one RTTY bit after start bit.
       * The start bit detection algorithm puts the start bit
       * position about 1-2 signal samples before the first data bit
       * so a correction is added to take us to the first data bit */
      char_filter_idx = startbit_idx + STARTBIT_CORRECTION;
      if( char_filter_idx >= buffer_len ) char_filter_idx -= buffer_len;
      char_filter_lev = 0;
      for( char_filter_sum_idx = 0;
          char_filter_sum_idx < DATABITS_PER_CHAR;
          char_filter_sum_idx++ )
      {
        if( char_filter_ita2_idx & (1 << char_filter_sum_idx) )
          char_filter_lev += -dsp_diversity +
            mark_lev[char_filter_idx] - space_lev[char_filter_idx];
        else
          char_filter_lev += dsp_diversity +
            space_lev[char_filter_idx] - mark_lev[char_filter_idx];

        char_filter_idx += LEVELS_PER_BIT;
        if( char_filter_idx >= buffer_len ) char_filter_idx -= buffer_len;
      }

      if( char_filter_max <= char_filter_lev )
      {
        char_filter_max     = char_filter_lev;
        char_filter_max_idx = char_filter_ita2_idx;
      }

    } // char_filter_ita2_idx = 0; char_filter_ita2_idx < NUM_BAUDOT_CHARS; ...

    // Go to FIGS shift characters
    if( !Flag[RTTY_LTRS_SHIFT] )
      char_filter_max_idx += NUM_BAUDOT_CHARS;

    // Process decoded character
    *dec_char = ita2_code[char_filter_max_idx];

    switch( *dec_char )
    {
      case 'v': // LTRS shift
        Flag[RTTY_LTRS_SHIFT] = True;
        g_idle_add( Rtty_Set_Shift_Label, NULL );
        *dec_char = NO_CHARACTER;
        break;

      case '^': // FIGS shift
        Flag[RTTY_LTRS_SHIFT] = False;
        g_idle_add( Rtty_Set_Shift_Label, NULL );
        *dec_char = NO_CHARACTER;
        break;

      case 'e': // ENQ character
        *dec_char = NO_CHARACTER;
        break;

      case '>': // LF
        if( last_char == LF )
          *dec_char = NO_CHARACTER;
        else
          *dec_char = LF;
        break;

      case '<': // CR
        *dec_char = NO_CHARACTER;
        break;

      case 'b': // BEL character
        *dec_char = NO_CHARACTER;
        break;

      case '|': // NUL character
        *dec_char = NO_CHARACTER;
        break;

      case ' ': // Unshift on space
        if( Flag[RTTY_ENABLE_USOS] )
        {
          Flag[RTTY_LTRS_SHIFT] = True;
          g_idle_add( Rtty_Set_Shift_Label, NULL );
        }
        break;
    } // switch( *dec_char )
    last_char = *dec_char;

    // Display sync train status
    sync_level = (uint8_t)( sync_filter_max / SYNCHRO_SCALE );
    if( !Flag[RTTY_ENABLE_SIGNAL] )
      Queue_Draw( rtty_gui.rtty_scope );

    // Reset samples input counter
    samples_cnt = 0;

  } // if( samples_cnt >= rtty_rc.samples_per_char )
  else *dec_char = NO_CHARACTER;

  // Keep count of element samples stored
  samples_cnt++;

  // Don't print if signal level is below squelch
  if( (double)sync_level < rtty_rc.sqlch_thr )
    *dec_char = NO_CHARACTER;

  return( True );
} // Decode_RTTY_Character()

//------------------------------------------------------------------------

/* Rtty_Display_Scope1()
 *
 * Handles the draw signal for the upper scope (scope1)
 */
  void
Rtty_Display_Scope1( cairo_t *cr )
{
  if( Flag[RTTY_ENABLE_SIGNAL] )
    Rtty_Display_Signal( cr, points1 );
  else
    Rtty_Display_Synchro( cr, startbit_idx, sync_level );

} // Rtty_Display_Scope1()

//------------------------------------------------------------------------

/* Rtty_Display_Scope2()
 *
 * Handles the draw signal for the lower scope (scope2)
 */
  void
Rtty_Display_Scope2( cairo_t *cr )
{

  if( Flag[RTTY_ENABLE_SIGNAL] )
    Rtty_Display_Signal( cr, points2 );
  else if( rtty_wfall.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, rtty_wfall.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
  }

} // Rtty_Display_Scope2()

//------------------------------------------------------------------------

/* Phase_Omega()
 *
 * Provides the next value of sin(omega) as a double
 */
  static void
Phase_Omega( double *sine, double *cosine, uint8_t which )
{
  double dW = 0.0;

  switch( which )
  {
    case MARK:
      dW = rtty_rc.mark_dW;
      break;

    case SPACE:
      dW = rtty_rc.space_dW;
      break;
  }

  *sine   = sin( omega );
  *cosine = cos( omega );
  omega  += dW;
  CLAMP_2PI( omega );

} // Phase_Omega()

//------------------------------------------------------------------------

/* Cosine_Rising_Edge()
 *
 * Fills the Xmit buffer with a raised cosine rising edge
 */
  static void
Cosine_Rising_Edge( uint8_t which )
{
  // Index to cosine lookup table
  uint16_t cos_idx;

  for( cos_idx = 0; cos_idx < rtty_rc.tx_rtty_bit; cos_idx++ )
  {
    double sine, cosine, temp;
    Phase_Omega( &sine, &cosine, which );

    temp = sine * (double)(cosine_table[cos_idx]);
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)temp;

    temp = cosine * (double)(cosine_table[cos_idx]);
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)temp;

    xmit_buf_idx++;
  }

  xmit_buffer.xmit_buf_len += rtty_rc.tx_rtty_bit;
} // Cosine_Rising_Edge()

//------------------------------------------------------------------------

/* Cosine_Falling_Edge()
 *
 * Fills the Xmit buffer with a raised cosine falling edge
 */
  static void
Cosine_Falling_Edge( uint8_t which )
{
  // Index to cosine lookup table
  int16_t cos_idx;
  for( cos_idx = (int16_t)rtty_rc.tx_rtty_bit - 1; cos_idx >= 0; cos_idx-- )
  {
    double sine, cosine, temp;
    Phase_Omega( &sine, &cosine, which );
    temp = sine * (double)(cosine_table[cos_idx]);
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)temp;

    temp = cosine * (double)(cosine_table[cos_idx]);
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)temp;
    xmit_buf_idx++;
  }

  xmit_buffer.xmit_buf_len += rtty_rc.tx_rtty_bit;
} // Cosine_Falling_Edge()

//------------------------------------------------------------------------

/* Steady_Tone()
 *
 * Fills Xmit buffer with steady tone
 */
  static void
Steady_Tone( uint32_t tone_dur, uint8_t which )
{
  // Index to cosine lookup table
  uint32_t tone_idx;
  for( tone_idx = 0; tone_idx < tone_dur; tone_idx++ )
  {
    double sine, cosine, temp;
    Phase_Omega( &sine, &cosine, which );

    temp = sine * DUC_SAMPLE_MAX;
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)temp;

    temp = cosine * DUC_SAMPLE_MAX;
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)temp;

    xmit_buf_idx++;
  }

  xmit_buffer.xmit_buf_len += tone_dur;
} // Steady_Tone()

//------------------------------------------------------------------------

/* Rtty_Transmit_Preamble()
 *
 * Transmits a steady tone and then
 * diddles as the RTTY pre-amble
 */
  BOOLEAN
Rtty_Transmit_Preamble( void )
{
  Flag[RTTY_LTRS_SHIFT] = True;
  g_idle_add( Rtty_Set_Shift_Label, NULL );

  // Initialize buffer index
  xmit_buf_idx = 0;
  xmit_buffer.xmit_buf_len = 0;

  // Fill buffer with cosine rising edge
  Cosine_Rising_Edge( MARK );

  // Fill buffer with the steady tone samples
  Steady_Tone( rtty_rc.tx_tone_dur, MARK );

  // *** Write buffer to DUC, abort on error ***
  sem_wait( &duc_send_semaphore );
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  // Initialize buffer index
  xmit_buf_idx = 0;
  xmit_buffer.xmit_buf_len = 0;

  // Transmit some diddles (LTRS Shift)
  if( !Transmit_RTTY_Character('v') ||
      !Transmit_RTTY_Character('v') )
    return( False );

  return( True );
} // Rtty_Transmit_Preamble()

//------------------------------------------------------------------------

/* Rtty_Transmit_Postamble()
 *
 * Transmits a steady tone and then a
 * cosine falling edge for the postamble
 */
  BOOLEAN
Rtty_Transmit_Postamble( void )
{
  // Initialize buffer index
  xmit_buf_idx = 0;
  xmit_buffer.xmit_buf_len = 0;

  // Fill buffer with the steady tone samples
  Steady_Tone( rtty_rc.tx_tone_dur, MARK );

  // Fill buffer with cosine falling edge
  Cosine_Falling_Edge( MARK );

  // *** Write buffer to DUC, abort on error ***
  sem_wait( &duc_send_semaphore );
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  // Clear Transmit buffers
  for( uint32_t idx = 0; idx < rtty_rc.tx_tone_dur; idx++ )
  {
    xmit_buffer.xmit_buf_i[idx] = 0;
    xmit_buffer.xmit_buf_q[idx] = 0;
  }
  xmit_buffer.xmit_buf_len = rtty_rc.tx_tone_dur;

  // *** Write buffer to DUC, abort on error ***
  sem_wait( &duc_send_semaphore );
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  return( True );
} // Rtty_Transmit_Postamble()

//------------------------------------------------------------------------

/* Rtty_Transmit_Element()
 *
 * Transmits an RTTY element (bit)
 */
  static BOOLEAN
Rtty_Transmit_Element( uint8_t mark )
{
  uint16_t idx;
  double sine, cosine, temp;

  // Send a mark tone
  switch( mark )
  {
    case MARK:
      // Fill buffer with samples at mark tone rate
      for( idx = 0; idx < rtty_rc.tx_rtty_bit; idx++ )
      {
        Phase_Omega( &sine, &cosine, MARK );

        temp = sine * DUC_SAMPLE_MAX;
        xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)temp;

        temp = cosine * DUC_SAMPLE_MAX;
        xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)temp;

        xmit_buf_idx++;
      }
      break;

    case SPACE:
      // Fill buffer with samples at space tone rate
      for( idx = 0; idx < rtty_rc.tx_rtty_bit; idx++ )
      {
        Phase_Omega( &sine, &cosine, SPACE );

        temp = sine * DUC_SAMPLE_MAX;
        xmit_buffer.xmit_buf_i[xmit_buf_idx] = (int16_t)temp;

        temp = cosine * DUC_SAMPLE_MAX;
        xmit_buffer.xmit_buf_q[xmit_buf_idx] = (int16_t)temp;

        xmit_buf_idx++;
      }
  } // switch( mark )

  xmit_buffer.xmit_buf_len += rtty_rc.tx_rtty_bit;
  return( True );
} // Rtty_Transmit_Element()

//------------------------------------------------------------------------

/* Rtty_Character()
 *
 * Prepares an RTTY character in the Xmit buffer
 */
  static BOOLEAN
Rtty_Character( uint32_t index )
{
  // Add a start bit
  Rtty_Transmit_Element( SPACE );

  // Send the 5 RTTY character bits
  for( uint8_t idx = 0; idx < 5; idx++ )
  {
    if( index & 0x01 )
      Rtty_Transmit_Element( MARK );
    else
      Rtty_Transmit_Element( SPACE );
    index >>= 1;
  }

  // Add a stop bit
  Steady_Tone( rtty_rc.tx_stop_bit_len, MARK );

  // *** Write buffer to DUC, abort on error ***
  sem_wait( &duc_send_semaphore );
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  // Initialize buffer idx
  xmit_buf_idx = 0;
  xmit_buffer.xmit_buf_len = 0;

  return( True );
} // Rtty_Character()

//------------------------------------------------------------------------

/* Transmit_RTTY_Character()
 *
 * Handles the transmission of characters in RTTY code
 */
  BOOLEAN
Transmit_RTTY_Character( uint32_t ascii_chr )
{
  const uint8_t ita2_code[NUM_ITA2_CHARS] = ITA2_CODE;
  uint8_t ita2_idx;

  // If character is idle (diddle) character
  if( ascii_chr == 'v' )
  {
    // Transmit Morse code CW ID
    if( Flag[GUEST_TRANSMIT_ID] )
    {
      Flag[GUEST_TRANSMIT_ID] = False;

      // Initialize buffer idx
      xmit_buf_idx = 0;

      // Transmit postamble
      Rtty_Transmit_Postamble();

      // Send Morse code ID
      Flag[HERMES2_SEND_DUC_PACKET] = False;
      snprintf( hermes2_rc.morse_mesg,
          sizeof(hermes2_rc.morse_mesg), " DE %s TU", op_data.call );
      Flag[TRANSMIT_MORSE_MESG] = True;
      Modulator = Morse_Transmit;
      sem_wait( &duc_send_semaphore );

      // Restore DUC Send modulator
      Modulator = DUC_Buffer_Transmit;
      Flag[HERMES2_SEND_DUC_PACKET] = True;

      // Initialize buffer idx
      xmit_buf_idx = 0;

      // Transmit preamble
      Rtty_Transmit_Preamble();

      return( True );
    } // if( Flag[GUEST_TRANSMIT_ID] )
  } // if( ascii_chr == 'v' )

  // CR and LF must be converted to ita2 code
  if( ascii_chr == CR ) ascii_chr = '<';
  if( ascii_chr == LF ) ascii_chr = '>';

  // Find the ascii character in the ITA2 array
  for( ita2_idx = 0; ita2_idx < NUM_ITA2_CHARS; ita2_idx++ )
    if( ita2_code[ita2_idx] == ascii_chr )
      break;

  // Ignore non-ITA2 characters
  if( ita2_idx >= NUM_ITA2_CHARS )
    return( True );

  /* Do not toggle LTRS/FIGS Shift for common
   * characters (NUL, LF, CR, FIGS, LTRS) */
  if( (ascii_chr != 0x00) &&
      (ascii_chr != '>' ) &&
      (ascii_chr != '<' ) &&
      (ascii_chr != ' ' ) &&
      (ascii_chr != '^' ) &&
      (ascii_chr != 'v' ) )
  {
    // Reduce the ita2 index to < 32 and set FIGS shift
    if( ita2_idx >= NUM_BAUDOT_CHARS )
    {
      ita2_idx -= NUM_BAUDOT_CHARS;

      // Set FIGS shift if needed
      if( Flag[RTTY_LTRS_SHIFT] )
      {
        Flag[RTTY_LTRS_SHIFT] = False;
        g_idle_add( Rtty_Set_Shift_Label, NULL );
        if( !Rtty_Character(FIGS_SHIFT_IDX) ||
            !Rtty_Character(FIGS_SHIFT_IDX) )
          return( False );
      }
    } // if( ita2_idx >= NUM_BAUDOT_CHARS )
    else
    {
      // Set LTRS shift if needed
      if( !Flag[RTTY_LTRS_SHIFT] )
      {
        Flag[RTTY_LTRS_SHIFT] = True;
        g_idle_add( Rtty_Set_Shift_Label, NULL );
        if( !Rtty_Character(LTRS_SHIFT_IDX) ||
            !Rtty_Character(LTRS_SHIFT_IDX) )
          return( False );
      }
    } // if( ita2_idx >= NUM_BAUDOT_CHARS )
  } // if( (ascii_chr != 0x00) && ...

  // Prepare RTTY character
  Rtty_Character( ita2_idx );

  return( True );
} // Transmit_RTTY_Character()

//------------------------------------------------------------------------

/* Rtty_Make_Cosine_Table()
 *
 * Builds a raised cosine wavetable for tone generation
 */
  BOOLEAN
Rtty_Make_Cosine_Table( void )
{
  size_t siz = (size_t)rtty_rc.tx_rtty_bit * sizeof(uint16_t);
  Mem_Alloc( (void **) &cosine_table, siz );

  // This makes a raised cosine table
  double step = M_PI_2 / (double)rtty_rc.tx_rtty_bit;
  for( uint16_t idx = 0; idx < rtty_rc.tx_rtty_bit; idx++ )
    cosine_table[idx] = (uint16_t)( DUC_SAMPLE_MAX * sin(step * (double)idx) );

  return(True);
} // Rtty_Make_Cosine_Table()

//------------------------------------------------------------------------

// Free resources
  void
Rtty_Free_Codec( void )
{
  Mem_Free( (void **) &points1 );
  Mem_Free( (void **) &points2 );
  Mem_Free( (void **) &cosine_table );
}

//------------------------------------------------------------------------

