/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "callback_func.h"
#include "codec.h"
#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../common/guest_utils.h"
#include "../common/utils.h"
#include "../Hermes2/interface.h"
#include "../rsid/rsid_modes.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>

// For clearing text buffers
static GtkTextBuffer *text_buffer;

#define RTTY_SQUELCH_MULT  3.0

//------------------------------------------------------------------------

// All the callback functions of the RTTY client

  void
on_rtty_window_destroy(
    GObject  *object,
    gpointer  user_data)
{
  Rtty_Cleanup();
}

//------------------------------------------------------------------------

  gboolean
on_rtty_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data)
{
  Save_Window_Position( rtty_gui.window, rtty_rc.rc_fpath );
  Guest_Quit_Activate( rtty_gui.window );
  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_rtty_quit_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Save_Window_Position( rtty_gui.window, rtty_rc.rc_fpath );
  Guest_Quit_Activate( rtty_gui.window );
}

//------------------------------------------------------------------------

  gboolean
on_rtty_tx_textview_key_press(
    GtkWidget   *widget,
    const GdkEventKey *event,
    gpointer     user_data)
{
  if( Flag[GUEST_TRANSMIT_MODE] )
    return( Rtty_Tx_Textview_Key_Press(event) );
  else return( FALSE );
}

//------------------------------------------------------------------------

  gboolean
on_rtty_tx_textview_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  // Needed for 'Clear Window'
  text_buffer = rtty_gui.tx_text_buffer;
  return( Rtty_Textview_Button_Press(event) );
}

//------------------------------------------------------------------------

  gboolean
on_rtty_rx_textview_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  // Needed for 'Clear Window'
  text_buffer = rtty_gui.rx_text_buffer;
  return( Rtty_Textview_Button_Press(event) );
}

//------------------------------------------------------------------------

  void
on_rtty_transmit_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  if( !Transceiver[Indices.TRx_Index]->receive_active ||
      !ddv->tx_pa_enable )
    return;

  Rtty_Change_Modes();
}

//------------------------------------------------------------------------

  void
on_rtty_receive_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;
  Rtty_Receive_Clicked();
}

//------------------------------------------------------------------------

  void
on_rtty_squelch_value_changed(
    GtkRange *range,
    gpointer  user_data)
{
  GtkWidget *hscale;
  GtkAdjustment *adjustment;

  // Read and enter squelch level
  hscale = Builder_Get_Object( rtty_gui.window_builder, "rtty_squelch_value" );
  adjustment = gtk_range_get_adjustment( GTK_RANGE(hscale) );
  rtty_rc.sqlch_thr =
    (uint8_t)(gtk_adjustment_get_value(adjustment) * RTTY_SQUELCH_MULT);
}

//------------------------------------------------------------------------

  gboolean
on_rtty_scope1_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  Rtty_Display_Scope1( cr );
  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_rtty_scope2_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  Rtty_Display_Scope2( cr );
  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_rtty_scope2_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  if( Flag[GUEST_RECEIVING] )
    Rtty_Tune_to_Monitor(
        event, CLICK_TUNE_RANGE, rtty_wfall.width,
        rtty_rc.tone_freq, &rtty_ifft_data );
  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_rtty_scope_label_button_press(
    GtkWidget *widget,
    const GdkEventButton *event,
    gpointer   user_data)
{
  if( Flag[RTTY_ENABLE_SIGNAL] )
  {
    gtk_label_set_text( GTK_LABEL(rtty_gui.scope_label), _("Spectrum/Synchro") );
    Flag[RTTY_ENABLE_SIGNAL] = False;
  }
  else
  {
    gtk_label_set_text( GTK_LABEL(rtty_gui.scope_label), _("Mark/Space") );
    Flag[RTTY_ENABLE_SIGNAL] = True;
  }
}

//------------------------------------------------------------------------

  void
on_rtty_callsign_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Callsign_Changed( editable, &qso_record, "RTTY" );
}

//------------------------------------------------------------------------

  void
on_rtty_rst_out_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_RST( editable );
  Strlcpy( qso_record.dx_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)),
      sizeof(qso_record.dx_rst) );
}

//------------------------------------------------------------------------

  void
on_rtty_rst_in_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_RST( editable );
  Strlcpy( qso_record.my_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)),
      sizeof(qso_record.my_rst) );
}

//------------------------------------------------------------------------

  void
on_rtty_op_name_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field QSO record structure
  Strlcpy( qso_record.dx_name, buff, sizeof(qso_record.dx_name) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_name );
}

//------------------------------------------------------------------------

  void
on_rtty_qth_name_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field and QSO record structure
  Strlcpy( qso_record.dx_qth, buff, sizeof(qso_record.dx_qth) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_qth );
}

//------------------------------------------------------------------------

  void
on_rtty_locator_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Locator_Changed( editable, &qso_record, "RTTY" );
}

//------------------------------------------------------------------------

  void
on_rtty_band_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Band_Changed( editable, &qso_record, "RTTY" );
}

//------------------------------------------------------------------------

  void
on_rtty_new_record_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  New_Record( &qso_record, rtty_gui.window_builder, "RTTY" );
  Strlcpy( qso_record.mode_dir, "rtty", sizeof(qso_record.mode_dir) );
}

//------------------------------------------------------------------------

  void
on_rtty_save_record_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  // Save record if needed (and valid)
  if( Read_QSO_Record(&qso_record, rtty_gui.window_builder, "RTTY") )
    Guest_Save_Dialog( _("Save QSO Record to file?") );
  else
    Guest_Save_Dialog(
        _("QSO Record is incomplete\n"\
          "Do you still want to save?") );
}

//------------------------------------------------------------------------

  void
on_rtty_clear_fields_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Clear_Record_Fields( True, rtty_gui.window_builder, "RTTY" );
}

//------------------------------------------------------------------------

  void
on_rtty_clear_window_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  gtk_text_buffer_set_text( text_buffer, "", -1 );

  gtk_text_buffer_get_iter_at_offset(
      rtty_gui.tx_text_buffer, &rtty_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(rtty_gui.tx_text_buffer) );
  gtk_text_buffer_get_iter_at_offset(
      rtty_gui.rx_text_buffer, &rtty_gui.rx_text_buffer_iter,
      gtk_text_buffer_get_char_count(rtty_gui.rx_text_buffer) );

  gtk_widget_grab_focus(
      Builder_Get_Object(rtty_gui.window_builder, "rtty_tx_textview") );
}

//------------------------------------------------------------------------

  void
on_rtty_clear_windows_clicked(
    GtkButton       *button,
    gpointer         user_data)
{
  gtk_text_buffer_set_text( rtty_gui.tx_text_buffer, "", -1 );
  gtk_text_buffer_set_text( rtty_gui.rx_text_buffer, "", -1 );

  gtk_text_buffer_get_iter_at_offset(
      rtty_gui.tx_text_buffer, &rtty_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(rtty_gui.tx_text_buffer) );
  gtk_text_buffer_get_iter_at_offset(
      rtty_gui.rx_text_buffer, &rtty_gui.rx_text_buffer_iter,
      gtk_text_buffer_get_char_count(rtty_gui.rx_text_buffer) );

  gtk_widget_grab_focus(
      Builder_Get_Object(rtty_gui.window_builder, "rtty_tx_textview") );
}

//------------------------------------------------------------------------

  void
on_rtty_macro_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  if( ddv->tx_pa_enable ) Rtty_Select_Macro( button );
}

//------------------------------------------------------------------------

  void
on_rtty_bd45_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.baud_rate = 45.45;
    Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_RTTY_45;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_bd50_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.baud_rate = 50.0;
    Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_RTTY_50;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_bd75_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.baud_rate = 75.0;
    Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_RTTY_75;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_bd100_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.baud_rate = 100.0;
    Transceiver[Indices.TRx_Index]->RSID_Mode = NUM_MODES;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_afsk85_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.afsk_shift = 85;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_afsk170_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.afsk_shift = 170;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_afsk200_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.afsk_shift = 200;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_afsk250_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.afsk_shift = 250;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_afsk450_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.afsk_shift = 450;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_afsk850_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.afsk_shift = 850;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_stop10_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.stop_bit = 1.0;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_stop15_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.stop_bit = 1.5;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_stop20_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    rtty_rc.stop_bit = 2.0;
    Rtty_Init_Config();
  }
}

//------------------------------------------------------------------------

  void
on_rtty_usos_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[RTTY_ENABLE_USOS] = True;
  else
    Flag[RTTY_ENABLE_USOS] = False;
}

//------------------------------------------------------------------------

  void
on_rtty_record_qsos_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[GUEST_RECORD_QSO] = True;
  else
    Flag[GUEST_RECORD_QSO] = False;
}

//------------------------------------------------------------------------

  void
on_rtty_identify_in_cw_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Flag[GUEST_TRANSMIT_ID] = True;
}

//------------------------------------------------------------------------

  void
on_rtty_reverse_afsk_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[RTTY_REVERSE_AFSK] = True;
  else
    Flag[RTTY_REVERSE_AFSK] = False;
  Rtty_Init_Config();
}

//------------------------------------------------------------------------

  void
on_rtty_shift_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  if( Flag[RTTY_LTRS_SHIFT] )
    Flag[RTTY_LTRS_SHIFT] = False;
  else
    Flag[RTTY_LTRS_SHIFT] = True;
  g_idle_add( Rtty_Set_Shift_Label, NULL );
}

//------------------------------------------------------------------------

  void
on_rtty_enable_diversity_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[RTTY_ENABLE_DIVERSITY] = True;
  else
    Flag[RTTY_ENABLE_DIVERSITY] = False;
}

//------------------------------------------------------------------------

