/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callback_func.h"
#include "codec.h"
#include "display.h"
#include "operation.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/ifft.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../common/guest_utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/modulate.h"
#include "../hpsdr/settings.h"
#include <ctype.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>

//------------------------------------------------------------------------

/* Rtty_Change_Modes()
 *
 * Toggles down Macro TX and toggles
 * Keyboard Tx and Receive mode
 */
  void
Rtty_Change_Modes( void )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Toggle Tx/Rx/Macro modes
  if( Flag[GUEST_TRANSMIT_MACRO] )
  {
    Flag[GUEST_TRANSMIT_MACRO] = False;
  }
  else if( Flag[GUEST_TRANSMIT_KEYBD] )
  {
    Flag[GUEST_TRANSMIT_KEYBD]  = False;
  }
  else if( !ddv->transmit_on ) // MOX or Tune not on
  {
    // Set up control flags for Keyboard transmit
    Flag[GUEST_TRANSMIT_KEYBD] = True;

    // Wait for Rx mode to exit
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      pthread_join( hermes2_rc.guest_rx_thread, NULL );
    }

    // Clear keyboard buffer data
    keybd_buf.key_count    = 0;
    keybd_buf.char_pointer = 0;
    keybd_buf.space_count  = 0;
    keybd_buf.char_count   = 0;

    // Create a thread to transmit characters from keyboard
    if( !Rtty_Initialize_Transmit() ) return;
    if( !Pthread_Create(
          &hermes2_rc.guest_tx_thread, NULL, Rtty_Transmit_Keybd, NULL,
          _("Failed to create Transmit Keyboard thread")) )
      return;
  }
} // Rtty_Change_Modes()

//------------------------------------------------------------------------

/* Rtty_Select_Macro()
 *
 * Selects a macro to be transmitted
 */
  void
Rtty_Select_Macro( GtkButton *button )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;

  // Clear keyboard buffer data
  keybd_buf.key_count    = 0;
  keybd_buf.char_pointer = 0;
  keybd_buf.space_count  = 0;
  keybd_buf.char_count   = 0;

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(rtty_gui.window_builder, "rtty_tx_textview") );

  // Transmit a macro
  if( !Flag[GUEST_TRANSMIT_MACRO] )
  {
    // Wait for Rx mode to exit
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      pthread_join( hermes2_rc.guest_rx_thread, NULL );
    }

    // Enable macro transmission if we are not
    // in transmit mode (MOX on or Tune active)
    if( !ddv->transmit_on || Flag[GUEST_TRANSMIT_KEYBD] )
    {
      if( !Flag[GUEST_TRANSMIT_KEYBD] &&
          !Rtty_Initialize_Transmit() )
        return;

      Flag[GUEST_TRANSMIT_MACRO] = True;
      g_idle_add( Rtty_Set_TxRx_Labels, NULL );

      // Get macro index from the name of the macro button
      rtty_macro_idx =
        (uint8_t)( atoi(gtk_widget_get_name(GTK_WIDGET(button))) - 1 );

      // Create a thread to run Macro Transmit if not in Keyboard mode
      if( !Flag[GUEST_TRANSMIT_KEYBD] )
      {
        if( !Pthread_Create(
              &hermes2_rc.guest_tx_thread, NULL, Rtty_Transmit_Macro, NULL,
              _("Failed to create Transmit_Macro thread")) )
          return;
      }

      // Will start Macro thread from Keybd thread
      Flag[GUEST_TRANSMIT_KEYBD] = False;

    } // if( !ddv->transmit_on ... )
  } // if( !Flag[GUEST_TRANSMIT_MACRO] )
  else
  {
    // Exit Macro mode
    Flag[GUEST_TRANSMIT_MACRO] = False;
  }

} // Rtty_Select_Macro()

//------------------------------------------------------------------------

/* Rtty_Receive_Clicked()
 *
 * Handles the on_rtty_receive_clicked callback
 */
  void
Rtty_Receive_Clicked( void )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  if( Flag[GUEST_RECEIVING] ) // If in Rx mode, terminated it
  {
    Flag[GUEST_RECEIVE_MODE] = False;
    pthread_join( hermes2_rc.guest_rx_thread, NULL );
  }
  else // If not in Tx mode, start Rx mode
  {
    if( !Flag[GUEST_TRANSMIT_MACRO] &&
        !Flag[GUEST_TRANSMIT_KEYBD] &&
        !ddv->transmit_on )
    {
      Flag[GUEST_RECEIVE_MODE] = True;
      rx_print_chr.printchr = LF;
      if( !Pthread_Create(
            &hermes2_rc.guest_rx_thread, NULL, Rtty_Receive_Mode, NULL,
            _("Failed to create Receive thread")) )
      {
        Flag[GUEST_RECEIVE_MODE] = False;
        g_idle_add( Rtty_Set_TxRx_Labels, NULL );
        return;
      }
    }
    else // Wait for Tx mode to exit
    {
      Flag[GUEST_TRANSMIT_MACRO] = False;
      Flag[GUEST_TRANSMIT_KEYBD] = False;
      if( Flag[GUEST_TRANSMITTING] )
        pthread_join( hermes2_rc.guest_tx_thread, NULL );
    }
  } // else of if( Flag[GUEST_RECEIVE_MODE] )

  g_idle_add( Rtty_Set_TxRx_Labels, NULL );
} // Rtty_Receive_Clicked()

//------------------------------------------------------------------------

/* Rtty_Textview_Button_Press()
 *
 * Handles button press events on Rx and Tx textview
 */
  gboolean
Rtty_Textview_Button_Press( const GdkEventButton *event )
{
  switch( event->button )
  {
    case 1: // Focus to widget that received the button press
      return( FALSE );

    case 2: // Toggle Tx/Rx/Macro modes
      Rtty_Change_Modes();
      break;

    case 3: // Popup menu. "Sensitize Identify in CW" in keybord mode
      if( Flag[GUEST_TRANSMIT_KEYBD] )
        gtk_widget_set_sensitive(
            Builder_Get_Object( rtty_gui.popup_menu_builder, "rtty_identify_in_cw"), TRUE );
      else
        gtk_widget_set_sensitive(
            Builder_Get_Object( rtty_gui.popup_menu_builder, "rtty_identify_in_cw"), FALSE );

      gtk_menu_popup_at_pointer( GTK_MENU(rtty_gui.popup_menu), NULL );
  } // switch( event->button )

  return( TRUE );
} // Rtty_Textview_Button_Press()

//------------------------------------------------------------------------

/* Rtty_Tx_Textview_Key_Press()
 *
 * Handles keyboard input during transmit
 */
  gboolean
Rtty_Tx_Textview_Key_Press( const GdkEventKey *event )
{
  if( (event->keyval == GDK_KEY_Shift_L) ||
      (event->keyval == GDK_KEY_Shift_R) )
    return( FALSE );

  // Do nothing if not in keybd mode
  if( !Flag[GUEST_TRANSMIT_KEYBD] && !Flag[GUEST_RECEIVING] )
    return( FALSE );

  // Deal with Backspace key
  if( event->keyval == GDK_KEY_BackSpace )
  {
    // Roll back word wrap counter
    keybd_buf.char_count--;
    if( keybd_buf.char_count < 0 )
      keybd_buf.char_count += rtty_rc.word_wrap;

    // Enter Back Space code to key buffer
    keybd_buf.key_buf[keybd_buf.key_count] = BS;
    keybd_buf.key_count++;
    if( keybd_buf.key_count >= KEY_BUF_SIZE )
      keybd_buf.key_count = 0;

    // Print Backspace on text view
    tx_print_chr.printchr = GDK_KEY_BackSpace;
    Queue_Character( &tx_print_chr );

    return( TRUE );
  } // if( event->keyval == GDK_KEY_BackSpace )

  // Print Return on Return key or on Word Wrap
  keybd_buf.char_count++;
  if( (event->keyval == GDK_KEY_Return) ||
      ((keybd_buf.char_count >= rtty_rc.word_wrap) &&
       (event->keyval == ' ')) )
  {
    // For word wrap case add a GDK_KEY_Return entry to buffer
    keybd_buf.key_buf[keybd_buf.key_count] = GDK_KEY_Return;
    keybd_buf.key_count++;
    if( keybd_buf.key_count >= KEY_BUF_SIZE )
      keybd_buf.key_count = 0;

    // Print GDK_KEY_Return to text view
    tx_print_chr.printchr = GDK_KEY_Return;
    Queue_Character( &tx_print_chr );

    // Reset Word Wrap counter
    keybd_buf.char_count = 0;
    return( TRUE );
  }

  // Don't process non-ASCII characters
  if( (event->keyval & 0xFF) > 0x7F )
    return( TRUE );

  // Capitalize letters
  keybd_buf.key_buf[keybd_buf.key_count] = (guint)toupper( event->keyval );

  // Print character and increment keystroke count
  tx_print_chr.printchr = keybd_buf.key_buf[keybd_buf.key_count];
  Queue_Character( &tx_print_chr );
  keybd_buf.key_count++;
  if( keybd_buf.key_count >= KEY_BUF_SIZE )
    keybd_buf.key_count = 0;

  return( TRUE );
} // Rtty_Tx_Textview_Key_Press()

//------------------------------------------------------------------------

/* Rtty_Tune_to_Monitor
 *
 * Tunes the transceiver to the frequency of the strongest
 * signal near a mouse click in the waterfall window
 */
  void
Rtty_Tune_to_Monitor(
    const GdkEventButton *event,
    uint8_t tune_range,
    uint16_t wfall_width,
    uint16_t center_freq,
    ifft_data_t *ifft_data )
{
  uint16_t
    from, to,   // Range to scan for a max bin value
    bin_max,    // Max bin value found in this range
    max_idx,    // ifft idx at which the max is found
    ifft_idx;   // Idx used to search ifft bin values

  // Frequency correction
  int16_t
    freq_correction,
    freq1,
    freq3,
    audio_freq; // Audio frequency in waterfal window

  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  /* Calculate ifft index corresponding to pointer x
   * in waterfall window. This is +1 over the pointer
   * position since the first FFT bin is not used */
  ifft_idx = (uint16_t)( event->x + 1.5 );

  // Look above and below click point for max bin val
  if( ifft_idx < tune_range )
    from = 0;
  else
    from = ifft_idx - tune_range;
  to = ifft_idx + tune_range;
  if( to >= wfall_width ) to = wfall_width - 1;

  // Find max bin value around click point
  bin_max = 0;
  max_idx = ifft_idx;
  for( ifft_idx = from; ifft_idx < to; ifft_idx++ )
    if( bin_max < ifft_data->bin_ave[ifft_idx] )
    {
      bin_max = ifft_data->bin_ave[ifft_idx];
      max_idx = ifft_idx;
    }

  /* Audio frequency corresponding to ifft index. Since the
   * waterfall width has to be an odd number so that a center
   * line may be present, and since the FFT has to be a power
   * of 2 (e.g. even), the first (DC) output bin of the FFT is
   * not used and hence max_idx has to be increased by 1 and
   * the waterfall width also increased by one to correspond
   * with the FFT width. Only this way the audio frequency that
   * corresponds to FFT index can be calculated accurately. */
  audio_freq = (int16_t)( (2 * (max_idx + 1) * center_freq) / (wfall_width + 1) );

  // Chose right mark/space frequencies
  if( Flag[RTTY_REVERSE_AFSK] )
  {
    freq1 = (int16_t)rtty_rc.mark_freq;
    freq3 = (int16_t)rtty_rc.space_freq;
  }
  else
  {
    freq1 = (int16_t)rtty_rc.space_freq;
    freq3 = (int16_t)rtty_rc.mark_freq;
  }

  /* Tune TCVR to put lower AFSK freq under
   * left Waterfall line if left button clicked.
   * Tune TCVR to put higher AFSK freq under
   * right Waterfall line if right button clicked. */
  int rx_freq = (int)TRx->rx_frequency;
  freq_correction = audio_freq;
  if( event->button == 1 )
    freq_correction -= freq1;
  else if( event->button == 3 )
    freq_correction -= freq3;

  rx_freq += freq_correction;
  if( rx_freq < 0 ) rx_freq = 0;
  TRx->rx_frequency = (uint32_t)rx_freq;
  Hermes2_Set_Center_Frequency( TRx, RX_FLAG );

  // Link Tx frequency to Rx
  if( TRx->link_tx_rx_freq && !TRx->tx_freq_lock )
  {
    TRx->tx_frequency = (uint32_t)rx_freq;
    Hermes2_Set_Center_Frequency( TRx, TX_FLAG );
  }

} // Rtty_Tune_to_Monitor

//-------------------------------------------------------------------------

