/*
 *  This file was derived from the C++ source code files in src/rsid/
 *  of the fldigi package. Below is the original copyright notice.
 *  fldigi is a GNU GPL licensed application and so is hermes2.
 *
 *  Copyright (C) 2008-2012
 *        Dave Freese, W1HKJ
 *   Copyright (C) 2009-2012
 *        Stelios Bounanos, M0GLD
 *   Copyright (C) 2012
 *        John Douyere, VK2ETA
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef RSID_H
#define RSID_H  1

#include "rsid_modes.h"
#include "../common/common.h"
#include "../common/utils.h"
#include <math.h>
#include <string.h>
#include <samplerate.h>

#define RSID_FFT_SAMPLES 	512         // Batch size of samples analyzed
#define RSID_SAMPLE_RATE    11025       // Internal RSID sample rate
#define RSID_SRATE_RATIO    4.353741497 // SND_DSP_RATE / RSID_SAMPLE_RATE
#define RSID_FFT_LENGTH     1024         // Length of FFT (number of points)
#define RSID_ARRAY_SIZE     2048         // (RSID_FFT_LENGTH * 2)
#define RSID_BUFFER_SIZE    4096         // (RSID_ARRAY_SIZE * 2)
#define RSID_LEFT_SHIFT     3584        // RSID_BUFFER_SIZE - RSID_FFT_SAMPLES
#define RSID_NSYMBOLS       15          // Number of Symbols (RSID tones)
#define RSID_NTIMES         30          // (RSID_NSYMBOLS * 2)
#define RSID_RESOLUTION     5.3833      // Frequency resolution (RSID_SAMPLE_RATE/RSID_ARRAY_SIZE)

// Each rsid symbol has a duration equal to 1024 samples
// at 11025 Hz sample rate = 0.092879819 sec.
#define RSID_SYM_DUR     0.092879819

//------------------------------------------------------------------------

enum
{
  RSID_BANDWIDTH_500 = 0,
  RSID_BANDWIDTH_1K,
  RSID_BANDWIDTH_WIDE,
};

typedef struct _RSIDS
{
  uint16_t    rs;
  int         mode;
  const char *name;
} RSIDs_t;

//------------------------------------------------------------------------

typedef struct _RSID
{
  // Table of precalculated Reed Solomon symbols
  uint8_t *pCodes1;
  uint8_t *pCodes2;

  BOOLEAN found1;
  BOOLEAN found2;

  const RSIDs_t *rsid_ids_1;
  int   rsid_ids_size1;

  const RSIDs_t *rsid_ids_2;
  int   rsid_ids_size2;

  const int *Squares;
  const int *indices;

  double rsid_secondary_time_out;

  int hamming_resolution;

  // Span of FFT bins, in which the RSID will be searched for
  int nBinLow;
  int nBinHigh;

  double aFFT_real[RSID_ARRAY_SIZE], aFFT_imag[RSID_ARRAY_SIZE];
  double aInputSamples[RSID_BUFFER_SIZE];
  double fftwindow[RSID_ARRAY_SIZE];
  double aFFTAmpl[RSID_FFT_LENGTH];
  int    fft_buckets[RSID_NTIMES][RSID_FFT_LENGTH];

  BOOLEAN bPrevTimeSliceValid;
  int  iPrevDistance;
  int  iPrevBin;
  int  iPrevSymbol;

  BOOLEAN bPrevTimeSliceValid2;
  int  iPrevDistance2;
  int  iPrevBin2;
  int  iPrevSymbol2;

  // Transmit buffers to hold the calculated values of I/Q samples
  double  *outbuf;
  size_t   symlen, outbuf_size;
  uint32_t outbuf_len;

  uint16_t rmode1;
  uint16_t rmode2;

  BOOLEAN rxid_initialized, txid_initialized;

  // Function pointers
  void (*Encode)(int code, uint8_t *rsid);
  void (*Search)( void );
  void (*Setup_Mode)(int m);
  void (*CalculateBuckets)(const double *pSpectrum, int iBegin, int iEnd);
  BOOLEAN (*Search_Amp)(int *bin_out, int *symbol_out, uint8_t *pcode_table);
  void (*Apply)(int iBin, int iSymbol, BOOLEAN extended);
  void (*Setup_TRx)(void);
  void (*Free)(void);
  void (*Reset)(void);
  void *(*Receive)(void *arg);
  void (*Send)(uint16_t mode);
  void (*Transmit)(void *arg);
  BOOLEAN (*Assigned)(uint16_t mode);
  void (*Reset_Rsid)(void *who);

} RsId_t;

extern RsId_t RsId;

//------------------------------------------------------------------------

#define ELEM1_(code_, tag_, mode_) RSID_ ## tag_ = code_,
enum { RSID_LIST1 };
#undef ELEM1_

#define ELEM1_(code_, tag_, mode_) { RSID_ ## tag_, mode_, #tag_ },
static const RSIDs_t rsid_ids1[] = { RSID_LIST1 };
#undef ELEM1_

#define ELEM2_(code_, tag_, mode_) RSID_ ## tag_ = code_,
enum { RSID_LIST2 };
#undef ELEM2_

#define ELEM2_(code_, tag_, mode_) { RSID_ ## tag_, mode_, #tag_ },
static const RSIDs_t rsid_ids2[] = { RSID_LIST2 };
#undef ELEM2_

//------------------------------------------------------------------------

  static inline int
RsId_HammingDistance( int iBucket, const uint8_t *p2 )
{
  int dist = 0;
  for( int i = 0, j = 1; i < RSID_NSYMBOLS; i++, j += 2 )
  {
    if( RsId.fft_buckets[j][iBucket] != p2[i] )
    {
      ++dist;
      if( dist > RsId.hamming_resolution )
        break;
    }
  }

  return( dist );
} //RsId_HammingDistance()

//------------------------------------------------------------------------

/* rsid.c */
void RsId_Send(uint16_t mode);
void RsId_Init(void);

//------------------------------------------------------------------------

#endif

