/*
 *  This file was derived from the C++ source code files in src/rsid/
 *  of the fldigi package. Below is the original copyright notice.
 *  fldigi is a GNU GPL licensed application and so is hermes2.
 *
 *  Copyright (C) 2008-2012
 *        Dave Freese, W1HKJ
 *   Copyright (C) 2009-2012
 *        Stelios Bounanos, M0GLD
 *   Copyright (C) 2012
 *        John Douyere, VK2ETA
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "rsid.h"
#include "rsid_modes.h"
#include "../hpsdr/pc_to_hw.h"
#include "../hpsdr/settings.h"
#include "../common/common.h"
#include "../common/cfft.h"
#include "../common/shared.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/display.h"
#include "../Hermes2/modulate.h"
#include "../Hermes2/sound.h"
#include "../olivia/operation.h"

#include <config.h>
#include <complex.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>

//------------------------------------------------------------------------

// NOTE originally was static const int
static const uint8_t Squares[] =
{
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
  0,  2,  4,  6,  8, 10, 12, 14,  9, 11, 13, 15,  1,  3,  5,  7,
  0,  3,  6,  5, 12, 15, 10,  9,  1,  2,  7,  4, 13, 14, 11,  8,
  0,  4,  8, 12,  9, 13,  1,  5, 11, 15,  3,  7,  2,  6, 10, 14,
  0,  5, 10, 15, 13,  8,  7,  2,  3,  6,  9, 12, 14, 11,  4,  1,
  0,  6, 12, 10,  1,  7, 13, 11,  2,  4, 14,  8,  3,  5, 15,  9,
  0,  7, 14,  9,  5,  2, 11, 12, 10, 13,  4,  3, 15,  8,  1,  6,
  0,  8,  9,  1, 11,  3,  2, 10, 15,  7,  6, 14,  4, 12, 13,  5,
  0,  9, 11,  2, 15,  6,  4, 13,  7, 14, 12,  5,  8,  1,  3, 10,
  0, 10, 13,  7,  3,  9, 14,  4,  6, 12, 11,  1,  5, 15,  8,  2,
  0, 11, 15,  4,  7, 12,  8,  3, 14,  5,  1, 10,  9,  2,  6, 13,
  0, 12,  1, 13,  2, 14,  3, 15,  4,  8,  5,  9,  6, 10,  7, 11,
  0, 13,  3, 14,  6, 11,  5,  8, 12,  1, 15,  2, 10,  7,  9,  4,
  0, 14,  5, 11, 10,  4, 15,  1, 13,  3,  8,  6,  7,  9,  2, 12,
  0, 15,  7,  8, 14,  1,  9,  6,  5, 10,  2, 13, 11,  4, 12,  3
};

//------------------------------------------------------------------------

// NOTE originally was static const int
static const uint8_t RS_Indices[] =
{
  2, 4, 8, 9, 11, 15, 7, 14, 5, 10, 13, 3
};

//------------------------------------------------------------------------

// The RsId object struct
RsId_t RsId;

//------------------------------------------------------------------------

// Reset RsId
  static void
RsId_Reset( void )
{
  RsId.iPrevDistance  = 99;
  RsId.iPrevDistance2 = 99;
  RsId.bPrevTimeSliceValid  = False;
  RsId.bPrevTimeSliceValid2 = False;
  RsId.found1 = False;
  RsId.found2 = False;
  RsId.rsid_secondary_time_out = 0;

  memset( RsId.aInputSamples, 0, RSID_BUFFER_SIZE * sizeof(double) );
  memset( RsId.aFFT_real,     0, RSID_ARRAY_SIZE  * sizeof(double) );
  memset( RsId.aFFT_imag,     0, RSID_ARRAY_SIZE  * sizeof(double) );
  memset( RsId.aFFTAmpl,      0, RSID_FFT_LENGTH    * sizeof(double) );
  memset( RsId.fft_buckets,   0, RSID_NTIMES * RSID_FFT_LENGTH * sizeof(int) );

  // Seems to be error tolerance - low, medium, high
  RsId.hamming_resolution = 1;

} // RsId_Reset()

//------------------------------------------------------------------------

// Free RSID objects
  static void
RsId_Free( void )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  if( Flag[RSIDTX_ACTIVE] ) return;

  if( !TRx->rx_rsid_enable && !TRx->tx_rsid_enable )
  {
    if( TRx->rx_rsid_running )
      pthread_join( TRx->rsid_thread, NULL );
    Mem_Free( (void **) &(RsId.pCodes1) );
    Mem_Free( (void **) &(RsId.pCodes2) );
    Mem_Free( (void **) &(RsId.outbuf) );
  }

  if( !TRx->rx_rsid_enable )
    RsId.rxid_initialized = False;

  if( !TRx->tx_rsid_enable )
    RsId.txid_initialized = False;

} // RsId_Free()

//------------------------------------------------------------------------

  static void
RsId_Encode( int code, uint8_t *rsid )
{
  rsid[0] = (uint8_t)(  code >> 8 );
  rsid[1] = (uint8_t)( (code >> 4) & 0x0f );
  rsid[2] = (uint8_t)(  code & 0x0f );

  for( int i = 3; i < RSID_NSYMBOLS; i++ )
    rsid[i] = 0;

  for( int i = 0; i < 12; i++ )
  {
    for( int j = RSID_NSYMBOLS - 1; j > 0; j-- )
    {
      rsid[j] = rsid[j - 1] ^ Squares[ (rsid[j] << 4) + RS_Indices[i] ];
    }
    rsid[0] = Squares[ (rsid[0] << 4) + RS_Indices[i] ];
  }
} // RsId_Encode()

//------------------------------------------------------------------------

  static void
RsId_CalculateBuckets( const double *pSpectrum, int iBegin, int iEnd )
{
  double Amp, AmpMax = 0.0;
  int iBucketMax  = iBegin - 2;
  int j;

  for( int i = iBegin; i < iEnd; i += 2 )
  {
    if( iBucketMax == i - 2 )
    {
      AmpMax = pSpectrum[i];
      iBucketMax = i;
      for( j = i + 2; j < i + RSID_NTIMES + 2; j += 2 )
      {
        Amp = pSpectrum[j];
        if( Amp > AmpMax )
        {
          AmpMax     = Amp;
          iBucketMax = j;
        }
      }
    }
    else
    {
      j = i + RSID_NTIMES;
      Amp = pSpectrum[j];
      if( Amp > AmpMax )
      {
        AmpMax     = Amp;
        iBucketMax = j;
      }
    }

    RsId.fft_buckets[RSID_NTIMES - 1][i] = ( iBucketMax - i ) >> 1;
  }
} // RsId_CalculateBuckets()

//------------------------------------------------------------------------

  static void *
RsId_Receive( void *arg )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  uint8_t  digi_buf_cnt = 0;
  uint16_t digi_buf_idx = 0;
  BOOLEAN  wait = TRUE;

  // We use a floating point index because the ratio
  // of incoming digimode sample rate to the RSID
  // sample rate is not an integer (48000 / 11025)
  double decim_idx = 0.0;

  // Keep looking for RSID signals while enabled
  TRx->rx_rsid_running = True;
  while( TRx->rx_rsid_enable )
  {
    // Fill the RSID samples buffer's last half from the digimode buffer
    uint16_t input_idx = RSID_ARRAY_SIZE;
    while( input_idx < RSID_BUFFER_SIZE )
    {
      // Decimate the incoming sample rate to near enough the
      // RSID sample rate by averaging 4 or 5 incoming samples
      RsId.aInputSamples[input_idx] = 0.0;
      while( decim_idx < RSID_SRATE_RATIO )
      {
        // Wait for digimode buffer to be filled
        if( wait )
        {
          sem_wait( &rsid_semaphore );
          wait = False;

          // If a secondary RSID tone sequence is expected, time it.
          // Subtract the time it takes for digimode buffer to fill.
          if( RsId.rsid_secondary_time_out > 0 )
          {
            RsId.rsid_secondary_time_out -= (double)DIGIMODE_BUFFER_SIZE / (double)SND_DSP_RATE;
            if( RsId.rsid_secondary_time_out <= 0 )
            {
              Error_Dialog( "Secondary RSId timed out", SHOW_OK );
              RsId.Reset();
            }
          }
        } // if( wait )

        // Summate incoming samples scaled to reduce magnitudes
        static const double pscale = 4.0 / RSID_FFT_LENGTH / RSID_FFT_LENGTH;
        RsId.aInputSamples[input_idx] +=
          (double)digimode_buffer[digi_buf_cnt][digi_buf_idx] * pscale;
          // <scaled by the FFT (Hamming) window> * RsId.fftwindow[digi_buf_idx];

        digi_buf_idx++;
        if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
        {
          digi_buf_idx = 0;
          digi_buf_cnt++;
          if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
            digi_buf_cnt = 0;

          /* Wait for digimode buffer to fill with audio samples from demodulator.
           * When this function is slow to respond due to processor loading, then
           * the sem_post() in Hermes2/detect.c is executed more than once, so the
           * wait function must be repeated according to the semaphore's value
           */
          if( digi_buf_cnt == digi_buf_input )
            wait = True;
         }

        decim_idx += 1.0;
      } // while( decim_idx < RSID_SRATE_RATIO )
      decim_idx -= RSID_SRATE_RATIO;

      // Average of summed samples
      RsId.aInputSamples[input_idx] /= RSID_SRATE_RATIO;

      input_idx++;
    } // while( input_idx < RSID_BUFFER_SIZE )

    for( uint8_t ls = 0; ls < 4; ls++ )
    {
      RsId.Search();
      memmove(
          (void *)RsId.aInputSamples,
          (const void *) &RsId.aInputSamples[RSID_FFT_SAMPLES],
          RSID_LEFT_SHIFT * sizeof(double) );

      memcpy( (void *)RsId.aFFT_real, (const void *)RsId.aInputSamples,
          RSID_ARRAY_SIZE * sizeof(double) );
      memset( (void *)RsId.aFFT_imag, 0, RSID_ARRAY_SIZE * sizeof(double) );
    }

  } // while( TRx->rx_rsid_enable )

  // Destroy RSID semaphore
  Init_Semaphore( &rsid_semaphore, False );

  TRx->rx_rsid_running = False;
  return( NULL );

} // RsId_Receive()

//------------------------------------------------------------------------

  static BOOLEAN
RsId_Search_Amp( int *bin_out, int *symbol_out, uint8_t *pcode )
{
  int i, j;
  int iDistanceMin = 1000;  // infinity
  int iDistance;
  int iBin    = -1;
  int iSymbol = -1;

  int tblsize;
  const RSIDs_t *prsid;

  if( pcode == RsId.pCodes1 )
  {
    tblsize = RsId.rsid_ids_size1;
    prsid = RsId.rsid_ids_1;
  }
  else
  {
    tblsize = RsId.rsid_ids_size2;
    prsid = RsId.rsid_ids_2;
  }

  for( i = 0; i < tblsize; i++ )
  {
    const uint8_t *pc = pcode + i * RSID_NSYMBOLS;
    for( j = RsId.nBinLow; j < RsId.nBinHigh - RSID_NTIMES; j++ )
    {
      iDistance = RsId_HammingDistance( j, pc );
      if( iDistance < iDistanceMin )
      {
        iDistanceMin = iDistance;
        iSymbol      = prsid[i].rs;
        iBin         = j;
        if( iDistanceMin == 0 ) break;
      }
    }
  }

  if( iDistanceMin <= RsId.hamming_resolution )
  {
    *symbol_out = iSymbol;
    *bin_out    = iBin;
    return( True );
  }

  return( False );
} // RsId_Search_Amp()

//------------------------------------------------------------------------

// FFT order (log2 RSID_ARRAY_SIZE)
#define RSID_FFT_ORDER   11

  static void
RsId_Search( void )
{
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  double f_low, f_high;

  /* The search range is determined by the audio frequency range
   * which is determined by the Bandpass filter and BFO settings */
  if( TRx->rx_cw_modes ) // For CW demodulator modes
  {
    f_low  = TRx->rx_weaver_frequency - TRx->rx_bandwidth / 2.0;
    f_high = TRx->rx_weaver_frequency + TRx->rx_bandwidth / 2.0;
  }
  else // For SSB demodulator modes
  {
    f_low  = ( 2 * TRx->rx_weaver_frequency - TRx->rx_bandwidth ) / 2.0;
    f_high = f_low + TRx->rx_bandwidth;
  }
  RsId.nBinLow  = (int)( f_low  / RSID_RESOLUTION ) - 32; // To allow a search nargin
  RsId.nBinHigh = (int)( f_high / RSID_RESOLUTION ) + 32; // for tuning discrepancies

  // Keep search bins in range
  if( RsId.nBinLow < 3 ) RsId.nBinLow = 3;
  if( RsId.nBinHigh > RSID_FFT_LENGTH - 32 )
    RsId.nBinHigh = RSID_FFT_LENGTH - 32;

  // Do an in-place FFT on the input arrays
  cFFT( RSID_ARRAY_SIZE, RSID_FFT_ORDER, RsId.aFFT_real, RsId.aFFT_imag );

  // Clear FFT amplitudes array
  memset( RsId.aFFTAmpl, 0, sizeof(RsId.aFFTAmpl) );

  // The pscale factor moved to Receive()
  for( int i = 0; i < RSID_FFT_LENGTH; i++ )
  {
    RsId.aFFTAmpl[i] = norm( RsId.aFFT_real[i], RsId.aFFT_imag[i] );
  }

  int bucket_low  = 3;
  int bucket_high = RSID_FFT_LENGTH - 32;
  memmove( RsId.fft_buckets, &(RsId.fft_buckets[1][0]),
      (RSID_NTIMES - 1) * RSID_FFT_LENGTH * sizeof(int) );
  memset( &(RsId.fft_buckets[RSID_NTIMES - 1][0]), 0, RSID_FFT_LENGTH * sizeof(int) );

  RsId.CalculateBuckets( RsId.aFFTAmpl, bucket_low,     bucket_high - RSID_NTIMES );
  RsId.CalculateBuckets( RsId.aFFTAmpl, bucket_low + 1, bucket_high - RSID_NTIMES );

  int symbol_out_1 = -1;
  int bin_out_1    = -1;
  int symbol_out_2 = -1;
  int bin_out_2    = -1;

  if( RsId.rsid_secondary_time_out <= 0 )
  {
    RsId.found1 = RsId.Search_Amp( &bin_out_1, &symbol_out_1, RsId.pCodes1 );
    if( RsId.found1 )
    {
      if( symbol_out_1 != RSID_ESCAPE )
      {
        RsId.Apply( bin_out_1, symbol_out_1, False );
        RsId.Reset();
        return;
      }
      else
      {
        // 10 rsid_gap + 15 symbols + 2 for timing errors
        RsId.rsid_secondary_time_out = 27 * RSID_SYM_DUR;
        return;
      }
    }
    else return;
  }

  RsId.found2 = RsId.Search_Amp( &bin_out_2, &symbol_out_2, RsId.pCodes2 );
  if( RsId.found2 )
  {
    if( symbol_out_2 != RSID_NONE2 )
    {
      RsId.Apply( bin_out_2, symbol_out_2, True );
      RsId.Reset();
    }
  }
} // RsId_Search()

//------------------------------------------------------------------------

// Saved values for RsId_Setup_TRx()
static uint16_t mode_num  = 0;
static uint32_t mode_freq = 0;

  static void
RsId_Apply( int iBin, int iSymbol, BOOLEAN extended )
{
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  int rsid_idx, mbin = NUM_MODES;
  int tblsize;
  const RSIDs_t *p_rsid;

  // Detected RSID frequency from the FFT bins
  double rsid_freq = ( (double)iBin + RSID_NSYMBOLS - 0.5 ) * RSID_RESOLUTION;
  if( TRx->rx_cw_modes ) rsid_freq -= TRx->rx_weaver_frequency;

  // Extended RSID code detected
  if( extended )
  {
    tblsize = RsId.rsid_ids_size2;
    p_rsid  = RsId.rsid_ids_2;
  }
  else // Primary RSID code
  {
    tblsize = RsId.rsid_ids_size1;
    p_rsid  = RsId.rsid_ids_1;
  }

  // Look up the RSID mode number
  for( rsid_idx = 0; rsid_idx < tblsize; rsid_idx++ )
  {
    if( p_rsid[rsid_idx].rs == iSymbol )
    {
      mbin = p_rsid[rsid_idx].mode;
      break;
    }
  }

  // Error conditions
  if( mbin == NUM_MODES )
  {
    char msg[50];
    if( rsid_idx < tblsize ) // RSID known but unimplemented
      snprintf (msg, sizeof(msg), "RSID: %s unimplemented", p_rsid[rsid_idx].name );
    else // RSID unknown; shouldn't  happen
      snprintf( msg, sizeof(msg), "RSID: code %d unknown", iSymbol );
    Error_Dialog( msg, SHOW_OK );
    return;
  }

  // Display RSID mode name and detected frequency if it matches
  // selection of mode to watch for or if no mode selected by user
  if( strstr(p_rsid[rsid_idx].name, TRx->RSID_select) != NULL )
  {
    // Current Rx center frequency
    mode_freq = (uint32_t)( TRx->rx_frequency + rsid_freq );
    mode_num  = (uint16_t)p_rsid[rsid_idx].mode;
    RSID_Mode( (char *)(p_rsid[rsid_idx].name), mode_freq );

    // Make a beep on Transceiver's audio O/P
    Flag[BEEP_ACTIVE] = True;
  }

} // RsId_Apply()

//------------------------------------------------------------------------

/* RsId_Setup_TRx()
 *
 * Sets up the Mode and Frequency of the
 * Transceiver according to the detected RSID mode
 */
  static void
RsId_Setup_TRx( void )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  uint8_t mode;


  // Set up the Rx Modulation mode.
  // Tx will be set if linked to Rx.
  switch( mode_num )
  {
    case 0: // No mode detected
      gtk_image_set_from_icon_name(
          GTK_IMAGE(TRx->rsid_image), "gtk-dialog-error", GTK_ICON_SIZE_BUTTON );
      return;
    case MODE_PSK31: case MODE_QPSK31:
      mode = RX_MODE_PSK31U;
      break;
    case MODE_RTTY_45:
      mode = RX_MODE_RTTYN;
      break;
    case MODE_RTTY_50:
      mode = RX_MODE_RTTYM;
      break;
    case MODE_RTTY_75:
      mode = RX_MODE_RTTYW;
      break;
    case MODE_CONTESTIA_4_125:   case MODE_OLIVIA_4_125:
      mode = RX_MODE_4_125;
      break;
    case MODE_CONTESTIA_4_250:   case MODE_OLIVIA_4_250:
      mode = RX_MODE_4_250;
      break;
    case MODE_CONTESTIA_8_250:   case MODE_OLIVIA_8_250:
      mode = RX_MODE_8_250;
      break;
    case MODE_CONTESTIA_8_500:   case MODE_OLIVIA_8_500:
      mode = RX_MODE_8_500;
      break;
    case MODE_CONTESTIA_16_500:  case MODE_OLIVIA_16_500:
      mode = RX_MODE_16_500;
      break;
    case MODE_CONTESTIA_16_1000: case MODE_OLIVIA_16_1000:
      mode = RX_MODE_16_1000;
      break;
    case MODE_CONTESTIA_32_1000: case MODE_OLIVIA_32_1000:
      mode = RX_MODE_32_1000;
      break;
    case MODE_FELDHELL:
      mode = RX_MODE_HELLM;
      break;
    case MODE_FSKH105:
      mode = RX_MODE_HELLM;
      break;
    case MODE_CW:
      mode = RX_MODE_CWUM;
      break;
    case MODE_VOICE:
      mode = RX_MODE_USBM;
      break;
    default: mode = RX_MODE_CWUW;
  }

  //Set up Rx frequency. Tx will be set if linked to Rx.
  // Set the Rx Band combobox first
  Set_Rx_Band( TRx, mode_freq );

  // Enter the Receiver frequency
  TRx->rx_frequency = mode_freq;

  // Link Tx frequency to Rx
  if( TRx->link_tx_rx_freq && !TRx->tx_freq_lock )
  {
    TRx->tx_frequency = TRx->rx_frequency;
    Hermes2_Set_Center_Frequency( TRx, TX_FLAG );
  }

  // Set the Center Frequency of the Transceiver
  Hermes2_Set_Center_Frequency( TRx, RX_FLAG );

  // Set the detected RSId mode
  Rx_Modulation_Mode_Changed( Indices.TRx_Index, mode );

  // Show search for RSID is active or not
  g_idle_add( RSID_Status, NULL );

  // Cancel mode number
  mode_num = 0;

} // RsId_Setup_TRx()

//=============================================================================
// Transmit rsid code for current mode
//=============================================================================

  static BOOLEAN
RsId_Assigned( uint16_t mode )
{
  RsId.rmode1 = RSID_NONE1;
  RsId.rmode2 = RSID_NONE2;

  switch( mode )
  {
    case MODE_RTTY_ASCII_7:
      RsId.rmode1 = RSID_RTTY_ASCII_7;
      break;
    case MODE_RTTY_ASCII_8:
      RsId.rmode1 = RSID_RTTY_ASCII_8;
      break;
    case MODE_RTTY_45:
      RsId.rmode1 = RSID_RTTY_45;
      break;
    case MODE_RTTY_50:
      RsId.rmode1 = RSID_RTTY_50;
      break;
    case MODE_RTTY_75:
      RsId.rmode1 = RSID_RTTY_75;
      break;

    case MODE_OLIVIA_4_125:
      RsId.rmode1 = RSID_OLIVIA_4_125;
      break;
    case MODE_OLIVIA_4_250:
      RsId.rmode1 = RSID_OLIVIA_4_250;
      break;
    case MODE_OLIVIA_4_500:
      RsId.rmode1 = RSID_OLIVIA_4_500;
      break;
    case MODE_OLIVIA_4_1000:
      RsId.rmode1 = RSID_OLIVIA_4_1000;
      break;
    case MODE_OLIVIA_4_2000:
      RsId.rmode1 = RSID_OLIVIA_4_2000;
      break;
    case MODE_OLIVIA_8_125:
      RsId.rmode1 = RSID_OLIVIA_8_125;
      break;
    case MODE_OLIVIA_8_250:
      RsId.rmode1 = RSID_OLIVIA_8_250;
      break;
    case MODE_OLIVIA_8_500:
      RsId.rmode1 = RSID_OLIVIA_8_500;
      break;
    case MODE_OLIVIA_8_1000:
      RsId.rmode1 = RSID_OLIVIA_8_1000;
      break;
    case MODE_OLIVIA_8_2000:
      RsId.rmode1 = RSID_OLIVIA_8_2000;
      break;
    case MODE_OLIVIA_16_500:
      RsId.rmode1 = RSID_OLIVIA_16_500;
      break;
    case MODE_OLIVIA_16_1000:
      RsId.rmode1 = RSID_OLIVIA_16_1000;
      break;
    case MODE_OLIVIA_16_2000:
      RsId.rmode1 = RSID_OLIVIA_16_2000;
      break;
    case MODE_OLIVIA_32_1000:
      RsId.rmode1 = RSID_OLIVIA_32_1000;
      break;
    case MODE_OLIVIA_32_2000:
      RsId.rmode1 = RSID_OLIVIA_32_2000;
      break;
    case MODE_OLIVIA_64_2000:
      RsId.rmode1 = RSID_OLIVIA_64_2000;
      break;

    case MODE_CONTESTIA_4_125:
      RsId.rmode1 = RSID_CONTESTIA_4_125;
      break;
    case MODE_CONTESTIA_4_250:
      RsId.rmode1 = RSID_CONTESTIA_4_250;
      break;
    case MODE_CONTESTIA_4_500:
      RsId.rmode1 = RSID_CONTESTIA_4_500;
      break;
    case MODE_CONTESTIA_4_1000:
      RsId.rmode1 = RSID_CONTESTIA_4_1000;
      break;
    case MODE_CONTESTIA_4_2000:
      RsId.rmode1 = RSID_CONTESTIA_4_2000;
      break;
    case MODE_CONTESTIA_8_125:
      RsId.rmode1 = RSID_CONTESTIA_8_125;
      break;
    case MODE_CONTESTIA_8_250:
      RsId.rmode1 = RSID_CONTESTIA_8_250;
      break;
    case MODE_CONTESTIA_8_500:
      RsId.rmode1 = RSID_CONTESTIA_8_500;
      break;
    case MODE_CONTESTIA_8_1000:
      RsId.rmode1 = RSID_CONTESTIA_8_1000;
      break;
    case MODE_CONTESTIA_8_2000:
      RsId.rmode1 = RSID_CONTESTIA_8_2000;
      break;
    case MODE_CONTESTIA_16_500:
      RsId.rmode1 = RSID_CONTESTIA_16_500;
      break;
    case MODE_CONTESTIA_16_1000:
      RsId.rmode1 = RSID_CONTESTIA_16_1000;
      break;
    case MODE_CONTESTIA_16_2000:
      RsId.rmode1 = RSID_CONTESTIA_16_2000;
      break;
    case MODE_CONTESTIA_32_1000:
      RsId.rmode1 = RSID_CONTESTIA_32_1000;
      break;
    case MODE_CONTESTIA_32_2000:
      RsId.rmode1 = RSID_CONTESTIA_32_2000;
      break;
    case MODE_CONTESTIA_64_2000:
      RsId.rmode1 = RSID_CONTESTIA_64_2000;
      break;

    case MODE_DOMINOEX4:
      RsId.rmode1 = RSID_DOMINOEX_4_FEC;
      break;
    case MODE_DOMINOEX5:
      RsId.rmode1 = RSID_DOMINOEX_5_FEC;
      break;
    case MODE_DOMINOEX8:
      RsId.rmode1 = RSID_DOMINOEX_8_FEC;
      break;
    case MODE_DOMINOEX11:
      RsId.rmode1 = RSID_DOMINOEX_11_FEC;
      break;
    case MODE_DOMINOEX16:
      RsId.rmode1 = RSID_DOMINOEX_16_FEC;
      break;
    case MODE_DOMINOEX22:
      RsId.rmode1 = RSID_DOMINOEX_22_FEC;
      break;

    case MODE_MT63_500S:
      RsId.rmode1 = RSID_MT63_500_ST;
      break;
    case MODE_MT63_500L:
      RsId.rmode1 = RSID_MT63_500_LG;
      break;
    case MODE_MT63_1000S:
      RsId.rmode1 = RSID_MT63_1000_ST;
      break;
    case MODE_MT63_1000L:
      RsId.rmode1 = RSID_MT63_1000_LG;
      break;
    case MODE_MT63_2000S:
      RsId.rmode1 = RSID_MT63_2000_ST;
      break;
    case MODE_MT63_2000L:
      RsId.rmode1 = RSID_MT63_2000_LG;
      break;
  }

  // If RsId.rmode1 is still unset, look it up
  // Try secondary table first
  if( RsId.rmode1 == RSID_NONE1 )
  {
    uint16_t idx = 0;
    while( RsId.rsid_ids_2[idx].rs )
    {
      if( mode == RsId.rsid_ids_2[idx].mode )
      {
        RsId.rmode1 = RSID_ESCAPE;
        RsId.rmode2 = RsId.rsid_ids_2[idx].rs;
        break;
      }
      idx++;
    }

    // Try primary table also
    if( RsId.rmode2 == RSID_NONE2 )
    {
      idx = 0;
      while( RsId.rsid_ids_1[idx].rs )
      {
        if( mode == RsId.rsid_ids_1[idx].mode )
        {
          RsId.rmode1 = RsId.rsid_ids_1[idx].rs;
          break;
        }
        idx++;
      }
    } //if( RsId.rmode2 == RSID_NONE2 )
  } // if( RsId.rmode1 == RSID_NONE1 )

  // Not really needed in my version of this function
  if( RsId.rmode1 == RSID_NONE1 )
    return( False );
  else
    return( True );

} // RsId_Assigned()

//------------------------------------------------------------------------

// The SSB modulator requires tone data in batches
// of 126 samples at a rate of 48000 samples/sec
#define LOC_BUF_SIZE   DUC_NUM_IQ_SAMPLES

// Sends the RSId I/Q samples produced in RsId_Send() to the DUC
  static void
RsId_Transmit( void *arg )
{
  static double   ssb_buf[LOC_BUF_SIZE] = { 0 };
  uint16_t        ssb_buf_idx = 0;
  static uint32_t buf_in_idx  = 0;

  // Fill the SSB modulator buffer with samples
  while( buf_in_idx < RsId.outbuf_len )
  {
    ssb_buf[ssb_buf_idx] = RsId.outbuf[buf_in_idx];
    ssb_buf_idx++;
    buf_in_idx++;

    // Send samples to DUC for transmission when buffers full
    if( ssb_buf_idx >= LOC_BUF_SIZE )
    {
      Modulate_SSB( (void *)ssb_buf );
      return;
    }
  }

  // Hand over to the Modulator of the Mode that sent RSID
  Modulator = Next_Modulator;
  Flag[RSIDTX_ACTIVE] = False;

  buf_in_idx = 0;
} // RsId_Transmit()

//----------------------------------------------------------------------

// Empirical, to produce near 100% O/P power
#define RSID_AMPLITUDE   28000.0

// Generates RSId IQ samples for the RSID tones sequence
  void
RsId_Send( uint16_t mode )
{
  // Sample Rate = 48kHz * 1000, fixed in hermes2
  static double sample_rate = DUC_SAMPLE_RATE;
  uint8_t rsid[RSID_NSYMBOLS];
  int     iTone;
  double  freq, center_freq;
  double  phase, dphase;

  // size_incr is the increase in RsId.outbuf allocation for new samples
  // next_seg is the offset in bytes from RsId.outbuf[0] to input new bytes
  // offset is the offset in doubles from RsId.outbuf[0] to input new samples
  size_t size_incr, next_seg, offset;
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Assign the RSId code to the specified mode
  if( !RsId.Assigned(mode) ) return;

  // Encode the rsid mode into rsid symbols (I think!)
  RsId.Encode( RsId.rmode1, rsid );

  // Initialize before transmissions
  RsId.outbuf_size = 0;
  Mem_Free( (void **)&(RsId.outbuf) );

  // Fill out buffer with zero samples for 5 symbol periods
  size_incr = sizeof(double) * RsId.symlen * 5;
  RsId.outbuf_size += size_incr;
  Mem_Realloc( (void **)&(RsId.outbuf), RsId.outbuf_size );
  memset( (void *)(RsId.outbuf), 0, RsId.outbuf_size );
  next_seg = RsId.outbuf_size;

  // Fill out buffer with samples for 15 (RSID_NSYMBOLS) symbol periods
  size_incr = sizeof(double) * RsId.symlen * RSID_NSYMBOLS;
  RsId.outbuf_size += size_incr;
  Mem_Realloc( (void **)&(RsId.outbuf), RsId.outbuf_size );

  // Transmit sequence of RSID_NSYMBOLS (15) symbols (tones)
  // RSId tones appear to be Orthogonal: Frequency increment is
  // inverse of tone duration. I replaced RSID_SAMPLE_RATE / 1024
  // (= RSID_SYMLEN) with RSID_SYM_DUR to avoid some confusion.
  // Center frequency is the center of the Transmit bandpass filter.
  center_freq =
    (double)TRx->tx_weaver_frequency - RSID_NSYMBOLS / 2.0 / RSID_SYM_DUR;
  phase  = 0.0;
  offset = next_seg / sizeof(double);
  for( int i = 0; i < RSID_NSYMBOLS; i++)
  {
    // Center frequency is the center of the Transmit bandpass filter.
    // Division iTone / RSID_SYM_DUR gives frequency of the RSID tone.
    iTone  = rsid[i];
    freq   = center_freq + iTone / RSID_SYM_DUR;
    dphase = M_2PI * freq / sample_rate;
    for( size_t j = 0; j < RsId.symlen; j++ )
    {
      RsId.outbuf[offset + j] = RSID_AMPLITUDE * sin( phase );
      phase += dphase;
      CLAMP_2PI( phase );
    }

    offset += RsId.symlen;
  } // for( int i = 0; i < RSID_NSYMBOLS; i++)
  next_seg = RsId.outbuf_size;

  // If no primary RSID code was assigned but a secondary
  if( (RsId.rmode1 == RSID_ESCAPE) && (RsId.rmode2 != RSID_NONE2) )
  {
    // Transmit 10 symbol periods of silence between RSID sequences
    size_incr = sizeof(double) * RsId.symlen * 10;
    RsId.outbuf_size += size_incr;
    Mem_Realloc( (void **)&(RsId.outbuf), RsId.outbuf_size );
    offset = next_seg / sizeof(double);
    memset( (void *)(RsId.outbuf + offset), 0, size_incr );
    next_seg = RsId.outbuf_size;

    // Encode from the secondary RSId codes
    RsId.Encode( RsId.rmode2, rsid );

    // Fill out buffer with samples for 15 (RSID_NSYMBOLS) symbol periods
    size_incr = sizeof(double) * RsId.symlen * RSID_NSYMBOLS;
    RsId.outbuf_size += size_incr;
    Mem_Realloc( (void **)&(RsId.outbuf), RsId.outbuf_size );

    // Transmit sequence of RSID_NSYMBOLS (15) symbols (tones)
    // RSId tones appear to be Orthogonal: Frequency increment is
    // inverse of tone duration. I have replaced RSID_SAMPLE_RATE / 1024
    // (= RSID_SYMLEN) with RSID_SYM_DUR to avoid some confusion.
    phase  = 0.0;
    offset = next_seg / sizeof(double);
    for( int i = 0; i < RSID_NSYMBOLS; i++ )
    {
      // Center frequency is the center of the Transmit bandpass filter.
      // Division iTone / RSID_SYM_DUR gives frequency of the RSID tone.
      iTone  = rsid[i];
      freq   = center_freq + iTone / RSID_SYM_DUR;
      dphase = M_2PI * freq / sample_rate;
      for( size_t j = 0; j < RsId.symlen; j++ )
      {
        RsId.outbuf[offset + j] = RSID_AMPLITUDE * sin( phase );
        phase += dphase;
        CLAMP_2PI( phase );
      }

      offset += RsId.symlen;
    } // for( int i = 0; i < RSID_NSYMBOLS; i++ )
    next_seg = RsId.outbuf_size;

  } // if( (RsId.rmode1 == RSID_ESCAPE) && (RsId.rmode2 != RSID_NONE2) )

  // Transmit 5 symbol periods of silence at end of
  // transmission and between RSID and the data signal
  size_incr = sizeof(double) * RsId.symlen * 5;
  RsId.outbuf_size += size_incr;
  Mem_Realloc( (void **)&(RsId.outbuf), RsId.outbuf_size );
  offset = next_seg / sizeof(double);
  memset( (void *)(RsId.outbuf + offset), 0, size_incr );

  // Finalize out RSID buffer length and transmit it
  RsId.outbuf_len = (uint32_t)( RsId.outbuf_size / sizeof(double) );
  Modulator = RsId.Transmit;

} // RsId_Send()

//------------------------------------------------------------------------

// Initialize RSID
  void
RsId_Init( void )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];


  // Initialize objects common to both RxID and TxID if not done
  if( !RsId.rxid_initialized || !RsId.txid_initialized )
  {
    RsId.Free   = RsId_Free;
    RsId.Reset  = RsId_Reset;
    RsId.Encode = RsId_Encode;

    // *** Set up RSID code buffers ***
    // rsid_ids1 and rsid_ids2 are set in rsid.h
    RsId.rsid_ids_1 = rsid_ids1;
    RsId.rsid_ids_2 = rsid_ids2;

    RsId.rsid_ids_size1 = sizeof( rsid_ids1 ) / sizeof( rsid_ids1[0] ) - 1;
    RsId.rsid_ids_size2 = sizeof( rsid_ids2 ) / sizeof( rsid_ids2[0] ) - 1;

    RsId.pCodes1 = NULL;
    RsId.pCodes2 = NULL;
    Mem_Alloc( (void **)&(RsId.pCodes1), (size_t)(RsId.rsid_ids_size1 * RSID_NSYMBOLS) );
    Mem_Alloc( (void **)&(RsId.pCodes2), (size_t)(RsId.rsid_ids_size2 * RSID_NSYMBOLS) );

    // Initialization of assigned mode/submode IDs.
    uint8_t *c;
    for( int i = 0; i < RsId.rsid_ids_size1; i++ )
    {
      c = RsId.pCodes1 + i * RSID_NSYMBOLS;
      RsId.Encode( RsId.rsid_ids_1[i].rs, c );
    }
    for( int i = 0; i < RsId.rsid_ids_size2; i++ )
    {
      c = RsId.pCodes2 + i * RSID_NSYMBOLS;
      RsId.Encode( RsId.rsid_ids_2[i].rs, c );
    }

    RsId.symlen = (size_t)( DUC_SAMPLE_RATE * RSID_SYM_DUR + 0.5 );
    RsId.Reset();

  } // if( !RsId.rxid_initialized || !RsId.txid_initialized )

  // Initialize RxID objects if not already done
  if( !RsId.rxid_initialized && TRx->rx_rsid_enable )
  {
    // Set Function pointers for RxId functions
    RsId.CalculateBuckets = RsId_CalculateBuckets;
    RsId.Apply            = RsId_Apply;
    RsId.Setup_TRx        = RsId_Setup_TRx;
    RsId.Receive          = RsId_Receive;
    RsId.Search           = RsId_Search;
    RsId.Search_Amp       = RsId_Search_Amp;

    // Create a thread to run RxID function
    if( !Pthread_Create( &TRx->rsid_thread, NULL, RsId.Receive, NULL,
          _("Failed to create RSID Receive thread")) )
    {
      TRx->rx_rsid_enable = False;
      Init_Semaphore( &rsid_semaphore, False );
      return;
    }

    // Default FFT search of bins range
    RsId.nBinLow  = 3;
    RsId.nBinHigh = RSID_FFT_LENGTH - 32; // - RSID_NTIMES - 2

    /*
       Hamming window for FFT samples
       for( int i = 0; i < RSID_ARRAY_SIZE; i++ )
       {
         double w = M_2PI * (double)i / RSID_ARRAY_SIZE;
         RsId.fftwindow[i] = 0.54 - 0.46 * cos( w );
       }
     */

    RsId.rxid_initialized = True;
  } // if( !RsId.rxid_initialized )

  // Initialize TxID objects if not already done
  if( !RsId.txid_initialized && TRx->tx_rsid_enable )
  {
    RsId.Assigned = RsId_Assigned;
    RsId.Send     = RsId_Send;
    RsId.Transmit = RsId_Transmit;

    if( !Flag[RSIDTX_ACTIVE] ) RsId.outbuf = NULL;
    RsId.txid_initialized = True;
  }

} // RsId_Init()

//------------------------------------------------------------------------

