/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "callbacks.h"
#include "display.h"
#include "operation.h"
#include "codec.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../common/guest_utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/sound.h"
#include "../rsid/rsid_modes.h"
#include <gtk/gtk.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdlib.h>
#include <stdlib.h>
#include <string.h>

//------------------------------------------------------------------------

// General definitions for signal processing
#define BAUD_RATE   31.25  // PSK baude rate

//------------------------------------------------------------------------

/* Psk31_Read_Config()
 *
 * Loads the psk31rc configuration file
 */
  gboolean
Psk31_Read_Config( gpointer data )
{
  uint16_t idx;

  // Buffer for Read_Line
  char line[READ_LINE_BUF_SIZE];

  // Config file pointer
  FILE *psk31rc;

  // For setting squelch level
  GtkWidget *hscale;
  GtkAdjustment *adjustment;

  // Setup file path to psk31rc
  // Create the psk31 config file name
  Strlcpy( psk31_rc.rc_fpath, getenv("HOME"),
      sizeof(psk31_rc.rc_fpath) );
  Strlcat( psk31_rc.rc_fpath, "/.hermes2/psk31/psk31.config",
      sizeof(psk31_rc.rc_fpath) );

  // Open psk31.config file
  psk31rc = fopen( psk31_rc.rc_fpath, "r" );
  if( psk31rc == NULL )
  {
    perror( psk31_rc.rc_fpath );
    Error_Dialog(
        _("Failed to open psk31.config file\n"\
          "Quit PSK31 and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read squelch threshold, abort if EOF
  if( Read_Line(line, psk31rc, _("Squelch Threshold") ) != SUCCESS )
    return( FALSE );
  psk31_rc.squelch_threshold = (uint16_t)atoi( line );

  // Enter squelch level to hscale
  hscale = Builder_Get_Object( psk31_gui.window_builder, "psk31_squelch_value" );
  adjustment = gtk_range_get_adjustment( GTK_RANGE(hscale) );
  gtk_adjustment_set_value( adjustment, (gdouble)psk31_rc.squelch_threshold );

  // Read Magniphase reverse flag, abort if EOF
  if( Read_Line(line, psk31rc, _("Reverse Magniphase") ) != SUCCESS )
    return( FALSE );
  psk31_rc.rev_magniphase = (BOOLEAN)atoi( line );

  // Read tone duration, abort if EOF
  if( Read_Line(line, psk31rc, _("Tone Duration") ) != SUCCESS )
    return( FALSE );
  psk31_rc.tx_preamble_tone = (uint8_t)atoi( line );

  // Read number of reversals, abort if EOF
  if( Read_Line(line, psk31rc, _("Number of Reversals") ) != SUCCESS )
    return( FALSE );
  psk31_rc.tx_preamble_rev = (uint8_t)atoi( line );

  // Read word wrap column, abort if EOF
  if( Read_Line(line, psk31rc, _("Word Wrap Column") ) != SUCCESS )
    return( FALSE );
  psk31_rc.word_wrap = (uint8_t)atoi( line );

  // *** Setup default main menu items ***
  // Read Record QSO enable flag, abort if EOF
  if( Read_Line(line, psk31rc, _("Record QSOs Enable") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(psk31_gui.popup_menu_builder, "psk31_record_qsos")), TRUE );
    Flag[GUEST_RECORD_QSO] = True;
  }
  else if( strcmp(line, "no") == 0 )
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(psk31_gui.popup_menu_builder, "psk31_record_qsos")), FALSE );
  else
  {
    Close_File( &psk31rc );
    Error_Dialog(
        _("Error reading psk31.config\n"\
          "Unrecognized menu option\n"\
          "Quit psk31 and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Capitalize enable flag, abort if EOF
  if( Read_Line(line, psk31rc, _("Capitalize Enable") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(psk31_gui.popup_menu_builder,
            "psk31_capitalize_letters")), TRUE );
    Flag[GUEST_CAPITALIZE] = True;
  }
  else if( strcmp(line, "no") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(psk31_gui.popup_menu_builder,
            "psk31_capitalize_letters")), FALSE );
  }
  else
  {
    Close_File( &psk31rc );
    Error_Dialog(
        _("Error reading psk31.config\n"\
          "Unrecognized menu option\n"\
          "Quit psk31 and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Station info and Macros
  if( !Read_Station_Info( psk31rc, psk31_macro,
        "psk31_label", psk31_gui.window_builder) )
    return( FALSE );

  // Read main Window's position and place it
  if( Read_Line(line, psk31rc, _("Window Position") ) != SUCCESS )
    return( FALSE );
  gint x = (gint)atoi( line );
  gint y = (gint)atoi( &line[5] );
  gtk_window_move( GTK_WINDOW(psk31_gui.window), x, y );

  // Duration of a PSK pulse, in Rx ADC samples
  psk31_rc.rx_psk_element  = (uint16_t)( SND_DSP_RATE / BAUD_RATE );
  psk31_rc.rx_psk_element2 = psk31_rc.rx_psk_element / 2;

  // Duration of a PSK pulse, in Tx DAC samples
  psk31_rc.tx_psk_element  = (uint16_t)( DUC_SAMPLE_RATE / BAUD_RATE );
  psk31_rc.tx_psk_element2 = psk31_rc.tx_psk_element / 2;

  // Duration of preamble tone in Tx DAC samples
  psk31_rc.tx_tone_period = DUC_SAMPLE_RATE / PSK31_AUDIO_FREQUENCY;

  // Puts the tone freq in middle of waterfall
  idx = (2 * psk31_ifft_data.data_len) / psk31_wfall.width;
  if( idx )
    psk31_ifft_data.ifft_stride = SND_DSP_RATE / PSK31_AUDIO_FREQUENCY / idx;

  // Calculate phase quadrants
  psk31_rc.quadrant_4 = SND_DSP_RATE / PSK31_AUDIO_FREQUENCY;
  psk31_rc.quadrant_1 = psk31_rc.quadrant_4 / 4;
  psk31_rc.quadrant_2 = psk31_rc.quadrant_4 / 2;
  psk31_rc.quadrant_3 = ( 3 * psk31_rc.quadrant_4 ) / 4;

  // Make cosine & sine wave table
  Psk31_Make_Cosine_Table();

  // Set default RSID mode
  Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_PSK31;

  Close_File( &psk31rc );
  return( FALSE );
} // End of Psk31_Read_Config()

//------------------------------------------------------------------

/* PSK31_Cleanup()
 *
 * Cleans up before quitting
 */
  void
PSK31_Cleanup( void )
{
  // Save outstanding records
  if( Flag[GUEST_SAVE_RECORD] )
    Save_QSO_Record( &qso_record );

  // Close open files
  Close_File( &(qso_record.qso_record_fp) );
  Close_File( &(qso_record.adif_log_fp) );
  Close_File( &(qso_record.station_log_fp) );

  // Free Digimode semaphores
  Init_Semaphore( &digimode_semaphore, False );
  Init_Semaphore( &duc_send_semaphore, False );

  // Free buffer allocations
  Psk31_Free_Codec();
  Free_Xmit_Buffers();
  Free_iFFT( &psk31_ifft_data );

  // Free macro buffer pointers
  for( uint8_t idx = 0; idx < NUM_OF_LABELS; idx++ )
    Mem_Free( (void **) &psk31_macro[idx] );

  // Free waterfall pixbuf
  if( psk31_wfall.pixbuf )
    g_object_unref( psk31_wfall.pixbuf );
  psk31_wfall.pixbuf = NULL;

  // Close some GUI objects
  g_object_unref( psk31_gui.window_builder );
  psk31_gui.window = NULL;
  hermes2_gui.guest_window = NULL;
  psk31_gui.window_builder = NULL;

  if( psk31_gui.popup_menu_builder != NULL )
  {
    g_object_unref( psk31_gui.popup_menu_builder );
    psk31_gui.popup_menu_builder = NULL;
  }

  // Clear all relevant flags
  Flag[GUEST_RECORD_QSO]        = False;
  Flag[GUEST_RECEIVE_MODE]      = False;
  Flag[GUEST_TRANSMIT_MODE]     = False;
  Flag[GUEST_TRANSMIT_TAG]      = False;
  Flag[GUEST_TRANSMIT_MACRO]    = False;
  Flag[GUEST_TRANSMIT_KEYBD]    = False;
  Flag[GUEST_KEYBD_BUSY]        = False;
  Flag[GUEST_CAPITALIZE]        = False;
  Flag[GUEST_SAVE_RECORD]       = False;
  Flag[GUEST_QUIT]              = False;
  Flag[HERMES2_SEND_DUC_PACKET] = False;
  Flag[TRANSMIT_MORSE_MESG]     = False;
  Flag[GUEST_RECEIVING]         = False;
  Flag[GUEST_TRANSMITTING]      = False;

  //Clear PSK31 flags
  Flag[PSK31_SQUELCH_OPEN]     = False;
  Flag[PSK31_MODE_BPSK]        = False;
  Flag[PSK31_MODE_QPSK_USB]    = False;
  Flag[PSK31_MODE_QPSK_LSB]    = False;
  Flag[PSK31_CHANNEL_A_SELECT] = False;
  Flag[PSK31_ENABLE_AFC]       = False;

  // Disable the common Guest modulation mode
  if( Transceiver[Indices.TRx_Index] != NULL )
    Transceiver[Indices.TRx_Index]->guest_modulation_mode = RX_MODE_ITEMS;

} // PSK31_Cleanup( void )

//------------------------------------------------------------------------
