/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "phase.h"
#include "display.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/ifft.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include <cairo/cairo.h>
#include <math.h>
#include <semaphore.h>
#include <stdint.h>
#include <stdlib.h>

static int
  sel_level_i,   // In-phase level from selected channel
  sel_level_q;   // Quadrature level from selected channel

// Receive freq error correction factor
double freq_error;

//------------------------------------------------------------------------

// Receive magniphase scale factor
#define MAGNIPHASE_SCALE    220

#define PHASE_AVE_WINDOW    5  // Length of phase averager window
#define PHASE_AVE_WINDOW_1  4  // Length of phase averager window - 1
#define SQUELCH_WINDOW      10 // Length of squelch averager window
#define SQUELCH_WINDOW_1     9 // Length of squelch averager window - 1

// Max value of frequency correction factor
#define MAX_FREQ_ERROR      10.0

// Degrees-to-rads
#define RAD2DEG   57.295779513082

//------------------------------------------------------------------------

/* PSK31_Phase()
 *
 * Detects phase changes of the PSK carrier
 */
  BOOLEAN
PSK31_Phase( int16_t *phase_chg )
{
  // Phase in samples of phase detector
  static double det_phase;

  static int
    last_lev_i,  // Last phase detector in-phase signal level
    last_lev_q,  // Last phase detector quadrature signal level
    det_level_i, // Phase detector in-phase signal level
    det_level_q, // Phase detector quadrature signal level
    chA_level_i, // 'A' channel in-phase signal level
    chA_level_q, // 'A' channel quadrature signal level
    chB_level_i, // 'B' channel in-phase signal level
    chB_level_q; // 'B' channel quadrature signal level

  static uint32_t
    norm_squelch, // Normalized squelch value
    chA_squelch,  // Channel A squelch (signal) level
    chB_squelch;  // Channel B squelch (signal) level

  // Counter used for producing the baud rate
  static uint16_t elem_timer;

  static int16_t
    carrier_phase, // Phase angle of audio signal from receiver
    last_phase,    // Last carrier phase from detector o/p
    phase_slip,    // Average and normalized carrier phase slip
    max_slip;      // Maximum phase shift taken as phase slip

  // Counts slices (= 1/2 PSK31 elements) done
  static uint8_t slice_cnt = 0;

  // Normalized values of detector channels
  int
    chA_norm_lev_i,
    chA_norm_lev_q,
    chB_norm_lev_i,
    chB_norm_lev_q;

  // Wait for sample buffers full
  static BOOLEAN wait = True;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;


  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return( False );

  /* Wait for digimode buffer to fill with audio samples from demodulator.
   * When this function is slow to respond due to processor loading, then
   * the sem_post() in Hermes2/detect.c is executed more than once, so the
   * wait function must be repeated according to the semaphore's value
   */
  if( wait )
  {
    if( !Flag[GUEST_QUIT] ) sem_wait( &digimode_semaphore );
    wait = False;
  }

  // *** Following code acts like a synchronous detector,   ***
  // *** working at a nominal 500 Hz. To avoid the nulling  ***
  // *** of the o/p levels when a phase reversal occurs in  ***
  // *** the middle of a PSK31 element, two 'channels' are  ***
  // *** used separated by a 1/2 element time delay and the ***
  // *** one with the best squelch level is selected in use ***

  // Return psk phase after two PSK31 'slices' eg 1 element period
  for( ; slice_cnt < 2; slice_cnt++ )
  {
    // Summate samples for each half element
    for( ; elem_timer < psk31_rc.rx_psk_element2; elem_timer++ )
    {
      // Digimode buffer index
      static uint16_t digi_buf_idx = 0;

      // Get a audio sample from the digimodes buffer
      int16_t signal_sample =
        digimode_buffer[digi_buf_cnt][digi_buf_idx];

      // Display waterfall when input buffer full
      if( iFFT_Data(signal_sample, &psk31_ifft_data) )
        Display_Waterfall( &psk31_wfall, &psk31_ifft_data );

      // *** Calculate phase detector in-phase level ***
      // Samples are accumulated over a PSK31 element
      /* From 0 to 180 deg use sample level as is.
       * From 180 to 360 deg use rectified sample */
      if( det_phase < psk31_rc.quadrant_2 )
        det_level_i += signal_sample;
      else
        det_level_i -= signal_sample;

      // *** Calculate phase detector quadrature level ***
      // Samples are accumulated over a PSK31 element
      /* From 90 to 270 deg use sample level as is.
       * From 270 to 90 deg use rectified sample */
      if( (det_phase >= psk31_rc.quadrant_1) &&
          (det_phase <  psk31_rc.quadrant_3) )
        det_level_q += signal_sample;
      else
        det_level_q -= signal_sample;

      // Advance detector phase by one DSP sample count
      det_phase += 1.0 + freq_error / PSK31_AUDIO_FREQUENCY;

      // At a full signal cycle, reset the
      // phase count of the phase detector
      if( det_phase >= psk31_rc.quadrant_4 )
        det_phase -= psk31_rc.quadrant_4;

      // Wait for digimode buffer to be filled
      digi_buf_idx++;
      if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
      {
        digi_buf_idx = 0;

        // Increment current audio sample output buffer index
        digi_buf_cnt++;
        if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
          digi_buf_cnt = 0;

        if( digi_buf_cnt == digi_buf_input )
        {
          wait = True;
          return( False );
        }
      } // if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )

    } // for(elem_timer = 0; elem_timer < rc_data.psk_elem2;...
    elem_timer = 0;

    // Channel A sums current detector levels
    chA_level_i += det_level_i;
    chA_level_q += det_level_q;

    // Channel B sums last detector levels
    chB_level_i += last_lev_i;
    chB_level_q += last_lev_q;

    // Save current levels
    last_lev_i = det_level_i;
    last_lev_q = det_level_q;

  } // for( slice_cnt = 0; slice_cnt < 2; slice_cnt++ )
  slice_cnt = 0;

  // Normalize detector channel levels
  chA_norm_lev_i = chA_level_i / psk31_rc.rx_psk_element;
  chA_norm_lev_q = chA_level_q / psk31_rc.rx_psk_element;
  chB_norm_lev_i = chB_level_i / psk31_rc.rx_psk_element;
  chB_norm_lev_q = chB_level_q / psk31_rc.rx_psk_element;

  // Calculate sliding window average values for the channel
  // squelch. This is the vector magnitude of in-phase and
  // quadrature signals from the phase detector channels
  double dsquelch;
  dsquelch     = hypot( (double)chA_norm_lev_i, (double)chA_norm_lev_q );
  chA_squelch *= SQUELCH_WINDOW_1;
  chA_squelch += (uint32_t)dsquelch;
  chA_squelch /= SQUELCH_WINDOW;

  dsquelch     = hypot( (double)chB_norm_lev_i, (double)chB_norm_lev_q );
  chB_squelch *= SQUELCH_WINDOW_1;
  chB_squelch += (uint32_t)dsquelch;
  chB_squelch /= SQUELCH_WINDOW;

  // Set best channel select flag according to squelch
  if( 4 * chA_squelch > 5 * chB_squelch )
    Flag[PSK31_CHANNEL_A_SELECT] = True;
  else if( 4 * chB_squelch > 5 * chA_squelch )
    Flag[PSK31_CHANNEL_A_SELECT] = False;

  // Select best channel data according to flag
  if( Flag[PSK31_CHANNEL_A_SELECT] )
  {
    // Normalize squelch value and select level
    norm_squelch = chA_squelch;
    sel_level_i  = chA_norm_lev_i;
    sel_level_q  = chA_norm_lev_q;
  }
  else
  {
    // Normalize squelch value and select level
    norm_squelch = chB_squelch;
    sel_level_i  = chB_norm_lev_i;
    sel_level_q  = chB_norm_lev_q;
  }

  // Set squelch open flag
  if( norm_squelch > psk31_rc.squelch_threshold )
    Flag[PSK31_SQUELCH_OPEN] = True;
  else
    Flag[PSK31_SQUELCH_OPEN] = False;

  // Calculate new carrier phase if squelch is
  // open. Calculate carrier phase from arctan
  // of in-phase and quadrature detector levels
  double dphase;
  if( Flag[PSK31_SQUELCH_OPEN] )
    dphase = atan2( (double)sel_level_q, (double)sel_level_i );
  else
    dphase = 0.0;
  carrier_phase = (int16_t)( dphase * RAD2DEG );

  // Phase change between consecutive phase values
  *phase_chg = last_phase - carrier_phase;
  last_phase = carrier_phase;

  // Correct for discontinuity at +-180 deg
  if( *phase_chg > 180 )
    *phase_chg -= 360;
  else if( *phase_chg < -180 )
    *phase_chg += 360;

  // For BPSK phase slip limit = 90
  // For QPSK phase slip limit = 45
  if( Flag[PSK31_MODE_BPSK] )
    max_slip = 90;
  else
    max_slip = 45;

  // Limit phase changes to +-max_slip for slip rate calculations.
  // Caclulate average phase change with a sliding window averager
  if( abs(*phase_chg) < max_slip )
  {
    phase_slip *= PHASE_AVE_WINDOW_1;
    phase_slip += *phase_chg;
    phase_slip /= PHASE_AVE_WINDOW;
  }

  // Do Auto Freq Control
  if( Flag[PSK31_ENABLE_AFC] )
  {
    if( (phase_slip > 1) && (freq_error < MAX_FREQ_ERROR) )
      freq_error += 0.05;
    else if( (phase_slip < -1) && (freq_error > -MAX_FREQ_ERROR) )
      freq_error -= 0.05;
  }

  // Plot carrier phase on magniphase
  Queue_Draw( psk31_gui.psk31_scope );

  // Initialize levels
  chA_level_i = chA_level_q = 0;
  chB_level_i = chB_level_q = 0;
  det_level_i = det_level_q = 0;

  return( True );
} // End of PSK31_Phase()

//------------------------------------------------------------------------

/* Update_Magniphase()
 *
 * Displays carrier phase and magnitude
 */
  void
Update_Magniphase( cairo_t *cr )
{
  double x, y, e;

  // Factors needed to draw freq error pointer
  double
    scope_radius   = (double)(psk31_gui.psk31_scope_width - 16) / 2.0,
    scope_center_x = (double)psk31_gui.psk31_scope_width  / 2.0,
    scope_center_y = (double)psk31_gui.psk31_scope_height / 2.0;

  int
    level_i = sel_level_i / MAGNIPHASE_SCALE,
    level_q = sel_level_q / MAGNIPHASE_SCALE;

  // Draw a dark green filled circle, background for scope
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_arc( cr,
      scope_center_x,
      scope_center_y,
      scope_radius,
      0.0, M_2PI );
  cairo_fill( cr );

  // Reverse magniphase sense
  if( psk31_rc.rev_magniphase )
    level_q = -level_q;

  // Draw the signal magniphase vector
  cairo_set_source_rgb( cr, SCOPE_FOREGND );
  cairo_move_to( cr, scope_center_x, scope_center_y );
  cairo_line_to( cr,
      scope_center_x + (double)level_q,
      scope_center_y + (double)level_i );

  // Frequency error pointer co-ordinates
  e = freq_error / RAD2DEG * 18.0; // Error in rads
  x = scope_center_x + scope_radius * sin( e );
  y = scope_center_y - scope_radius * cos( e );

  // Reverse magniphase sense
  if( psk31_rc.rev_magniphase ) x = -x;

  // Draw frequency error pointer line
  cairo_move_to( cr, scope_center_x, scope_center_y );
  cairo_line_to( cr, x, y );

  // Stroke paths
  cairo_stroke( cr );

} // End of Update_Magniphase()

//------------------------------------------------------------------------

