/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "codec.h"
#include "phase.h"
#include "shared.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/modulate.h"
#include "../Hermes2/sound.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include <math.h>
#include <semaphore.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

// Trellis data structure for the Viterbi decoder
typedef struct _VITERBI_DEC
{
  int      distance; // Total excess distance to this state
  long int estimate; // Estimate of transmitted bit sequence
} trellis_state_t;

//-------------------------------------------------------------------

/*
   The VARICODE Alphabet

   The codes are transmitted left bit first, with '0' representing
   a phase reversal on BPSK and '1' representing a steady carrier.
   A minimum of two zeros is inserted between characters.
   Some implementations may not handle all the codes below 32.
 */

#define VARICODE_ALPHABET \
{ \
  /* ASCII   VARICODE       HEX */ \
  /* NUL   10 1010 1011 */  0x2ab, \
  /* SOH   10 1101 1011 */  0x2db, \
  /* STX   10 1110 1101 */  0x2ed, \
  /* ETX   11 0111 0111 */  0x377, \
  /* EOT   10 1110 1011 */  0x2eb, \
  /* ENQ   11 0101 1111 */  0x35f, \
  /* ACK   10 1110 1111 */  0x2ef, \
  /* BEL   10 1111 1101 */  0x2fd, \
  /* BS    10 1111 1111 */  0x2ff, \
  /* TAB      1110 1111 */  0x0ef, \
  /* LF          1 1101 */  0x01d, \
  /* VT    11 0110 1111 */  0x36f, \
  /* FF    10 1101 1101 */  0x2dd, \
  /* CR          1 1111 */  0x01f, \
  /* SO    11 0111 0101 */  0x375, \
  /* SI    11 1010 1011 */  0x3ab, \
  /* DLE   10 1111 0111 */  0x2f7, \
  /* DC1   10 1111 0101 */  0x2f5, \
  /* DC2   11 1010 1101 */  0x3ad, \
  /* DC3   11 1010 1111 */  0x3af, \
  /* DC4   11 0101 1011 */  0x35b, \
  /* NAK   11 0110 1011 */  0x36b, \
  /* SYN   11 0110 1101 */  0x36d, \
  /* ETB   11 0101 0111 */  0x357, \
  /* CAN   11 0111 1011 */  0x37b, \
  /* EM    11 0111 1101 */  0x37d, \
  /* SUB   11 1011 0111 */  0x3b7, \
  /* ESC   11 0101 0101 */  0x355, \
  /* FS    11 0101 1101 */  0x35d, \
  /* GS    11 1011 1011 */  0x3bb, \
  /* RS    10 1111 1011 */  0x2fb, \
  /* US    11 0111 1111 */  0x37f, \
  /*                  1 */  0x001, \
  /* !      1 1111 1111 */  0x1ff, \
  /* "      1 0101 1111 */  0x15f, \
  /* #      1 1111 0101 */  0x1f5, \
  /* $      1 1101 1011 */  0x1db, \
  /* %     10 1101 0101 */  0x2d5, \
  /* &     10 1011 1011 */  0x2bb, \
  /* '      1 0111 1111 */  0x17f, \
  /* (        1111 1011 */  0x0fb, \
  /* )        1111 0111 */  0x0f7, \
  /* *      1 0110 1111 */  0x16f, \
  /* +      1 1101 1111 */  0x1df, \
  /* ,         111 0101 */  0x075, \
  /* -          11 0101 */  0x035, \
  /* .         101 0111 */  0x057, \
  /* /      1 1010 1111 */  0x1af, \
  /* 0        1011 0111 */  0x0b7, \
  /* 1        1011 1101 */  0x0bd, \
  /* 2        1110 1101 */  0x0ed, \
  /* 3        1111 1111 */  0x0ff, \
  /* 4      1 0111 0111 */  0x177, \
  /* 5      1 0101 1011 */  0x15b, \
  /* 6      1 0110 1011 */  0x16b, \
  /* 7      1 1010 1101 */  0x1ad, \
  /* 8      1 1010 1011 */  0x1ab, \
  /* 9      1 1011 0111 */  0x1b7, \
  /* :        1111 0101 */  0x0f5, \
  /* ;      1 1011 1101 */  0x1bd, \
  /* <      1 1110 1101 */  0x1ed, \
  /* =         101 0101 */  0x055, \
  /* >      1 1101 0111 */  0x1d7, \
  /* ?     10 1010 1111 */  0x2af, \
  /* @     10 1011 1101 */  0x2bd, \
  /* A         111 1101 */  0x07d, \
  /* B        1110 1011 */  0x0eb, \
  /* C        1010 1101 */  0x0ad, \
  /* D        1011 0101 */  0x0b5, \
  /* E         111 0111 */  0x077, \
  /* F        1101 1011 */  0x0db, \
  /* G        1111 1101 */  0x0fd, \
  /* H      1 0101 0101 */  0x155, \
  /* I         111 1111 */  0x07f, \
  /* J      1 1111 1101 */  0x1fd, \
  /* K      1 0111 1101 */  0x17d, \
  /* L        1101 0111 */  0x0d7, \
  /* M        1011 1011 */  0x0bb, \
  /* N        1101 1101 */  0x0dd, \
  /* O        1010 1011 */  0x0ab, \
  /* P        1101 0101 */  0x0d5, \
  /* Q      1 1101 1101 */  0x1dd, \
  /* R        1010 1111 */  0x0af, \
  /* S         110 1111 */  0x06f, \
  /* T         110 1101 */  0x06d, \
  /* U      1 0101 0111 */  0x157, \
  /* V      1 1011 0101 */  0x1b5, \
  /* W      1 0101 1101 */  0x15d, \
  /* X      1 0111 0101 */  0x175, \
  /* Y      1 0111 1011 */  0x17b, \
  /* Z     10 1010 1101 */  0x2ad, \
  /* [      1 1111 0111 */  0x1f7, \
  /* \      1 1110 1111 */  0x1ef, \
  /* ]      1 1111 1011 */  0x1fb, \
  /* ^     10 1011 1111 */  0x2bf, \
  /* _      1 0110 1101 */  0x16d, \
  /* `     10 1101 1111 */  0x2df, \
  /* a             1011 */  0x00b, \
  /* b         101 1111 */  0x05f, \
  /* c          10 1111 */  0x02f, \
  /* d          10 1101 */  0x02d, \
  /* e               11 */  0x003, \
  /* f          11 1101 */  0x03d, \
  /* g         101 1011 */  0x05b, \
  /* h          10 1011 */  0x02b, \
  /* i             1101 */  0x00d, \
  /* j      1 1110 1011 */  0x1eb, \
  /* k        1011 1111 */  0x0bf, \
  /* l           1 1011 */  0x01b, \
  /* m          11 1011 */  0x03b, \
  /* n             1111 */  0x00f, \
  /* o              111 */  0x007, \
  /* p          11 1111 */  0x03f, \
  /* q      1 1011 1111 */  0x1bf, \
  /* r           1 0101 */  0x015, \
  /* s           1 0111 */  0x017, \
  /* t              101 */  0x005, \
  /* u          11 0111 */  0x037, \
  /* v         111 1011 */  0x07b, \
  /* w         110 1011 */  0x06b, \
  /* x        1101 1111 */  0x0df, \
  /* y         101 1101 */  0x05d, \
  /* z      1 1101 0101 */  0x1d5, \
  /* {     10 1011 0111 */  0x2b7, \
  /* |      1 1011 1011 */  0x1bb, \
  /* }     10 1011 0101 */  0x2b5, \
  /* ~     10 1101 0111 */  0x2d7, \
  /* DEL   11 1011 0101 */  0x3b5  \
}

//-------------------------------------------------------------------

// Polynomials for the convolution encoder
#define POLYNOM1    0x19
#define POLYNOM2    0x17

/*
   The CONVOLUTIONAL Code

   The left column contains the 32 combinations of a run of five
   Varicode bits, transmitted left bit first. The right column is
   the corresponding phase shift to be applied to the carrier, with
   0 meaning no shift, 1 meaning advance by 90, 2 meaning polarity
   reversal and 3 meaning retard by 90.  A continuous phase advance
   is the same as an HF frequency shift.
 */

#define CONVOLUTIONAL_CODE \
{ \
  /* Bits      Phase */ \
  /* 00000 */  2,       \
  /* 00001 */  1,       \
  /* 00010 */  3,       \
  /* 00011 */  0,       \
  /* 00100 */  3,       \
  /* 00101 */  0,       \
  /* 00110 */  2,       \
  /* 00111 */  1,       \
  /* 01000 */  0,       \
  /* 01001 */  3,       \
  /* 01010 */  1,       \
  /* 01011 */  2,       \
  /* 01100 */  1,       \
  /* 01101 */  2,       \
  /* 01110 */  0,       \
  /* 01111 */  3,       \
  /* 10000 */  1,       \
  /* 10001 */  2,       \
  /* 10010 */  0,       \
  /* 10011 */  3,       \
  /* 10100 */  0,       \
  /* 10101 */  3,       \
  /* 10110 */  1,       \
  /* 10111 */  2,       \
  /* 11000 */  3,       \
  /* 11001 */  0,       \
  /* 11010 */  2,       \
  /* 11011 */  1,       \
  /* 11100 */  2,       \
  /* 11101 */  1,       \
  /* 11110 */  3,       \
  /* 11111 */  0        \
}

/*
   For example, the "space" symbol - a single 1 preceded and followed
   by zeros - would be represented by successive run-of-five groups
   00000, 00001, 00010, 00100, 01000, 10000, 00000.
   The transmitter will send the QPSK pattern .. 2,1,3,3,0,1,2, ..
   Note that a continuous sequence of zeros (the idle sequence)
   gives continuous reversals, the same as BPSK.
 */

//-------------------------------------------------------------------

#define PHASE_SHIFT_0       0       // PSK phase reference
#define PHASE_SHIFT_90      1       // PSK 90 phase advance
#define PHASE_SHIFT_180     2       // PSK 180 phase reversal
#define PHASE_SHIFT_270     3       // PSK 90 phase retard

#define NUM_VARICODE_CHARS  128

//------------------------------------------------------------------------

// Varicode alphabet in HEX
static uint16_t varicode_table[NUM_VARICODE_CHARS] = VARICODE_ALPHABET;

// Index to transmit samples buffer
static uint32_t xmit_buf_idx = 0;

// cos lookup wavetable
static int16_t *cosine_table = NULL;

// Phasing signs
static int16_t Sign_I = 1, Sign_Q = 1;

//------------------------------------------------------------------------

/* Distance()
 *
 * Distance meteric for the Viterbi decoder
 */
  static int
Distance( int A, int B)
{
  int dist;

  // Shortest distance around the circumference
  // (phase circle) between two points A and B
  dist = abs(A - B);
  if( dist > 180 ) dist = 360 - dist;

  return( dist );
} // Distance()

//------------------------------------------------------------------------

// Viterbi decoder trellis
static trellis_state_t trellis[16];

// Convolutional code symbols
static int symbols[32];

/* Viterbi_Decoder()
 *
 * The Viterbi Decoder
 */
  static uint8_t
Viterbi_Decoder( int16_t phase_chg )
{
  int
    min_dist,  // Tracks minimum distance in trellis
    dists[32], // Distance metrics for the Viterbi
    idx;

  // Bit sequence esimates
  long int ests[32];

  uint8_t
    select, // Select smallest distance
    vote;   // Vote counter for bit guess


  // Reverse phase for receiving on LSB
  if( Flag[PSK31_MODE_QPSK_LSB] )
    phase_chg = 360 - phase_chg;

  // Calculate distances for all states & both current data values
  min_dist = 65535;
  for( idx = 0; idx < 32; idx++ )
  {
    // Added distance = distance between rcvd phase and predicted symbol
    dists[idx] = trellis[idx/2].distance + Distance(phase_chg, symbols[idx]);

    // Keep track of the smallest distance
    if( dists[idx] < min_dist ) min_dist = dists[idx];

    // New bit sequence estimate
    ests[idx] = (trellis[idx/2].estimate << 1) + (idx & 1);

  } // for( idx = 0; idx < 32; idx++ )

  // For each state in the new trellis array
  for( idx = 0; idx < 16; idx++ )
  {
    // Select lowest distance
    if( dists[idx] < dists[16 + idx] ) select = 0;
    else select = 16;

    // Update excess distances
    trellis[idx].distance = dists[select + idx] - min_dist;

    // Keep the new estimate
    trellis[idx].estimate = ests[select + idx];

  } // for( idx = 0; idx < 16; idx++ )

  // Take a vote of the 20th bits
  vote = 0;
  for( idx = 0; idx < 16; idx++ )
    if( (trellis[idx].estimate & (1 << 20)) > 0 )
      vote++;

  // 'phase_chg' is used as a random number
  if( vote == 8 ) return( phase_chg & 1 );
  else return( vote < 8 );

} // Viterbi_Decoder()

//------------------------------------------------------------------------

/* Varicode_Bit()
 *
 * Returns Varicode bits decoded from carrier phase shifts
 */
  static BOOLEAN
Varicode_Bit( uint8_t *varicode_bit )
{
  // Index to a bit in 3-bit sequence
  uint8_t bit_idx;

  // Phase change from one PSK31 element to the next
  int16_t phase_chg;

  /* Ring buffer and its index, for storing
   * 3 consecutive phase change values */
  static int16_t ring_buffer[3];


  /* Get PSK31 phase change. A false return means the function
   * is waiting for sound data and it should be called again */
  if( !PSK31_Phase(&phase_chg) )
  {
    if( !PSK31_Phase(&phase_chg) )
      return( False );
  }

  /* Simulated PSK31 phase changes for testing the
   * decoder. They produce the characters cdfhmpu
   {
   int16_t phs;
   static uint8_t txt_idx = 0, bit_idx = 0, ave = 0;
   long rnd;
   static uint8_t txt[7] = { 0x2f, 0x2d, 0x3d, 0x2b, 0x3b, 0x03f, 0x037 };

   rnd = rand();
   phs = (int16_t)( 60 - (120 * rnd) / RANDOM_MAX );
   if( !(txt[txt_idx] & (0x80 >> bit_idx)) )
   {
   phs = 180 - phs;
   if( phs > 180 ) phs = phs - 360;
   ave *= 19;
   ave += abs( phs );
   ave /= 20;
   }
   phase_chg = phs;

   printf( "Bit: %3d  Phase chg: %3d  Average chg: %3d\n",
   txt[txt_idx] & (0x80 >> bit_idx), phase_chg, ave );

   bit_idx++;
   if( bit_idx >= 8 )
   {
   bit_idx = 0;
   txt_idx++;
   if( txt_idx >= 7 ) txt_idx = 0;
   }
   } */

  // If mode is BPSK, use 3-bit sequence detector
  if( Flag[PSK31_MODE_BPSK] )
  {
    int16_t
      diff,      // To calculate the phase value for bits in 3-bit sequence
      phase_min; // Minimum value of phase change sum

    uint8_t
      bit_seq,   // The bit sequence selected for having min phase ave
      seq_idx,   // Index for giving a 3-bit seq 000-111
      bit_cnt;   // Count of number of bits tested in 3-bit sequence

    static uint8_t ring_idx = 0;

    // Store phase change in ring buffer
    ring_buffer[ring_idx] = (int16_t)abs( phase_chg );
    ring_idx++;
    if( ring_idx >= 3 ) ring_idx = 0;

    /* For a '0' bit in the 3-bit sequence, 180 deg
     * is subtracted from the value in the ring buffer,
     * and for a '1' bit the value is not changed */
    bit_seq   = 0;
    phase_min = 541; // 3*180+1

    // Produce all combinations 000-111 of a 3-bit field
    for( seq_idx = 0; seq_idx < 8; seq_idx++ )
    {
      bit_idx = 1;
      int16_t phase_sum = 0;

      // For each combination, calculate phase change sum
      for( bit_cnt = 0; bit_cnt < 3; bit_cnt++ )
      {
        /* If bit is 0, subtract 150 from the ring buffer.
         * This is the estimated average phase change for 0 */
        diff = ring_buffer[ring_idx];
        if( !(seq_idx & bit_idx) ) diff -= 150;
        phase_sum += abs( diff );
        bit_idx <<= 1;
        ring_idx++;
        if( ring_idx >= 3 ) ring_idx = 0;
      } // for( bit_cnt = 0; bit_cnt < 3; bit_cnt++ )

      /* Find which 3-bit combination has the lowest
       * minimum phase sum. This is the 3-bit field
       * that is the nearest match to the incoming
       * sequence of phase change values */
      if( phase_min > phase_sum )
      {
        bit_seq = seq_idx;
        phase_min = phase_sum;
      }
    } // for( seq_idx = 0; seq_idx < 8; seq_idx++ )

    /* The left most bit in the 3-bit
     * field is the last bit received */
    *varicode_bit = bit_seq >> 2;
  }
  else   // If in QPSK mode
    *varicode_bit = Viterbi_Decoder( phase_chg );

  return( True );
} // Varicode_Bit( void )

//------------------------------------------------------------------------

/* Decode_PSK_Character()
 *
 * Decodes a PSK31-coded character from
 * the sequence of phase reversals in PSK31
 * signal, producing an Ascii equivalent
 */
  BOOLEAN
Decode_PSK_Character( uint16_t *dec_char )
{
  // One bit of a Varicode character
  uint8_t varicode_bit;

  // Assembles varicode character
  static uint16_t varicode = 0;


  // Get next Varicode bit
  if( !Varicode_Bit(&varicode_bit) )
  {
    *dec_char = NO_CHARACTER;
    return( False );
  }

  // Enter new varicode bit
  varicode <<= 1;
  varicode |= varicode_bit;

  // Return char on two consecutive 0's (char space)
  if( (varicode & 0x03) == 0 )
  {
    // New decoded ASCII character
    uint16_t ascii_char;

    // Dump trailing 0's of SPACE
    varicode >>= 2;

    // Convert varicode to ascii
    // Look up varicode hex code in table
    for( ascii_char = 0; ascii_char < NUM_VARICODE_CHARS; ascii_char++ )
      if( varicode_table[ascii_char] == varicode )
        break;

    varicode = 0;

    // Change CR to LF, seems needed
    if( ascii_char == CR ) ascii_char = LF;
    *dec_char = ascii_char;

  } // if( (varicode & 0x03) == 0 )
  else *dec_char = NO_CHARACTER;

  return( True );
} // Decode_PSK_Character()

//------------------------------------------------------------------------

/* Cosine_Rising_Edge()
 *
 * Fills the Xmit buffer with a raised cosine rising edge
 */
  static void
Cosine_Rising_Edge( void )
{
  // Index to cosine lookup table
  uint16_t cos_idx;
  for( cos_idx = 0; cos_idx < psk31_rc.tx_psk_element2; cos_idx++ )
  {
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = Sign_I * cosine_table[cos_idx];
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = Sign_Q * cosine_table[cos_idx];
    xmit_buf_idx++;
  }

} // Cosine_Rising_Edge()

//------------------------------------------------------------------------

/* Cosine_Falling_Edge()
 *
 * Fills the Xmit buffer with a raised cosine falling edge
 */
  static void
Cosine_Falling_Edge( void )
{
  // Index to cosine lookup table
  int16_t cos_idx;
  for( cos_idx = (int16_t)psk31_rc.tx_psk_element2 - 1; cos_idx >= 0; cos_idx-- )
  {
    xmit_buffer.xmit_buf_i[xmit_buf_idx] = Sign_I * cosine_table[cos_idx];
    xmit_buffer.xmit_buf_q[xmit_buf_idx] = Sign_Q * cosine_table[cos_idx];
    xmit_buf_idx++;
  }

} // Cosine_Falling_Edge()

//------------------------------------------------------------------------

/* Steady_Tone()
 *
 * Fills Xmit buffer with steady tone
 */
  static void
Steady_Tone( uint32_t tone_dur, BOOLEAN clear )
{
  uint32_t idx;

  // Clear flag indicates steady tone
  if( clear )
    for( idx = 0; idx < tone_dur; idx++ )
    {
      xmit_buffer.xmit_buf_i[xmit_buf_idx] = Sign_I * IQ_SAMPLE_MAX;
      xmit_buffer.xmit_buf_q[xmit_buf_idx] = Sign_Q * IQ_SAMPLE_MAX;
      xmit_buf_idx++;
    }
  else // Indicates clearing of buffer with zero level
    for( idx = 0; idx < tone_dur; idx++ )
    {
      xmit_buffer.xmit_buf_i[xmit_buf_idx] = 0;
      xmit_buffer.xmit_buf_q[xmit_buf_idx] = 0;
      xmit_buf_idx++;
    }

} // Steady_Tone()

//------------------------------------------------------------------------

/* Psk31_Transmit_Preamble()
 *
 * Transmits a steady tone and then reversals as the PSK31 pre-amble
 */
  BOOLEAN
Psk31_Transmit_Preamble( void )
{
  // Tone duration in samples of audio
  uint32_t tone_duration;

  // *** Transmit a steady tone for psk31_rc.tx_preamble_tone ***
  tone_duration = DUC_SAMPLE_RATE * psk31_rc.tx_preamble_tone;

  // Initialize buffer idx
  xmit_buf_idx = 0;

  // Fill buffer with cosine rising edge
  Cosine_Rising_Edge();

  // Fill buffer with the steady tone samples
  Steady_Tone( tone_duration, True );

  // *** Write buffer to DUC, abort on error ***
  xmit_buffer.xmit_buf_len = tone_duration;
  if( !Flag[GUEST_QUIT] ) sem_wait( &duc_send_semaphore );
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  // *** Transmit psk31_rc.num_rev reversals ***
  for( uint8_t idx = 0; idx < psk31_rc.tx_preamble_rev; idx++ )
    if( !Psk31_Transmit_Element(PHASE_SHIFT_180) )
      return( False );

  return( True );
} // Psk31_Transmit_Preamble()

//------------------------------------------------------------------------

/* Psk31_Transmit_Postamble()
 *
 * Transmits reversals followed by steady tone
 * for the tail end of a PSK31 transmission
 */
  BOOLEAN
Psk31_Transmit_Postamble( void )
{
  // Tone duration in samples of audio
  uint32_t tone_dur;

  // *** Transmit psk31_rc.num_rev reversals ***
  for( uint8_t idx = 0; idx < psk31_rc.tx_preamble_rev; idx++ )
    if( !Psk31_Transmit_Element(PHASE_SHIFT_180) )
      return( False );

  // *** Transmit a steady tone for psk31_rc.tone_duration sec ***
  // Tone duration in samples
  tone_dur = DUC_SAMPLE_RATE * psk31_rc.tx_preamble_tone;

  // Initialize buffer index
  xmit_buf_idx = 0;

  // Fill buffer with the steady tone samples
  Steady_Tone( tone_dur, True );

  // Fill buffer with cosine falling edge
  Cosine_Falling_Edge();

  // *** Write samples buffer to DUC ***
  xmit_buffer.xmit_buf_len = tone_dur;
  if( !Flag[GUEST_QUIT] ) sem_wait( &duc_send_semaphore );
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  // Initialize buffer index
  xmit_buf_idx = 0;

  // Fill buffer with zero levels
  Steady_Tone( tone_dur / 2, False );

  // *** Write samples buffer to DUC ***
  xmit_buffer.xmit_buf_len = tone_dur / 2;
  if( !Flag[GUEST_QUIT] ) sem_wait( &duc_send_semaphore );
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  return( True );
} // Psk31_Transmit_Postamble()

//------------------------------------------------------------------------

/* Psk31_Transmit_Element()
 *
 * Transmits a PSK31 element with the specified phase shift
 */
  BOOLEAN
Psk31_Transmit_Element( uint8_t phase_shift )
{
  // Initialize buffer idx according to stereo/mono mode
  xmit_buf_idx = 0;

  // Phase shift carrier as requested
  switch( phase_shift )
  {
    case PHASE_SHIFT_0: // *** Send a steady tone ***
                        // Fill buffer with the steady tone samples
      Steady_Tone( psk31_rc.tx_psk_element, True );
      break;

    case PHASE_SHIFT_180: // *** Send a phase reversal ***
                          // Fill 1/2 buffer with cosine shaped falling edge
      Cosine_Falling_Edge();

      // Reverse phase by changing appropriate sign(s)
      Sign_I = -Sign_I;
      Sign_Q = -Sign_Q;

      // Fill rest of buffer with cosine shaped rising edge
      Cosine_Rising_Edge();
      break;

    case PHASE_SHIFT_90: // Send a +90 degree phase shift
                         // Fill 1/2 buffer with cosine shaped falling edge
      Cosine_Falling_Edge();

      // Change phase +90 deg by changing appropriate sign(s)
      Sign_Q = -Sign_Q;

      // Fill rest of buffer with cosine shaped rising edge
      Cosine_Rising_Edge();
      break;

    case PHASE_SHIFT_270: // Send a -90 degree phase shift
      // Fill 1/2 buffer with cosine shaped falling edge

      // Change phase -90 deg by changing appropriate sign(s)
      Sign_I = -Sign_I;

      // Fill rest of buffer with cosine shaped rising edge
      Cosine_Rising_Edge();
  } // switch( phase_shift )

  // *** Write samples buffer to DUC ***
  xmit_buffer.xmit_buf_len = psk31_rc.tx_psk_element;
  if( !Flag[GUEST_QUIT] ) sem_wait( &duc_send_semaphore );
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  return( True );
} // Psk31_Transmit_Element()

//------------------------------------------------------------------------

/* Transmit_PSK_Character()
 *
 * Transmits the PSK31 elements of an ascii character
 */
  BOOLEAN
Transmit_PSK_Character( guint ascii_chr )
{
  // Varicode character in Hex
  uint16_t varicode_chr;

  uint16_t idx;

  // Required PSK31 phase shift
  uint8_t phase_shift;

  // Phase shift mappings
  static const uint8_t conv_code[32] = CONVOLUTIONAL_CODE;

  // QPSK encoder shift register
  static uint16_t shift_reg = 0;

  // *** If ascii_chr is NULL_CHAR (='last character') flag ***
  if( ascii_chr == PSK31_NULL_CHAR )
  {
    // Transmit Morse code CW ID
    if( Flag[GUEST_TRANSMIT_ID] )
    {
      Flag[GUEST_TRANSMIT_ID] = False;

      // Initialize buffer idx
      xmit_buf_idx = 0;

      // Fill buffer with cosine falling edge
      Cosine_Falling_Edge();

      // *** Write samples buffer to DUC ***
      xmit_buffer.xmit_buf_len = psk31_rc.tx_psk_element2;
      if( !Flag[GUEST_QUIT] ) sem_wait( &duc_send_semaphore );
      if( !xmit_buffer.status )
      {
        MOX_Control( MOX_OFF );
        return( False );
      }

      // Send Morse code ID
      Clear_DUC();
      Flag[HERMES2_SEND_DUC_PACKET] = False;
      snprintf( hermes2_rc.morse_mesg,
          sizeof(hermes2_rc.morse_mesg), " DE %s TU", op_data.call );
      Flag[TRANSMIT_MORSE_MESG] = True;
      Modulator = Morse_Transmit;
      if( !Flag[GUEST_QUIT] ) sem_wait( &duc_send_semaphore );

      // Restore DUC Send modulator
      Modulator = DUC_Buffer_Transmit;
      Flag[HERMES2_SEND_DUC_PACKET] = True;

      // Initialize buffer idx
      xmit_buf_idx = 0;

      // Fill buffer with cosine shaped rising edge
      Cosine_Rising_Edge();

      // *** Write samples buffer to DUC ***
      xmit_buffer.xmit_buf_len = psk31_rc.tx_psk_element;
      if( !Flag[GUEST_QUIT] ) sem_wait( &digimode_semaphore );
      if( !xmit_buffer.status )
      {
        MOX_Control( MOX_OFF );
        return( False );
      }
    } // if( Flag[PSK31_TRANSMIT_MORSE_ID] )

    // In BPSK, just transmit reversals
    if( Flag[PSK31_MODE_BPSK] )
    {
      if( !Psk31_Transmit_Element(PHASE_SHIFT_180) )
        return( False );
      else
        return( True );
    }
    else
    {
      // Flush shift register
      for( idx = 0; idx < 5; idx++ )
      {
        shift_reg <<= 1;

        // Only need the last 5 bits
        shift_reg &= 0x1F;

        // Find required QPSK phase shift
        phase_shift = conv_code[shift_reg];

        // Reverse odd phases for LSB
        if( (Flag[PSK31_MODE_QPSK_LSB]) && (phase_shift & 1) )
          phase_shift = 4 - phase_shift;

        if( !Psk31_Transmit_Element(phase_shift) )
          return( False );

      } // for( idx = 0; idx < 5; idx++ )
    } // else of: if( isFlagSet(MODE_BPSK) )
  } // if( ascii_chr == PSK31_NULL_CHAR )
  else
  {
    // Count of consecutive zeroes
    uint8_t zero_cnt;

    // Convert ASCII to Varicode
    if( ascii_chr>= NUM_VARICODE_CHARS ) return( True );
    varicode_chr = varicode_table[ascii_chr];

    /* Find beginning of varicode character
     * including the trailing zeroes below */
    idx = 1;
    zero_cnt = 0;
    while( zero_cnt < 2 )
    {
      if( (varicode_chr & idx) == 0 )
        zero_cnt++;
      else
        zero_cnt = 0;

      idx <<= 1;
    }

    // Add two trailing zeroes
    varicode_chr <<= 2;

    // Bring idx back to 1st '1'
    idx >>= 1;

    // *** Transmit character in BPSK or QPSK ***
    if( Flag[PSK31_MODE_BPSK] )
    {
      // Transmit varicode character bits as phase reversals
      while( idx )
      {
        if( idx & varicode_chr ) // A '1' bit
        {
          if( !Psk31_Transmit_Element(PHASE_SHIFT_0) )
            return( False );
        }
        else // A '0' bit
        {
          if( !Psk31_Transmit_Element(PHASE_SHIFT_180) )
            return( False );
        }

        // Go down the varicode character
        idx >>= 1;
      } // while( idx )
    } // if( Flag[PSK31_MODE_BPSK] )
    else // If QPSK
    {
      while( idx )
      {
        shift_reg <<= 1;

        // Enter bits from Varicode char to shift register
        if( idx & varicode_chr ) // A '1' bit
          shift_reg |= 1;
        else // A '0' bit
          shift_reg |= 0;

        // Only need the last 5 bits
        shift_reg &= 0x1f;

        // Find required QPSK phase shift
        phase_shift = conv_code[shift_reg];

        // Reverse odd phases for LSB
        if( (Flag[PSK31_MODE_QPSK_LSB]) && (phase_shift & 1) )
          phase_shift = 4 - phase_shift;

        if( !Psk31_Transmit_Element(phase_shift) )
          return( False );

        // Go down the varicode character
        idx >>= 1;
      } // while( idx )
    } // else of: if( isFlagSet(MODE_BPSK) )
  } // else of: if( ascii_chr == PSK31_NULL_CHAR )

  return( True );
} // Transmit_PSK_Character()

//------------------------------------------------------------------------

/* Parity()
 *
 * Returns odd (?) parity
 */
  static unsigned char
Parity( unsigned char data )
{
  unsigned char count;

  count = 0;
  while( data > 0 )
  {
    if( data & 1 ) count++;
    data >>= 1;
  }

  return( count & 1 );
} // Parity()

//------------------------------------------------------------------------

/* Psk31_Init_Viterbi()
 *
 * Initializes the Viterbi decoder
 */
  void
Psk31_Init_Viterbi( void )
{
  const int dprime[16] =
  {
    450, 540, 270, 360, 360, 270, 360, 450,
    360, 270, 360, 270, 270, 180,  90,   0
  };
  int idx;

  // Generate the encode table
  for( idx = 0; idx < 32; idx++ )
    symbols[idx] = 90*(2*Parity(idx & POLYNOM1)+Parity(idx & POLYNOM2) );

  for( idx = 0; idx < 16; idx++ )
  {
    trellis[idx].distance = dprime[idx];
    trellis[idx].estimate = 0xFFFFFFF0 + (unsigned int)idx;
  }

} // Psk31_Init_Viterbi( void )

//------------------------------------------------------------------------

/* Psk31_Make_Cosine_Table()
 *
 * Builds a raised cosine wavetable for tone generation
 */
  void
Psk31_Make_Cosine_Table( void )
{
  uint16_t idx;
  double step = M_PI_2 / (double)psk31_rc.tx_psk_element2;

  Mem_Alloc((void **) &cosine_table, (size_t)psk31_rc.tx_psk_element2 * sizeof(int16_t) );

  // This makes a raised cosine table
  for( idx = 0; idx < psk31_rc.tx_psk_element2; idx++ )
    cosine_table[idx] = (int16_t)( IQ_SAMPLE_MAX * sin(step * (double)idx) );

  return;
} // Psk31_Make_Cosine_Table()

//------------------------------------------------------------------------

// Free resources
  void
Psk31_Free_Codec( void )
{
  Mem_Free( (void **) &cosine_table );
}

//------------------------------------------------------------------------

