/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "callback_func.h"
#include "display.h"
#include "interface.h"
#include "operation.h"
#include "phase.h"
#include "codec.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/interface.h"
#include "../rsid/rsid_modes.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdlib.h>
#include <pthread.h>

// For clearing text buffers
static GtkTextBuffer *text_buffer;

#define PSK31_SQUELCH_MULT  1200.0

// All the callback functions of the PSK31 client

//------------------------------------------------------------------------

  void
on_psk31_window_destroy(
    GObject  *object,
    gpointer  user_data)
{
  PSK31_Cleanup();
}

//------------------------------------------------------------------------

  gboolean
on_psk31_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data)
{
  Save_Window_Position( psk31_gui.window, psk31_rc.rc_fpath );
  Guest_Quit_Activate( psk31_gui.window );
  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_psk31_quit_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Save_Window_Position( psk31_gui.window, psk31_rc.rc_fpath );
  Guest_Quit_Activate( psk31_gui.window );
}

//------------------------------------------------------------------------

  gboolean
on_psk31_tx_textview_key_press(
    GtkWidget   *widget,
    const GdkEventKey *event,
    gpointer     user_data)
{
  if( Flag[GUEST_TRANSMIT_MODE] )
    return( Psk31_Tx_Textview_Key_Press(event) );
  else return( FALSE );
}

//------------------------------------------------------------------------

  gboolean
on_psk31_tx_textview_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  // Needed for 'Clear Window'
  text_buffer = psk31_gui.tx_text_buffer;
  return( Psk31_Textview_Button_Press(event) );
}

//------------------------------------------------------------------------

  gboolean
on_psk31_rx_textview_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  // Needed for 'Clear Window'
  text_buffer = psk31_gui.rx_text_buffer;
  return( Psk31_Textview_Button_Press(event) );
}

//------------------------------------------------------------------------

  void
on_psk31_transmit_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(psk31_gui.window_builder, "psk31_tx_textview") );
  Psk31_Change_Modes();
}

//------------------------------------------------------------------------

  void
on_psk31_receive_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;
  Psk31_Receive_Clicked();
}

//------------------------------------------------------------------------

  void
on_psk31_squelch_value_changed(
    GtkRange *range,
    gpointer  user_data)
{
  GtkWidget *hscale;
  GtkAdjustment *adjustment;

  // Read and enter squelch level
  hscale = Builder_Get_Object( psk31_gui.window_builder, "psk31_squelch_value" );
  adjustment = gtk_range_get_adjustment( GTK_RANGE(hscale) );
  psk31_rc.squelch_threshold =
    (uint16_t)( gtk_adjustment_get_value(adjustment) * PSK31_SQUELCH_MULT );
}

//------------------------------------------------------------------------

  gboolean
on_psk31_wfall_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  if( psk31_wfall.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, psk31_wfall.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
  }

  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_psk31_wfall_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  if( event->button == 1 )
  {
    if( Flag[GUEST_RECEIVING] )
      Tune_to_Monitor(
          event->x, CLICK_TUNE_RANGE, psk31_wfall.width,
          PSK31_AUDIO_FREQUENCY, &psk31_ifft_data );
  }

  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_psk31_scope_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  Update_Magniphase( cr );
  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_psk31_callsign_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Callsign_Changed( editable, &qso_record, "PSK31" );
}

//------------------------------------------------------------------------

  void
on_psk31_rst_out_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_RST( editable );
  Strlcpy( qso_record.dx_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)),
      sizeof(qso_record.dx_rst) );
}

//------------------------------------------------------------------------

  void
on_psk31_rst_in_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_RST( editable );
  Strlcpy( qso_record.my_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)),
      sizeof(qso_record.my_rst) );
}

//------------------------------------------------------------------------

  void
on_psk31_op_name_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field QSO record structure
  Strlcpy( qso_record.dx_name, buff, sizeof(qso_record.dx_name) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_name );
}

//------------------------------------------------------------------------

  void
on_psk31_qth_name_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field and QSO record structure
  Strlcpy( qso_record.dx_qth, buff, sizeof(qso_record.dx_qth) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_qth );
}

//------------------------------------------------------------------------

  void
on_psk31_locator_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Locator_Changed( editable, &qso_record, "HELL" );
}

//------------------------------------------------------------------------

  void
on_psk31_band_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Band_Changed( editable, &qso_record, "HELL" );
}

//------------------------------------------------------------------------

  void
on_psk31_new_record_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  New_Record( &qso_record, psk31_gui.window_builder, "PSK31" );
  Strlcpy( qso_record.mode_dir, "psk31", sizeof(qso_record.mode_dir) );
}

//------------------------------------------------------------------------

  void
on_psk31_save_record_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  // Save record if needed (and valid)
  if( Read_QSO_Record(&qso_record, psk31_gui.window_builder, "PSK31") )
    Guest_Save_Dialog( _("Save QSO Record to file?") );
  else
    Guest_Save_Dialog(
        _("QSO Record is incomplete\n"\
          "Do you still want to save?") );
}

//------------------------------------------------------------------------

  void
on_psk31_clear_fields_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Clear_Record_Fields( True, psk31_gui.window_builder, "PSK31" );
}

//------------------------------------------------------------------------

  void
on_psk31_clear_window_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  gtk_text_buffer_set_text( text_buffer, "", -1 );

  gtk_text_buffer_get_iter_at_offset(
      psk31_gui.tx_text_buffer, &psk31_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(psk31_gui.tx_text_buffer) );
  gtk_text_buffer_get_iter_at_offset(
      psk31_gui.rx_text_buffer, &psk31_gui.rx_text_buffer_iter,
      gtk_text_buffer_get_char_count(psk31_gui.rx_text_buffer) );

  gtk_widget_grab_focus(
      Builder_Get_Object(psk31_gui.window_builder, "psk31_tx_textview") );
}

//------------------------------------------------------------------------

  void
on_psk31_clear_windows_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_text_buffer_set_text(psk31_gui.tx_text_buffer, "", -1);
  gtk_text_buffer_set_text(psk31_gui.rx_text_buffer, "", -1);

  gtk_text_buffer_get_iter_at_offset(
      psk31_gui.tx_text_buffer, &psk31_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(psk31_gui.tx_text_buffer) );
  gtk_text_buffer_get_iter_at_offset(
      psk31_gui.rx_text_buffer, &psk31_gui.rx_text_buffer_iter,
      gtk_text_buffer_get_char_count(psk31_gui.rx_text_buffer) );

  gtk_widget_grab_focus(
      Builder_Get_Object(psk31_gui.window_builder, "psk31_tx_textview") );
}

//------------------------------------------------------------------------

  void
on_psk31_macro_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Psk31_Select_Macro( button );
}

//------------------------------------------------------------------------

  void
on_psk31_bpsk_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Flag[PSK31_MODE_BPSK]     = True;
    Flag[PSK31_MODE_QPSK_USB] = False;
    Flag[PSK31_MODE_QPSK_LSB] = False;
    Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_PSK31U );
    Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_PSK31;
    g_idle_add( Psk31_Set_TxRx_Labels, NULL );
  }
}

//------------------------------------------------------------------------

  void
on_psk31_qpsk_lsb_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Flag[PSK31_MODE_QPSK_LSB] = True;
    Flag[PSK31_MODE_QPSK_USB] = False;
    Flag[PSK31_MODE_BPSK]     = False;
    Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_PSK31L );
    Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_QPSK31;
    g_idle_add( Psk31_Set_TxRx_Labels, NULL );
    Psk31_Init_Viterbi();
  }
}

//------------------------------------------------------------------------

  void
on_psk31_qpsk_usb_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Flag[PSK31_MODE_QPSK_USB] = True;
    Flag[PSK31_MODE_QPSK_LSB] = False;
    Flag[PSK31_MODE_BPSK]     = False;
    Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_PSK31U );
    Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_QPSK31;
    g_idle_add( Psk31_Set_TxRx_Labels, NULL );
    Psk31_Init_Viterbi();
  }
}

//------------------------------------------------------------------------

  void
on_psk31_capitalize_letters_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[GUEST_CAPITALIZE] = True;
  else
    Flag[GUEST_CAPITALIZE] = False;
}

//------------------------------------------------------------------------

  void
on_psk31_record_qsos_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[GUEST_RECORD_QSO] = True;
  else
  {
    Flag[GUEST_RECORD_QSO] = False;
    if( qso_record.qso_record_fp != NULL )
    {
      Close_File( &qso_record.qso_record_fp );
      qso_record.qso_record_fp = NULL;
    }
  }
}

//------------------------------------------------------------------------

  void
on_psk31_identify_in_cw_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Flag[GUEST_TRANSMIT_ID] = True;
}

//------------------------------------------------------------------------

  void
on_psk31_freq_adjust_value_changed(
    GtkRange *range,
    gpointer  user_data)
{
  GtkWidget     *hscale;
  GtkAdjustment *adjustment;

  // Read and enter freq correction
  hscale = Builder_Get_Object( psk31_gui.window_builder, "psk31_freq_adjust" );
  adjustment = gtk_range_get_adjustment( GTK_RANGE(hscale) );
  freq_error = gtk_adjustment_get_value( adjustment );
}

//------------------------------------------------------------------------

  void
on_psk31_afc_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
    Flag[PSK31_ENABLE_AFC] = True;
  else
  {
    Flag[PSK31_ENABLE_AFC] = False;
    freq_error = 0.0;
  }
}

//------------------------------------------------------------------------

