/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callback_func.h"
#include "callbacks.h"
#include "display.h"
#include "operation.h"
#include "codec.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../common/guest_utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/modulate.h"
#include <ctype.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdlib.h>
#include <pthread.h>

//------------------------------------------------------------------------

/* Psk31_Change_Modes()
 *
 * Toggles down Macro TX and toggles
 * Keyboard Tx and Receive mode
 */
  void
Psk31_Change_Modes( void )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Toggle Tx/Rx/Macro modes
  if( Flag[GUEST_TRANSMIT_MACRO] )
  {
    Flag[GUEST_TRANSMIT_MACRO] = False;
  }
  else if( Flag[GUEST_TRANSMIT_KEYBD] )
  {
    Flag[GUEST_TRANSMIT_KEYBD]  = False;
  }
  else if( !ddv->transmit_on ) // MOX or Tune not on
  {
    // Set up control flags for Keyboard transmit
    Flag[GUEST_TRANSMIT_KEYBD] = True;

    // Wait for Rx mode to exit
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      pthread_join( hermes2_rc.guest_rx_thread, NULL );
    }

    // Clear keyboard buffer data
    keybd_buf.key_count    = 0;
    keybd_buf.char_pointer = 0;
    keybd_buf.space_count  = 0;
    keybd_buf.char_count   = 0;

    // Create a thread to transmit characters from keyboard
    if( !Psk31_Initialize_Transmit() ) return;
    if( !Pthread_Create(
          &hermes2_rc.guest_tx_thread, NULL, Psk31_Transmit_Keybd, NULL,
          _("Failed to create Transmit Keyboard thread")) )
      return;
  }
} // Psk31_Change_Modes()

//------------------------------------------------------------------------

/* Psk31_Select_Macro()
 *
 * Selects a macro to be transmitted
 */
  void
Psk31_Select_Macro( GtkButton *button )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;

  // Clear keyboard buffer data
  keybd_buf.key_count    = 0;
  keybd_buf.char_pointer = 0;
  keybd_buf.space_count  = 0;
  keybd_buf.char_count   = 0;

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(psk31_gui.window_builder, "psk31_tx_textview") );

  // Transmit a macro
  if( !Flag[GUEST_TRANSMIT_MACRO] )
  {
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      pthread_join( hermes2_rc.guest_rx_thread, NULL );
    }

    // Enable macro transmission if we are not
    // in transmit mode (MOX on or Tune active)
    if( !ddv->transmit_on || Flag[GUEST_TRANSMIT_KEYBD] )
    {
      if( !Flag[GUEST_TRANSMIT_KEYBD] && !Psk31_Initialize_Transmit() )
        return;

      Flag[GUEST_TRANSMIT_MACRO] = True;
      g_idle_add( Psk31_Set_TxRx_Labels, NULL );

      // Get macro index from the name of the macro button
      psk31_macro_idx =
        (uint8_t)( atoi(gtk_widget_get_name(GTK_WIDGET(button))) - 1 );

      // Create a thread to run Macro Transmit if not in Keyboard mode
      if( !Flag[GUEST_TRANSMIT_KEYBD] )
      {
        if( !Pthread_Create(
              &hermes2_rc.guest_tx_thread, NULL, Psk31_Transmit_Macro, NULL,
              _("Failed to create Transmit_Macro thread")) )
          return;
      }

      // Will start Macro thread from Keybd thread
      Flag[GUEST_TRANSMIT_KEYBD] = False;

    } // if( !ddv->transmit_on ... )
  } // if( !Flag[GUEST_TRANSMIT_MACRO] )
  else
  {
    Flag[GUEST_TRANSMIT_MACRO] = False;
    if( Flag[GUEST_TRANSMITTING] )
      pthread_join( hermes2_rc.guest_tx_thread, NULL );
  }

} // Psk31_Select_Macro()

//------------------------------------------------------------------------

/* Psk31_Receive_Clicked()
 *
 * Handles the on_psk31_receive_clicked callback
 */
  void
Psk31_Receive_Clicked( void )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Toggle Tx/Rx/Macro modes
  if( Flag[GUEST_RECEIVING] ) // If in Rx mode, terminated it
  {
    Flag[GUEST_RECEIVE_MODE] = False;
    pthread_join( hermes2_rc.guest_rx_thread, NULL );
  }
  else // Enter Rx mode if not in Tx mode
  {
    if( !Flag[GUEST_TRANSMIT_MACRO] &&
        !Flag[GUEST_TRANSMIT_KEYBD] &&
        !ddv->transmit_on )
    {
      Flag[GUEST_RECEIVE_MODE] = True;
      rx_print_chr.printchr = LF;
      if( !Pthread_Create(
            &hermes2_rc.guest_rx_thread, NULL, Psk31_Receive_Mode, NULL,
            _("Failed to create Receive thread")) )
      {
        Flag[GUEST_RECEIVE_MODE] = False;
        g_idle_add( Psk31_Set_TxRx_Labels, NULL );
        return;
      }
    }
    else // Wait for Tx mode to exit
    {
      Flag[GUEST_TRANSMIT_MACRO] = False;
      Flag[GUEST_TRANSMIT_KEYBD] = False;
      if( Flag[GUEST_TRANSMITTING] )
        pthread_join( hermes2_rc.guest_tx_thread, NULL );
    }
  } // if( Flag[GUEST_RECEIVE_MODE] )

  g_idle_add( Psk31_Set_TxRx_Labels, NULL );

} // Psk31_Receive_Clicked()

//------------------------------------------------------------------------

/* Psk31_Textview_Button_Press()
 *
 * Handles button press events on Rx and Tx textview
 */
  gboolean
Psk31_Textview_Button_Press( const GdkEventButton *event )
{
  switch( event->button )
  {
    case 1: // Focus to widget that received the button press
      return( FALSE );

    case 2: // Toggle Tx/Rx/Macro modes
      Psk31_Change_Modes();
      break;

    case 3: // Popup menu. "Sensitize Identify in CW" in keybord mode
      if( Flag[GUEST_TRANSMIT_KEYBD] )
        gtk_widget_set_sensitive(
            Builder_Get_Object( psk31_gui.popup_menu_builder, "psk31_identify_in_cw"), TRUE );
      else
        gtk_widget_set_sensitive(
            Builder_Get_Object( psk31_gui.popup_menu_builder, "psk31_identify_in_cw"), FALSE );

      gtk_menu_popup_at_pointer( GTK_MENU(psk31_gui.popup_menu), NULL );
  } // switch( event->button )

  return( TRUE );
} // Psk31_Textview_Button_Press()

//------------------------------------------------------------------------

/* Psk31_Tx_Textview_Key_Press()
 *
 * Handles keyboard input during transmit
 */
  gboolean
Psk31_Tx_Textview_Key_Press( const GdkEventKey *event )
{
  if( (event->keyval == GDK_KEY_Shift_L) ||
      (event->keyval == GDK_KEY_Shift_R) )
    return( FALSE );

  // Deal with Backspace key
  if( event->keyval == GDK_KEY_BackSpace )
  {
    // Roll back word wrap counter
    keybd_buf.char_count--;
    if( keybd_buf.char_count < 0 )
      keybd_buf.char_count += psk31_rc.word_wrap;

    // Enter Back Space code to key buffer
    keybd_buf.key_buf[keybd_buf.key_count] = BS;
    keybd_buf.key_count++;
    if( keybd_buf.key_count >= KEY_BUF_SIZE )
      keybd_buf.key_count = 0;

    // Print Backspace on text view
    tx_print_chr.printchr = GDK_KEY_BackSpace;
    Queue_Character( &tx_print_chr );

    return( TRUE );
  } // if( event->keyval == GDK_KEY_BackSpace )

  // Print Return on Return key or on Word Wrap
  keybd_buf.char_count++;
  if( (event->keyval == GDK_KEY_Return) ||
      ((keybd_buf.char_count >= psk31_rc.word_wrap) &&
       (event->keyval == ' ')) )
  {
    // For word wrap case add a GDK_KEY_Return entry to buffer
    keybd_buf.key_buf[keybd_buf.key_count] = GDK_KEY_Return;
    keybd_buf.key_count++;
    if( keybd_buf.key_count >= KEY_BUF_SIZE )
      keybd_buf.key_count = 0;

    // Print GDK_KEY_Return to text view
    tx_print_chr.printchr = GDK_KEY_Return;
    Queue_Character( &tx_print_chr );

    // Reset Word Wrap counter
    keybd_buf.char_count = 0;
    return( TRUE );
  }

  // Don't process non-ASCII characters
  if( (event->keyval & 0xFF) > 0x7F )
    return( TRUE );

  // Capitalize letters if enabled
  if( Flag[GUEST_CAPITALIZE] )
    keybd_buf.key_buf[keybd_buf.key_count] = (guint)toupper( event->keyval );
  else
    keybd_buf.key_buf[keybd_buf.key_count] = event->keyval;

  // Print character and increment keystroke count
  tx_print_chr.printchr = keybd_buf.key_buf[keybd_buf.key_count];
  Queue_Character( &tx_print_chr );
  keybd_buf.key_count++;
  if( keybd_buf.key_count >= KEY_BUF_SIZE )
    keybd_buf.key_count = 0;

  return( TRUE );
} // Psk31_Tx_Textview_Key_Press()

//------------------------------------------------------------------------

