/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "callbacks.h"
#include "display.h"
#include "interface.h"
#include "operation.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/sound.h"
#include "../common/utils.h"
#include "../common/guest_utils.h"
#include <gtk/gtk.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdlib.h>
#include <stdlib.h>
#include <string.h>

//------------------------------------------------------------------------

#define OLIVIA_NUM_OF_MODES     7

/* Olivia_Read_Config()
 *
 * Loads the oliviarc configuration file
 */
  gboolean
Olivia_Read_Config( gpointer data )
{
   // Buffer for Read_Line
  char line[81];

  // Config file pointer
  FILE *oliviarc;

  // Setup file path to oliviarc
  // Create the olivia config file name
  Strlcpy( olivia_rc_data.rc_fpath,
      getenv("HOME"), sizeof(olivia_rc_data.rc_fpath) );
  Strlcat( olivia_rc_data.rc_fpath,
      "/.hermes2/olivia/olivia.config", sizeof(olivia_rc_data.rc_fpath) );

  // Open olivia.config file
  oliviarc = fopen( olivia_rc_data.rc_fpath, "r" );
  if( oliviarc == NULL )
  {
    perror( olivia_rc_data.rc_fpath );
    Error_Dialog(
        _("Failed to open olivia.config file\n"\
          "Quit Olivia and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read word wrap column, abort if EOF
  if( Read_Line(line, oliviarc, _("Word Wrap Column") ) != SUCCESS )
    return( FALSE );
  olivia_rc_data.word_wrap = (uint8_t)atoi( line );

  // Read Receiver synchronizer integration length, abort if EOF
  if( Read_Line(line, oliviarc, _("Synchronizer Integration Length") ) != SUCCESS )
    return( FALSE );
  olivia_rc_data.sync_integ_len = (uint8_t)atoi( line );

  // Read Receiver synchronizer search margin, abort if EOF
  if( Read_Line(line, oliviarc, _("Synchronizer Search Margin") ) != SUCCESS )
    return( FALSE );
  olivia_rc_data.sync_margin = (uint8_t)atoi( line );

  // Read Receiver sync detection threshold, abort if EOF
  if( Read_Line(line, oliviarc, _("Sync Detection Threshold") ) != SUCCESS )
    return( FALSE );
  olivia_rc_data.sync_threshold = Atof( line );

  // Initialize fixed and permanent settings
  // in Parameters, Transmitter and Receiver
  MFSK_Params_Initialize( &Olivia_Parameters );
  MFSK_Transmitter_Initialize( &Transmitter );
  MFSK_Receiver_Initialize( &Receiver );

  // Read default Olivia mode
  if( Read_Line(line, oliviarc, _("Default Mode") ) != SUCCESS )
    return( FALSE );
  olivia_rc_data.mode = (uint8_t)atoi( line );
  if( olivia_rc_data.mode >= OLIVIA_NUM_OF_MODES )
  {
    Close_File( &oliviarc );
    Error_Dialog(
        _("Error reading olivia.config\n"\
          "Incorrect Olivia Mode\n"\
          "Quit olivia and correct"), HIDE_OK );
    return( FALSE );
  }

  // Activate default Olivia mode menu item
  gchar *modes[OLIVIA_NUM_OF_MODES] =
  {
    "olivia_4_125",   \
    "olivia_4_250",   \
    "olivia_8_250",   \
    "olivia_8_500",   \
    "olivia_16_500",  \
    "olivia_16_1000", \
    "olivia_32_1000"
  };
  gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
        Builder_Get_Object(olivia_gui.popup_menu_builder,
          modes[olivia_rc_data.mode])), TRUE );

  // *** Setup default main menu items ***
  // Read Record QSO enable flag, abort if EOF
  if( Read_Line(line, oliviarc, _("Record QSOs Enable") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(olivia_gui.popup_menu_builder, "olivia_record_qsos")), TRUE );
    Flag[GUEST_RECORD_QSO] = True;
  }
  else if( strcmp(line, "no") == 0 )
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(olivia_gui.popup_menu_builder, "olivia_record_qsos")), FALSE );
  else
  {
    Close_File( &oliviarc );
    Error_Dialog(
        _("Error reading olivia.config\n"\
          "Unrecognized menu option\n"\
          "Quit olivia and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Capitalize enable flag, abort if EOF
  if( Read_Line(line, oliviarc, _("Capitalize Enable") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(olivia_gui.popup_menu_builder, "olivia_capitalize_letters")), TRUE );
    Flag[GUEST_CAPITALIZE] = True;
  }
  else if( strcmp(line, "no") == 0 )
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(olivia_gui.popup_menu_builder, "olivia_capitalize_letters")), FALSE );
  else
  {
    Close_File( &oliviarc );
    Error_Dialog(
        _("Error reading olivia.config\n"\
          "Unrecognized menu option\n"\
          "Quit olivia and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Station info and Macros
  if( !Read_Station_Info(
        oliviarc, olivia_macro, "olivia_label", olivia_gui.window_builder) )
    return( FALSE );

  // Read main Window's position and place it
  if( Read_Line(line, oliviarc, _("Window Position") ) != SUCCESS )
    return( FALSE );
  gint x = (gint)atoi( line );
  gint y = (gint)atoi( &line[5] );
  gtk_window_move( GTK_WINDOW(olivia_gui.window), x, y );

  // Wait for GTK to complete its tasks
  while( g_main_context_iteration(NULL, FALSE) );

  Close_File( &oliviarc );
  return( FALSE );
} // End of Olivia_Read_Config()

//------------------------------------------------------------------

/* Olivia_Cleanup()
 *
 * Cleans up before quitting
 */
  void
Olivia_Cleanup( void )
{
  // Save outstanding records
  if( Flag[GUEST_SAVE_RECORD] ) Save_QSO_Record( &qso_record );

  // Close open files
  Close_File( &(qso_record.qso_record_fp) );
  Close_File( &(qso_record.adif_log_fp) );
  Close_File( &(qso_record.station_log_fp) );

  // Free Transmitter and Receiver objects
  Transmitter.Free( &Transmitter );
  Receiver.Free( &Receiver );

  // Free Digimode semaphores
  Init_Semaphore( &digimode_semaphore, False );
  Init_Semaphore( &duc_send_semaphore, False );

  // Free buffer allocations
  Free_iFFT( &olivia_ifft_data );

  // Free macro buffer pointers
  for( uint8_t idx = 0; idx < NUM_OF_LABELS; idx++ )
    Mem_Free( (void **) &olivia_macro[idx] );

  // Free waterfall pixbuf
  if( olivia_wfall.pixbuf ) g_object_unref( olivia_wfall.pixbuf );
  olivia_wfall.pixbuf = NULL;

  // Close some GUI objects
  g_object_unref( olivia_gui.window_builder );
  g_object_unref( olivia_gui.popup_menu_builder );
  olivia_gui.window_builder     = NULL;
  olivia_gui.popup_menu_builder = NULL;
  olivia_gui.window             = NULL;
  hermes2_gui.guest_window      = NULL;

  // Clear general flags
  Flag[GUEST_RECORD_QSO]        = False;
  Flag[GUEST_RECEIVE_MODE]      = False;
  Flag[GUEST_TRANSMIT_MODE]     = False;
  Flag[GUEST_TRANSMIT_TAG]      = False;
  Flag[GUEST_TRANSMIT_MACRO]    = False;
  Flag[GUEST_TRANSMIT_KEYBD]    = False;
  Flag[GUEST_KEYBD_BUSY]        = False;
  Flag[GUEST_CAPITALIZE]        = False;
  Flag[GUEST_SAVE_RECORD]       = False;
  Flag[GUEST_QUIT]              = False;
  Flag[HERMES2_SEND_DUC_PACKET] = False;
  Flag[TRANSMIT_MORSE_MESG]     = False;
  Flag[GUEST_RECEIVING]         = False;
  Flag[GUEST_TRANSMITTING]      = False;

  // Clear Olivia flags
  Flag[OLIVIA_MODE_4_125]   = False;
  Flag[OLIVIA_MODE_4_250]   = False;
  Flag[OLIVIA_MODE_8_250]   = True;
  Flag[OLIVIA_MODE_8_500]   = False;
  Flag[OLIVIA_MODE_16_500]  = False;
  Flag[OLIVIA_MODE_16_1000] = False;
  Flag[OLIVIA_MODE_32_1000] = False;
  Flag[BLOCK_FUNCTION]      = False;

  // Disable the common Guest modulation mode
  if( Transceiver[Indices.TRx_Index] != NULL )
    Transceiver[Indices.TRx_Index]->guest_modulation_mode = RX_MODE_ITEMS;

} // Olivia_Cleanup()

//------------------------------------------------------------------------

