/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "rateconv.h"
#include "struc.h"
#include "../common/utils.h"
#include <math.h>
#include <stdint.h>
#include <stdlib.h>

  static void
CRateConverter_Init( CRateConverter_t *Self )
{
  Self->FilterShape = NULL;
  Self->InputTap    = NULL;
}

  static void
CRateConverter_Free( CRateConverter_t *Self )
{
  Mem_Free( (void **) &(Self->FilterShape) );
  Mem_Free( (void **) &(Self->InputTap) );
}

//----------------------------------------------------------------------

  static void
CRateConverter_Default( CRateConverter_t *Self )
{
  Self->TapLen       = 16;
  Self->OverSampling = 16;
  Self->UpperFreq    = 3.0 / 8.0;
  Self->OutputRate   = 1.0;
}

//----------------------------------------------------------------------

  static void
CRateConverter_Preset( CRateConverter_t *Self )
{
  uint32_t Idx;

  // TapLen=Exp2(Log2(TapLen ));
  Self->FilterLen = Self->TapLen * Self->OverSampling;
  Mem_Realloc( (void **) &(Self->FilterShape), sizeof(double) * Self->FilterLen );
  Mem_Realloc( (void **) &(Self->InputTap),    sizeof(double) * Self->TapLen );

  for( Idx = 0; Idx < Self->FilterLen; Idx++ )
  {
    double Phase =
      ( M_PI * (double)(2 * (int)Idx - (int)Self->FilterLen) ) /
      (double)Self->FilterLen;

    // Blackman-Harris
    double Window =
      0.35875 +
      0.48829 * cos( Phase ) +
      0.14128 * cos( 2.0 * Phase ) +
      0.01168 * cos( 3.0 * Phase );

    double Filter = 1.0;
    if( Phase != 0.0 )
    {
      Phase *= Self->UpperFreq * (double)Self->TapLen;
      Filter = sin( Phase ) / Phase;
    }
    Self->FilterShape[Idx] = Window * Filter;
  }

  Self->Reset( Self );
}

//----------------------------------------------------------------------

  static void
CRateConverter_Reset( CRateConverter_t *Self )
{
  uint32_t Idx;

  Self->InputWrap = Self->TapLen - 1;
  for( Idx = 0; Idx < Self->TapLen; Idx++ )
  {
    Self->InputTap[Idx] = 0;
  }
  Self->InputTapPtr = 0;

  Self->OutputTime   = 0;
  Self->OutputPeriod = (double)Self->OverSampling / Self->OutputRate;
  Self->OutputBefore = 0;
  Self->OutputAfter  = 0;
  Self->OutputPtr    = 0;
}

//----------------------------------------------------------------------

  static double
CRateConverter_Convolute( CRateConverter_t *Self, size_t Shift )
{
  double Sum = 0.0;
  Shift = ( Self->OverSampling - 1 ) - Shift;

  uint32_t Idx = Self->InputTapPtr;
  for( ; Shift < Self->FilterLen; Shift += Self->OverSampling )
  {
    Sum += Self->InputTap[Idx] * Self->FilterShape[Shift];
    Idx++;
    Idx &= Self->InputWrap;
  }
  return( Sum );
}

//----------------------------------------------------------------------

  static void
CRateConverter_NewInput( CRateConverter_t *Self, double Input )
{
  Self->InputTap[Self->InputTapPtr] = Input;
  Self->InputTapPtr++;
  Self->InputTapPtr &= Self->InputWrap;
}

//----------------------------------------------------------------------

// Process samples, store output at a pointer ( you need to ensure enough storage )
  static int
CRateConverter_Process(
    CRateConverter_t *Self,
    double  *Input,
    uint32_t InputLen,
    double  *Output )
{
  int OutputLen = 0;

  while( True )
  {
    if( Self->OutputPtr )
    {
      size_t Idx = (size_t)( floor(Self->OutputTime) ) + 1;
      if( Idx >= Self->OverSampling )
      {
        if( InputLen == 0 ) break;
        Self->NewInput( Self, *Input );
        Input++;
        InputLen--;
        Idx -= Self->OverSampling;
        Self->OutputTime -= (double)Self->OverSampling;
      }

      Self->OutputAfter = Self->Convolute( Self, Idx );
      double Weight = (double)Idx - Self->OutputTime;
      *Output = Weight * Self->OutputBefore + (1.0 - Weight) * Self->OutputAfter;
      Output++;
      OutputLen++;
      Self->OutputPtr = 0;
    }
    else
    {
      size_t Idx = (size_t)( floor(Self->OutputTime + Self->OutputPeriod) );
      if( Idx >= Self->OverSampling )
      {
        if( InputLen == 0 ) break;
        Self->NewInput( Self, *Input );
        Input++;
        InputLen--;
        Idx -= Self->OverSampling;
        Self->OutputTime -= (double)Self->OverSampling;
      }

      Self->OutputBefore = Self->Convolute( Self, Idx );
      Self->OutputTime  += Self->OutputPeriod;
      Self->OutputPtr    = 1;
    }
  }

  return( OutputLen );
}

//----------------------------------------------------------------------

// Process samples. store output in a Seq<> with automatic allocation
  static int
CRateConverter_Process_Seq(
    CRateConverter_t *Self,
    double *Input,
    uint32_t InputLen,
    Seq_t *Output )
{
  uint32_t OutPtr = Output->Len;
  uint32_t MaxOutLen =
    (uint32_t)( ceil((double)InputLen * Self->OutputRate + 2.0) );
  Output->EnsureSpace( Output, OutPtr + MaxOutLen );
  int OutLen = Self->Process( Self, Input, InputLen, Output->Elem + OutPtr );
  Output->Len = OutPtr + (uint32_t)OutLen;
  return( OutLen );
}

//----------------------------------------------------------------------

// Initialize CRateConverter - my function
  void
CRateConverter_Initialize( CRateConverter_t *Self )
{
  Self->Init        = CRateConverter_Init;
  Self->Free        = CRateConverter_Free;
  Self->Default     = CRateConverter_Default;
  Self->Preset      = CRateConverter_Preset;
  Self->Reset       = CRateConverter_Reset;
  Self->Convolute   = CRateConverter_Convolute;
  Self->NewInput    = CRateConverter_NewInput;
  Self->Process     = CRateConverter_Process;
  Self->Process_Seq = CRateConverter_Process_Seq;

  CRateConverter_Init( Self );
  CRateConverter_Default( Self );
}

//----------------------------------------------------------------------

