/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "callback_func.h"
#include "operation.h"
#include "display.h"
#include "mfsk.h"
#include "shared.h"
#include "../common/shared.h"
#include "../common/guest_utils.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/modulate.h"
#include "../Hermes2/sound.h"
#include "../rsid/rsid.h"
#include <ctype.h>
#include <gtk/gtk.h>
#include <semaphore.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <math.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

//------------------------------------------------------------------------

// Index to macros
uint8_t olivia_macro_idx;

//------------------------------------------------------------------------

// Buffer for the Transmitter's output
static double *Tx_Buffer = NULL;

// Character string for Transmitter Char Monitor
#define MONITOR_STR_SIZE   31

//------------------------------------------------------------------------

/* Enter_Tx_Monitor_Char()
 *
 * Enters a character to the Tx Monitor string
 */
  void
Enter_Tx_Monitor_Char( uint8_t chr )
{
  static gchar monitor_str[MONITOR_STR_SIZE + 1] =
  { "                               " };

  // Clear string if chr == 0xFF
  if( chr == 0xFF )
  {
    for( uint8_t idx = 0; idx < MONITOR_STR_SIZE; idx++ )
      monitor_str[idx] = ' ';
    monitor_str[MONITOR_STR_SIZE] = '\0';
    return;
  }

  // Move string contents left one position
  for( uint8_t idx = 1; idx < MONITOR_STR_SIZE; idx++ )
    monitor_str[idx - 1] = monitor_str[idx];

  // Enter new character from Monitor if not BS
  if( chr == BS )
    monitor_str[MONITOR_STR_SIZE - 1] = '<';
  else
    monitor_str[MONITOR_STR_SIZE - 1] = (gchar)chr;
  monitor_str[MONITOR_STR_SIZE] = '\0';

  // Display characters from Transmitter's monitor FIFO
  if( olivia_gui.window != NULL )  // Can happen somehow
    Set_Entry_Text( GTK_ENTRY(olivia_gui.tx_monitor_string), monitor_str );

} // Enter_Tx_Monitor_Char()

//------------------------------------------------------------------------

/* Tx_Input_Empty()
 *
 * Displays status of Transmitter's Input FIFO (empty or not)
 */
  static gboolean
Tx_Input_Empty( gpointer arg )
{
  // Current status (True = Not empty)
  static BOOLEAN status = False;
  BOOLEAN empty = Transmitter.Input.Empty( &(Transmitter.Input) );
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Show green "LED"
  if( empty && status )
  {
    status = False;
    gtk_image_set_from_icon_name(
        GTK_IMAGE(olivia_gui.tx_input_status), "gtk-yes", GTK_ICON_SIZE_BUTTON );
    Flag[GUEST_TRANSMIT_MACRO] = False;

    // Make a beep on Transceiver's audio O/P
    Flag[BEEP_ACTIVE] = True;
    TRx->receiver_muted = False;
  }
  else if( !empty && !status ) // Show red "LED"
  {
    status = True;
    gtk_image_set_from_icon_name(
        GTK_IMAGE(olivia_gui.tx_input_status), "gtk-no", GTK_ICON_SIZE_BUTTON );
  }

  return( FALSE );
} // Tx_Input_Empty()

//------------------------------------------------------------------------

/* Olivia_Initialize_Transmit()
 *
 * Initializes the Transmit mode
 */
  BOOLEAN
Olivia_Initialize_Transmit( void )
{
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Open qso record file
  if( !Open_Record_File("olivia") )
  {
    Flag[HERMES2_SEND_DUC_PACKET] = False;
    Flag[GUEST_TRANSMIT_KEYBD]    = False;
    Flag[GUEST_TRANSMIT_MACRO]    = False;
    Flag[GUEST_TRANSMIT_MODE]     = False;
    Flag[OLIVIA_LABELS_CB] = True;

    g_idle_add_full(
        G_PRIORITY_DEFAULT_IDLE,
        Olivia_Set_TxRx_Labels,
        NULL, Olivia_g_Idle_cb );

    MOX_Control( MOX_OFF );
    return( False );
  }

  // Set up Transmit text view
  gtk_text_buffer_get_iter_at_offset(
      olivia_gui.tx_text_buffer, &olivia_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(olivia_gui.tx_text_buffer) );
  tx_print_chr.iter = &( olivia_gui.tx_text_buffer_iter );

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(olivia_gui.window_builder, "olivia_tx_textview") );

  // Set transmit mode
  Flag[GUEST_TRANSMIT_MODE] = True;
  Flag[OLIVIA_LABELS_CB]    = True;
  Flag[RSIDTX_ACTIVE]       = False;
  g_idle_add_full(
      G_PRIORITY_DEFAULT_IDLE,
      Olivia_Set_TxRx_Labels,
      NULL, Olivia_g_Idle_cb );

  // Enable Transmit if needed
  MOX_Control( MOX_ON );
  Flag[HERMES2_SEND_DUC_PACKET] = True;

  // Send RSID tones or Olivia two-tone preamble
  if( TRx->tx_rsid_enable )
    Next_Modulator = Two_Tone_Marker;
  else
    Modulator = Two_Tone_Marker;

  // Start Olivia Transmitter
  Transmitter.Start( &Transmitter );

  return( True );
} // Olivia_Initialize_Transmit()

//------------------------------------------------------------------------

/* Exit_Transmit()
 *
 * Carries out actions needed when exiting the Transmit mode
 */
  void
Exit_Transmit( void )
{
  if( Flag[GUEST_QUIT] )
  {
    // Clear Tx monitor string
    Enter_Tx_Monitor_Char( 0xFF );
    MOX_Control( MOX_OFF );
    Modulator = NULL;
    Flag[GUEST_TRANSMIT_MODE]  = False;
    g_idle_add( Olivia_Set_TxRx_Labels, NULL );
    return;
  }

  // Stop and flush Transmitter
  Transmitter.Stop( &Transmitter );
  Transmitter.Input.Clear( &(Transmitter.Monitor) );
  while( Transmitter.Running(&Transmitter) )
    Transmitter.Output_Float( &Transmitter, &Tx_Buffer );
  Transmitter.Reset( &Transmitter );
  Transmitter.Monitor.Clear( &(Transmitter.Monitor) );

  // Display the status of Transmitter FIFO input buffer
  g_idle_add( Tx_Input_Empty, NULL );
  while( g_main_context_iteration(NULL, FALSE) );

  // Enter ' -- ' to Transmit Monitor display
  Enter_Tx_Monitor_Char( ' ' );
  Enter_Tx_Monitor_Char( '-' );
  Enter_Tx_Monitor_Char( '-' );
  Enter_Tx_Monitor_Char( ' ' );

  Flag[GUEST_TRANSMIT_MACRO] = False;
  Flag[GUEST_TRANSMIT_KEYBD] = False;
  Flag[GUEST_TRANSMIT_MODE]  = False;
  Olivia_Transmit_Keybd();  // Resets function

  // Send Olivia two-tone postamble
  Modulator = Two_Tone_Marker;

} // Exit_Transmit()

//------------------------------------------------------------------------

/* Exit_Two_Tone_Marker()
 *
 * Helper thread to exit the Two_Tone_Marker() function
 */
  static void *
Exit_Two_Tone_Marker( void *data )
{
  // Wait for Two_Tone_Marker function to finish
  sem_wait( &duc_send_semaphore );

  // Set Olivia to Receive mode
  Modulator = NULL;
  Flag[GUEST_TRANSMIT_KEYBD] = False;
  Flag[GUEST_TRANSMIT_MACRO] = False;
  MOX_Control( MOX_OFF );
  if( !Flag[GUEST_RECEIVING] )
  {
    if( !Create_Rx_Thread() ) return( NULL );
  }

  return( NULL );
} // Exit_Two_Tone_Marker()

//------------------------------------------------------------------------

#define TONE_BUFFER_SIZE     DUC_NUM_IQ_SAMPLES
#define NUM_TONE_BUFFERS    3
#define NUM_TONE_PAIRS      2

// Empirical, to give near 100% power O/P during tone transmission
#define TONE_AMPLITUDE      28000.0

/* Two_Tone_Marker()
 *
 * Generates the two marker tones used as
 * preamble and postmble in Olivia transmissions
 */
  void
Two_Tone_Marker( void *arg )
{
  const MFSK_Parameters_t *P = &Olivia_Parameters;

  double
    lower_freq,    // Marker tone lower frequency
    upper_freq,    // Marker tone upper frequency
    margin,        // Margin of marker tones from band edges
    dPhase_lower,  // Delta of lower tone phase per DUC sample
    dPhase_upper;  // Delta of upper tone phase per DUC sample

  static double phase = 0.0;  // Phase angle of tone

  // Samples buffer for the SSB modulator
  static double   tone_buffer[NUM_TONE_BUFFERS][TONE_BUFFER_SIZE];
  static uint8_t  buffer_cnt = 0;
  static uint32_t
    num_lower_samples = 0,  // Number of lower tone samples produced
    num_upper_samples = 0,  // Number of upper tone samples produced
    buffer_idx_lower  = 0,  // Tone buffer index for lower tone
    buffer_idx_upper  = 0;  // Tone buffer index for upper tone

  // Duration of each tone in DUC samples
  static uint32_t tone_dur = DUC_SAMPLE_RATE / 4;

  // Number of tone pairs generated
  static uint8_t tone_count = 0;

  // The margin of tones from band edges is half the carrier separation
  margin = (double)P->Bandwidth / (double)P->Carriers / 2.0;

  // Lower and upper tone frequencies inside the bandwidth
  lower_freq   = P->LowerBandEdge + margin;
  upper_freq   = P->LowerBandEdge + (double)P->Bandwidth - margin;
  dPhase_lower = M_2PI * lower_freq / DUC_SAMPLE_RATE;
  dPhase_upper = M_2PI * upper_freq / DUC_SAMPLE_RATE;

  // Produce a sequence of lower-upper-lower-upper tones
  while( tone_count < NUM_TONE_PAIRS )
  {
    // Produce the lower tone samples
    while( num_lower_samples < tone_dur )
    {
      tone_buffer[buffer_cnt][buffer_idx_lower] = TONE_AMPLITUDE * sin( phase );
      phase += dPhase_lower;
      CLAMP_2PI( phase );

      buffer_idx_lower++;
      if( buffer_idx_lower >= TONE_BUFFER_SIZE )
      {
        // Supply the SSB Modulator with tone samples
        Modulate_SSB( (void *)tone_buffer[buffer_cnt] );
        buffer_cnt++;
        if( buffer_cnt >= NUM_TONE_BUFFERS ) buffer_cnt = 0;
        buffer_idx_lower = 0;
        return;
      }

      num_lower_samples++;
    } // while( num_lower_samples < tone_dur )

    // Produce the upper tone samples
    while( num_upper_samples < tone_dur )
    {
      tone_buffer[buffer_cnt][buffer_idx_upper] = TONE_AMPLITUDE * sin( phase );
      phase += dPhase_upper;
      CLAMP_2PI( phase );

      buffer_idx_upper++;
      if( buffer_idx_upper >= TONE_BUFFER_SIZE )
      {
        // Supply the SSB Modulator with tone samples
        Modulate_SSB( (void *)tone_buffer[buffer_cnt] );
        buffer_cnt++;
        if( buffer_cnt >= NUM_TONE_BUFFERS ) buffer_cnt = 0;
        buffer_idx_upper = 0;
        return;
      }

      num_upper_samples++;
    } // while( num_upper_samples < tone_dur )

    num_lower_samples = 0;
    num_upper_samples = 0;
    tone_count++;
  } // while( tone_cnt < NUM_TONE_PAIRS )

  // Clear the DUC/DDC of tone samples
  static uint32_t clear_cnt = 0;
  if( !Flag[GUEST_TRANSMITTING] )
  {
    if( !clear_cnt )
      memset( tone_buffer[0], 0, TONE_BUFFER_SIZE * sizeof(double) );
    while( clear_cnt < tone_dur )
    {
      Modulate_SSB( (void *) tone_buffer[0] );
      clear_cnt += TONE_BUFFER_SIZE;
      return;
    }
  }
  clear_cnt  = 0;
  tone_count = 0;

  // Enable Olivia Modulator if in Transmit mode
  if( Flag[GUEST_TRANSMIT_MODE] )
  {
    Modulator = Modulate_Olivia;
  }
  else // Put the SDR Transceiver out of transmit mode
  {
    // Send CW Morse code ID
    if( Flag[GUEST_TRANSMIT_ID] )
    {
      Clear_DUC();
      snprintf( hermes2_rc.morse_mesg,
          sizeof(hermes2_rc.morse_mesg), " DE %s TU", op_data.call );
      Flag[HERMES2_SEND_DUC_PACKET] = False;
      Flag[TRANSMIT_MORSE_MESG]     = True;
      Modulator = Morse_Transmit;

      // Create a thread to exit Transmit mode when Morse ID has finished.
      // The conditional is redundant but needed to silence Coverity Scan warnings.
      pthread_t thrd;
      if( !Pthread_Create(
            &thrd, NULL, Exit_Two_Tone_Marker, NULL,
            "Failed to create Exit_Two_Tone_Marker thread") )
      {
        Flag[GUEST_TRANSMIT_ID] = False;
        return;
      }

      Flag[GUEST_TRANSMIT_ID] = False;
      return;
    } // if( Flag[GUEST_TRANSMIT_ID] )

    // Set Olivia to Receive mode
    Modulator = NULL;
    Flag[GUEST_TRANSMIT_KEYBD]   = False;
    Flag[GUEST_TRANSMIT_MACRO]   = False;
    MOX_Control( MOX_OFF );
    if( !Flag[GUEST_RECEIVING] )
    {
      if( !Create_Rx_Thread() ) return;
    }
  } // else of if( Flag[GUEST_TRANSMITTING] )

} // Two_Tone_Marker()

//------------------------------------------------------------------------

#define UPSAMPLE_RATIO       6        // 48kHz / 8kHz (DUC rate / Olivia rate)
#define LOCAL_BUFFER_SIZE    256      // Max number of samples from Transmitter
#define MOD_BUFFER_SIZE      1536     // LOCAL_BUFFER_SIZE * UPSAMPLE_RATIO
#define NUM_MOD_BUFFERS      3        // Modulator buffers in the Ring
#define OLIVIA_AMPLITUDE     30000.0  // Empirical, to give near 100% O/P power

/* Modulate_Olivia()
 *
 * Reads characters from the Transmitter's FIFO and generates DDC
 * samples that synthesize the Olivia MFSK Tones to be transmitted
 */
  void
Modulate_Olivia( void *arg )
{
  /* Ring Buffer for up sampled Transmitter output, to be supplied
   * to the SSB Modulator. The length of each buffer is upsampling
   * from 8kHz to 48kHz (6) x Number of samples (256) returned
   * from Transmitter.Output_Int() e.g 6 * 256 = 1536 samples
   */
  static double mod_buffer[NUM_MOD_BUFFERS][MOD_BUFFER_SIZE];

  // Index to buffer used to store new samples
  static uint8_t mod_buf_in = 0;

  /* Index to buffer used to supply the modulator.
   * This is initiated to the last buffer so that
   * it is one behind the above input buffer index */
  static uint8_t mod_buf_out = NUM_MOD_BUFFERS - 1;

  // Local Tansmitter samples buffer
  static double local_buffer[LOCAL_BUFFER_SIZE];

  // Display the status of Transmitter FIFO input buffer
  g_idle_add( Tx_Input_Empty, NULL );
  while( g_main_context_iteration(NULL, FALSE) );

  if( Flag[GUEST_TRANSMIT_ID] ) Exit_Transmit();

  // Read next batch of Transmitter samples if input index
  // has not reached the index of the output buffer in use
  if( mod_buf_in != mod_buf_out )
  {
    // Character read out from the Transmitter's monitor FIFO
    uint8_t chr_out;

    // Enter characters from Transmitter's FIFO to the Monitor
    // unless special control characters are encountered
    while( Transmitter.GetChar(&Transmitter, &chr_out) )
    {
      // Signal transmit Morse ID
      if( chr_out == '*' ) Flag[GUEST_TRANSMIT_ID] = True;

      // If character string ends in '_' or 'Escape', switch over to Receive
      if( (chr_out == (GDK_KEY_underscore & 0xFF)) ||
          (chr_out == (GDK_KEY_Escape     & 0xFF)) ||
          !Flag[GUEST_TRANSMIT_MODE]               ||
           Flag[GUEST_TRANSMIT_ID] )
      {
        // Stop and flush Transmitter
        Exit_Transmit();
        return;
      } // if( chr_out == (GDK_KEY_underscore & 0xFF) ... )

      // Enable Keyboard transmit if character is '~'
      if( chr_out == '~' )
      {
        // Set up control flags for Keyboard transmit
        Flag[GUEST_TRANSMIT_KEYBD] = True;

        // Clear keyboard buffer data
        keybd_buf.key_count    = 0;
        keybd_buf.char_pointer = 0;
        keybd_buf.space_count  = 0;
        keybd_buf.char_count   = 0;
        return;
      }

      // Enter and display character in Tx Monitor
      Enter_Tx_Monitor_Char( chr_out );

    } // while( Transmitter.GetChar(&Transmitter, &chr_out) )

    // Repeat while the Local buffer is not full
    int avail_space = LOCAL_BUFFER_SIZE;
    while( avail_space != 0 )
    {
      int copy;
      static int avail_samples = 0;

      // If there are no available samples in the Transmitter buffer, read a new batch
      if( avail_samples == 0 )
        avail_samples = Transmitter.Output_Float( &Transmitter, &Tx_Buffer );

      // If available space in the Local buffer is >= to the available
      // samples in the Transmitter buffer, copy all available samples.
      // If available space in the Local buffer is < the available samples,
      // then only copy no more samples than the available space in buffer.
      if( avail_space >= avail_samples )
        copy = avail_samples;
      else
        copy = avail_space;

      // Copy samples to available space in local buffer
      int idx = LOCAL_BUFFER_SIZE - avail_space;
      memcpy( &local_buffer[idx], Tx_Buffer, (size_t)copy * sizeof(double) );

      // Reduce available space and available
      // samples by the amount of samples copied
      avail_space   -= copy;
      avail_samples -= copy;

    } // while( avail_space != 0 )

    /* Up Transmitter sample rate from 8kHz to 48kHz. out_idx points
     * to a location in the up-sampled modulator buffer and in_idx to a
     * location in the Transmitter samples buffer with data at 8 kHz */
    uint16_t out_idx = 0;
    static double last_sample = 0.0;
    for( uint16_t in_idx = 0; in_idx < LOCAL_BUFFER_SIZE; in_idx++ )
    {
      // Save previous new sample to modulator buffer
      mod_buffer[mod_buf_in][out_idx] = last_sample;

      // Read new sample from the Transmitter's buffer and upscale
      double new_sample = local_buffer[in_idx] * OLIVIA_AMPLITUDE;

      // Slope of linear splice between two Tx samples
      double slope = ( new_sample - last_sample ) / UPSAMPLE_RATIO;
      last_sample  = new_sample;

      // Extrapolate new out samples between the two input samples
      for( uint8_t idx = 1; idx < UPSAMPLE_RATIO; idx++ )
      {
        mod_buffer[mod_buf_in][out_idx + 1] = mod_buffer[mod_buf_in][out_idx] + slope;
        out_idx++;
      }

      out_idx++;
    } // for( uint16_t in_idx = 0; in_idx < LOCAL_BUFFER_SIZE; in_idx++ )

    // Increment and reset input buffer index
    mod_buf_in++;
    if( mod_buf_in >= NUM_MOD_BUFFERS ) mod_buf_in = 0;

  } // if( mod_buf_in != mod_buf_out )

  // Supply the SSB Modulator with upscaled samples in packets of DUC_NUM_IQ_SAMPLES
  static uint16_t mod_buf_idx = 0;
  double temp_buf[DUC_NUM_IQ_SAMPLES];
  for( uint8_t cnt = 0; cnt < DUC_NUM_IQ_SAMPLES; cnt++ )
  {
    // Copy smaples from the outgoing mod buffer section to temp buffer
    temp_buf[cnt] = mod_buffer[mod_buf_out][mod_buf_idx];

    // Move to next buffer in ring
    mod_buf_idx++;
    if( mod_buf_idx >= MOD_BUFFER_SIZE )
    {
      mod_buf_idx = 0;
      mod_buf_out++;
      if( mod_buf_out >= NUM_MOD_BUFFERS ) mod_buf_out = 0;
    }
  }
  Modulate_SSB( (void *)temp_buf );

  // If Quit activate, exit Transmit mode
  if( Flag[GUEST_QUIT] || !Flag[GUEST_TRANSMIT_MODE] )
  {
    Exit_Transmit();
  }

} // Modulate_Olivia()

//------------------------------------------------------------------------

/* Olivia_Transmit_Keybd()
 *
 * Transmits directly from keyboard entries
 */
  void
Olivia_Transmit_Keybd( void )
{
  // *** Process keystrokes from user ***
  if( Flag[GUEST_TRANSMIT_KEYBD] )
  {
    // Send LF on GDK_KEY_Return
    if( key_buf == GDK_KEY_Return ) key_buf = LF;

    // Send BS on GDK_KEY_BackSpace
    if( key_buf == GDK_KEY_BackSpace ) key_buf = BS;

    // Transmit Keyboard character and record
    Transmitter.PutChar( &Transmitter, (uint8_t)key_buf );

  } // if( Flag[GUEST_TRANSMIT_KEYBD] )
  else
  {
    // Print a LF to Tx text view
    tx_print_chr.printchr = LF;
    Queue_Character( &tx_print_chr );
  }

  return;
} // End of Olivia_Transmit_Keybd()

//------------------------------------------------------------------------

/* Olivia_Transmit_Macro()
 *
 * Transmits a pre-recorded Macro
 */
  void
Olivia_Transmit_Macro( void )
{
  uint8_t
    macro_cnt = 0, // Count macro characters tx'd
    tag_idx   = 0; // Index for transmitting tags

  // Tags for entering fields in macros
  const char *tags[NUM_OF_TAGS] = TAG_TABLE;

  // String within a tag <>
  char tag_string[13];

  // Tag replacement string
  static char tag_replace[22];


  /* Continue entering characters into the Tx buffer till one of
   * the special caharacters '~', '_', * or '\0' are encountered */
  while( (olivia_macro[olivia_macro_idx][macro_cnt] != '~') &&
         (olivia_macro[olivia_macro_idx][macro_cnt] != '_') &&
         (olivia_macro[olivia_macro_idx][macro_cnt] != '*') &&
         (olivia_macro[olivia_macro_idx][macro_cnt] != '\0') )
  {
    // If Macro begins with a ^ transmit new line
    if( olivia_macro[olivia_macro_idx][macro_cnt] == '^' )
    {
      Transmitter.PutChar( &Transmitter, LF );
      tx_print_chr.printchr = LF;
      Queue_Character( &tx_print_chr );
      macro_cnt++;
      continue;
    }

    // *** Look for tags and substitude accordingly ***
    if( (olivia_macro[olivia_macro_idx][macro_cnt] == '<') &&
        !Flag[GUEST_TRANSMIT_TAG] )
    {
      uint16_t idx;

      // *** Copy tag string to buffer and identify ***
      macro_cnt++; // Move into tag field
      Flag[GUEST_TRANSMIT_TAG] = True;

      // Copy line to tag, max 12 chars or EOS
      idx = 0;
      uint16_t siz = (uint16_t)sizeof( tag_string ) - 1;
      while( (idx < siz) && (olivia_macro[olivia_macro_idx][macro_cnt] != '>') )
      {
        tag_string[idx] = olivia_macro[olivia_macro_idx][macro_cnt];
        idx++;
        macro_cnt++;
      }

      // Terminate tag buffer and advance macro buffer
      tag_string[idx] = '\0';
      macro_cnt++;

      // Identify tag and substitude accordingly
      idx = 0;
      while( idx < NUM_OF_TAGS )
      {
        if( strcmp(tag_string, tags[idx]) == 0 )
          break;
        idx++;
      }

      // Abort if tag unidentified
      if( idx == NUM_OF_TAGS )
      {
        MOX_Control( MOX_OFF );
        Error_Dialog(
            _("Error reading olivia.config file\n"\
              "A tag in a macro is invalid\n"\
              "Quit olivia and correct"), HIDE_OK);
        return;
      }

      // Replace tags
      siz = sizeof( tag_replace );
      switch( idx )
      {
        case 0: // own-call
          Strlcpy( tag_replace, op_data.call, siz );
          break;

        case 1: // own-name
          Strlcpy( tag_replace, op_data.name, siz );
          break;

        case 2: // own-qth
          Strlcpy( tag_replace, op_data.qth, siz );
          break;

        case 3: // own-loc
          Strlcpy( tag_replace, op_data.loc, siz );
          break;

        case 4: // own-rst
          Strlcpy( tag_replace, qso_record.my_rst, siz );
          break;

        case 5: // rem-call
          Strlcpy( tag_replace, qso_record.dx_call, siz );
          break;

        case 6: // rem-name
          Strlcpy( tag_replace, qso_record.dx_name, siz );
          break;

        case 7: // rem-qth
          Strlcpy( tag_replace, qso_record.dx_qth, siz );
          break;

        case 8: // rem-loc
          Strlcpy( tag_replace, qso_record.dx_loc, siz );
          break;

        case 9: // rem-rst
          Strlcpy( tag_replace, qso_record.dx_rst, siz );
          break;

        case 10: // date-time
          Strlcpy( tag_replace, qso_record.date, siz );
          Strlcat ( tag_replace, " ", siz );
          Strlcat( tag_replace, qso_record.time, siz );
          break;

        case 11: // op-freq, strip leading spaces
          Strlcpy( tag_replace, qso_record.freq, siz );
          break;

        case 12: // app-version
          snprintf( tag_replace, siz, "%s", PACKAGE_STRING );
          break;

      } // switch( idx )
    } // if( (olivia_macro[macro_idx][macro_cnt] == '<') && )

    // *** Transmit tag replacement ***
    if( Flag[GUEST_TRANSMIT_TAG] )
    {
      if( tag_replace[tag_idx] != '\0' )
      {
        // Capitalize letters if enabled
        if( Flag[GUEST_CAPITALIZE] )
          tag_replace[tag_idx] = (char)toupper( tag_replace[tag_idx] );
        Transmitter.PutChar( &Transmitter, (uint8_t)tag_replace[tag_idx] );
        tx_print_chr.printchr = (guint)tag_replace[tag_idx];
        tag_idx++;
        Queue_Character( &tx_print_chr );
      } // if( (tag_replace[tag_idx] != '\0') )
      else
      {
        Flag[GUEST_TRANSMIT_TAG] = False;
        tag_idx = 0;
      }

      continue;
    } // if( Flag[GUEST_TRANSMIT_TAG) &&

    // Capitalize letters if enabled
    if( Flag[GUEST_CAPITALIZE] )
      olivia_macro[olivia_macro_idx][macro_cnt] =
        (char)toupper( olivia_macro[olivia_macro_idx][macro_cnt] );

    // Transmit a char from Macro
    Transmitter.PutChar( &Transmitter, (uint8_t)olivia_macro[olivia_macro_idx][macro_cnt] );

    tx_print_chr.printchr = (guint)olivia_macro[olivia_macro_idx][macro_cnt];
    Queue_Character( &tx_print_chr );
    macro_cnt++;

  } // while( (olivia_macro[olivia_macro_idx][macro_cnt] != '~') && ...

  // Flush qso record file
  if( Flag[GUEST_RECORD_QSO] )
    fflush( qso_record.qso_record_fp );

  // Transmit a final space
  Transmitter.PutChar( &Transmitter, ' ' );

  // Put last Macro character into Transmit buffer.
  // Needed in Modulate_Olivia() to switch over to Receive
  Transmitter.PutChar( &Transmitter, (uint8_t)olivia_macro[olivia_macro_idx][macro_cnt] );

  Flag[GUEST_TRANSMIT_TAG] = False;

} // Olivia_Transmit_Macro()

//------------------------------------------------------------------------

/* Ratio of Rx audio sample rate to
 * Olivia sample rate (48kHz/8kHz) */
#define DOWN_SAMPLE    6

// Size of Olivia Receiver input buffer
#define INPUT_SIZE  1024

/* Olivia_Receive_Mode()
 *
 * Decodes incoming Olivia signals and
 * displays resulting ASCII characters
 */
  void *
Olivia_Receive_Mode( void *data )
{
  // A character decoded from Olivia signals
  uint16_t dec_chr;

  static uint8_t
    samp_cnt = 0, // Count of audio samples added to Receiver input buffer
    chr_cnt  = 0; // Received character count

  // Index to Receiver's Input buffer
  static uint16_t samp_idx = 0;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;

  // *** Initialization Block ***

  // Set some Receive mode flags
  digi_buf_cnt = 0;
  Flag[OLIVIA_FLUSH_RECEIVER] = False;
  Flag[OLIVIA_EXIT_RECEIVE]   = False;
  Flag[GUEST_RECEIVING]       = True;

  // Free and (Re)-initialize and preset/allocate Receiver object
  Receiver.Free( &Receiver );
  if( !Receiver.Preset(&Receiver, &Olivia_Parameters) )
  {
    Error_Dialog(
        _("Failed to Preset Receiver Parameters\n"\
          "Quit Olivia and correct"), HIDE_OK );
    Flag[GUEST_RECEIVE_MODE] = False;
    Flag[GUEST_RECEIVING]    = False;
    return( NULL );
  }

  // Abort if QSO record file fails to open
  if( !Open_Record_File("olivia") )
  {
    Flag[GUEST_RECEIVE_MODE] = False;
    Flag[OLIVIA_LABELS_CB]   = True;
    g_idle_add_full(
        G_PRIORITY_DEFAULT_IDLE,
        Olivia_Set_TxRx_Labels,
        NULL,
        Olivia_g_Idle_cb );

    Flag[GUEST_RECEIVING] = False;
    return( NULL );
  }

  // Set up Receive text view
  gtk_text_buffer_get_iter_at_offset(
      olivia_gui.rx_text_buffer, &olivia_gui.rx_text_buffer_iter,
      gtk_text_buffer_get_char_count(olivia_gui.rx_text_buffer) );
  rx_print_chr.iter = &( olivia_gui.rx_text_buffer_iter );

  // Print a LF to Rx window
  rx_print_chr.printchr = LF;
  Queue_Character( &rx_print_chr );

  Flag[GUEST_RECEIVE_MODE] = True;
  Flag[OLIVIA_LABELS_CB]   = True;
  g_idle_add_full(
      G_PRIORITY_DEFAULT_IDLE,
      Olivia_Set_TxRx_Labels,
      NULL,
      Olivia_g_Idle_cb );

  // *** End of Initialization Block ***

  // Cancel Receive if Rx flag cleared or quit flag set
  do
  {
    // Wait for digimode buffer to fill with audio samples from demodulator.
    sem_wait( &digimode_semaphore );

    // Process samples from the SDR receiver
    static double Input[INPUT_SIZE];
    for( int16_t idx = 0; idx < DIGIMODE_BUFFER_SIZE; idx++ )
    {
      int16_t signal_sample = digimode_buffer[digi_buf_cnt][idx];

      Input[samp_idx] += (double)signal_sample;
      samp_cnt++;
      if( samp_cnt >= DOWN_SAMPLE )
      {
        samp_cnt = 0;
        samp_idx++;
        if( samp_idx >= INPUT_SIZE )
        {
          samp_idx = 0;

          // *** Print characters to Receive window & file ***
          // Process Rx signal
          Receiver.Process( &Receiver, Input, INPUT_SIZE );

          // Flush Receiver and print characters
          if( Flag[OLIVIA_FLUSH_RECEIVER] )
          {
            Flag[OLIVIA_FLUSH_RECEIVER] = False;
            Receiver.Flush( &Receiver );
          }

          // Keep receiving decoded characters till the buffer empties
          uint8_t rx_chr;
          while( Receiver.GetChar(&Receiver, &rx_chr) )
          {
            dec_chr = (uint16_t)rx_chr;

            // If decoded character is valid ASCII code
            if( dec_chr <= 127 )
            {
              // Capitalize letters if enabled
              if( Flag[GUEST_CAPITALIZE] )
                dec_chr = (uint16_t)toupper( dec_chr );

              // Replace BS character
              if( dec_chr == BS ) dec_chr = GDK_KEY_BackSpace;

              // Print character to Rx viewer
              rx_print_chr.printchr = dec_chr;
              Queue_Character( &rx_print_chr );

              // Reset char count on line feed
              if( dec_chr == LF )
                chr_cnt = 0;
              else if( (dec_chr == GDK_KEY_BackSpace) && chr_cnt )
                chr_cnt--;
              else
              {
                // Do word wrapping
                chr_cnt++;
                if( (chr_cnt >= olivia_rc_data.word_wrap) &&
                    (dec_chr == ' ') )
                {
                  rx_print_chr.printchr = LF;
                  Queue_Character( &rx_print_chr );
                  chr_cnt = 0;
                }
              } // else
            } // if( dec_chr <= 127 )
          } // while( Receiver.GetChar(&Receiver, &rx_chr) )
        } // if( samp_idx >= INPUT_SIZE )

        Input[samp_idx] = 0.0;
      } // if( samp_cnt >= DOWN_SAMPLE )

      // Display signal waterfall when buffer ready
      if( iFFT_Data(signal_sample, &olivia_ifft_data) )
      {
        Display_Waterfall( &olivia_wfall, &olivia_ifft_data );
        Queue_Draw( olivia_gui.scope );
      }

    } // for( int16_t idx = 0; idx < DIGIMODE_BUFFER_SIZE; idx++ )

    // Increment current audio sample output buffer index
    digi_buf_cnt++;
    if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
    {
      digi_buf_cnt = 0;
      g_idle_add( Olivia_Show_Decoder_Status, NULL );
    }

  } // do
  while( !Flag[GUEST_QUIT] && !Flag[OLIVIA_EXIT_RECEIVE] );

  Flag[GUEST_RECEIVE_MODE] = False;
  Flag[OLIVIA_LABELS_CB]   = True;
  g_idle_add_full(
      G_PRIORITY_DEFAULT_IDLE,
      Olivia_Set_TxRx_Labels,
      NULL,
      Olivia_g_Idle_cb );

  Flag[GUEST_RECEIVING] = False;
  return( NULL );
} // End of Olivia_Receive_Mode()

//------------------------------------------------------------------------

/* Create_Rx_Thread()
 *
 * Creates a thread to run the Receive mode functions
 */
  BOOLEAN
Create_Rx_Thread( void )
{
  if( !Pthread_Create(
        &hermes2_rc.guest_rx_thread, NULL, Olivia_Receive_Mode, NULL,
        _("Failed to create Receive thread")) )
  {
    return( False );
  }

  Flag[GUEST_TRANSMIT_MACRO] = False;
  Flag[OLIVIA_LABELS_CB]     = True;
  g_idle_add_full(
      G_PRIORITY_DEFAULT_IDLE,
      Olivia_Set_TxRx_Labels,
      NULL,
      Olivia_g_Idle_cb );

  return( True );
} /* Create_Rx_Thread() */

//------------------------------------------------------------------------

