/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef MFSK_H
#define MFSK_H    1

#include "buffer.h"
#include "rateconv.h"
#include "../common/common.h"
#include <complex.h>
#include <stdint.h>
#include <stdio.h>

//----------------------------------------------------------------------

#define OLIVIA_SAMPLE_RATE  8000
#define S16_SCALE   32767.0

//----------------------------------------------------------------------

// The basic MFSK parameters
typedef struct _MFSK_PARAMETERS
{
  // Primary parameters
  uint32_t BitsPerSymbol;     // [Bits]
  uint32_t Bandwidth;         // [Hz]
  uint32_t SampleRate;        // [Hz]
  double   LowerBandEdge;     // [Hz]
  double   InputSampleRate;   // [Hz]
  double   OutputSampleRate;  // [Hz]
  uint32_t RxSyncMargin;      // [MFSK carriers]
  uint32_t RxSyncIntegLen;    // [FEC Blocks]
  double   RxSyncThreshold;   // [S/N]

  // Fixed parameters.
  uint32_t BitsPerCharacter;  // [Bits]
  uint32_t SymbolsPerBlock;
  uint32_t CarrierSepar;      // [FFT bins]
  uint32_t SpectraPerSymbol;  // [Spectral (FFT) slices]
  uint32_t SpectraPerBlock;
  uint32_t UseGrayCode;
  uint32_t PhaseDiffer;
  uint32_t RxSyncSquareEnergy;
  uint32_t DecodeSquareEnergy;
  uint64_t ScramblingCode;

  // Secondary parameters
  uint32_t Carriers;
  uint32_t SymbolSepar;      // [Samples]
  uint32_t SymbolLen;        // [Samples]
  uint32_t FirstCarrier;     // [FFT bins]

  /* Pointers to functions that were
   * within the class MFSK_Parameters */
  void   (*Default)(struct _MFSK_PARAMETERS *);
  void   (*Preset)(struct _MFSK_PARAMETERS *);
  void   (*Set_Fixed)(struct _MFSK_PARAMETERS *);
  double (*BaudRate)(struct _MFSK_PARAMETERS *);
  double (*FFTbinBandwidth)(struct _MFSK_PARAMETERS *);
  double (*CarrierBandwidth)(struct _MFSK_PARAMETERS *);
  double (*TuneMargin)(struct _MFSK_PARAMETERS *);
  double (*BlockPeriod)(struct _MFSK_PARAMETERS *);
  double (*CharactersPerSecond)(struct _MFSK_PARAMETERS *);

  // Flag to signal change in Parameters
  BOOLEAN New_Parameters;

} MFSK_Parameters_t;

void MFSK_Params_Initialize(struct _MFSK_PARAMETERS *);

//----------------------------------------------------------------------

extern struct _MFSK_MODULATOR MFSK_Modulator;
void MFSK_Modulator_Initialize(void);

//----------------------------------------------------------------------

extern struct _MFSK_INPUTPROCESSOR MFSK_InputProcessor;
extern void MFSK_InputProcessor_Initialize(void);

//----------------------------------------------------------------------

extern struct _MFSK_DEMODULATOR MFSK_Demodulator;
extern void MFSK_Demodulator_Initialize(void);

//----------------------------------------------------------------------

extern struct _MFSK_ENCODER MFSK_Encoder;
void MFSK_Encoder_Initialize(void);

//----------------------------------------------------------------------

// Soft FEC decoder ( calls are similar to the MFSK_HardDecoder class )
typedef struct _MFSK_SOFTDECODER
{
  MFSK_Parameters_t *Parameters;

  uint32_t BitsPerSymbol;
  uint32_t SymbolsPerBlock;
  uint32_t SpectraPerSymbol;

  uint32_t InputBufferLen;
  double  *InputBuffer;
  uint32_t InputPtr;

  double *FHT_Buffer;

  double   Signal, NoiseEnergy;
  uint8_t *OutputBlock;

  // Pointers to functions in class MFSK_Softdecoder
  void     (*Init)(struct _MFSK_SOFTDECODER *Self);
  void     (*Free)(struct _MFSK_SOFTDECODER *Self);
  void     (*Reset)(struct _MFSK_SOFTDECODER *Self);
  void     (*Preset)(struct _MFSK_SOFTDECODER *Self, struct _MFSK_PARAMETERS *);
  void     (*SpectralInput)(struct _MFSK_SOFTDECODER *Self, const double *);
  void     (*Input)(struct _MFSK_SOFTDECODER *Self, const double *);
  void     (*DecodeCharacter)(struct _MFSK_SOFTDECODER *Self, uint32_t);
  void     (*Process)(struct _MFSK_SOFTDECODER *Self);
  uint32_t (*Output)(struct _MFSK_SOFTDECODER *Self, uint8_t *);

} MFSK_SoftDecoder_t;

void MFSK_SoftDecoder_Initialize(MFSK_SoftDecoder_t *Self);

//----------------------------------------------------------------------

extern struct _MFSK_SOFTITERDECODER MFSK_SoftIterDecoder;
void MFSK_SoftIterDecoder_Initialize(void);

//----------------------------------------------------------------------

// MFSK transmitter (FEC encoder + MFSK modulator + rate corrector )
typedef struct _MFSK_TRANSMITTER
{
  struct _MFSK_PARAMETERS *Parameters;
  FIFO_t Input;             // Buffer( queue ) for the characters to be encoded
  FIFO_t Monitor;           // Buffer for monitoring the characters being sent
  CRateConverter_t RateConverter; // Output rate converter

  uint32_t MaxOutputLen;    // Maximum length of the audio batch returned by Output()
  uint32_t BitsPerSymbol;
  uint32_t SymbolsPerBlock;
  uint32_t SymbolPtr;

  int State_Running;
  int State_StopReq;
  int State;

  uint8_t InputBlock[8];    // FEC code block buffer

  double *ModulatorOutput;  // Modulator output
  double *ConverterOutput;  // Rate converter output

  // Pointers to functions that were in the MFSK_Transmitter class
  void    (*Init)(struct _MFSK_TRANSMITTER *);
  void    (*Free)(struct _MFSK_TRANSMITTER *);
  void    (*Preset)(struct _MFSK_TRANSMITTER *, struct _MFSK_PARAMETERS *);
  void    (*Reset)(struct _MFSK_TRANSMITTER *);
  void    (*Start)(struct _MFSK_TRANSMITTER *);
  void    (*Stop)(struct _MFSK_TRANSMITTER *);
  int     (*Running)(const struct _MFSK_TRANSMITTER *);
  BOOLEAN (*PutChar)(struct _MFSK_TRANSMITTER *, uint8_t);
  BOOLEAN (*GetChar)(struct _MFSK_TRANSMITTER *, uint8_t *);
  int     (*Output_Float)(struct _MFSK_TRANSMITTER *, double **);
  int     (*Output_Int)(struct _MFSK_TRANSMITTER *, int16_t *);

} MFSK_Transmitter_t;

void MFSK_Transmitter_Initialize( MFSK_Transmitter_t *Self );

//----------------------------------------------------------------------

extern struct _MFSK_SYNCHRONIZER MFSK_Synchronizer;
void MFSK_Synchronizer_Initialize( void );

//----------------------------------------------------------------------

typedef struct _MFSK_RECEIVER
{
  MFSK_Parameters_t *Parameters;
  CRateConverter_t RateConverter;
  Seq_t  InputBuffer;
  FIFO_t Output;       // Buffer for decoded characters

  // Pointers to functions that were in class MFSK_Receiver
  void    (*Free)(struct _MFSK_RECEIVER *);
  BOOLEAN (*Preset)(struct _MFSK_RECEIVER *, struct _MFSK_PARAMETERS *);
  void    (*Reset)(struct _MFSK_RECEIVER *);
  double  (*SyncSNR)(void);
  double  (*FrequencyOffset)(void);
  double  (*FrequencyDrift)(void);
  double  (*TimeDrift)(void);
  double  (*InputSNRdB)(void);
  BOOLEAN (*Process)(struct _MFSK_RECEIVER *, double *, uint32_t);
  void    (*Flush)(struct _MFSK_RECEIVER *);
  BOOLEAN (*GetChar)(struct _MFSK_RECEIVER *, uint8_t *);
  void    (*ProcessInputBuffer)(struct _MFSK_RECEIVER *);
  void    (*ProcessSymbol)(struct _MFSK_RECEIVER *, double *);

} MFSK_Receiver_t;

void MFSK_Receiver_Initialize( MFSK_Receiver_t *Self );

//----------------------------------------------------------------------

// Fast ( integer ) power of two
static inline uint32_t Exp2( uint32_t X )
{
  return( (uint32_t) 1 << X );
}

// Fast ( integer ) base-2 logarithm
static inline uint32_t Log2( uint32_t X )
{
  uint32_t Y;
  for( Y = 0; X > 1; X >>= 1 ) Y++;
  return( Y );
}

//----------------------------------------------------------------------

  static inline uint8_t
BinaryCode_Uint8( uint8_t Gray )
{
  Gray ^= (Gray >> 4 );
  Gray ^= (Gray >> 2 );
  Gray ^= (Gray >> 1 );
  return( Gray );
}

/*----------------------------------------------------------------------

  inline uint16_t
BinaryCode_Uint16( uint16_t Gray )
{
  Gray ^= (Gray >> 8 );
  Gray ^= (Gray >> 4 );
  Gray ^= (Gray >> 2 );
  Gray ^= (Gray >> 1 );
  return( Gray );
}

  inline uint32_t
BinaryCode_Uint32( uint32_t Gray )
{
  Gray ^= (Gray >> 16 );
  Gray ^= (Gray >> 8 );
  Gray ^= (Gray >> 4 );
  Gray ^= (Gray >> 2 );
  Gray ^= (Gray >> 1 );
  return( Gray );
}

----------------------------------------------------------------------*/

// Function prototypes
BOOLEAN FitPeak(double *PeakPos, double *Peak, double Left, double Center, double Right);
void MFSK_SoftDemodulate(double *Symbol, const double *SpectraEnergy, uint32_t BitsPerSymbol, uint32_t CarrierSepar, int UseGrayCode, int SquareEnergy);
void MFSK_SoftModulate(double *CarrierProb, const double *Symbol, uint32_t BitsPerSymbol, int UseGrayCode);
void WhiteNoise(complex double *Noise, double Amplitude);
void BoxFilter_Initialize(void);
void MFSK_Params_Preset(MFSK_Parameters_t *Self);
void MFSK_Params_Initialize(MFSK_Parameters_t *Self);

#endif

