/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

// MFSK trasnmitter and receiver code,
// ( c ) Pawel Jalocha, April 2006

#include "mfsk.h"
#include "buffer.h"
#include "fht.h"
#include "gray.h"
#include "lowpass3.h"
#include "rateconv.h"
#include "shared.h"
#include "struc.h"
#include "../common/common.h"
#include "../common/rfft.h"
#include "../common/utils.h"
#include "../Hermes2/sound.h"
#include <complex.h>
#include <math.h>
#include <stddef.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

//----------------------------------------------------------------------

/*
   The convention to pass parameters to the objects:

   User-setable parameters are listed on top of the class.
   When an object is being created it is given certain default
   parameters by the Default() call. If the user wishes to modify
   some ( or all ) of them he should refer to them directly.
   Then, the user must call Preset() which will preset the
   internal object structures including dependend parameters
   and dynamic arrays. Only then calls line Input() and
   Process() can be executed.

   Preset() may return a negative number in case some parameter
   are not valid. It can as well adjust some parameters to the
   closest valid values.

   If the user wishes to change the parameters he should do so
   and then call Preset() again, however, the data accumulated
   in the internal structures are lost.

   When the user wants to save memory and free the internal
   storage allocated by an object he can call Free() and then
   later he can call Preset() again to re-use this object.
 */

//----------------------------------------------------------------------

// The symbol shape described in frequency domain
static const double MFSK_SymbolFreqShape[] =
{
  +1.0000000000, +2.1373197349, +1.1207588117, -0.0165609232
};

static const uint32_t MFSK_SymbolFreqShapeLen =
sizeof( MFSK_SymbolFreqShape ) / sizeof( double );

//----------------------------------------------------------------------

// Helper functions
  BOOLEAN
FitPeak( double *PeakPos, double *Peak, double Left, double Center, double Right )
{
  double A = (Right + Left) / 2.0 - Center;
  if( A >= 0.0 ) return( False );
  double B = ( Right - Left ) / 2.0;
  *PeakPos = ( -B / (2.0 * A) );
  *Peak = A * *PeakPos * *PeakPos + B * *PeakPos + Center;
  return( True );
}

//----------------------------------------------------------------------

  static void
ConvertToS16( const double *Input, int16_t *Output, uint32_t Len )
{
  uint32_t Idx;

  for( Idx = 0; Idx < Len; Idx++ )
  {
    int Out = (int)( floor(S16_SCALE * Input[Idx] + 0.5) );
    CLAMP_S16( Out );
    Output[Idx] = (int16_t)Out;
  }
}

//----------------------------------------------------------------------

  static double
Complex_Energy( const complex double *Cmpx )
{
  double Re = creal( *Cmpx );
  double Im = cimag( *Cmpx );
  return( Re * Re + Im * Im );
}

//----------------------------------------------------------------------

  static double
Complex_Mag2( const complex double *Cmpx )
{
  double Re = creal( *Cmpx );
  double Im = cimag( *Cmpx );
  return( Re * Re + Im * Im );
}

// ***------------------------------------------------------------------***

// Set the fixed parameter values. My function
  static void
MFSK_Params_Set_Fixed( MFSK_Parameters_t *Self )
{
  if( olivia_rc_data.mode_olivia )
  {
    Self->BitsPerCharacter = 7; // [Bits] For Olivia Mode
    Self->ScramblingCode   = 0xE257E6D0291574ECLL;
  }
  else
  {
    Self->BitsPerCharacter = 6; // [Bits] For CONTESTIA mode
    Self->ScramblingCode   = 0xEDB88320LL;
  }

  Self->SymbolsPerBlock    = Exp2( Self->BitsPerCharacter - 1 );
  Self->CarrierSepar       = 4; // [FFT bins]
  Self->SpectraPerSymbol   = 4; // [Spectral (FFT) slices]
  Self->SpectraPerBlock    = Self->SpectraPerSymbol * Self->SymbolsPerBlock;
  Self->UseGrayCode        = 1;
  Self->PhaseDiffer        = 1;
  Self->RxSyncSquareEnergy = 1;
  Self->DecodeSquareEnergy = 1;
}

//----------------------------------------------------------------------

// Set the default parameter values
  static void
MFSK_Params_Default( MFSK_Parameters_t *Self )
{
  Self->BitsPerSymbol     = 4;
  Self->SampleRate        = OLIVIA_SAMPLE_RATE;
  Self->Bandwidth         = 500;
  Self->LowerBandEdge     = (double)( Self->SampleRate / 16 );
  Self->InputSampleRate   = (double)( Self->SampleRate );
  Self->OutputSampleRate  = (double)( Self->SampleRate );
  Self->RxSyncIntegLen    = olivia_rc_data.sync_integ_len;
  Self->RxSyncMargin      = olivia_rc_data.sync_margin;
  Self->RxSyncThreshold   = olivia_rc_data.sync_threshold;
}

//----------------------------------------------------------------------

// Preset MFSK parameters
  void
MFSK_Params_Preset( MFSK_Parameters_t *Self )
{
  if( Self->BitsPerSymbol > 8 )
    Self->BitsPerSymbol = 8;
  else if( Self->BitsPerSymbol < 1 )
    Self->BitsPerSymbol = 1;
  Self->Carriers = Exp2( Self->BitsPerSymbol );

  uint32_t MinBandwidth = Self->SampleRate / 64;
  uint32_t MaxBandwidth = Self->SampleRate / 4;

  if( Self->Bandwidth < MinBandwidth )
    Self->Bandwidth = MinBandwidth;
  else if( Self->Bandwidth > MaxBandwidth )
    Self->Bandwidth = MaxBandwidth;
  Self->Bandwidth   = MinBandwidth * Exp2( Log2(Self->Bandwidth / MinBandwidth) );

  Self->SymbolSepar = (Self->SampleRate / Self->Bandwidth) * Self->Carriers;
  Self->SymbolLen   = Self->SymbolSepar * Self->CarrierSepar;

  Self->FirstCarrier  = (uint32_t)( floor(
        (Self->LowerBandEdge / (double)Self->SampleRate) *
        (double)Self->SymbolLen + 0.5) ) + ( Self->CarrierSepar / 2 );

  if( (Self->FirstCarrier + Self->Carriers * Self->CarrierSepar) >= (Self->SymbolLen / 2) )
    Self->FirstCarrier = ( Self->SymbolLen / 2 ) - Self->Carriers * Self->CarrierSepar;

  /* This is not used anywhere???
     Self->LowerBandEdge =
     (double)( Self->FirstCarrier - Self->CarrierSepar / 2 ) / (double)Self->SymbolLen;
   */

  if( Self->RxSyncMargin > (Self->FirstCarrier / Self->CarrierSepar) )
    Self->RxSyncMargin = ( Self->FirstCarrier / Self->CarrierSepar );

  // Signal new parameters
  Self->New_Parameters = True;
}

//----------------------------------------------------------------------

// Helper functions to return MFSK parameters
  static double
MFSK_Params_BaudRate( MFSK_Parameters_t *Self )
{
  return( (double)Self->SampleRate / (double)Self->SymbolSepar );
}

  static double
MFSK_Params_FFTbinBandwidth( MFSK_Parameters_t *Self )
{
  return( (double)Self->SampleRate / (double)Self->SymbolLen );
}

  static double
MFSK_Params_CarrierBandwidth( MFSK_Parameters_t *Self )
{
  return( (double)Self->SampleRate / (double)Self->SymbolLen * (double)Self->CarrierSepar );
}

  static double
MFSK_Params_TuneMargin( MFSK_Parameters_t *Self )
{
  return( Self->CarrierBandwidth(Self) * (double)Self->RxSyncMargin );
}

  static double
MFSK_Params_BlockPeriod( MFSK_Parameters_t *Self )
{
  return( ((double)Self->SymbolsPerBlock * (double)Self->SymbolSepar) /
      (double)Self->SampleRate );
}

  static double
MFSK_Params_CharactersPerSecond( MFSK_Parameters_t *Self )
{
  return( (double)Self->BitsPerSymbol * (double)Self->SampleRate /
      ((double)Self->SymbolsPerBlock  * (double)Self->SymbolSepar) );
}

//----------------------------------------------------------------------

// Initialize the MFSK parameters. My function
  void
MFSK_Params_Initialize( MFSK_Parameters_t *Self )
{
  // Set MFSK parameters function pointers
  Self->Default   = MFSK_Params_Default;
  Self->Preset    = MFSK_Params_Preset;
  Self->Set_Fixed = MFSK_Params_Set_Fixed;

  // Helper function that return values or print
  Self->BaudRate            = MFSK_Params_BaudRate;
  Self->FFTbinBandwidth     = MFSK_Params_FFTbinBandwidth;
  Self->CarrierBandwidth    = MFSK_Params_CarrierBandwidth;
  Self->TuneMargin          = MFSK_Params_TuneMargin;
  Self->BlockPeriod         = MFSK_Params_BlockPeriod;
  Self->CharactersPerSecond = MFSK_Params_CharactersPerSecond;

  // Initialize MFSK parameters
  Self->Set_Fixed( Self );
  Self->Default( Self );
}

// ***------------------------------------------------------------------***

// Soft-demodulate an MFSK symbol
  void
MFSK_SoftDemodulate(
    double *Symbol,
    const double *SpectraEnergy,
    uint32_t BitsPerSymbol,
    uint32_t CarrierSepar,
    int      UseGrayCode,
    int      SquareEnergy )
{
  uint32_t Bit, Idx;

  for( Bit = 0; Bit < BitsPerSymbol; Bit++ )
    Symbol[Bit] = 0.0;

  uint32_t Carriers = Exp2( BitsPerSymbol );
  double TotalEnergy = 0.0;
  uint32_t Freq = 0;

  for( Idx = 0; Idx < Carriers; Idx++ )  // Loop over carriers
  {
    uint8_t SymbIdx = (uint8_t)Idx;
    if( UseGrayCode ) SymbIdx = BinaryCode_Uint8( SymbIdx );

    double Energy = SpectraEnergy[Freq];  // Energy for given carrier
    if( SquareEnergy ) Energy *= Energy;  // Square the energy ( works better, but why ?)
    TotalEnergy += Energy;

    uint8_t Mask = 1;
    for( Bit = 0; Bit < BitsPerSymbol; Bit++ )  // Soft decision (contribution) for every bit
    {
      if( SymbIdx & Mask ) Symbol[Bit] -= Energy;  // Add or subtract the contribution
      else Symbol[Bit] += Energy;                  // Depending on bit value
      Mask <<= 1;
    }
    Freq += CarrierSepar;
  }

  if( TotalEnergy > 0.0 )  // Normalize the soft decisions
  {
    for( Bit = 0; Bit < BitsPerSymbol; Bit++ )
      Symbol[Bit] /= TotalEnergy;
  }
}

// ***------------------------------------------------------------------***

// Soft-modulate an MFSK symbol
  void
MFSK_SoftModulate(
    double   *CarrierProb,
    const double *Symbol,
    uint32_t BitsPerSymbol,
    int      UseGrayCode )
{
  uint32_t Carriers = Exp2( BitsPerSymbol );

  uint8_t Idx;
  for( Idx = 0; Idx < Carriers; Idx++ )  // Loop over carriers
  {
    uint8_t SymbIdx = Idx;
    if( UseGrayCode ) SymbIdx = BinaryCode_Uint8( SymbIdx );

    double   Prob = 1.0;
    uint32_t Bit;
    uint8_t  Mask = 1;

    for( Bit = 0; Bit < BitsPerSymbol; Bit++ )
    {
      double BitProb = 1.0;
      if( SymbIdx & Mask )
        BitProb -= Symbol[Bit];
      else
        BitProb += Symbol[Bit];

      Prob *= ( BitProb / 2.0 );
      Mask <<= 1;
    }

    CarrierProb[Idx] = Prob;
  }
}

//----------------------------------------------------------------------

// MFSK modulator, synthesis of the MFSK signal
struct _MFSK_MODULATOR
{
  MFSK_Parameters_t *Parameters;
  uint32_t OutputLen;    // Output length per transmitted symbol [samples]
  uint32_t SymbolLen;
  uint32_t SymbolSepar;

  double  *CosineTable;  // Cosine table for fast cos/sin calculation
  double  *SymbolShape;  // The shape of the symbol
  int      SymbolPhase;  // The phase of the tone being transmitted
  double  *OutTap;       // Output tap ( buffer )
  uint32_t TapPtr;
  uint32_t WrapMask;

  // *** Pointers to functions in class MFSK_Modulator ***
  void     (*Init)(void);
  void     (*Free)(void);
  void     (*Preset)(MFSK_Parameters_t *NewParameters);
  void     (*Send)(uint8_t Symbol);
  uint32_t (*Output_Int16)(int16_t *Buffer);
  uint32_t (*Output_Float)(double *Buffer);
  void     (*AddSymbol)(int Freq, int Phase);

} MFSK_Modulator;

//----------------------------------------------------------------------

  static void
MFSK_Modulator_Init( void )
{
  MFSK_Modulator.CosineTable = NULL;
  MFSK_Modulator.SymbolShape = NULL;
  MFSK_Modulator.OutTap      = NULL;
}

//----------------------------------------------------------------------

  static void
MFSK_Modulator_Free( void )
{
  Mem_Free( (void **) &(MFSK_Modulator.CosineTable) );
  Mem_Free( (void **) &(MFSK_Modulator.SymbolShape) );
  Mem_Free( (void **) &(MFSK_Modulator.OutTap) );
}

//----------------------------------------------------------------------

  static void
MFSK_Modulator_Preset( MFSK_Parameters_t *NewParameters )
{
  MFSK_Modulator.Parameters  = NewParameters;
  MFSK_Modulator.SymbolLen   = MFSK_Modulator.Parameters->SymbolLen;
  MFSK_Modulator.SymbolSepar = MFSK_Modulator.Parameters->SymbolSepar;

  uint32_t Idx;

  // If realloc fails, program will exit(-1)
  size_t siz = sizeof(double) * (size_t)MFSK_Modulator.SymbolLen;
  Mem_Realloc( (void **) &(MFSK_Modulator.CosineTable), siz );
  for( Idx = 0; Idx < MFSK_Modulator.SymbolLen; Idx++ )
    MFSK_Modulator.CosineTable[Idx] =
      cos( (M_2PI * (double)Idx) / (double)MFSK_Modulator.SymbolLen );

  // If realloc fails, program will exit(-1)
  Mem_Realloc( (void **) &(MFSK_Modulator.SymbolShape), siz );
  {
    uint32_t Time;
    double Ampl = MFSK_SymbolFreqShape[0];
    for( Time = 0; Time < MFSK_Modulator.SymbolLen; Time++ )
      MFSK_Modulator.SymbolShape[Time] = Ampl;
  }

  uint32_t Freq;
  for( Freq = 1; Freq < MFSK_SymbolFreqShapeLen; Freq++ )
  {
    uint32_t Time;
    double Ampl = MFSK_SymbolFreqShape[Freq];
    if( Freq & 1 ) Ampl = -Ampl;

    uint32_t Phase = 0;
    for( Time = 0; Time < MFSK_Modulator.SymbolLen; Time++ )
    {
      MFSK_Modulator.SymbolShape[Time] +=
        Ampl * MFSK_Modulator.CosineTable[Phase];
      Phase += Freq;
      if( Phase >= MFSK_Modulator.SymbolLen )
        Phase -= MFSK_Modulator.SymbolLen;
    }
  }

  {
    uint32_t Time;
    double Scale = 1.0 / (double)( 2 * MFSK_Modulator.Parameters->CarrierSepar );
    for( Time = 0; Time < MFSK_Modulator.SymbolLen; Time++ )
      MFSK_Modulator.SymbolShape[Time] *= Scale;
  }

  // If realloc fails, program will exit(-1)
  siz = sizeof( double ) * MFSK_Modulator.SymbolLen;
  Mem_Realloc( (void **) &(MFSK_Modulator.OutTap), siz );

  for( Idx = 0; Idx < MFSK_Modulator.SymbolLen; Idx++ )
    MFSK_Modulator.OutTap[Idx] = 0.0;
  MFSK_Modulator.TapPtr = 0;

  MFSK_Modulator.WrapMask = MFSK_Modulator.SymbolLen - 1;
  MFSK_Modulator.SymbolPhase = 0;
  MFSK_Modulator.OutputLen = MFSK_Modulator.SymbolSepar;
}

//----------------------------------------------------------------------

  static void
MFSK_Modulator_Send( uint8_t Symbol )
{
  if( MFSK_Modulator.Parameters->UseGrayCode )
    GrayCode( Symbol );

  int SymbolFreq = (int)( MFSK_Modulator.Parameters->FirstCarrier +
      MFSK_Modulator.Parameters->CarrierSepar * Symbol );
  int TimeShift = MFSK_Modulator.SymbolSepar / 2 - MFSK_Modulator.SymbolLen / 2;

  MFSK_Modulator.SymbolPhase += SymbolFreq * TimeShift;
  MFSK_Modulator.SymbolPhase &= MFSK_Modulator.WrapMask;

  MFSK_Modulator.AddSymbol( SymbolFreq, MFSK_Modulator.SymbolPhase );

  TimeShift = MFSK_Modulator.SymbolSepar / 2 + MFSK_Modulator.SymbolLen / 2;
  MFSK_Modulator.SymbolPhase += SymbolFreq * TimeShift;
  MFSK_Modulator.SymbolPhase &= MFSK_Modulator.WrapMask;

  if( MFSK_Modulator.Parameters->PhaseDiffer )
  {
    int PhaseShift = MFSK_Modulator.SymbolLen / 4;
    if( rand() & 1 ) PhaseShift = -PhaseShift;
    MFSK_Modulator.SymbolPhase += PhaseShift;
  }

  MFSK_Modulator.SymbolPhase &= MFSK_Modulator.WrapMask;
}

//----------------------------------------------------------------------

// Get output as double data
  static uint32_t
MFSK_Modulator_Output_Float(double *Buffer )
{
  uint32_t Idx;
  for( Idx = 0; Idx < MFSK_Modulator.SymbolSepar; Idx++ )
  {
    Buffer[Idx] = MFSK_Modulator.OutTap[MFSK_Modulator.TapPtr];
    MFSK_Modulator.OutTap[MFSK_Modulator.TapPtr] = 0.0;
    MFSK_Modulator.TapPtr++;
    MFSK_Modulator.TapPtr &= MFSK_Modulator.WrapMask;
  }

  return( MFSK_Modulator.SymbolSepar );
}

//----------------------------------------------------------------------

// Get output as 16-bit signed data
  static uint32_t
MFSK_Modulator_Output_Int16( int16_t *Buffer )
{
  uint32_t Idx;

  for( Idx = 0; Idx < MFSK_Modulator.SymbolSepar; Idx++ )
  {
    double Ampl = MFSK_Modulator.OutTap[MFSK_Modulator.TapPtr];
    Ampl *= S16_SCALE;
    int32_t Out = (int32_t)( floor(Ampl + 0.5) );
    CLAMP_S16( Out );
    Buffer[Idx] = (int16_t)Out;
    MFSK_Modulator.OutTap[MFSK_Modulator.TapPtr] = 0;
    MFSK_Modulator.TapPtr++;
    MFSK_Modulator.TapPtr &= MFSK_Modulator.WrapMask;
  }

  return( MFSK_Modulator.SymbolSepar );
}

//----------------------------------------------------------------------

  static void
MFSK_Modulator_AddSymbol( int Freq, int Phase )
{
  uint32_t Time;
  for( Time = 0; Time < MFSK_Modulator.SymbolLen; Time++ )
  {
    MFSK_Modulator.OutTap[MFSK_Modulator.TapPtr] +=
      MFSK_Modulator.CosineTable[Phase] * MFSK_Modulator.SymbolShape[Time];
    Phase += Freq;
    Phase &= MFSK_Modulator.WrapMask;
    MFSK_Modulator.TapPtr++;
    MFSK_Modulator.TapPtr &= MFSK_Modulator.WrapMask;
  }
}

//----------------------------------------------------------------------

// Initialize the MFSK modulator. My function
  void
MFSK_Modulator_Initialize( void )
{
  MFSK_Modulator.Init         = MFSK_Modulator_Init;
  MFSK_Modulator.Free         = MFSK_Modulator_Free;
  MFSK_Modulator.Preset       = MFSK_Modulator_Preset;
  MFSK_Modulator.Send         = MFSK_Modulator_Send;
  MFSK_Modulator.Output_Int16 = MFSK_Modulator_Output_Int16;
  MFSK_Modulator.Output_Float = MFSK_Modulator_Output_Float;
  MFSK_Modulator.AddSymbol    = MFSK_Modulator_AddSymbol;

  MFSK_Modulator_Init();
}

// ***------------------------------------------------------------------***

// A running-box, low pass filter
static struct _BOXFILTER
{
  uint32_t Len;
  double  *Tap;
  uint32_t Ptr;
  double   Output;

  void (*Free)(void);
  void (*Preset)(void);
  void (*Clear)(void);
  void (*Process)(double Input);

} BoxFilter;

//----------------------------------------------------------------------

// Functions that were in class BoxFilter
  static void
BoxFilter_Free( void )
{
  Mem_Free( (void **) &(BoxFilter.Tap) );
}

//----------------------------------------------------------------------

  static void
BoxFilter_Preset( void )
{
  // If realloc fails, the program will exit(-1)
  Mem_Realloc( (void **) &(BoxFilter.Tap), sizeof(double) * BoxFilter.Len );
  BoxFilter.Clear();
}

//----------------------------------------------------------------------

  static void
BoxFilter_Clear( void )
{
  uint32_t Idx;
  for( Idx = 0; Idx < BoxFilter.Len; Idx++ )
    BoxFilter.Tap[Idx] = 0.0;
  BoxFilter.Ptr    = 0;
  BoxFilter.Output = 0.0;
}

//----------------------------------------------------------------------

  static void
BoxFilter_Process( double Input )
{
  BoxFilter.Output -= BoxFilter.Tap[BoxFilter.Ptr];
  BoxFilter.Output += Input;
  BoxFilter.Tap[BoxFilter.Ptr] = Input;
  BoxFilter.Ptr++;
  if( BoxFilter.Ptr >= BoxFilter.Len )
    BoxFilter.Ptr -= BoxFilter.Len;
}

//----------------------------------------------------------------------

// Initialize Box Filter - My function
  void
BoxFilter_Initialize( void )
{
  BoxFilter.Tap     = NULL;
  BoxFilter.Free    = BoxFilter_Free;
  BoxFilter.Preset  = BoxFilter_Preset;
  BoxFilter.Clear   = BoxFilter_Clear;
  BoxFilter.Process = BoxFilter_Process;
}

// ***------------------------------------------------------------------***

// Input processor, removes coherent interference and pulse noise
struct _MFSK_INPUTPROCESSOR
{
  // The user-settable parameters:
  uint32_t WindowLen;   // Spectral analysis (FFT) window length

  // Limiter level ( amplitude ) to reduce time
  // and frequency localised interference
  double LimiterLevel;

  uint32_t WrapMask;   // Wrap mask for buffer addressing

  double  *InpTap;     // Input buffer for analysis window
  uint32_t InpTapPtr;

  double  *OutTap;     // Output buffer for reconstruction window
  uint32_t OutTapPtr;

  double *WindowShape;  // Analysis/reconstruction window shape

  uint32_t SliceSepar;  // Time separation between analysis/reconstruction slices

  struct _R2FFT FFT;         // FFT engine
  complex double *FFT_Buff;  // FFT buffer

  uint32_t SpectraLen;         // Number of spectral points after FFT
  complex double *Spectra[2];  // Spectra buffer

  double *Output;  // ( Final ) output buffer after pulse limiter
  double *Energy;  // Energy vs frequency

  /* Pointers to functions that were
   * in class MFSK_InputProcessor */
  void     (*Init)(void);
  void     (*Free)(void);
  void     (*Default)(void);
  void     (*Preset)(void);
  void     (*Reset)(void);
  void     (*LimitSpectraPeaks)(complex double *, uint32_t);
  void     (*LimitOutputPeaks)(void);
  void     (*AverageEnergy)(uint32_t);
  void     (*ProcessSpectra)(complex double *);
  void     (*ProcessInpTap)(const double *);
  void     (*ProcessInpWindow_Re)(void);
  void     (*ProcessInpWindow_Im)(void);
  void     (*ProcessOutWindow_Re)(void);
  void     (*ProcessOutWindow_Im)(void);
  void     (*ProcessOutTap)(double *);
  uint32_t (*Process)(double *);
  //int      (*GetOutput)(int16_t *);

} MFSK_InputProcessor;

//----------------------------------------------------------------------

/* These are the functions that were
 * in the class MFSK_InputProcessor */

  static void
MFSK_InputProcessor_Init( void )
{
  MFSK_InputProcessor.InpTap      = NULL;
  MFSK_InputProcessor.OutTap      = NULL;
  MFSK_InputProcessor.WindowShape = NULL;
  MFSK_InputProcessor.FFT_Buff    = NULL;
  MFSK_InputProcessor.Spectra[0]  = NULL;
  MFSK_InputProcessor.Spectra[1]  = NULL;
  MFSK_InputProcessor.Output      = NULL;
  MFSK_InputProcessor.Energy      = NULL;
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_Free( void )
{
  Mem_Free( (void **) &(MFSK_InputProcessor.InpTap) );
  Mem_Free( (void **) &(MFSK_InputProcessor.OutTap) );
  Mem_Free( (void **) &(MFSK_InputProcessor.WindowShape) );
  Mem_Free( (void **) &(MFSK_InputProcessor.FFT_Buff) );
  Mem_Free( (void **) &(MFSK_InputProcessor.Spectra[0]) );
  Mem_Free( (void **) &(MFSK_InputProcessor.Spectra[1]) );
  Mem_Free( (void **) &(MFSK_InputProcessor.Output) );
  Mem_Free( (void **) &(MFSK_InputProcessor.Energy) );

  if( MFSK_InputProcessor.FFT.Free )
    MFSK_InputProcessor.FFT.Free( &(MFSK_InputProcessor.FFT) );
  if( BoxFilter.Free )
    BoxFilter.Free();
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_Default( void )
{
  MFSK_InputProcessor.WindowLen    = 8192;
  MFSK_InputProcessor.LimiterLevel = 2.5;
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_Preset( void )
{
  uint32_t Idx;

  MFSK_InputProcessor.WrapMask = MFSK_InputProcessor.WindowLen - 1;
  double ShapeScale = 2.0 / (double)MFSK_InputProcessor.WindowLen;

  // If any realloc fails, the program will exit(-1)
  size_t siz = sizeof(double) * MFSK_InputProcessor.WindowLen;
  Mem_Realloc( (void **) &(MFSK_InputProcessor.InpTap), siz );
  memset( MFSK_InputProcessor.InpTap, 0, siz );
  MFSK_InputProcessor.InpTapPtr = 0;

  Mem_Realloc( (void **) &(MFSK_InputProcessor.OutTap), siz );
  memset( MFSK_InputProcessor.OutTap, 0, siz );
  MFSK_InputProcessor.OutTapPtr = 0;
  r2FFT_Initialize( &(MFSK_InputProcessor.FFT) );

  MFSK_InputProcessor.FFT.Preset(
      &(MFSK_InputProcessor.FFT), MFSK_InputProcessor.WindowLen );

  siz = sizeof(complex double) * MFSK_InputProcessor.WindowLen;
  Mem_Realloc( (void **) &(MFSK_InputProcessor.FFT_Buff), siz );
  memset( MFSK_InputProcessor.FFT_Buff, 0, siz ); // My addition
  MFSK_InputProcessor.SliceSepar = MFSK_InputProcessor.WindowLen / 2;

  siz = sizeof(double) * MFSK_InputProcessor.WindowLen;
  Mem_Realloc( (void **) &(MFSK_InputProcessor.WindowShape), siz );
  for( Idx = 0; Idx < MFSK_InputProcessor.WindowLen; Idx++ )
    MFSK_InputProcessor.WindowShape[Idx] =
      ShapeScale * sqrt( 1.0 - creal(MFSK_InputProcessor.FFT.Twiddle[Idx]) );

  Mem_Realloc( (void **) &(MFSK_InputProcessor.Output), siz );
  memset( MFSK_InputProcessor.Output, 0, siz );

  MFSK_InputProcessor.SpectraLen = MFSK_InputProcessor.WindowLen / 2;
  siz = sizeof(complex double) * MFSK_InputProcessor.SpectraLen;
  Mem_Realloc( (void **) &(MFSK_InputProcessor.Spectra[0]), siz );
  Mem_Realloc( (void **) &(MFSK_InputProcessor.Spectra[1]), siz );

  siz = sizeof(double) * MFSK_InputProcessor.SpectraLen;
  Mem_Realloc( (void **) &(MFSK_InputProcessor.Energy), siz );

  BoxFilter_Initialize();
  BoxFilter.Len = MFSK_InputProcessor.WindowLen / 16;
  BoxFilter.Preset();
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_Reset( void )
{
  size_t siz = sizeof(double) * MFSK_InputProcessor.WindowLen;
  memset( MFSK_InputProcessor.InpTap, 0, siz );
  MFSK_InputProcessor.InpTapPtr = 0;
  memset( MFSK_InputProcessor.OutTap, 0, siz );
  MFSK_InputProcessor.OutTapPtr = 0;
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_LimitSpectraPeaks( complex double *Spectra, uint32_t BoxLen )
{
  BoxFilter.Len = BoxLen;
  BoxFilter.Preset();

  uint32_t MaxFreq = 3 * ( MFSK_InputProcessor.SpectraLen / 4 );
  uint32_t Freq, Idx;

  for( Freq = 0; Freq < BoxLen; Freq++ )
    BoxFilter.Process( MFSK_InputProcessor.Energy[Freq] );

  double Threshold =
    MFSK_InputProcessor.LimiterLevel * MFSK_InputProcessor.LimiterLevel;
  for( Idx = BoxLen / 2; Freq < MaxFreq; Freq++, Idx++)
  {
    BoxFilter.Process( MFSK_InputProcessor.Energy[Freq] );

    double Signal = MFSK_InputProcessor.Energy[Idx];
    double Limit  = ( BoxFilter.Output / (double)BoxLen ) * Threshold;
    if( Signal > Limit )
    {
      Spectra[Idx] *= sqrt( Limit / Signal );
      MFSK_InputProcessor.Energy[Idx] = Limit;
    }
  }
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_LimitOutputPeaks( void )
{
  uint32_t Idx;
  double   RMS = 0.0;
  for( Idx = 0; Idx < MFSK_InputProcessor.WindowLen; Idx++ )
  {
    double Signal = MFSK_InputProcessor.Output[Idx];
    RMS += Signal * Signal;
  }

  RMS = sqrt( RMS / (double)MFSK_InputProcessor.WindowLen );
  double Limit = RMS * MFSK_InputProcessor.LimiterLevel;

  for( Idx = 0; Idx < MFSK_InputProcessor.WindowLen; Idx++ )
  {
    double Signal = MFSK_InputProcessor.Output[Idx];
    if( Signal < -Limit ) Signal = -Limit;
    else if( Signal > Limit ) Signal = Limit;
    MFSK_InputProcessor.Output[Idx] = Signal;
  }
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_AverageEnergy( uint32_t Len )
{
  BoxFilter.Len = Len;
  BoxFilter.Preset();

  uint32_t MaxFreq =  3 * ( MFSK_InputProcessor.SpectraLen / 4 );
  double   Scale   = 1.0 / (double)Len;
  uint32_t Len2    = Len / 2;
  uint32_t Idx, Freq;

  for( Freq = 0; Freq < Len; Freq++ )
    BoxFilter.Process( MFSK_InputProcessor.Energy[Freq] );

  for( Idx = 0; Idx < Len2; Idx++ )
    MFSK_InputProcessor.Energy[Idx] = BoxFilter.Output * Scale;

  for( ; Freq < MaxFreq; Freq++, Idx++ )
  {
    BoxFilter.Process( MFSK_InputProcessor.Energy[Freq] );
    MFSK_InputProcessor.Energy[Idx] = BoxFilter.Output * Scale;
  }

  for( ; Idx < MFSK_InputProcessor.SpectraLen; Idx++ )
    MFSK_InputProcessor.Energy[Idx] = BoxFilter.Output * Scale;
}

//----------------------------------------------------------------------

// Here we process the spectral data
  static void
MFSK_InputProcessor_ProcessSpectra( complex double *Spectra )
{
  uint32_t Freq;

  for( Freq = 0; Freq < MFSK_InputProcessor.SpectraLen; Freq++ )
  {
    MFSK_InputProcessor.Energy[Freq] = Complex_Energy( &(Spectra[Freq]) );
  }

  uint32_t limit = MFSK_InputProcessor.WindowLen / 64;
  MFSK_InputProcessor.LimitSpectraPeaks( Spectra, limit );
  MFSK_InputProcessor.LimitSpectraPeaks( Spectra, limit );
  MFSK_InputProcessor.LimitSpectraPeaks( Spectra, limit );

  MFSK_InputProcessor.AverageEnergy( MFSK_InputProcessor.WindowLen / 96 );
  MFSK_InputProcessor.AverageEnergy( MFSK_InputProcessor.WindowLen / 64 );

  for( Freq = 0; Freq < MFSK_InputProcessor.SpectraLen; Freq++ )
  {
    double Corr = MFSK_InputProcessor.Energy[Freq];
    if( Corr <= 0 ) continue;
    Corr = 1.0 / sqrt( Corr );
    Spectra[Freq] *= Corr;
  }
}

//----------------------------------------------------------------------

// Combined two "overloaded" functions into this one
  static void
MFSK_InputProcessor_ProcessInpTap( const double *Input )
{
  uint32_t InpIdx;
  for( InpIdx = 0; InpIdx < MFSK_InputProcessor.SliceSepar; InpIdx++ )
  {
    if( Input )
      MFSK_InputProcessor.InpTap[MFSK_InputProcessor.InpTapPtr] = Input[InpIdx];
    else
      MFSK_InputProcessor.InpTap[MFSK_InputProcessor.InpTapPtr] = 0.0;
    MFSK_InputProcessor.InpTapPtr++;
    MFSK_InputProcessor.InpTapPtr &= MFSK_InputProcessor.WrapMask;
  }
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_ProcessInpWindow_Re( void )
{
  uint32_t Time;
  for( Time = 0; Time < MFSK_InputProcessor.WindowLen; Time++ )
  {
    MFSK_InputProcessor.FFT_Buff[Time] =
      MFSK_InputProcessor.InpTap[MFSK_InputProcessor.InpTapPtr] *
      MFSK_InputProcessor.WindowShape[Time] +
      (complex double)I * cimag( MFSK_InputProcessor.FFT_Buff[Time] );

    MFSK_InputProcessor.InpTapPtr++;
    MFSK_InputProcessor.InpTapPtr &= MFSK_InputProcessor.WrapMask;
  }
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_ProcessInpWindow_Im( void )
{
  uint32_t Time;
  for( Time = 0; Time < MFSK_InputProcessor.WindowLen; Time++ )
  {
    MFSK_InputProcessor.FFT_Buff[Time] =
      creal( MFSK_InputProcessor.FFT_Buff[Time] ) +
      (complex double)I * MFSK_InputProcessor.InpTap[MFSK_InputProcessor.InpTapPtr] *
      MFSK_InputProcessor.WindowShape[Time];

    MFSK_InputProcessor.InpTapPtr++;
    MFSK_InputProcessor.InpTapPtr &= MFSK_InputProcessor.WrapMask;
  }
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_ProcessOutWindow_Re( void )
{
  uint32_t Time;
  for( Time = 0; Time < MFSK_InputProcessor.WindowLen; Time++ )
  {
    MFSK_InputProcessor.OutTap[MFSK_InputProcessor.OutTapPtr] +=
      creal( MFSK_InputProcessor.FFT_Buff[Time] ) *
      MFSK_InputProcessor.WindowShape[Time];

    MFSK_InputProcessor.OutTapPtr++;
    MFSK_InputProcessor.OutTapPtr &= MFSK_InputProcessor.WrapMask;
  }
}

//----------------------------------------------------------------------

  static void
MFSK_InputProcessor_ProcessOutWindow_Im( void )
{
  uint32_t Time;
  for( Time = 0; Time < MFSK_InputProcessor.WindowLen; Time++ )
  {
    MFSK_InputProcessor.OutTap[MFSK_InputProcessor.OutTapPtr] +=
      cimag( MFSK_InputProcessor.FFT_Buff[Time] ) *
      MFSK_InputProcessor.WindowShape[Time];

    MFSK_InputProcessor.OutTapPtr++;
    MFSK_InputProcessor.OutTapPtr &= MFSK_InputProcessor.WrapMask;
  }
}

//-- --------------------------------------------------------------------

  static void
MFSK_InputProcessor_ProcessOutTap( double *Output )
{
  uint32_t OutIdx;
  for( OutIdx = 0; OutIdx < MFSK_InputProcessor.SliceSepar; OutIdx++ )
  {
    Output[OutIdx] = MFSK_InputProcessor.OutTap[MFSK_InputProcessor.OutTapPtr];
    MFSK_InputProcessor.OutTap[MFSK_InputProcessor.OutTapPtr] = 0;
    MFSK_InputProcessor.OutTapPtr++;
    MFSK_InputProcessor.OutTapPtr &= MFSK_InputProcessor.WrapMask;
  }
}

//----------------------------------------------------------------------

  static uint32_t
MFSK_InputProcessor_Process( double *Input )
{
  if( Input )
    MFSK_InputProcessor.ProcessInpTap( Input );
  else
    MFSK_InputProcessor.ProcessInpTap( NULL );
  MFSK_InputProcessor.ProcessInpWindow_Re();

  if( Input )
    MFSK_InputProcessor.ProcessInpTap( Input + MFSK_InputProcessor.SliceSepar );
  else
    MFSK_InputProcessor.ProcessInpTap( NULL );
  MFSK_InputProcessor.ProcessInpWindow_Im();

  MFSK_InputProcessor.FFT.Process(
      &(MFSK_InputProcessor.FFT), MFSK_InputProcessor.FFT_Buff );

  MFSK_InputProcessor.FFT.SeparTwoReals(
      &(MFSK_InputProcessor.FFT),
      MFSK_InputProcessor.FFT_Buff,
      MFSK_InputProcessor.Spectra[0],
      MFSK_InputProcessor.Spectra[1] );

  MFSK_InputProcessor.ProcessSpectra( MFSK_InputProcessor.Spectra[0] );
  MFSK_InputProcessor.ProcessSpectra( MFSK_InputProcessor.Spectra[1] );

  MFSK_InputProcessor.FFT.JoinTwoReals(
      &(MFSK_InputProcessor.FFT),
      MFSK_InputProcessor.Spectra[0],
      MFSK_InputProcessor.Spectra[1],
      MFSK_InputProcessor.FFT_Buff );

  MFSK_InputProcessor.FFT.Process(
      &(MFSK_InputProcessor.FFT), MFSK_InputProcessor.FFT_Buff );

  MFSK_InputProcessor.ProcessOutWindow_Re();
  MFSK_InputProcessor.ProcessOutTap( MFSK_InputProcessor.Output );
  MFSK_InputProcessor.ProcessOutWindow_Im();
  MFSK_InputProcessor.ProcessOutTap(
      MFSK_InputProcessor.Output + MFSK_InputProcessor.SliceSepar );

  MFSK_InputProcessor.LimitOutputPeaks();
  MFSK_InputProcessor.LimitOutputPeaks();

  return( MFSK_InputProcessor.WindowLen );
}

//----------------------------------------------------------------------

// Get output as 16-bit signed data
/*
   static int
   MFSK_InputProcessor_GetOutput( int16_t *Buffer )
   {
   const double Scale   = 32768.0;
   const int32_t Limit = 0x7FFF;
   uint32_t Idx;

   for( Idx = 0; Idx < MFSK_InputProcessor.WindowLen; Idx++ )
   {
   double Ampl = MFSK_InputProcessor.Output[Idx];
   Ampl *= Scale;
   int32_t Out = (int32_t)( floorf(Ampl + 0.5f) );
   if( Out > Limit ) Out = Limit;
   else if( Out < -Limit ) Out = -Limit;
   Buffer[Idx] = (int16_t)Out;
   }

   return( MFSK_InputProcessor.WindowLen );
   }
 */
//----------------------------------------------------------------------

// Initializes the MFSK_InputProcessor - My function
  void
MFSK_InputProcessor_Initialize( void )
{
  MFSK_InputProcessor.Init                = MFSK_InputProcessor_Init;
  MFSK_InputProcessor.Free                = MFSK_InputProcessor_Free;
  MFSK_InputProcessor.Default             = MFSK_InputProcessor_Default;
  MFSK_InputProcessor.Preset              = MFSK_InputProcessor_Preset;
  MFSK_InputProcessor.Reset               = MFSK_InputProcessor_Reset;
  MFSK_InputProcessor.LimitSpectraPeaks   = MFSK_InputProcessor_LimitSpectraPeaks;
  MFSK_InputProcessor.LimitOutputPeaks    = MFSK_InputProcessor_LimitOutputPeaks;
  MFSK_InputProcessor.AverageEnergy       = MFSK_InputProcessor_AverageEnergy;
  MFSK_InputProcessor.ProcessSpectra      = MFSK_InputProcessor_ProcessSpectra;
  MFSK_InputProcessor.ProcessInpTap       = MFSK_InputProcessor_ProcessInpTap;
  MFSK_InputProcessor.ProcessInpWindow_Re = MFSK_InputProcessor_ProcessInpWindow_Re;
  MFSK_InputProcessor.ProcessInpWindow_Im = MFSK_InputProcessor_ProcessInpWindow_Im;
  MFSK_InputProcessor.ProcessOutWindow_Re = MFSK_InputProcessor_ProcessOutWindow_Re;
  MFSK_InputProcessor.ProcessOutWindow_Im = MFSK_InputProcessor_ProcessOutWindow_Im;
  MFSK_InputProcessor.ProcessOutTap       = MFSK_InputProcessor_ProcessOutTap;
  MFSK_InputProcessor.Process             = MFSK_InputProcessor_Process;
  //MFSK_InputProcessor.GetOutput           = MFSK_InputProcessor_GetOutput;

  MFSK_InputProcessor_Init();
}

// ***------------------------------------------------------------------***

// Front-end spectral analysis and soft decoder for MFSK
struct _MFSK_DEMODULATOR
{
  MFSK_Parameters_t *Parameters;

  uint32_t InputLen;       // Input must be provided in batches of that length [samples]
  uint32_t SymbolSepar;
  uint32_t SymbolLen;
  uint32_t SpectraPerSymbol;

  uint32_t DecodeMargin;  // Frequency margin for decoding the signal [FFT bins]
  uint32_t DecodeWidth;   // Spectra width                            [FFT bins]

  uint32_t SliceSepar;    // Time separation between samples          [samples]

  uint32_t WrapMask;      // Wrapping mask for the FFT window

  double  *InpTap;        // Input buffer
  uint32_t InpTapPtr;

  double *SymbolShape;    // The shape of the symbol and the FFT window

  r2FFT_t FFT;  // FFT engine
  complex double *FFT_Buff;  // FFT buffer

  uint32_t SpectraLen;         // Number of spectral points per FFT
  complex double *Spectra[2];  // 2 buffers for FFT spectra

  CircularBuffer_Float_t History;  // Spectra history

  // Pointers to functions in class MFSK_Demodulator
  void     (*Init)(void);
  void     (*Free)(void);
  void     (*Preset)(MFSK_Parameters_t *);
  void     (*Reset)(void);
  double   *(*HistoryPtr)(int);
  uint32_t (*SlideOneSlice)(const double *);
  void     (*Process)(double *);
  BOOLEAN  (*PickBlock)(double *, int, int);

} MFSK_Demodulator;

//----------------------------------------------------------------------

  static void
MFSK_Demodulator_Init( void )
{
  MFSK_Demodulator.InpTap      = NULL;
  MFSK_Demodulator.SymbolShape = NULL;
  MFSK_Demodulator.FFT_Buff    = NULL;
  MFSK_Demodulator.Spectra[0]  = NULL;
  MFSK_Demodulator.Spectra[1]  = NULL;
}

//----------------------------------------------------------------------

  static void
MFSK_Demodulator_Free( void )
{
  Mem_Free( (void **) &(MFSK_Demodulator.InpTap) );
  Mem_Free( (void **) &(MFSK_Demodulator.SymbolShape) );
  Mem_Free( (void **) &(MFSK_Demodulator.FFT_Buff) );
  Mem_Free( (void **) &(MFSK_Demodulator.Spectra[0]) );
  Mem_Free( (void **) &(MFSK_Demodulator.Spectra[1]) );

  if( MFSK_Demodulator.FFT.Free )
    MFSK_Demodulator.FFT.Free( &(MFSK_Demodulator.FFT) );
  if( MFSK_Demodulator.History.Free )
    MFSK_Demodulator.History.Free( &(MFSK_Demodulator.History) );
}

//----------------------------------------------------------------------

  static void
MFSK_Demodulator_Preset( MFSK_Parameters_t *NewParameters )
{
  MFSK_Demodulator.Parameters = NewParameters;

  MFSK_Demodulator.SymbolSepar      = MFSK_Demodulator.Parameters->SymbolSepar;
  MFSK_Demodulator.SymbolLen        = MFSK_Demodulator.Parameters->SymbolLen;
  MFSK_Demodulator.SpectraPerSymbol = MFSK_Demodulator.Parameters->SpectraPerSymbol;

  MFSK_Demodulator.InputLen = MFSK_Demodulator.SymbolSepar;
  MFSK_Demodulator.DecodeMargin =
    MFSK_Demodulator.Parameters->RxSyncMargin *
    MFSK_Demodulator.Parameters->CarrierSepar;

  MFSK_Demodulator.WrapMask = MFSK_Demodulator.SymbolLen - 1;

  double ShapeScale = 1.0 / (double)MFSK_Demodulator.SymbolLen;

  size_t siz = sizeof(double) * MFSK_Demodulator.SymbolLen;
  Mem_Realloc( (void **) &(MFSK_Demodulator.InpTap), siz );
  memset( MFSK_Demodulator.InpTap, 0, siz );
  MFSK_Demodulator.InpTapPtr = 0;

  r2FFT_Initialize( &(MFSK_Demodulator.FFT) );
  MFSK_Demodulator.FFT.Preset(
      &(MFSK_Demodulator.FFT), MFSK_Demodulator.SymbolLen );

  siz = sizeof(complex double) * MFSK_Demodulator.SymbolLen;
  Mem_Realloc( (void **) &(MFSK_Demodulator.FFT_Buff), siz );
  MFSK_Demodulator.SliceSepar =
    MFSK_Demodulator.SymbolSepar / MFSK_Demodulator.SpectraPerSymbol;

  siz = sizeof(double) * MFSK_Demodulator.SymbolLen;
  Mem_Realloc( (void **) &(MFSK_Demodulator.SymbolShape), siz );
  {
    uint32_t Time;
    double Ampl = MFSK_SymbolFreqShape[0];
    for( Time = 0; Time < MFSK_Demodulator.SymbolLen; Time++ )
      MFSK_Demodulator.SymbolShape[Time] = Ampl;
  }

  uint32_t Freq;
  for( Freq = 1; Freq < MFSK_SymbolFreqShapeLen; Freq++ )
  {
    uint32_t Time;
    double Ampl = MFSK_SymbolFreqShape[Freq];
    if( Freq & 1 ) Ampl = -Ampl;

    uint32_t Phase = 0;
    for( Time = 0; Time < MFSK_Demodulator.SymbolLen; Time++ )
    {
      MFSK_Demodulator.SymbolShape[Time] +=
        Ampl * creal( MFSK_Demodulator.FFT.Twiddle[Phase] );
      Phase += Freq;
      if( Phase >= MFSK_Demodulator.SymbolLen )
        Phase -= MFSK_Demodulator.SymbolLen;
    }
  }

  {
    uint32_t Time;
    for( Time = 0; Time < MFSK_Demodulator.SymbolLen; Time++ )
      MFSK_Demodulator.SymbolShape[Time] *= ShapeScale;
  }

  MFSK_Demodulator.SpectraLen = MFSK_Demodulator.SymbolLen / 2;
  siz = sizeof( complex double ) * MFSK_Demodulator.SpectraLen;
  Mem_Realloc( (void **) &(MFSK_Demodulator.Spectra[0]), siz );
  Mem_Realloc( (void **) &(MFSK_Demodulator.Spectra[1]), siz );

  MFSK_Demodulator.DecodeWidth =
    ( (MFSK_Demodulator.Parameters->Carriers - 1) *
      MFSK_Demodulator.Parameters->CarrierSepar + 1 ) +
    2 * MFSK_Demodulator.DecodeMargin;

  CircularBuffer_Float_Initialize( &(MFSK_Demodulator.History) );
  MFSK_Demodulator.History.Len =
    ( MFSK_Demodulator.Parameters->RxSyncIntegLen + 2 ) *
    MFSK_Demodulator.Parameters->SpectraPerBlock;
  MFSK_Demodulator.History.Width = MFSK_Demodulator.DecodeWidth;
  MFSK_Demodulator.History.Preset( &MFSK_Demodulator.History );
  MFSK_Demodulator.History.Clear( &(MFSK_Demodulator.History) );
}

//----------------------------------------------------------------------

  static void
MFSK_Demodulator_Reset( void )
{
  MFSK_Demodulator.History.Clear( &(MFSK_Demodulator.History) );
}

//----------------------------------------------------------------------

  static double
*MFSK_Demodulator_HistoryPtr( int Idx )
{
  return( MFSK_Demodulator.History.OffsetPtr(&(MFSK_Demodulator.History), Idx) );
}

//----------------------------------------------------------------------

  static uint32_t
MFSK_Demodulator_SlideOneSlice( const double *Input )
{
  uint32_t InpIdx;
  for( InpIdx = 0; InpIdx < MFSK_Demodulator.SliceSepar; InpIdx++ )
  {
    MFSK_Demodulator.InpTap[MFSK_Demodulator.InpTapPtr] = Input[InpIdx];
    MFSK_Demodulator.InpTapPtr++;
    MFSK_Demodulator.InpTapPtr &= MFSK_Demodulator.WrapMask;
  }

  return( MFSK_Demodulator.SliceSepar );
}

//----------------------------------------------------------------------

//template <class InpType>
  static void
MFSK_Demodulator_Process( double *Input )
{
  uint32_t InpIdx, Time, Slice;

  for( InpIdx = 0, Slice = 0; Slice < MFSK_Demodulator.SpectraPerSymbol; Slice += 2 )
  {
    InpIdx += MFSK_Demodulator.SlideOneSlice( Input + InpIdx );

    for( Time = 0; Time < MFSK_Demodulator.SymbolLen; Time++ )
    {
      MFSK_Demodulator.FFT_Buff[Time] =
        MFSK_Demodulator.InpTap[MFSK_Demodulator.InpTapPtr] *
        MFSK_Demodulator.SymbolShape[Time] + (complex double)I * 0.0;

      MFSK_Demodulator.InpTapPtr++;
      MFSK_Demodulator.InpTapPtr &= MFSK_Demodulator.WrapMask;
    }

    InpIdx += MFSK_Demodulator.SlideOneSlice( Input + InpIdx );

    for( Time = 0; Time < MFSK_Demodulator.SymbolLen; Time++ )
    {
      MFSK_Demodulator.FFT_Buff[Time] =
        MFSK_Demodulator.FFT_Buff[Time] +
        (complex double)I * MFSK_Demodulator.InpTap[MFSK_Demodulator.InpTapPtr] *
        MFSK_Demodulator.SymbolShape[Time];

      MFSK_Demodulator.InpTapPtr++;
      MFSK_Demodulator.InpTapPtr &= MFSK_Demodulator.WrapMask;
    }

    MFSK_Demodulator.FFT.Process(
        &(MFSK_Demodulator.FFT), MFSK_Demodulator.FFT_Buff );
    MFSK_Demodulator.FFT.SeparTwoReals(
        &(MFSK_Demodulator.FFT),
        MFSK_Demodulator.FFT_Buff,
        MFSK_Demodulator.Spectra[0],
        MFSK_Demodulator.Spectra[1] );

    double *Data0 =
      MFSK_Demodulator.History.OffsetPtr( &(MFSK_Demodulator.History), 0 );
    double *Data1 =
      MFSK_Demodulator.History.OffsetPtr( &(MFSK_Demodulator.History), 1 );

    uint32_t Idx;
    uint32_t Freq =
      MFSK_Demodulator.Parameters->FirstCarrier - MFSK_Demodulator.DecodeMargin;

    for( Idx = 0; Idx < MFSK_Demodulator.DecodeWidth; Idx++, Freq++ )
    {
      Data0[Idx] = Complex_Energy( &(MFSK_Demodulator.Spectra[0][Freq]) );
      Data1[Idx] = Complex_Energy( &(MFSK_Demodulator.Spectra[1][Freq]) );
    }

    MFSK_Demodulator.History.IncrPtr(
        &MFSK_Demodulator.History, &(MFSK_Demodulator.History.Ptr), 2 );
  }
}

//----------------------------------------------------------------------

  static BOOLEAN
MFSK_Demodulator_PickBlock( double *Spectra, int TimeOffset, int FreqOffset )
{
  int SpectraPerBlock = (int)MFSK_Demodulator.Parameters->SpectraPerBlock;
  if( ( TimeOffset > -SpectraPerBlock) ||
      (-TimeOffset > (int)MFSK_Demodulator.History.Len) )
    return( False );

  uint32_t Carriers     = MFSK_Demodulator.Parameters->Carriers;
  uint32_t CarrierSepar = MFSK_Demodulator.Parameters->CarrierSepar;

  if( (FreqOffset < 0) ||
      ((FreqOffset + ((int)Carriers - 1) * (int)CarrierSepar) >=
       (int)MFSK_Demodulator.DecodeWidth) )
    return( False );

  uint32_t SymbolsPerBlock = MFSK_Demodulator.Parameters->SymbolsPerBlock;
  uint32_t Symbol;
  for( Symbol = 0;
      Symbol < SymbolsPerBlock;
      Symbol++, TimeOffset += MFSK_Demodulator.SpectraPerSymbol )
  {
    double *Hist = MFSK_Demodulator.History.OffsetPtr(
        &(MFSK_Demodulator.History), TimeOffset ) + FreqOffset;

    uint32_t Freq;
    for( Freq = 0; Freq < Carriers; Freq++, Hist += CarrierSepar )
    {
      *( Spectra++ ) = *Hist;
    }
  }

  return( True );
}

//----------------------------------------------------------------------

// Initializes the MFSK_Demodulator - My function
  void
MFSK_Demodulator_Initialize( void )
{
  MFSK_Demodulator.Init          = MFSK_Demodulator_Init;
  MFSK_Demodulator.Free          = MFSK_Demodulator_Free;
  MFSK_Demodulator.Preset        = MFSK_Demodulator_Preset;
  MFSK_Demodulator.Reset         = MFSK_Demodulator_Reset;
  MFSK_Demodulator.HistoryPtr    = MFSK_Demodulator_HistoryPtr;
  MFSK_Demodulator.SlideOneSlice = MFSK_Demodulator_SlideOneSlice;
  MFSK_Demodulator.Process       = MFSK_Demodulator_Process;
  MFSK_Demodulator.PickBlock     = MFSK_Demodulator_PickBlock;

  MFSK_Demodulator_Init();
}

// ***------------------------------------------------------------------***

// FEC block encoder
struct _MFSK_ENCODER
{
  // Parameters to be set before calling Preset()
  // Number of bits per MFSK symbol
  // (default is 5 bits thus 32 possible symbols)
  uint32_t BitsPerSymbol;

  // Number of bits per transmitted character
  // (default is 7 bits for ASCII code)
  uint32_t BitsPerCharacter;

  uint32_t Symbols;          // Number of possible MFSK symbols
  uint32_t SymbolsPerBlock;  // Number of MFSK symbols per FEC block

  uint64_t ScramblingCode;

  int8_t *FHT_Buffer;    // Temporary buffer for ( inverse ) Fast Hadamard Transform

  uint8_t *OutputBlock;  // Encoded block is stored here

  // Pointers to functions in class MFSK_Encoder
  void (*Init)(void);
  void (*Free)(void);
  void (*Default)(void);
  void (*Preset)(void);
  void (*EncodeCharacter)(uint8_t);
  void (*ScrambleFHT)(uint32_t);
  void (*EncodeBlock)(uint8_t *);

} MFSK_Encoder;

//----------------------------------------------------------------------

  static void
MFSK_Encoder_Default( void )
{
  MFSK_Encoder.BitsPerSymbol = 5;
  if( olivia_rc_data.mode_olivia )
    MFSK_Encoder.BitsPerCharacter = 7;  // For Olivia mode
  else
    MFSK_Encoder.BitsPerCharacter = 6;  // For CONTESTIA mode
}

//----------------------------------------------------------------------

  static void
MFSK_Encoder_Init( void )
{
  MFSK_Encoder.FHT_Buffer  = NULL;
  MFSK_Encoder.OutputBlock = NULL;
  if( olivia_rc_data.mode_olivia )
    MFSK_Encoder.ScramblingCode = 0xE257E6D0291574ECLL;
  else
    MFSK_Encoder.ScramblingCode = 0xEDB88320LL;
}

//----------------------------------------------------------------------

  static void
MFSK_Encoder_Free( void )
{
  Mem_Free( (void **) &(MFSK_Encoder.FHT_Buffer) );
  Mem_Free( (void **) &(MFSK_Encoder.OutputBlock) );
}

//----------------------------------------------------------------------

  static void
MFSK_Encoder_Preset( void )
{
  MFSK_Encoder.Symbols = Exp2( MFSK_Encoder.BitsPerSymbol );
  MFSK_Encoder.SymbolsPerBlock = Exp2( MFSK_Encoder.BitsPerCharacter - 1 );

  size_t siz = sizeof(int8_t) *  MFSK_Encoder.SymbolsPerBlock;
  Mem_Realloc( (void **) &(MFSK_Encoder.FHT_Buffer), siz );
  Mem_Realloc( (void **) &(MFSK_Encoder.OutputBlock), siz );
}

//----------------------------------------------------------------------

// Encode a single character with the IFHT
  static void
MFSK_Encoder_EncodeCharacter( uint8_t Char )
{
  uint32_t TimeBit;
  uint8_t Mask = ( (uint8_t)(MFSK_Encoder.SymbolsPerBlock << 1) ) - 1;

  if( olivia_rc_data.mode_olivia )  // For Olivia mode
    Char &= Mask;
  else   // For Contestia mode
  {
    if( (Char >= 'a') && (Char <= 'z') ) Char += 'A' - 'a';  // Capitalize
    if( Char == ' ' )     Char = ';';      // Space to ;
    else if( Char == LF ) Char = '<';      // Line Feed to <
    else if( Char == CR ) Char = 0;        // Carriage Return to NULL character
    else if( (Char >= 33) && (Char <= 90) ) Char -= 32;  // Reduce width to 6 bits
    else if( Char == BS ) Char = '=';      // Backspace to =
    else if( Char != 0 )  Char = '?' - 32; // Reduce width to 6 bits?
  }

  for( TimeBit = 0; TimeBit < MFSK_Encoder.SymbolsPerBlock; TimeBit++ )
    MFSK_Encoder.FHT_Buffer[TimeBit] = 0;
  if( Char < MFSK_Encoder.SymbolsPerBlock )
    MFSK_Encoder.FHT_Buffer[Char] = 1;
  else
    MFSK_Encoder.FHT_Buffer[Char - MFSK_Encoder.SymbolsPerBlock] = -1;
  IFHT_Int8( MFSK_Encoder.FHT_Buffer, MFSK_Encoder.SymbolsPerBlock );
}

//----------------------------------------------------------------------

// Scramble the codeword ( of a single character ) with the scrambling code
  static void
MFSK_Encoder_ScrambleFHT( uint32_t CodeOffset )
{
  uint32_t TimeBit;
  uint32_t CodeWrap = ( MFSK_Encoder.SymbolsPerBlock - 1 );
  uint32_t CodeBit  = CodeOffset & CodeWrap;
  for( TimeBit = 0; TimeBit < MFSK_Encoder.SymbolsPerBlock; TimeBit++ )
  {
    uint64_t CodeMask = 1;
    CodeMask <<= CodeBit;
    if( MFSK_Encoder.ScramblingCode & CodeMask )
      MFSK_Encoder.FHT_Buffer[TimeBit] = -MFSK_Encoder.FHT_Buffer[TimeBit];
    CodeBit++;
    CodeBit &= CodeWrap;
  }
}

//----------------------------------------------------------------------

// Encode a block of SymbolsPerBlock characters
  static void
MFSK_Encoder_EncodeBlock( uint8_t *InputBlock )
{
  uint32_t FreqBit, TimeBit;

  for( TimeBit = 0; TimeBit < MFSK_Encoder.SymbolsPerBlock; TimeBit++ )
    MFSK_Encoder.OutputBlock[TimeBit] = 0;

  for( FreqBit = 0; FreqBit < MFSK_Encoder.BitsPerSymbol; FreqBit++ )
  {
    uint32_t nShift;
    if( olivia_rc_data.mode_olivia )
      nShift = 13;  // For Olivia
    else
      nShift = 5;   // For Contestia

    MFSK_Encoder.EncodeCharacter( InputBlock[FreqBit] );
    MFSK_Encoder.ScrambleFHT( FreqBit * nShift );

    uint32_t Rotate = 0;
    for( TimeBit = 0; TimeBit < MFSK_Encoder.SymbolsPerBlock; TimeBit++ )
    {
      if( MFSK_Encoder.FHT_Buffer[TimeBit] < 0 )
      {
        uint32_t Bit = FreqBit + Rotate;
        if( Bit >= MFSK_Encoder.BitsPerSymbol )
          Bit -= MFSK_Encoder.BitsPerSymbol;
        uint8_t Mask = 1;
        Mask <<= Bit;
        MFSK_Encoder.OutputBlock[TimeBit] |= Mask;
      }

      Rotate++;
      if( Rotate >= MFSK_Encoder.BitsPerSymbol )
        Rotate -= MFSK_Encoder.BitsPerSymbol;
    }
  }
}

//----------------------------------------------------------------------

// Initialize MFSK_Encoder - my function
  void
MFSK_Encoder_Initialize( void )
{
  MFSK_Encoder.Default          = MFSK_Encoder_Default;
  MFSK_Encoder.Init             = MFSK_Encoder_Init;
  MFSK_Encoder.Free             = MFSK_Encoder_Free;
  MFSK_Encoder.Preset           = MFSK_Encoder_Preset;
  MFSK_Encoder.EncodeCharacter  = MFSK_Encoder_EncodeCharacter;
  MFSK_Encoder.ScrambleFHT      = MFSK_Encoder_ScrambleFHT;
  MFSK_Encoder.EncodeBlock      = MFSK_Encoder_EncodeBlock;

  MFSK_Encoder_Init();
  MFSK_Encoder_Default();
}

//----------------------------------------------------------------------

  static void
MFSK_SoftDecoder_Init( MFSK_SoftDecoder_t *Self )
{
  Self->InputBuffer = NULL;
  Self->FHT_Buffer  = NULL;
  Self->OutputBlock = NULL;
}

//----------------------------------------------------------------------

  static void
MFSK_SoftDecoder_Free( MFSK_SoftDecoder_t *Self )
{
  Mem_Free( (void **) &(Self->InputBuffer) );
  Mem_Free( (void **) &(Self->FHT_Buffer) );
  Mem_Free( (void **) &(Self->OutputBlock) );
}

//----------------------------------------------------------------------

  static void
MFSK_SoftDecoder_Reset( MFSK_SoftDecoder_t *Self )
{
  uint32_t Idx;
  for( Idx = 0; Idx < Self->InputBufferLen; Idx++ )
    Self->InputBuffer[Idx] = 0.0;
  Self->InputPtr = 0;
}

//----------------------------------------------------------------------

  static void
MFSK_SoftDecoder_Preset(
    MFSK_SoftDecoder_t *Self,
    MFSK_Parameters_t *NewParameters )
{
  Self->Parameters = NewParameters;

  Self->BitsPerSymbol    = Self->Parameters->BitsPerSymbol;
  Self->SymbolsPerBlock  = Self->Parameters->SymbolsPerBlock;
  Self->SpectraPerSymbol = Self->Parameters->SpectraPerSymbol;
  Self->InputBufferLen   =
    Self->SymbolsPerBlock * Self->SpectraPerSymbol * Self->BitsPerSymbol;

  Mem_Realloc( (void **) &(Self->InputBuffer), sizeof(double)  * Self->InputBufferLen );
  Mem_Realloc( (void **) &(Self->FHT_Buffer),  sizeof(double)  * Self->SymbolsPerBlock );
  Mem_Realloc( (void **) &(Self->OutputBlock), sizeof(uint8_t) * Self->BitsPerSymbol );
  Self->Reset( Self );
}

//----------------------------------------------------------------------

  static void
MFSK_SoftDecoder_SpectralInput(
    MFSK_SoftDecoder_t *Self,
    const double *SpectraEnergy )
{
  MFSK_SoftDemodulate(
      Self->InputBuffer + Self->InputPtr,
      SpectraEnergy,
      Self->BitsPerSymbol,
      Self->Parameters->CarrierSepar,
      (int)Self->Parameters->UseGrayCode,
      (int)Self->Parameters->RxSyncSquareEnergy );

  Self->InputPtr += Self->BitsPerSymbol;
  if( Self->InputPtr >= Self->InputBufferLen )
    Self->InputPtr -= Self->InputBufferLen;
}

//----------------------------------------------------------------------

  static void
MFSK_SoftDecoder_Input( MFSK_SoftDecoder_t *Self, const double *Symbol )
{
  uint32_t FreqBit;
  for( FreqBit = 0; FreqBit < Self->BitsPerSymbol; FreqBit++ )
  {
    Self->InputBuffer[Self->InputPtr] = Symbol[FreqBit];
    Self->InputPtr++;
  }
  if( Self->InputPtr >= Self->InputBufferLen )
    Self->InputPtr -= Self->InputBufferLen;
}

//----------------------------------------------------------------------

  static void
MFSK_SoftDecoder_DecodeCharacter( MFSK_SoftDecoder_t *Self, uint32_t FreqBit )
{
  uint32_t TimeBit;
  uint32_t Ptr      = Self->InputPtr;
  uint32_t Rotate   = FreqBit;
  uint32_t CodeWrap = ( Self->SymbolsPerBlock - 1 );
  uint32_t nShift, CodeBit;

  if( olivia_rc_data.mode_olivia )
    nShift = 13;  // For Olivia
  else
    nShift = 5;   // For Contestia
  CodeBit  = FreqBit * nShift;
  CodeBit &= CodeWrap;

  for( TimeBit = 0; TimeBit < Self->SymbolsPerBlock; TimeBit++ )
  {
    double Bit = Self->InputBuffer[Ptr + Rotate];
    uint64_t CodeMask = 1;
    CodeMask <<= CodeBit;

    if( Self->Parameters->ScramblingCode & CodeMask )
      Bit = -Bit;
    Self->FHT_Buffer[TimeBit] = Bit;
    CodeBit++;
    CodeBit &= CodeWrap;
    Rotate++;

    if( Rotate >= Self->BitsPerSymbol )
      Rotate -= Self->BitsPerSymbol;
    Ptr += Self->BitsPerSymbol * Self->SpectraPerSymbol;

    if( Ptr >= Self->InputBufferLen )
      Ptr -= Self->InputBufferLen;
  }

  FHT_Float( Self->FHT_Buffer, Self->SymbolsPerBlock );
  double   Peak    = 0.0;
  uint32_t PeakPos = 0;
  double   SqrSum  = 0.0;

  for( TimeBit = 0; TimeBit < Self->SymbolsPerBlock; TimeBit++ )
  {
    double Signal = Self->FHT_Buffer[TimeBit];
    SqrSum += Signal * Signal;
    if( fabs(Signal) > fabs(Peak) )
    {
      Peak    = Signal;
      PeakPos = TimeBit;
    }
  }

  uint8_t Char = (uint8_t)PeakPos;
  if( Peak < 0 )
    Char += Self->SymbolsPerBlock;
  SqrSum -= Peak * Peak;

  // For Contestia mode
  if( !olivia_rc_data.mode_olivia && (Char > 0) )
  {
    if( Char == ';' ) Char = ' ';       // ; to Space
    else if( Char == '<' ) Char = LF;   // < to Line Feed
    else if( Char == '=' ) Char = BS;   // = to Backspace
    else Char += 32;                    // Promote to ASCII
  }

  Self->OutputBlock[FreqBit] = Char;
  Self->NoiseEnergy += (double)SqrSum / (double)( Self->SymbolsPerBlock - 1 );
  Self->Signal += fabs( Peak );
}

//----------------------------------------------------------------------

  static void
MFSK_SoftDecoder_Process( MFSK_SoftDecoder_t *Self )
{
  uint32_t FreqBit;
  Self->Signal = 0.0;
  Self->NoiseEnergy = 0.0;

  for( FreqBit = 0; FreqBit < Self->BitsPerSymbol; FreqBit++ )
    Self->DecodeCharacter( Self, FreqBit );

  Self->Signal      /= (double)Self->BitsPerSymbol;
  Self->NoiseEnergy /= (double)Self->BitsPerSymbol;
}

//----------------------------------------------------------------------

  static uint32_t
MFSK_SoftDecoder_Output( MFSK_SoftDecoder_t *Self, uint8_t *Buffer )
{
  uint32_t FreqBit;
  for( FreqBit = 0; FreqBit < Self->BitsPerSymbol; FreqBit++ )
    Buffer[FreqBit] = Self->OutputBlock[FreqBit];
  return( Self->BitsPerSymbol );
}

//----------------------------------------------------------------------

// Initialize MFSK_SoftDecoder - my function
  void
MFSK_SoftDecoder_Initialize( MFSK_SoftDecoder_t *Self )
{
  Self->Init             = MFSK_SoftDecoder_Init;
  Self->Free             = MFSK_SoftDecoder_Free;
  Self->Reset            = MFSK_SoftDecoder_Reset;
  Self->Preset           = MFSK_SoftDecoder_Preset;
  Self->SpectralInput    = MFSK_SoftDecoder_SpectralInput;
  Self->Input            = MFSK_SoftDecoder_Input;
  Self->DecodeCharacter  = MFSK_SoftDecoder_DecodeCharacter;
  Self->Process          = MFSK_SoftDecoder_Process;
  Self->Output           = MFSK_SoftDecoder_Output;

  MFSK_SoftDecoder_Init( Self );
}

// ***------------------------------------------------------------------***

// Soft but iterative (!) FEC decoder
struct _MFSK_SOFTITERDECODER
{
  MFSK_Parameters_t *Parameters;

  double *Input;  // Demodulated spectra energies / tone probabilities

  uint32_t BitsPerSymbol;
  uint32_t BitsPerCharacter;
  uint32_t Symbols;
  uint32_t SymbolsPerBlock;

  double *InputExtrinsic;  // Extrinsic input information fed back from the decoder
  double *FHT_Codeword;    // FHT codewords to be decoded by FHT

  double Input_SignalEnergy;
  double Input_NoiseEnergy;
  double FEC_SignalEnergy;
  double FEC_NoiseEnergy;

  uint8_t *OutputBlock;

  // Pointers to functions that were in class MFSK_SoftIterDecoder
  void    (*Init)(void);
  void    (*Free)(void);
  void    (*Preset)(struct _MFSK_PARAMETERS *);
  void    (*SimulateInput)(const uint8_t *, double, double);
  BOOLEAN (*NormalizeSum)(double *, uint32_t, double);
  BOOLEAN (*NormalizeAbsSum)(double *Data, uint32_t Len, double Norm);
  void    (*ThirdPower)(double *, uint32_t);
  void    (*ScrambleCodeword)(double *, uint32_t);
  uint8_t (*DecodeChar)(const double *);
  //  void    (*Copy)(DstType *, SrcType *, uint32_t);
  //  void    (*Multiply)(DstType *, SrcType *, uint32_t);
  void    (*Process)(uint32_t);
  double  (*InputSNRdB)(void);
  int     (*WriteOutputBlock)(FIFO_t *);

} MFSK_SoftIterDecoder;

//----------------------------------------------------------------------

  static void
MFSK_SoftIterDecoder_Init( void )
{
  MFSK_SoftIterDecoder.Input          = NULL;
  MFSK_SoftIterDecoder.InputExtrinsic = NULL;
  MFSK_SoftIterDecoder.FHT_Codeword   = NULL;
  MFSK_SoftIterDecoder.OutputBlock    = NULL;
}

//----------------------------------------------------------------------

  static void
MFSK_SoftIterDecoder_Free( void )
{
  Mem_Free( (void **) &(MFSK_SoftIterDecoder.Input) );
  Mem_Free( (void **) &(MFSK_SoftIterDecoder.InputExtrinsic) );
  Mem_Free( (void **) &(MFSK_SoftIterDecoder.FHT_Codeword) );
  Mem_Free( (void **) &(MFSK_SoftIterDecoder.OutputBlock) );
}

//----------------------------------------------------------------------

  static void
MFSK_SoftIterDecoder_Preset(struct _MFSK_PARAMETERS *NewParameters )
{
  MFSK_SoftIterDecoder.Parameters = NewParameters;
  MFSK_SoftIterDecoder.BitsPerSymbol =
    MFSK_SoftIterDecoder.Parameters->BitsPerSymbol;
  MFSK_SoftIterDecoder.BitsPerCharacter =
    MFSK_SoftIterDecoder.Parameters->BitsPerCharacter;
  MFSK_SoftIterDecoder.Symbols =
    MFSK_SoftIterDecoder.Parameters->Carriers;
  MFSK_SoftIterDecoder.SymbolsPerBlock =
    MFSK_SoftIterDecoder.Parameters->SymbolsPerBlock;

  size_t siz = sizeof(double) *
    MFSK_SoftIterDecoder.SymbolsPerBlock * MFSK_SoftIterDecoder.Symbols;
  Mem_Realloc( (void **) &(MFSK_SoftIterDecoder.Input), siz );
  Mem_Realloc( (void **) &(MFSK_SoftIterDecoder.InputExtrinsic), siz );

  siz = sizeof(double) *
    MFSK_SoftIterDecoder.SymbolsPerBlock * MFSK_SoftIterDecoder.BitsPerSymbol;
  Mem_Realloc( (void **) &(MFSK_SoftIterDecoder.FHT_Codeword), siz );

  siz = sizeof(uint8_t) * MFSK_SoftIterDecoder.BitsPerSymbol;
  Mem_Realloc( (void **) &(MFSK_SoftIterDecoder.OutputBlock), siz );
}

//----------------------------------------------------------------------

  static double
UniformNoise( void )
{
  return( ((double)(rand()) + 1.0) / (RAND_MAX + 1.0) );
}

  void
WhiteNoise( complex double *Noise, double Amplitude  )
{
  double Phase;
  Amplitude *= sqrt( -2.0 * log(UniformNoise()) );
  Phase = M_2PI * UniformNoise();
  *Noise = cos( Phase ) * Amplitude + (complex double)I * sin( Phase ) * Amplitude;
}

//----------------------------------------------------------------------

  static void
MFSK_SoftIterDecoder_SimulateInput(
    const uint8_t *InputBlock,
    double SNR,
    double DeadCarrierSNR )
{
  double NoiseRMS = 1.0;
  double Signal = SNR * NoiseRMS * sqrt( 2.0 * (double)MFSK_SoftIterDecoder.Symbols );
  double DeadCarrier =
    DeadCarrierSNR * NoiseRMS * sqrt( 2.0 * (double)MFSK_SoftIterDecoder.Symbols );
  uint32_t DeadCarrierFreq = (uint32_t)(rand()) & ( MFSK_SoftIterDecoder.Symbols - 1 );

  uint32_t Symbol;
  uint32_t Idx;
  for( Symbol = 0, Idx = 0;
      Symbol < MFSK_SoftIterDecoder.SymbolsPerBlock;
      Symbol++, Idx += MFSK_SoftIterDecoder.Symbols )
  {
    uint32_t Freq;
    uint32_t SymbolFreq = InputBlock[Symbol];
    if( MFSK_SoftIterDecoder.Parameters->UseGrayCode )
      GrayCode( SymbolFreq );
    for( Freq = 0; Freq < MFSK_SoftIterDecoder.Symbols; Freq++ )
    {
      complex double Noise = 0.0 + (complex double)I * 0.0;
      WhiteNoise( &Noise, NoiseRMS );
      if( Freq == SymbolFreq ) Noise += Signal + (complex double)I * 0.0;
      if( Freq == DeadCarrierFreq ) Noise += 0.0 + (complex double)I * DeadCarrier;
      double Energy = Complex_Mag2( &Noise );
      MFSK_SoftIterDecoder.Input[Idx + Freq] = Energy * Energy;
    }
  }
}

//----------------------------------------------------------------------

  static BOOLEAN
MFSK_SoftIterDecoder_NormalizeSum(
    double *Data,
    uint32_t Len,
    double Norm )
{
  uint32_t Idx;
  double Sum = 0.0;
  for( Idx = 0; Idx < Len; Idx++ )
    Sum += Data[Idx];
  if( Sum <= 0.0 ) return( False );
  double Corr = Norm / Sum;
  for( Idx = 0; Idx < Len; Idx++ )
    Data[Idx] *= Corr;
  return( True );
}

//----------------------------------------------------------------------

  static BOOLEAN
MFSK_SoftIterDecoder_NormalizeAbsSum(
    double *Data,
    uint32_t Len,
    double Norm )
{
  uint32_t Idx;
  double Sum = 0.0;
  for( Idx = 0; Idx < Len; Idx++ )
    Sum += fabs( Data[Idx] );
  if( Sum <= 0.0 ) return( False );
  double Corr = Norm / Sum;
  for( Idx = 0; Idx < Len; Idx++ )
    Data[Idx] *= Corr;
  return( True );
}

//----------------------------------------------------------------------

  static void
MFSK_SoftIterDecoder_ThirdPower( double *Data, uint32_t Len )
{
  uint32_t Idx;
  for( Idx = 0; Idx < Len; Idx++ )
  {
    double Square = Data[Idx];
    // Square = Square * Square;
    Square = fabs( Square );  // <=  works better ( in simulations )
    Data[Idx] *= Square;
  }
}

//----------------------------------------------------------------------

  static void
MFSK_SoftIterDecoder_ScrambleCodeword( double *CodeWord, uint32_t ScrambleIdx )
{
  uint32_t Idx;
  uint32_t CodeWrap = ( MFSK_SoftIterDecoder.SymbolsPerBlock - 1 );
  ScrambleIdx &= CodeWrap;
  for( Idx = 0; Idx < MFSK_SoftIterDecoder.SymbolsPerBlock; Idx++ )
  {
    uint64_t CodeMask = 1;
    CodeMask <<= ScrambleIdx;
    if( MFSK_SoftIterDecoder.Parameters->ScramblingCode & CodeMask )
      CodeWord[Idx] = -CodeWord[Idx];
    ScrambleIdx++;
    ScrambleIdx &= CodeWrap;
  }
}

//----------------------------------------------------------------------

  static uint8_t
MFSK_SoftIterDecoder_DecodeChar( const double *FHT_Buffer )
{
  uint32_t TimeBit;
  double Peak = 0;
  uint32_t PeakPos   = 0;
  double NoiseEnergy = 0;

  for( TimeBit = 0; TimeBit < MFSK_SoftIterDecoder.SymbolsPerBlock; TimeBit++ )
  {
    double Signal = FHT_Buffer[TimeBit];
    NoiseEnergy += Signal * Signal;
    if( fabs(Signal) > fabs(Peak) )
    {
      Peak    = Signal;
      PeakPos = TimeBit;
    }
  }

  uint8_t Char = (uint8_t)PeakPos;
  if( Peak < 0 ) Char += MFSK_SoftIterDecoder.SymbolsPerBlock;

  double SignalEnergy = Peak * Peak;
  NoiseEnergy  -= SignalEnergy;
  SignalEnergy -= NoiseEnergy / (double)( MFSK_SoftIterDecoder.SymbolsPerBlock - 1 );
  NoiseEnergy  *= (double)MFSK_SoftIterDecoder.SymbolsPerBlock /
    (double)( MFSK_SoftIterDecoder.SymbolsPerBlock - 1 );

  MFSK_SoftIterDecoder.FEC_SignalEnergy += SignalEnergy;
  MFSK_SoftIterDecoder.FEC_NoiseEnergy  += NoiseEnergy;

  // For Contestia mode
  if( !olivia_rc_data.mode_olivia && (Char > 0) )
  {
    if( Char == ';' ) Char = ' ';       // ; to Space
    else if( Char == '<' ) Char = LF;   // < to Line Feed
    else if( Char == '=' ) Char = BS;   // = to Backspace
    else Char += 32;                    // Promote to ASCII
  }

  return( Char );
}

//----------------------------------------------------------------------

  static void
MFSK_SoftIterDecoder_Process( uint32_t MaxIter )
{
  uint32_t TimeBit;
  uint32_t Bit;
  uint32_t Freq;
  uint32_t InpIdx;
  uint32_t BlockIdx;
  double *SymbolBit = NULL;

  uint32_t nShift;
  if( olivia_rc_data.mode_olivia )
    nShift = 13;  // For Olivia
  else
    nShift = 5;   // For Contestia


  uint32_t InputSize =
    MFSK_SoftIterDecoder.Symbols * MFSK_SoftIterDecoder.SymbolsPerBlock;
  uint32_t BlockSize =
    MFSK_SoftIterDecoder.BitsPerSymbol * MFSK_SoftIterDecoder.SymbolsPerBlock;

  for( InpIdx = 0; InpIdx < InputSize; InpIdx++ )
    MFSK_SoftIterDecoder.InputExtrinsic[InpIdx] =
      1.0 / (double)MFSK_SoftIterDecoder.Symbols;

  for( ; MaxIter; MaxIter-- )
  {
    int SquareEnergy = (int)MFSK_SoftIterDecoder.Parameters->DecodeSquareEnergy;
    for( InpIdx = 0; InpIdx < InputSize; InpIdx++ )
    {
      double InputEnergy = MFSK_SoftIterDecoder.Input[InpIdx];
      if( SquareEnergy ) InputEnergy *= InputEnergy;
      MFSK_SoftIterDecoder.InputExtrinsic[InpIdx] *= InputEnergy;
    }

    Mem_Realloc( (void **) &SymbolBit,
        (size_t)MFSK_SoftIterDecoder.BitsPerSymbol * sizeof(double) );

    uint32_t Rotate = 0;
    for( TimeBit = 0, InpIdx = 0;
        TimeBit < MFSK_SoftIterDecoder.SymbolsPerBlock;
        TimeBit++, InpIdx += MFSK_SoftIterDecoder.Symbols )
    {
      MFSK_SoftDemodulate(
          SymbolBit, MFSK_SoftIterDecoder.InputExtrinsic + InpIdx,
          MFSK_SoftIterDecoder.BitsPerSymbol, 1,
          (int)MFSK_SoftIterDecoder.Parameters->UseGrayCode, 0 );

      BlockIdx = TimeBit + Rotate * MFSK_SoftIterDecoder.SymbolsPerBlock;
      for( Bit = 0; Bit < MFSK_SoftIterDecoder.BitsPerSymbol; Bit++ )
      {
        MFSK_SoftIterDecoder.FHT_Codeword[BlockIdx] = SymbolBit[Bit];
        BlockIdx += MFSK_SoftIterDecoder.SymbolsPerBlock;
        if( BlockIdx >= BlockSize ) BlockIdx -= BlockSize;
      }

      if( Rotate > 0 ) Rotate--;
      else Rotate += MFSK_SoftIterDecoder.BitsPerSymbol - 1;
    }

    MFSK_SoftIterDecoder.FEC_SignalEnergy = 0;
    MFSK_SoftIterDecoder.FEC_NoiseEnergy  = 0;
    for( Bit = 0, BlockIdx = 0;
        Bit < MFSK_SoftIterDecoder.BitsPerSymbol;
        Bit++, BlockIdx += MFSK_SoftIterDecoder.SymbolsPerBlock )
    {
      MFSK_SoftIterDecoder.ScrambleCodeword(
          MFSK_SoftIterDecoder.FHT_Codeword + BlockIdx, nShift * Bit );
      FHT_Float( MFSK_SoftIterDecoder.FHT_Codeword + BlockIdx,
          MFSK_SoftIterDecoder.SymbolsPerBlock );

      uint8_t Char = MFSK_SoftIterDecoder.DecodeChar(
          MFSK_SoftIterDecoder.FHT_Codeword + BlockIdx );
      MFSK_SoftIterDecoder.OutputBlock[Bit] = Char;

      MFSK_SoftIterDecoder.ThirdPower(
          MFSK_SoftIterDecoder.FHT_Codeword + BlockIdx,
          MFSK_SoftIterDecoder.SymbolsPerBlock );
      MFSK_SoftIterDecoder.NormalizeAbsSum(
          MFSK_SoftIterDecoder.FHT_Codeword + BlockIdx,
          MFSK_SoftIterDecoder.SymbolsPerBlock, 1.0 );

      IFHT_Float( MFSK_SoftIterDecoder.FHT_Codeword + BlockIdx,
          MFSK_SoftIterDecoder.SymbolsPerBlock );
      MFSK_SoftIterDecoder.ScrambleCodeword(
          MFSK_SoftIterDecoder.FHT_Codeword + BlockIdx, nShift * Bit );
    }

    Rotate = 0;
    for( TimeBit = 0, InpIdx = 0;
        TimeBit < MFSK_SoftIterDecoder.SymbolsPerBlock;
        TimeBit++, InpIdx += MFSK_SoftIterDecoder.Symbols )
    {
      BlockIdx = TimeBit + Rotate * MFSK_SoftIterDecoder.SymbolsPerBlock;
      for( Bit = 0; Bit < MFSK_SoftIterDecoder.BitsPerSymbol; Bit++ )
      {
        SymbolBit[Bit] = MFSK_SoftIterDecoder.FHT_Codeword[BlockIdx];
        BlockIdx += MFSK_SoftIterDecoder.SymbolsPerBlock;
        if( BlockIdx >= BlockSize ) BlockIdx -= BlockSize;
      }

      // CalculateFreqProb(InputExtrinsic+InpIdx,SymbolBit );
      MFSK_SoftModulate(
          MFSK_SoftIterDecoder.InputExtrinsic + InpIdx,
          SymbolBit, MFSK_SoftIterDecoder.BitsPerSymbol,
          (int)MFSK_SoftIterDecoder.Parameters->UseGrayCode );

      if( Rotate > 0 ) Rotate -= 1;
      else Rotate += MFSK_SoftIterDecoder.BitsPerSymbol - 1;
    }

    MFSK_SoftIterDecoder.Input_SignalEnergy = 0;
    MFSK_SoftIterDecoder.Input_NoiseEnergy  = 0;
    for( TimeBit = 0, InpIdx = 0;
        TimeBit < MFSK_SoftIterDecoder.SymbolsPerBlock;
        TimeBit++ )
    {
      for( Freq = 0; Freq < MFSK_SoftIterDecoder.Symbols; Freq++, InpIdx++ )
      {
        double Energy  = MFSK_SoftIterDecoder.Input[InpIdx];
        double SigProb = MFSK_SoftIterDecoder.InputExtrinsic[InpIdx];
        MFSK_SoftIterDecoder.Input_SignalEnergy += SigProb * Energy;
        MFSK_SoftIterDecoder.Input_NoiseEnergy  += ( 1 - SigProb ) * Energy;
      }
    }

    MFSK_SoftIterDecoder.Input_SignalEnergy -=
      MFSK_SoftIterDecoder.Input_NoiseEnergy /
      (double)( MFSK_SoftIterDecoder.Symbols - 1 );
    MFSK_SoftIterDecoder.Input_NoiseEnergy *=
      (double)MFSK_SoftIterDecoder.Symbols /
      (double)( MFSK_SoftIterDecoder.Symbols - 1 );
  }

  Mem_Free( (void **) &SymbolBit );
}

//----------------------------------------------------------------------

  static double
MFSK_SoftIterDecoder_InputSNRdB( void )
{
  return( 10.0 * log10(
        MFSK_SoftIterDecoder.Input_SignalEnergy /
        MFSK_SoftIterDecoder.Input_NoiseEnergy) );
}

//----------------------------------------------------------------------

  static int
MFSK_SoftIterDecoder_WriteOutputBlock( FIFO_t *Output )
{
  uint32_t Bit;
  int Written;
  for( Written = 0, Bit = 0; Bit < MFSK_SoftIterDecoder.BitsPerSymbol; Bit++ )
  {
    uint8_t Char = MFSK_SoftIterDecoder.OutputBlock[Bit];
    int Ret = Output->Write( Output, Char );
    if( Ret == 0 ) break;
    Written += Ret;
  }
  return( Written );
}

//----------------------------------------------------------------------

// Initialize MFSK_SoftIterDecoder - my function
  void
MFSK_SoftIterDecoder_Initialize( void )
{
  MFSK_SoftIterDecoder.Init             = MFSK_SoftIterDecoder_Init;
  MFSK_SoftIterDecoder.Free             = MFSK_SoftIterDecoder_Free;
  MFSK_SoftIterDecoder.Preset           = MFSK_SoftIterDecoder_Preset;
  MFSK_SoftIterDecoder.SimulateInput    = MFSK_SoftIterDecoder_SimulateInput;
  MFSK_SoftIterDecoder.NormalizeSum     = MFSK_SoftIterDecoder_NormalizeSum;
  MFSK_SoftIterDecoder.NormalizeAbsSum  = MFSK_SoftIterDecoder_NormalizeAbsSum;
  MFSK_SoftIterDecoder.ThirdPower       = MFSK_SoftIterDecoder_ThirdPower;
  MFSK_SoftIterDecoder.ScrambleCodeword = MFSK_SoftIterDecoder_ScrambleCodeword;
  MFSK_SoftIterDecoder.DecodeChar       = MFSK_SoftIterDecoder_DecodeChar;
  //  MFSK_SoftIterDecoder.Copy             = MFSK_SoftIterDecoder_Copy;
  //  MFSK_SoftIterDecoder.Multiply         = MFSK_SoftIterDecoder_Multiply;
  MFSK_SoftIterDecoder.Process          = MFSK_SoftIterDecoder_Process;
  MFSK_SoftIterDecoder.InputSNRdB       = MFSK_SoftIterDecoder_InputSNRdB;
  MFSK_SoftIterDecoder.WriteOutputBlock = MFSK_SoftIterDecoder_WriteOutputBlock;

  MFSK_SoftIterDecoder_Init();
}

/*----------------------------------------------------------------------

  static void
  MFSK_SoftIterDecoder_Copy( DstType *Dst, SrcType *Src, uint32_t Len )
  {
  uint32_t Idx;
  for( Idx = 0; Idx < Len; Idx++ )
  Dst[Idx] = Src[Idx];
  }
 */
/*----------------------------------------------------------------------

  static void
  MFSK_SoftIterDecoder_Multiply( DstType *Dst, SrcType *Src, uint32_t Len )
  {
  uint32_t Idx;
  for( Idx = 0; Idx < Len; Idx++ )
  Dst[Idx] *= Src[Idx];
  }
 */

// ***------------------------------------------------------------------***

  static void
MFSK_Transmitter_Init( MFSK_Transmitter_t *Self )
{
  Self->ModulatorOutput = NULL;
  Self->ConverterOutput = NULL;
  Self->State_Running   = 0x0001;
  Self->State_StopReq   = 0x0010;
}

//----------------------------------------------------------------------

  static void
MFSK_Transmitter_Free( MFSK_Transmitter_t *Self )
{
  if( Self->Input.Free )
    Self->Input.Free( &(Self->Input) );

  if( Self->Monitor.Free )
    Self->Monitor.Free( &(Self->Monitor) );

  if( MFSK_Encoder.Free )
    MFSK_Encoder.Free();

  if( MFSK_Modulator.Free )
    MFSK_Modulator.Free();
  Mem_Free( (void **) &(Self->ModulatorOutput) );

  if( Self->RateConverter.Free )
    Self->RateConverter.Free( &(Self->RateConverter) );
  Mem_Free( (void **) &(Self->ConverterOutput) );
}

//----------------------------------------------------------------------

// Preset internal arrays according to primary paramaters
  static void
MFSK_Transmitter_Preset(
    MFSK_Transmitter_t *Self,
    MFSK_Parameters_t *NewParameters )
{
  Self->Parameters = NewParameters;

  Self->BitsPerSymbol   = Self->Parameters->BitsPerSymbol;
  Self->SymbolsPerBlock = Self->Parameters->SymbolsPerBlock;

  // Preset the input character buffer
  // Self->Input.Len = 1024;
  FIFO_Initialize( &(Self->Input) );
  Self->Input.Preset( &(Self->Input), 1024 );

  // Self->Monitor.Len = 256;
  FIFO_Initialize( &(Self->Monitor) );
  Self->Monitor.Preset( &(Self->Monitor), 256 );

  // Preset the encoder
  MFSK_Encoder_Initialize();
  MFSK_Encoder.BitsPerSymbol = Self->BitsPerSymbol;
  MFSK_Encoder.Preset();

  // Preset the modulator
  MFSK_Modulator_Initialize();
  MFSK_Modulator.Preset( Self->Parameters );
  Mem_Realloc( (void **) &(Self->ModulatorOutput),
      sizeof(double) * MFSK_Modulator.OutputLen );

  // Preset the rate converter
  CRateConverter_Initialize( &(Self->RateConverter) );
  Self->RateConverter.OutputRate =
    Self->Parameters->OutputSampleRate / (double)Self->Parameters->SampleRate;
  Self->RateConverter.Preset(&(Self->RateConverter) );

  Self->MaxOutputLen = ( uint32_t )
    ( ceil((double)Self->Parameters->SymbolSepar *
           (double)Self->Parameters->OutputSampleRate /
           (double)Self->Parameters->SampleRate + 2.0) );
  Mem_Realloc( (void **) &(Self->ConverterOutput), sizeof(double) * Self->MaxOutputLen );
  Self->Reset( Self );
}

//----------------------------------------------------------------------

  static void
MFSK_Transmitter_Reset( MFSK_Transmitter_t *Self )
{
  Self->Input.Reset( &(Self->Input) );
  Self->Monitor.Reset( &(Self->Monitor) );
  Self->SymbolPtr = 0;
  Self->State     = 0;
  Self->RateConverter.Reset( &(Self->RateConverter) );
}

//----------------------------------------------------------------------

// Start the transmission
  static void
MFSK_Transmitter_Start( MFSK_Transmitter_t *Self )
{
  Self->State |= Self->State_Running;
}

//----------------------------------------------------------------------

// Request to stop ( and complete ) the transmission but the
// transmitter will only stop after transmitting all the data
  static void
MFSK_Transmitter_Stop( MFSK_Transmitter_t *Self )
{
  Self->State |= Self->State_StopReq;
}

//----------------------------------------------------------------------

// Check if the transmission is still running ( = not yet complete )
  static int
MFSK_Transmitter_Running( const MFSK_Transmitter_t *Self )
{
  return( Self->State & Self->State_Running );
}

//----------------------------------------------------------------------

// Put the character into the transmitter input queue
  static BOOLEAN
MFSK_Transmitter_PutChar( MFSK_Transmitter_t *Self, uint8_t Char )
{
  return( Self->Input.Write( &(Self->Input), Char) );
}

//----------------------------------------------------------------------

// Get one character from the monitor buffer
  static BOOLEAN
MFSK_Transmitter_GetChar( MFSK_Transmitter_t *Self, uint8_t *Char )
{
  return( Self->Monitor.Read( &(Self->Monitor), Char) );
}

//----------------------------------------------------------------------

// Get out the transmitter output ( audio )
  static int
MFSK_Transmitter_Output_Float( MFSK_Transmitter_t *Self, double **OutputPtr )
{
  // When at the block boundary
  if( Self->SymbolPtr == 0 )
  {
    // If the user requested to stop and no more characters
    if( (Self->State & Self->State_StopReq) &&
        Self->Input.Empty(&(Self->Input)) )
    {
      Self->State = 0;
    }  // Then simply stop. Otherwise when state is "running" then keep going
    else if( Self->State & Self->State_Running )
    {
      uint32_t Idx;   // Form a new block

      // Loop over characters in a block
      for( Idx = 0; Idx < Self->BitsPerSymbol; Idx++ )
      {
        // Get character from the input FIFO
        uint8_t Char;
        if( !Self->Input.Read(&(Self->Input), &Char) )
          break;

        // Put it into the block
        Self->InputBlock[Idx] = Char;

        // Put this character into the monitor FIFO
        Self->Monitor.Write( &(Self->Monitor), Char );
      }

      // Fill the unused part of the block
      for( ; Idx < Self->BitsPerSymbol; Idx++ )
        Self->InputBlock[Idx] = 0;

      // Encode the new block
      MFSK_Encoder.EncodeBlock( Self->InputBlock );
    }
  } // if( Self->SymbolPtr == 0 )

  // If state is "running" then
  if( Self->State & Self->State_Running )
  {
    // Send out the next symbol of encoded block through the modulator
    MFSK_Modulator.Send( MFSK_Encoder.OutputBlock[Self->SymbolPtr] );
    Self->SymbolPtr++;
    if( Self->SymbolPtr >= Self->SymbolsPerBlock )
      Self->SymbolPtr = 0;
  }

  // Get the modulator output
  uint32_t ModLen = MFSK_Modulator.Output_Float( Self->ModulatorOutput );

  // Correct the sampling rate
  int ConvLen = Self->RateConverter.Process(
      &(Self->RateConverter),
      Self->ModulatorOutput,
      (uint32_t)ModLen,
      Self->ConverterOutput );

  if( ConvLen <= 0 ) return( ConvLen );
  *OutputPtr = Self->ConverterOutput;

  return( ConvLen );
}

//----------------------------------------------------------------------

  static int
MFSK_Transmitter_Output_Int( MFSK_Transmitter_t *Self, int16_t *Buffer )
{
  double *OutputPtr = NULL;
  int OutputLen = Self->Output_Float( Self, &(OutputPtr) );
  ConvertToS16( Self->ConverterOutput, Buffer, (uint32_t)OutputLen );
  return( OutputLen );
}

//----------------------------------------------------------------------

// Initialize MFSK TRansmitter - my function
  void
MFSK_Transmitter_Initialize( MFSK_Transmitter_t *Self )
{
  Self->Init         = MFSK_Transmitter_Init;
  Self->Free         = MFSK_Transmitter_Free;
  Self->Preset       = MFSK_Transmitter_Preset;
  Self->Reset        = MFSK_Transmitter_Reset;
  Self->Start        = MFSK_Transmitter_Start;
  Self->Stop         = MFSK_Transmitter_Stop;
  Self->Running      = MFSK_Transmitter_Running;
  Self->PutChar      = MFSK_Transmitter_PutChar;
  Self->GetChar      = MFSK_Transmitter_GetChar;
  Self->Output_Float = MFSK_Transmitter_Output_Float;
  Self->Output_Int   = MFSK_Transmitter_Output_Int;

  MFSK_Transmitter_Init( Self );
}

// ***------------------------------------------------------------------***

struct _MFSK_SYNCHRONIZER
{
  struct _MFSK_PARAMETERS *Parameters;

  uint32_t FreqOffsets;         // Number of possible frequency offsets
  uint32_t BlockPhases;         // Number of possible time-phases within the FEC block
  MFSK_SoftDecoder_t *Decoder;  // Array of decoders

  uint32_t BlockPhase;  // Current running block time-phase

  CircularBuffer_Filter_t  SyncNoiseEnergy; // FEC noise integrators
  CircularBuffer_Filter_t  SyncSignal;      // FEC signal integrators
  double SyncFilterWeight;                  // Weight for the integrators

  double   SyncBestSignal;     // Best signal
  uint32_t SyncBestBlockPhase; // Time-phase of the best signal        [FFT bins]
  uint32_t SyncBestFreqOffset; // Frequency offset of the best signal  [FFT spectral slices]
  double   SyncSNR;            // S/N corresponding to the SyncBestSignal
  int      DecodeReference;    // When 0 then right in the middle of a FEC block

  double   PreciseFreqOffset;  // Precise interpolated frequency offset [FFT bins]
  double   PreciseBlockPhase;  // And the FEC block phase [spectral slices]
  uint32_t StableLock;         // Is 1 when the sync. looks stable
  LowPass3_Filter_t FreqDrift; // Frequency drift rate [FFT bin / FEC block]
  LowPass3_Filter_t TimeDrift; // Block phase ( time ) drift rate [ppm]

  // Pointers to functions that were in class MFSK_Synchronizer
  void    (*Init)(void);
  void    (*Free)(void);
  void    (*Preset)(struct _MFSK_PARAMETERS *);
  void    (*Reset)(void);
  void    (*Process)(double *);
  double  (*FEC_SNR)(void);
  double  (*FrequencyOffset)(void);
  double  (*FrequencyDriftRate)(void);
  double  (*TimeDriftRate)(void);

} MFSK_Synchronizer;

//----------------------------------------------------------------------

  static void
MFSK_Synchronizer_Init( void )
{
  MFSK_Synchronizer.Decoder = NULL;
}

//----------------------------------------------------------------------

  static void
MFSK_Synchronizer_Free( void )
{
  if( MFSK_Synchronizer.Decoder )
  {
    uint32_t Idx;
    for( Idx = 0; Idx < MFSK_Synchronizer.FreqOffsets; Idx++ )
    {
      if( MFSK_Synchronizer.Decoder[Idx].Free )
        MFSK_Synchronizer.Decoder[Idx].Free( &(MFSK_Synchronizer.Decoder[Idx]) );
    }
    Mem_Free( (void **) &(MFSK_Synchronizer.Decoder) );
  }

  if( MFSK_Synchronizer.SyncSignal.Free )
    MFSK_Synchronizer.SyncSignal.Free( &(MFSK_Synchronizer.SyncSignal) );

  if( MFSK_Synchronizer.SyncNoiseEnergy.Free )
    MFSK_Synchronizer.SyncNoiseEnergy.Free( &(MFSK_Synchronizer.SyncNoiseEnergy) );
}

//----------------------------------------------------------------------

// Resize internal arrays according to the parameters
  static void
MFSK_Synchronizer_Preset(struct _MFSK_PARAMETERS *NewParameters )
{
  MFSK_Synchronizer.Parameters = NewParameters;

  uint32_t Idx;

  if( MFSK_Synchronizer.Decoder )
  {
    for( Idx = 0; Idx < MFSK_Synchronizer.FreqOffsets; Idx++ )
    {
      if( MFSK_Synchronizer.Decoder[Idx].Free )
        MFSK_Synchronizer.Decoder[Idx].Free( &(MFSK_Synchronizer.Decoder[Idx]) );
    }
  }

  // Initialize and preset Synchroniser
  MFSK_Synchronizer.FreqOffsets =
    2 * MFSK_Synchronizer.Parameters->RxSyncMargin *
    MFSK_Synchronizer.Parameters->CarrierSepar + 1;

  MFSK_Synchronizer.BlockPhases =
    MFSK_Synchronizer.Parameters->SpectraPerSymbol *
    MFSK_Synchronizer.Parameters->SymbolsPerBlock;

  Mem_Realloc( (void **) &(MFSK_Synchronizer.Decoder),
      sizeof(MFSK_SoftDecoder_t) * MFSK_Synchronizer.FreqOffsets );

  // Initialize and preset Synchronizer Decoders
  for( Idx = 0; Idx < MFSK_Synchronizer.FreqOffsets; Idx++ )
  {
    MFSK_SoftDecoder_Initialize( &(MFSK_Synchronizer.Decoder[Idx]) );
    MFSK_Synchronizer.Decoder[Idx].Preset(
        &(MFSK_Synchronizer.Decoder[Idx]), MFSK_Synchronizer.Parameters );
  }

  // Initialize and preset Synchronizer SyncSignal filter
  CircularBuffer_Filter_Initialize( &(MFSK_Synchronizer.SyncSignal) );
  MFSK_Synchronizer.SyncSignal.Width = MFSK_Synchronizer.FreqOffsets;
  MFSK_Synchronizer.SyncSignal.Len   = MFSK_Synchronizer.BlockPhases;
  MFSK_Synchronizer.SyncSignal.Preset( &(MFSK_Synchronizer.SyncSignal) );

  // Initialize and preset Synchronizer SyncNoiseEnergy filter
  CircularBuffer_Filter_Initialize( &(MFSK_Synchronizer.SyncNoiseEnergy) );
  MFSK_Synchronizer.SyncNoiseEnergy.Width = MFSK_Synchronizer.FreqOffsets;
  MFSK_Synchronizer.SyncNoiseEnergy.Len   = MFSK_Synchronizer.BlockPhases;
  MFSK_Synchronizer.SyncNoiseEnergy.Preset(&(MFSK_Synchronizer.SyncNoiseEnergy) );

  MFSK_Synchronizer.SyncFilterWeight =
    1.0 / (double)MFSK_Synchronizer.Parameters->RxSyncIntegLen;

  MFSK_Synchronizer.Reset();
}

//----------------------------------------------------------------------

  static void
MFSK_Synchronizer_Reset( void )
{
  uint32_t Idx;

  for( Idx = 0; Idx < MFSK_Synchronizer.FreqOffsets; Idx++ )
    MFSK_Synchronizer.Decoder[Idx].Reset( &(MFSK_Synchronizer.Decoder[Idx]) );

  MFSK_Synchronizer.SyncSignal.Clear( &(MFSK_Synchronizer.SyncSignal) );
  MFSK_Synchronizer.SyncNoiseEnergy.Clear( &(MFSK_Synchronizer.SyncNoiseEnergy) );

  MFSK_Synchronizer.BlockPhase         = 0;
  MFSK_Synchronizer.SyncBestSignal     = 0;
  MFSK_Synchronizer.SyncBestBlockPhase = 0;
  MFSK_Synchronizer.SyncBestFreqOffset = 0;
  MFSK_Synchronizer.SyncSNR            = 0;
  MFSK_Synchronizer. DecodeReference   = -MFSK_Synchronizer.BlockPhases / 2;

  MFSK_Synchronizer.PreciseFreqOffset = 0;
  MFSK_Synchronizer.PreciseBlockPhase = 0;
  MFSK_Synchronizer.StableLock        = 0;
  MFSK_Synchronizer.FreqDrift.Out1    = 0;
  MFSK_Synchronizer.FreqDrift.Out2    = 0;
  MFSK_Synchronizer.FreqDrift.Output  = 0;
  MFSK_Synchronizer.TimeDrift.Out1    = 0;
  MFSK_Synchronizer.TimeDrift.Out2    = 0;
  MFSK_Synchronizer.TimeDrift.Output  = 0;
}

//----------------------------------------------------------------------

  static void
MFSK_Synchronizer_Process( double *Spectra )
{
  uint32_t Offset;

  Offset = MFSK_Synchronizer.BlockPhase * MFSK_Synchronizer.SyncSignal.Width;
  LowPass3_Filter_t *SignalPtr = MFSK_Synchronizer.SyncSignal.Data + Offset;

  Offset = MFSK_Synchronizer.BlockPhase * MFSK_Synchronizer.SyncNoiseEnergy.Width;
  LowPass3_Filter_t *NoiseEnergyPtr = MFSK_Synchronizer.SyncNoiseEnergy.Data + Offset;

  MFSK_SoftDecoder_t *DecoderPtr = MFSK_Synchronizer.Decoder;

  double BestSliceSignal   = 0.0;
  uint32_t BestSliceOffset = 0;

  for( Offset = 0; Offset < MFSK_Synchronizer.FreqOffsets; Offset++ )
  {
    DecoderPtr->SpectralInput( DecoderPtr, Spectra + Offset );
    DecoderPtr->Process( DecoderPtr );
    double NoiseEnergy = DecoderPtr->NoiseEnergy;
    double Signal      = DecoderPtr->Signal;

    NoiseEnergyPtr->Process(
        NoiseEnergyPtr, NoiseEnergy, MFSK_Synchronizer.SyncFilterWeight, 0.1 );
    SignalPtr->Process(
        SignalPtr, Signal, MFSK_Synchronizer.SyncFilterWeight, 0.1 );

    Signal = SignalPtr->Output;
    if( Signal > BestSliceSignal )
    {
      BestSliceSignal = Signal;
      BestSliceOffset = Offset;
    }

    DecoderPtr++;
    NoiseEnergyPtr++;
    SignalPtr++;
  }

  if( MFSK_Synchronizer.BlockPhase == MFSK_Synchronizer.SyncBestBlockPhase )
  {
    MFSK_Synchronizer.SyncBestSignal     = BestSliceSignal;
    MFSK_Synchronizer.SyncBestFreqOffset = BestSliceOffset;
  }
  else if( BestSliceSignal > MFSK_Synchronizer.SyncBestSignal )
  {
    MFSK_Synchronizer.SyncBestSignal     = BestSliceSignal;
    MFSK_Synchronizer.SyncBestBlockPhase = MFSK_Synchronizer.BlockPhase;
    MFSK_Synchronizer.SyncBestFreqOffset = BestSliceOffset;
  }

  MFSK_Synchronizer.DecodeReference =
    (int)MFSK_Synchronizer.BlockPhase - (int)MFSK_Synchronizer.SyncBestBlockPhase;

  if( MFSK_Synchronizer.DecodeReference < 0 )
    MFSK_Synchronizer.DecodeReference += MFSK_Synchronizer.BlockPhases;
  MFSK_Synchronizer.DecodeReference   -= (int)( MFSK_Synchronizer.BlockPhases / 2 );
  if( MFSK_Synchronizer.DecodeReference == 0 )
  {
    /*
     * My interpratation of the original stupid [] "operator overload"
     * Type BestNoise = SyncNoiseEnergy[SyncBestBlockPhase][SyncBestFreqOffset].Output;
     */
    LowPass3_Filter_t *Filt =
      MFSK_Synchronizer.SyncNoiseEnergy.Data +
      MFSK_Synchronizer.SyncBestBlockPhase *
      MFSK_Synchronizer.SyncNoiseEnergy.Width;

    double BestNoise = ( Filt + MFSK_Synchronizer.SyncBestFreqOffset )->Output;

    if( BestNoise > 0.0 ) BestNoise = sqrt( BestNoise );
    else BestNoise = 0.0;

    const double MinNoise =
      (double)MFSK_Synchronizer.Parameters->SymbolsPerBlock / 10000.0;
    if( BestNoise < MinNoise ) BestNoise = MinNoise;

    if( BestNoise != 0.0 ) // My addition, reported by Coverity Scan
      MFSK_Synchronizer.SyncSNR = MFSK_Synchronizer.SyncBestSignal / BestNoise;

    double NewPreciseFreqOffset = 0.0;
    double SignalPeak;

    const LowPass3_Filter_t *Signal = MFSK_Synchronizer.SyncSignal.Data +
      MFSK_Synchronizer.SyncBestBlockPhase * MFSK_Synchronizer.SyncSignal.Width;

    // Limit()
    uint32_t FitIdx = MFSK_Synchronizer.SyncBestFreqOffset;
    if( FitIdx < 1 ) FitIdx = 1;
    else if( FitIdx > MFSK_Synchronizer.FreqOffsets - 2 )
      FitIdx = MFSK_Synchronizer.FreqOffsets - 2;

    BOOLEAN FitOK = FitPeak(
        &(NewPreciseFreqOffset),
        &SignalPeak,
        Signal[FitIdx-1].Output,
        Signal[FitIdx].Output,
        Signal[FitIdx+1].Output );

    if( !FitOK )
      NewPreciseFreqOffset = (double)MFSK_Synchronizer.SyncBestFreqOffset;
    else
    {
      // Limit()
      double FrOff = NewPreciseFreqOffset;
      if( FrOff < -1.0 ) FrOff = -1.0;
      else if( FrOff > 1.0 ) FrOff = 1.0;
      NewPreciseFreqOffset = (double)FitIdx + FrOff;
    }

    double NewPreciseBlockPhase = 0.0;

    uint32_t FitIdxL = MFSK_Synchronizer.SyncBestBlockPhase;
    MFSK_Synchronizer.SyncSignal.DecrPtr(
        &(MFSK_Synchronizer.SyncSignal), &FitIdxL, 1 );

    uint32_t FitIdxR = MFSK_Synchronizer.SyncBestBlockPhase;
    MFSK_Synchronizer.SyncSignal.IncrPtr(
        &(MFSK_Synchronizer.SyncSignal), &FitIdxR, 1 );

    uint32_t FitIdxC = MFSK_Synchronizer.SyncBestBlockPhase;

    /* My interpratation of stupid [] "overloaded operator"
     * SyncSignal[FitIdxL][SyncBestFreqOffset].Output,
     */
    double OutputL, OutputC, OutputR;

    Offset  = FitIdxL * MFSK_Synchronizer.SyncSignal.Width;
    Filt    = MFSK_Synchronizer.SyncSignal.Data + Offset;
    OutputL = ( Filt + MFSK_Synchronizer.SyncBestFreqOffset )->Output;

    Offset  = FitIdxC * MFSK_Synchronizer.SyncSignal.Width;
    Filt    = MFSK_Synchronizer.SyncSignal.Data + Offset;
    OutputC = ( Filt + MFSK_Synchronizer.SyncBestFreqOffset )->Output;

    Offset  = FitIdxR * MFSK_Synchronizer.SyncSignal.Width;
    Filt    = MFSK_Synchronizer.SyncSignal.Data + Offset;
    OutputR = ( Filt + MFSK_Synchronizer.SyncBestFreqOffset )->Output;

    FitOK = FitPeak(
        &NewPreciseBlockPhase, &SignalPeak,
        OutputL, OutputC, OutputR );
    if( !FitOK )
    {
      NewPreciseBlockPhase = (double)MFSK_Synchronizer.SyncBestBlockPhase;
    }
    else
    {
      NewPreciseBlockPhase += (double)FitIdxC;
      MFSK_Synchronizer.SyncSignal.WrapPhase(
          &(MFSK_Synchronizer.SyncSignal), &NewPreciseBlockPhase );
    }

    double FreqDelta  = NewPreciseFreqOffset - MFSK_Synchronizer.PreciseFreqOffset;
    double PhaseDelta = NewPreciseBlockPhase - MFSK_Synchronizer.PreciseBlockPhase;
    MFSK_Synchronizer.SyncSignal.WrapDiffPhase(
        &(MFSK_Synchronizer.SyncSignal), &(PhaseDelta) );

    double DeltaDist2 = FreqDelta * FreqDelta + PhaseDelta * PhaseDelta;
    if( (DeltaDist2 <= 1.0) &&
        (MFSK_Synchronizer.SyncSNR >= MFSK_Synchronizer.Parameters->RxSyncThreshold) )
    {
      MFSK_Synchronizer.StableLock = 1;

      MFSK_Synchronizer.FreqDrift.Process(
          &(MFSK_Synchronizer.FreqDrift), FreqDelta,
          MFSK_Synchronizer.SyncFilterWeight, 0.1 );

      MFSK_Synchronizer.TimeDrift.Process(
          &(MFSK_Synchronizer.TimeDrift),
          PhaseDelta / (double)MFSK_Synchronizer.BlockPhases,
          MFSK_Synchronizer.SyncFilterWeight, 0.1 );
    }
    else
    {
      MFSK_Synchronizer.StableLock       = 0;
      MFSK_Synchronizer.FreqDrift.Out1   = 0;
      MFSK_Synchronizer.FreqDrift.Out2   = 0;
      MFSK_Synchronizer.FreqDrift.Output = 0;
      MFSK_Synchronizer.TimeDrift.Out1   = 0;
      MFSK_Synchronizer.TimeDrift.Out2   = 0;
      MFSK_Synchronizer.TimeDrift.Output = 0;
    }

    MFSK_Synchronizer.PreciseFreqOffset = NewPreciseFreqOffset;
    MFSK_Synchronizer.PreciseBlockPhase = NewPreciseBlockPhase;
  }

  MFSK_Synchronizer.SyncSignal.IncrPtr(
      &(MFSK_Synchronizer.SyncSignal), &(MFSK_Synchronizer.BlockPhase), 1 );
}

//----------------------------------------------------------------------

  static double
MFSK_Synchronizer_FEC_SNR( void )
{
  return( MFSK_Synchronizer.SyncSNR );
}

//----------------------------------------------------------------------

  static double
MFSK_Synchronizer_FrequencyOffset( void )
{
  return( (MFSK_Synchronizer.PreciseFreqOffset -
        (double)(MFSK_Synchronizer.FreqOffsets / 2)) *
      MFSK_Synchronizer.Parameters->FFTbinBandwidth(MFSK_Synchronizer.Parameters) );
}

//----------------------------------------------------------------------

  static double
MFSK_Synchronizer_FrequencyDriftRate( void )
{
  return( MFSK_Synchronizer.FreqDrift.Output *
      MFSK_Synchronizer.Parameters->FFTbinBandwidth(MFSK_Synchronizer.Parameters) /
      MFSK_Synchronizer.Parameters->BlockPeriod(MFSK_Synchronizer.Parameters) );
}

//----------------------------------------------------------------------

  static double
MFSK_Synchronizer_TimeDriftRate( void )
{
  return( MFSK_Synchronizer.TimeDrift.Output );
}

//----------------------------------------------------------------------

// Initialize MFSK_Synchronizer - my function
  void
MFSK_Synchronizer_Initialize( void )
{
  MFSK_Synchronizer.Init               = MFSK_Synchronizer_Init;
  MFSK_Synchronizer.Free               = MFSK_Synchronizer_Free;
  MFSK_Synchronizer.Preset             = MFSK_Synchronizer_Preset;
  MFSK_Synchronizer.Reset              = MFSK_Synchronizer_Reset;
  MFSK_Synchronizer.Process            = MFSK_Synchronizer_Process;
  MFSK_Synchronizer.FEC_SNR            = MFSK_Synchronizer_FEC_SNR;
  MFSK_Synchronizer.FrequencyOffset    = MFSK_Synchronizer_FrequencyOffset;
  MFSK_Synchronizer.FrequencyDriftRate = MFSK_Synchronizer_FrequencyDriftRate;
  MFSK_Synchronizer.TimeDriftRate      = MFSK_Synchronizer_TimeDriftRate;

  MFSK_Synchronizer_Init();
  LowPass3_Filter_Initialize( &(MFSK_Synchronizer.FreqDrift) );
  LowPass3_Filter_Initialize( &(MFSK_Synchronizer.TimeDrift) );
}

// ***------------------------------------------------------------------***

  static void
MFSK_Receiver_Free( MFSK_Receiver_t *Self )
{
  if( Self->RateConverter.Free )
    Self->RateConverter.Free( &(Self->RateConverter) );

  if( Self->InputBuffer.Free )
    Self->InputBuffer.Free( &(Self->InputBuffer) );

  if( MFSK_InputProcessor.Free )
    MFSK_InputProcessor.Free();

  if( MFSK_Demodulator.Free )
    MFSK_Demodulator.Free();

  if( MFSK_Synchronizer.Free )
    MFSK_Synchronizer.Free();

  if( MFSK_SoftIterDecoder.Free )
    MFSK_SoftIterDecoder.Free();

  if( Self->Output.Free )
    Self->Output.Free( &(Self->Output) );
}

//----------------------------------------------------------------------

// resize internal arrays according the parameters
  static BOOLEAN
MFSK_Receiver_Preset(
    MFSK_Receiver_t *Self,
    MFSK_Parameters_t *NewParameters )
{
  Self->Parameters = NewParameters;

  // Initialize and Preset Rate Converter
  CRateConverter_Initialize( &(Self->RateConverter) );
  Self->RateConverter.OutputRate =
    (double)Self->Parameters->SampleRate / Self->Parameters->InputSampleRate;
  Self->RateConverter.Preset(&(Self->RateConverter) );

  // Initialize and Preset InputProcessor
  MFSK_InputProcessor_Initialize();
  MFSK_InputProcessor.Default();
  MFSK_InputProcessor.WindowLen = 32 * Self->Parameters->SymbolSepar;
  MFSK_InputProcessor.Preset();

  // Initialize Input Buffer sequencer
  Seq_Initialize( &(Self->InputBuffer) );
  Self->InputBuffer.EnsureSpace(
      &(Self->InputBuffer), MFSK_InputProcessor.WindowLen + 2048 );

  // Initialize and Preset Demodulator
  MFSK_Demodulator_Initialize();
  MFSK_Demodulator.Preset( Self->Parameters );

  // Initialize and Preset Synchronizer
  MFSK_Synchronizer_Initialize();
  MFSK_Synchronizer.Preset( Self->Parameters );

  // Initialize and Preset Soft Iter Decoder
  MFSK_SoftIterDecoder_Initialize();
  MFSK_SoftIterDecoder.Preset( Self->Parameters );

  // Initialize and Preset Output FIFO
  FIFO_Initialize( &(Self->Output) );
  Self->Output.Preset( &(Self->Output), 1024 );

  return( True );
}

//----------------------------------------------------------------------

  static void
MFSK_Receiver_Reset( MFSK_Receiver_t *Self )
{
  Self->RateConverter.Reset( &(Self->RateConverter) );
  Self->InputBuffer.Len = 0;
  MFSK_InputProcessor.Reset();
  MFSK_Demodulator.Reset();
  MFSK_Synchronizer.Reset();
  Self->Output.Reset( &(Self->Output) );
}

//----------------------------------------------------------------------

  static double
MFSK_Receiver_SyncSNR( void )
{
  return( MFSK_Synchronizer.FEC_SNR() );
}

//----------------------------------------------------------------------

  static double
MFSK_Receiver_FrequencyOffset( void )
{
  return( MFSK_Synchronizer.FrequencyOffset() );
}

//----------------------------------------------------------------------

  static double
MFSK_Receiver_FrequencyDrift( void )
{
  return( MFSK_Synchronizer.FrequencyDriftRate() );
}

//----------------------------------------------------------------------

  static double
MFSK_Receiver_TimeDrift( void )
{
  return( MFSK_Synchronizer.TimeDriftRate() );
}

//----------------------------------------------------------------------

  static double
MFSK_Receiver_InputSNRdB( void )
{
  return( MFSK_SoftIterDecoder.InputSNRdB() );
}

//----------------------------------------------------------------------

// Process an audio batch: first the input processor, then the demodulator
  static BOOLEAN
MFSK_Receiver_Process(
    MFSK_Receiver_t *Self,
    double   *Input,
    uint32_t InputLen )
{
  if( Self->RateConverter.Process_Seq(
        &(Self->RateConverter),
        Input, InputLen,
        &Self->InputBuffer) <= 0 )
    return( False );

  Self->ProcessInputBuffer( Self );
  return( True );
}

//----------------------------------------------------------------------

  static void
MFSK_Receiver_Flush( MFSK_Receiver_t *Self )
{
  Self->ProcessInputBuffer( Self );
  uint32_t Idx;

  for( Idx = Self->InputBuffer.Len;
      Idx < MFSK_InputProcessor.WindowLen;
      Idx++ )
    Self->InputBuffer.Elem[Idx] = 0;

  Self->InputBuffer.Len = MFSK_InputProcessor.WindowLen;
  Self->ProcessInputBuffer( Self );

  for( Idx = 0; Idx < MFSK_InputProcessor.WindowLen; Idx++ )
    Self->InputBuffer.Elem[Idx] = 0;

  uint32_t FlushLen =
    Self->Parameters->SymbolSepar *
    Self->Parameters->SymbolsPerBlock *
    Self->Parameters->RxSyncIntegLen * 2;
  for( Idx = 0; Idx < FlushLen; Idx += MFSK_InputProcessor.WindowLen )
  {
    Self->InputBuffer.Len = MFSK_InputProcessor.WindowLen;
    Self->ProcessInputBuffer( Self );
  }
}

//----------------------------------------------------------------------

// Get one character from the output buffer
  static BOOLEAN
MFSK_Receiver_GetChar( MFSK_Receiver_t *Self, uint8_t *Char )
{
  return( Self->Output.Read(&(Self->Output), Char) );
}

//----------------------------------------------------------------------

// Process the input buffer: first the input processor, then the demodulator
  static void
MFSK_Receiver_ProcessInputBuffer( MFSK_Receiver_t *Self )
{
  while( Self->InputBuffer.Len >= MFSK_InputProcessor.WindowLen )
  {
    MFSK_InputProcessor.Process( Self->InputBuffer.Elem );
    Self->InputBuffer.Delete(
        &(Self->InputBuffer), 0, MFSK_InputProcessor.WindowLen );

    for( uint32_t Idx = 0;
        Idx < MFSK_InputProcessor.WindowLen;
        Idx += Self->Parameters->SymbolSepar )
      Self->ProcessSymbol( Self, MFSK_InputProcessor.Output + Idx );
  }
}

//----------------------------------------------------------------------

// Process ( through the demodulator ) an audio batch corresponding to one symbol
// ( demodulator always works with audio batches corresponding to one symbol period )
  static void
MFSK_Receiver_ProcessSymbol( MFSK_Receiver_t *Self, double *Input )
{
  uint32_t SpectraPerSymbol = Self->Parameters->SpectraPerSymbol;
  MFSK_Demodulator.Process( Input );

  int HistOfs;
  for( HistOfs = -(int)SpectraPerSymbol; HistOfs < 0; HistOfs++ )
  {
    double *Spectra = MFSK_Demodulator.HistoryPtr( HistOfs );
    MFSK_Synchronizer.Process( Spectra );

    uint32_t SpectraPerBlock = Self->Parameters->SpectraPerBlock;

    if( MFSK_Synchronizer.DecodeReference == 0 )
    {
      if( MFSK_Synchronizer.StableLock )
      {
        int TimeOffset  = ( HistOfs -
            ((int)(Self->Parameters->RxSyncIntegLen + 1) *
             (int)SpectraPerBlock + (int)SpectraPerBlock / 2 - 1) );
        int FreqOffset = (int)MFSK_Synchronizer.SyncBestFreqOffset;

        double BestSignal = 0.0;
        int BestTime = 0;
        int BestFreq = 0;
        int FreqSearch;

        for( FreqSearch = -1; FreqSearch <= 1; FreqSearch++ )
        {
          int TimeSearch;
          for( TimeSearch = -2; TimeSearch <= 2; TimeSearch++ )
          {
            BOOLEAN Ret = MFSK_Demodulator.PickBlock(
                MFSK_SoftIterDecoder.Input,
                TimeOffset + TimeSearch,
                FreqOffset + FreqSearch );

            if( !Ret  ) continue;

            MFSK_SoftIterDecoder.Process( 8 );
            double Signal = MFSK_SoftIterDecoder.Input_SignalEnergy;
            if( Signal > BestSignal )
            {
              BestSignal = Signal;
              BestFreq   = FreqSearch;
              BestTime   = TimeSearch;
            }
          }
        }

        MFSK_Demodulator.PickBlock(
            MFSK_SoftIterDecoder.Input,
            TimeOffset + BestTime,
            FreqOffset + BestFreq );
        MFSK_SoftIterDecoder.Process( 32 );
        MFSK_SoftIterDecoder.WriteOutputBlock( &(Self->Output) );
      }
    }
  }
}

//----------------------------------------------------------------------

// Initialize MFSK_Receiver - my function
  void
MFSK_Receiver_Initialize( MFSK_Receiver_t *Self )
{
  Self->Free               = MFSK_Receiver_Free;
  Self->Preset             = MFSK_Receiver_Preset;
  Self->Reset              = MFSK_Receiver_Reset;
  Self->SyncSNR            = MFSK_Receiver_SyncSNR;
  Self->FrequencyOffset    = MFSK_Receiver_FrequencyOffset;
  Self->FrequencyDrift     = MFSK_Receiver_FrequencyDrift;
  Self->TimeDrift          = MFSK_Receiver_TimeDrift;
  Self->InputSNRdB         = MFSK_Receiver_InputSNRdB;
  Self->Process            = MFSK_Receiver_Process;
  Self->Flush              = MFSK_Receiver_Flush;
  Self->GetChar            = MFSK_Receiver_GetChar;
  Self->ProcessInputBuffer = MFSK_Receiver_ProcessInputBuffer;
  Self->ProcessSymbol      = MFSK_Receiver_ProcessSymbol;
}

// ***------------------------------------------------------------------***

