/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "lowpass3.h"
#include "../common/utils.h"
#include <stdint.h>
#include <stdlib.h>

/* IIR low pass filter for integration ( averaging ) purposes
 * Overshoot is about 1% for Feedback = 0.5, and about 1e-6
 * for Feedback = 0.1. Weight is 1 / PeakingTime */

//----------------------------------------------------------------------

  static void
LowPass3_Filter_Process(
    LowPass3_Filter_t *Self,
    double Inp,
    double Weight,
    double Feedback )
{
  Weight *= 2.0;

  double DiffI1 = Inp;
  DiffI1 -= Self->Out1;

  double Diff12 = Self->Out1;
  Diff12 -= Self->Out2;

  double Diff23 = Self->Out2;
  Diff23 -= Self->Output;

  DiffI1       *= Weight;
  Self->Out1   += DiffI1;
  Diff12       *= Weight;
  Self->Out2   += Diff12;
  Diff23       *= Weight;
  Self->Output += Diff23;
  Diff23       *= Feedback;
  Self->Out2   += Diff23;
}

//----------------------------------------------------------------------

  static void
LowPass3_Filter_Set( LowPass3_Filter_t *Self, double Level )
{
  Self->Out1   = Level;
  Self->Out2   = Level;
  Self->Output = Level;
}

//----------------------------------------------------------------------

// Initialize LowPass3_Filter - my function
  void
LowPass3_Filter_Initialize( LowPass3_Filter_t *Self )
{
  Self->Set     = LowPass3_Filter_Set;
  Self->Process = LowPass3_Filter_Process;
}

// ***------------------------------------------------------------------***

  static void
CircularBuffer_Init( CircularBuffer_Filter_t *Self )
{
  Self->Data  = NULL;
  Self->Size  = 0;
  Self->Width = 1;
}

//----------------------------------------------------------------------

  static void
CircularBuffer_Free( CircularBuffer_Filter_t *Self )
{
  Mem_Free( (void **) &Self->Data );
  Self->Size = 0;
}

//----------------------------------------------------------------------

// Reset: set pointer to the beginning of the buffer
  static void
CircularBuffer_Reset( CircularBuffer_Filter_t *Self )
{
  Self->Ptr = 0;
}

//----------------------------------------------------------------------

/* Preset for given length and width.
 * Changed name to avoid stupid function overloads
 */
  static void
CircularBuffer_PresetNew(
    CircularBuffer_Filter_t *Self,
    uint32_t NewLen,
    uint32_t NewWidth )
{
  Self->Len   = NewLen;
  Self->Width = NewWidth;
  Self->Preset( Self );
}

//----------------------------------------------------------------------

  static void
CircularBuffer_Preset( CircularBuffer_Filter_t *Self )
{
  Self->Size = Self->Width * Self->Len;
  Mem_Realloc( (void **)
      &Self->Data, (size_t)Self->Size * sizeof(LowPass3_Filter_t) );
  Self->Ptr = 0;

  // My addition, initialize the data structures
  for( uint32_t Idx = 0; Idx < Self->Size; Idx++ )
    LowPass3_Filter_Initialize( &Self->Data[Idx] );
}

//----------------------------------------------------------------------

// Set all elements to given value
  static void
CircularBuffer_Set( CircularBuffer_Filter_t *Self, double *Value )
{
  uint32_t Idx;
  for( Idx = 0; Idx < Self->Size; Idx++ )
    Self->Data[Idx].Set( &Self->Data[Idx], *Value );
}

//----------------------------------------------------------------------

// Set all elements to zero
  static void
CircularBuffer_Clear( CircularBuffer_Filter_t *Self )
{
  double Zero;
  Zero = 0.0;
  Self->Set( Self, &Zero );
}

//----------------------------------------------------------------------

// Increment the pointer ( with wrapping around )
  static void
CircularBuffer_IncrPtr(
    const CircularBuffer_Filter_t *Self,
    uint32_t *Ptr,
    uint32_t Step )
{
  *Ptr += Step;
  if( *Ptr >= Self->Len )
    *Ptr -= Self->Len;
}

//----------------------------------------------------------------------

// Decrement the pointer ( with wrapping around )
  static void
CircularBuffer_DecrPtr(
    const CircularBuffer_Filter_t *Self,
    uint32_t *Ptr,
    uint32_t Step )
{
  if( *Ptr >= Step )
    *Ptr -= Step;
  else
    *Ptr += ( Self->Len - Step );
}

//----------------------------------------------------------------------

  static void
CircularBuffer_WrapPhase(
    CircularBuffer_Filter_t *Self,
    double *Phase )
{
  if( *Phase < 0.0 )
    *Phase += (double)Self->Len;
  else if( *Phase >= (double)Self->Len )
    *Phase -= (double)Self->Len;
}

//----------------------------------------------------------------------

  static void
CircularBuffer_WrapDiffPhase( CircularBuffer_Filter_t *Self, double *Phase )
{
  if( *Phase < (-(double)Self->Len / 2.0) )
    *Phase += (double)Self->Len;
  else if( *Phase >= ((double)Self->Len / 2.0) )
    *Phase -= (double)Self->Len;
}

//----------------------------------------------------------------------

// Get the current pipe pointer
  static LowPass3_Filter_t
*CircularBuffer_CurrPtr( const CircularBuffer_Filter_t *Self )
{
  return( Self->Data + (Self->Ptr * Self->Width) );
}

//----------------------------------------------------------------------

// Get storage pointer relative to current pointer +/- offset
  static LowPass3_Filter_t
*CircularBuffer_OffsetPtr( CircularBuffer_Filter_t *Self, int Offset )
{
  Offset += Self->Ptr;
  Offset *= Self->Width;
  if( Offset < 0 )
    Offset += Self->Size;
  else if( Offset >= (int)Self->Size )
    Offset -= Self->Size;
  return( Self->Data + Offset );
}

//----------------------------------------------------------------------

// My addition, initialize CircularBuffer
  void
CircularBuffer_Filter_Initialize( CircularBuffer_Filter_t *Self )
{
  Self->Init          = CircularBuffer_Init;
  Self->Free          = CircularBuffer_Free;
  Self->Reset         = CircularBuffer_Reset;
  Self->PresetNew     = CircularBuffer_PresetNew;
  Self->Preset        = CircularBuffer_Preset;
  Self->Set           = CircularBuffer_Set;
  Self->Clear         = CircularBuffer_Clear;
  Self->IncrPtr       = CircularBuffer_IncrPtr;
  Self->DecrPtr       = CircularBuffer_DecrPtr;
  Self->WrapPhase     = CircularBuffer_WrapPhase;
  Self->WrapDiffPhase = CircularBuffer_WrapDiffPhase;
  Self->CurrPtr       = CircularBuffer_CurrPtr;
  Self->OffsetPtr     = CircularBuffer_OffsetPtr;

  CircularBuffer_Init( Self );
}

//----------------------------------------------------------------------

