/*
 *  This program is free software; you can redistribute it and/or
*  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "callback_func.h"
#include "display.h"
#include "interface.h"
#include "mfsk.h"
#include "operation.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/interface.h"
#include "../Hermes2/sound.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdlib.h>

// For clearing text buffers
static GtkTextBuffer *text_buffer;

//------------------------------------------------------------------------

// All callback functions of the Olivia client
  void
on_olivia_window_destroy(
    GObject  *object,
    gpointer  user_data)
{
  Olivia_Cleanup();
}

//------------------------------------------------------------------------

  void
Olivia_g_Idle_cb( gpointer data )
{
  if( Flag[GUEST_QUIT] && (olivia_gui.window != NULL) )
    Guest_Quit_Activate( olivia_gui.window );
  Flag[OLIVIA_LABELS_CB] = False;
}

//------------------------------------------------------------------------

  gboolean
on_olivia_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data)
{
  Flag[GUEST_TRANSMIT_MODE] = FALSE;
  Flag[OLIVIA_EXIT_RECEIVE] = True;
  Flag[GUEST_QUIT] = True;
  Save_Window_Position( olivia_gui.window, olivia_rc_data.rc_fpath );

  // In case thread has already exited
  if( Flag[GUEST_RECEIVING] )
    pthread_join( hermes2_rc.guest_rx_thread, NULL );

  if( !Flag[OLIVIA_LABELS_CB] )
    Guest_Quit_Activate( olivia_gui.window );
  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_olivia_quit_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Flag[GUEST_TRANSMIT_MODE] = FALSE;
  Flag[OLIVIA_EXIT_RECEIVE] = True;
  Flag[GUEST_QUIT] = True;
  Save_Window_Position( olivia_gui.window, olivia_rc_data.rc_fpath );

  // In case thread has already exited
  if( Flag[GUEST_RECEIVING] )
    pthread_join( hermes2_rc.guest_rx_thread, NULL );

  if( !Flag[OLIVIA_LABELS_CB] )
    Guest_Quit_Activate( olivia_gui.window );
}

//------------------------------------------------------------------------

  gboolean
on_olivia_tx_textview_key_press(
    GtkWidget   *widget,
    GdkEventKey *event,
    gpointer     user_data)
{
  if( Flag[GUEST_TRANSMIT_MODE] )
    return( Olivia_Tx_Textview_Key_Press(event) );
  else return( FALSE );
}

//------------------------------------------------------------------------

  gboolean
on_olivia_tx_textview_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  // Needed for 'Clear Window'
  text_buffer = olivia_gui.tx_text_buffer;
  return( Olivia_Textview_Button_Press(event) );
}

//------------------------------------------------------------------------

  gboolean
on_olivia_rx_textview_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  // Needed for 'Clear Window'
  text_buffer = olivia_gui.rx_text_buffer;
  return( Olivia_Textview_Button_Press(event) );
}

//------------------------------------------------------------------------

  void
on_olivia_transmit_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  if( !Transceiver[Indices.TRx_Index]->receive_active ||
      !ddv->tx_pa_enable || Flag[BLOCK_FUNCTION] )
    return;

  Olivia_Change_Modes();
}

//------------------------------------------------------------------------

  void
on_olivia_macro_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  if( !Flag[BLOCK_FUNCTION] )
    Olivia_Select_Macro( button );
}

//------------------------------------------------------------------------

  void
on_olivia_receive_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  if( !Transceiver[Indices.TRx_Index]->receive_active ||
      Flag[BLOCK_FUNCTION] )
    return;
  Olivia_Receive_Clicked();
}

//------------------------------------------------------------------------

  gboolean
on_olivia_scope_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  Olivia_Display_Scope( widget, cr, &olivia_ifft_data );
  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_olivia_scope_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  if( event->button == 1 )
  {
    uint16_t freq =
      (uint16_t)Olivia_Parameters.LowerBandEdge +
      (uint16_t)Olivia_Parameters.Bandwidth / 2;

    if( Flag[GUEST_RECEIVE_MODE] )
      Tune_to_Monitor(
          event->x, CLICK_TUNE_RANGE,
          olivia_gui.scope_width,
          freq, &olivia_ifft_data );
  }

  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_olivia_wfall_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  if( olivia_wfall.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, olivia_wfall.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
  }

  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_olivia_wfall_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  if( event->button == 1 )
  {
    uint16_t freq =
      (uint16_t)Olivia_Parameters.LowerBandEdge +
      (uint16_t)Olivia_Parameters.Bandwidth / 2;

    if( Flag[GUEST_RECEIVE_MODE] )
      Tune_to_Monitor(
          event->x, CLICK_TUNE_RANGE,
          olivia_wfall.width,
          freq, &olivia_ifft_data );
  }

  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_olivia_callsign_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  const char *mode;
  if( olivia_rc_data.mode_olivia )
    mode = "OLIVIA";
  else
    mode = "CONTESTI"; // Somehow ADIF spec drops the "A" ??

  Callsign_Changed( editable, &qso_record, mode );
}

//------------------------------------------------------------------------

  void
on_olivia_rst_out_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_RST( editable );
  Strlcpy( qso_record.dx_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(qso_record.dx_rst) );
}

//------------------------------------------------------------------------

  void
on_olivia_rst_in_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  Process_RST( editable );
  Strlcpy( qso_record.my_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(qso_record.my_rst) );
}

//------------------------------------------------------------------------

  void
on_olivia_op_name_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field QSO record structure
  Strlcpy( qso_record.dx_name, buff, sizeof(qso_record.dx_name) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_name );
}

//------------------------------------------------------------------------

  void
on_olivia_qth_name_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field and QSO record structure
  Strlcpy( qso_record.dx_qth, buff, sizeof(qso_record.dx_qth) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_qth );
}

//------------------------------------------------------------------------

  void
on_olivia_locator_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  const char *mode;
  if( olivia_rc_data.mode_olivia )
    mode = "OLIVIA";
  else
    mode = "CONTESTI"; // Somehow ADIF spec drops the "A" ??

  Locator_Changed( editable, &qso_record, mode );
}

//------------------------------------------------------------------------

  void
on_olivia_band_changed(
    GtkEditable *editable,
    gpointer     user_data)
{
  const char *mode;
  if( olivia_rc_data.mode_olivia )
    mode = "OLIVIA";
  else
    mode = "CONTESTI"; // Somehow ADIF spec drops the "A" ??

  Band_Changed( editable, &qso_record, mode );
}

//------------------------------------------------------------------------

  void
on_olivia_new_record_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  const char *mode;
  if( olivia_rc_data.mode_olivia )
    mode = "OLIVIA";
  else
    mode = "CONTESTI"; // Somehow ADIF spec drops the "A" ??

  Strlcpy( qso_record.mode_dir, "olivia", sizeof(qso_record.mode_dir) );
  New_Record( &qso_record, olivia_gui.window_builder, mode );
}

//------------------------------------------------------------------------

  void
on_olivia_save_record_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  // Save record if needed (and valid)
  const char *mode;
  if( olivia_rc_data.mode_olivia )
    mode = "OLIVIA";
  else
    mode = "CONTESTI"; // Somehow ADIF spec drops the "A" ??

  if( Read_QSO_Record(&qso_record, olivia_gui.window_builder, mode) )
    Guest_Save_Dialog( _("Save QSO Record to file?") );
  else
    Guest_Save_Dialog(
        _("QSO Record is incomplete\n"\
          "Do you still want to save?") );
}

//------------------------------------------------------------------------

  void
on_olivia_clear_fields_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  const char *mode;
  if( olivia_rc_data.mode_olivia )
    mode = "OLIVIA";
  else
    mode = "CONTESTI"; // Somehow ADIF spec drops the "A" ??

  Clear_Record_Fields( True, olivia_gui.window_builder, mode );
}

//------------------------------------------------------------------------

  void
on_olivia_clear_window_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  gtk_text_buffer_set_text( text_buffer, "", -1 );

  gtk_text_buffer_get_iter_at_offset(
      olivia_gui.tx_text_buffer, &olivia_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(olivia_gui.tx_text_buffer) );
  gtk_text_buffer_get_iter_at_offset(
      olivia_gui.rx_text_buffer, &olivia_gui.rx_text_buffer_iter,
      gtk_text_buffer_get_char_count(olivia_gui.rx_text_buffer) );

  gtk_widget_grab_focus(
      Builder_Get_Object(olivia_gui.window_builder, "olivia_tx_textview") );
}

//------------------------------------------------------------------------

  void
on_olivia_clear_windows_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_text_buffer_set_text(olivia_gui.tx_text_buffer, "", -1);
  gtk_text_buffer_set_text(olivia_gui.rx_text_buffer, "", -1);

  gtk_text_buffer_get_iter_at_offset(
      olivia_gui.tx_text_buffer, &olivia_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(olivia_gui.tx_text_buffer) );
  gtk_text_buffer_get_iter_at_offset(
      olivia_gui.rx_text_buffer, &olivia_gui.rx_text_buffer_iter,
      gtk_text_buffer_get_char_count(olivia_gui.rx_text_buffer) );

  gtk_widget_grab_focus(
      Builder_Get_Object(olivia_gui.window_builder, "olivia_tx_textview") );
}

//------------------------------------------------------------------------

static uint8_t Mode;

  void
on_mode_olivia_toggled(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  // Do not change mode during Transmit or Receive mode
  if( Flag[GUEST_TRANSMIT_MODE] || Flag[GUEST_RECEIVE_MODE] ) return;

  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    olivia_rc_data.mode_olivia = True;
  else
    olivia_rc_data.mode_olivia = False;

  // Re-initialize parameters
  MFSK_Params_Initialize( &Olivia_Parameters );
  MFSK_Transmitter_Initialize( &Transmitter );
  MFSK_Receiver_Initialize( &Receiver );

  // Reset Mode parameters
  Set_Olivia_Mode( Mode );
}

//------------------------------------------------------------------------

  void
on_olivia_4_125_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Mode = OLIVIA_MODE_4_125;
    Set_Olivia_Mode( OLIVIA_MODE_4_125 );
  }
}

//------------------------------------------------------------------------

  void
on_olivia_4_250_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Mode = OLIVIA_MODE_4_250;
    Set_Olivia_Mode( OLIVIA_MODE_4_250 );
  }
}

//------------------------------------------------------------------------

  void
on_olivia_8_250_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Mode = OLIVIA_MODE_8_250;
    Set_Olivia_Mode( OLIVIA_MODE_8_250 );
  }
}

//------------------------------------------------------------------------

  void
on_olivia_8_500_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Mode = OLIVIA_MODE_8_500;
    Set_Olivia_Mode( OLIVIA_MODE_8_500 );
  }
}

//------------------------------------------------------------------------

  void
on_olivia_16_500_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Mode = OLIVIA_MODE_16_500;
    Set_Olivia_Mode( OLIVIA_MODE_16_500 );
  }
}

//------------------------------------------------------------------------

  void
on_olivia_16_1000_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Mode = OLIVIA_MODE_16_1000;
    Set_Olivia_Mode( OLIVIA_MODE_16_1000 );
  }
}

//------------------------------------------------------------------------

  void
on_olivia_32_1000_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Mode = OLIVIA_MODE_32_1000;
    Set_Olivia_Mode( OLIVIA_MODE_32_1000 );
  }
}

//------------------------------------------------------------------------

  void
on_olivia_capitalize_letters_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[GUEST_CAPITALIZE] = True;
  else
    Flag[GUEST_CAPITALIZE] = False;
}

//------------------------------------------------------------------------

  void
on_olivia_record_qsos_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[GUEST_RECORD_QSO] = True;
  else
  {
    Flag[GUEST_RECORD_QSO] = False;
    if( qso_record.qso_record_fp != NULL )
    {
      Close_File( &qso_record.qso_record_fp );
      qso_record.qso_record_fp = NULL;
    }
  }
}

//------------------------------------------------------------------------

  void
on_olivia_identify_in_cw_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data)
{
  Flag[GUEST_TRANSMIT_ID] = True;
}

//------------------------------------------------------------------------

  void
on_tune_freq_offset_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Flag[OLIVIA_TUNE_OFFSET] = True;
}

//------------------------------------------------------------------------

  void
on_olivia_flush_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Flag[OLIVIA_FLUSH_RECEIVER] = True;
}

//------------------------------------------------------------------------

