/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callback_func.h"
#include "callbacks.h"
#include "display.h"
#include "operation.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../common/guest_utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/modulate.h"
#include "../Hermes2/sound.h"
#include "../rsid/rsid.h"
#include <ctype.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdlib.h>

// Input keyboard character
guint key_buf;

//------------------------------------------------------------------------

/* Olivia_Change_Modes()
 *
 * Toggles down Macro TX and toggles
 * Keyboard Tx and Receive mode
 */
  void
Olivia_Change_Modes( void )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Toggle Tx/Rx modes
  Flag[BLOCK_FUNCTION] = True;
  if( Flag[GUEST_TRANSMIT_KEYBD] ) // In keyboard TX mode
  {
    Flag[GUEST_TRANSMIT_KEYBD] = False;
    Flag[GUEST_TRANSMIT_MODE]  = False;
  }
  else if( !ddv->transmit_on ) // MOX or Tune not on
  {
    // Set up control flags for Keyboard transmit
    Flag[GUEST_TRANSMIT_KEYBD] = True;

    // Clear keyboard buffer data
    keybd_buf.key_count    = 0;
    keybd_buf.char_pointer = 0;
    keybd_buf.space_count  = 0;
    keybd_buf.char_count   = 0;

    // Flush Receiver if running, will exit Receive mode
    // Wait for Rx mode to exit
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[OLIVIA_FLUSH_RECEIVER] = True;
      Flag[OLIVIA_EXIT_RECEIVE]   = True;

      // In case thread has not exited yet, join it
      if( Flag[GUEST_RECEIVING] )
        pthread_join( hermes2_rc.guest_rx_thread, NULL );
    }

    // Initialize transmission
    if( !Olivia_Initialize_Transmit() )
    {
      Flag[BLOCK_FUNCTION] = False;
      return;
    }

    Flag[OLIVIA_LABELS_CB] = True;
    g_idle_add_full(
        G_PRIORITY_DEFAULT_IDLE,
        Olivia_Set_TxRx_Labels,
        NULL,
        Olivia_g_Idle_cb );

  } // else of if( Flag[GUEST_TRANSMIT_KEYBD] )

  Flag[BLOCK_FUNCTION] = False;
} // Olivia_Change_Modes()

//------------------------------------------------------------------------

/* Olivia_Select_Macro()
 *
 * Selects a macro to be transmitted
 */
  void
Olivia_Select_Macro( GtkButton *button )
{
  Flag[BLOCK_FUNCTION] = True;
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // If the SDR client is not in Receive mode, abort
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(olivia_gui.window_builder, "olivia_tx_textview") );

  // Transmit a macro if not already in Macro mode
  if( !Flag[GUEST_TRANSMIT_MACRO] )
  {
    Flag[GUEST_TRANSMIT_MACRO] = True;

    // Clear keyboard buffer data
    keybd_buf.key_count    = 0;
    keybd_buf.char_pointer = 0;
    keybd_buf.space_count  = 0;
    keybd_buf.char_count   = 0;

    // Flush Receiver if running, will exit Receive mode
    // Wait for Rx mode to exit
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[OLIVIA_FLUSH_RECEIVER] = True;
      Flag[OLIVIA_EXIT_RECEIVE]   = True;

      // In case thread has already exited
      if( Flag[GUEST_RECEIVING] )
        pthread_join( hermes2_rc.guest_rx_thread, NULL );
    }

    // Enable macro transmission if we are not
    // in transmit mode (MOX on or Tune active).
    // If in keyboard mode, do not initialize.
    if( !ddv->transmit_on || Flag[GUEST_TRANSMIT_KEYBD] )
    {
      // Initialize Transmit if needed
      if( !Flag[GUEST_TRANSMIT_KEYBD] &&
          !Olivia_Initialize_Transmit() )
      {
        Flag[GUEST_TRANSMIT_MACRO] = False;
        Flag[BLOCK_FUNCTION] = False;
        return;
      }

      Flag[GUEST_TRANSMIT_KEYBD] = False;
      Flag[OLIVIA_LABELS_CB]     = True;
      g_idle_add_full(
          G_PRIORITY_DEFAULT_IDLE,
          Olivia_Set_TxRx_Labels,
          NULL,
          Olivia_g_Idle_cb );

      // Get macro index from the name of the macro button
      olivia_macro_idx =
        (uint8_t)( atoi(gtk_widget_get_name(GTK_WIDGET(button))) - 1 );

      // Fills the Tx buffer with the selected Macro
      Olivia_Transmit_Macro();

    } // if( !ddv->transmit_on ... )
  } // if( !Flag[GUEST_TRANSMIT_MACRO] )

  Flag[BLOCK_FUNCTION] = False;
} // Olivia_Select_Macro()

//------------------------------------------------------------------------

/* Olivia_Receive_Clicked()
 *
 * Handles the on_olivia_receive_clicked callback
 */
  void
Olivia_Receive_Clicked( void )
{
  // Toggle Tx/Rx/Macro modes
  Flag[BLOCK_FUNCTION] = True;
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // If in Rx mode, terminated it
  if( Flag[GUEST_RECEIVING] )
  {
    Flag[OLIVIA_FLUSH_RECEIVER] = True;
    Flag[OLIVIA_EXIT_RECEIVE]   = True;
  }
  else // If not in Tx mode, start Rx mode
  {
    if( !Flag[GUEST_TRANSMIT_MACRO] &&
        !Flag[GUEST_TRANSMIT_KEYBD] &&
        !ddv->transmit_on )
    {
      if( !Create_Rx_Thread() )
      {
        Flag[GUEST_RECEIVE_MODE] = False;
        Flag[BLOCK_FUNCTION] = False;
        return;
      }
    }
    else // In Tx mode, terminate it
    {
      // Stop and flush Transmitter
      Exit_Transmit();
    }
  } // if( Flag[GUEST_RECEIVING] )

  Flag[BLOCK_FUNCTION] = False;
} // Olivia_Receive_Clicked()

//------------------------------------------------------------------------

// Desensitizes menus when transmitting or receiving
  static void
Olivia_Menus_Activate( gboolean flag )
{
  gtk_widget_set_sensitive(
      Builder_Get_Object(olivia_gui.popup_menu_builder, "mode_olivia"), flag );
  gtk_widget_set_sensitive(
      Builder_Get_Object(olivia_gui.popup_menu_builder, "mode_contestia"), flag );
  gtk_widget_set_sensitive(
      Builder_Get_Object(olivia_gui.popup_menu_builder, "olivia_mode"), flag );
}

//------------------------------------------------------------------------

/* Olivia_Textview_Button_Press()
 *
 * Handles button press events on Rx and Tx textview
 */
  gboolean
Olivia_Textview_Button_Press( const GdkEventButton *event )
{
  switch( event->button )
  {
    // Focus to widget that received the button press
    case 1:
      return( FALSE );

    case 2: // Toggle Tx/Rx/Macro modes
      if( !Flag[BLOCK_FUNCTION] )
        Olivia_Change_Modes();
      break;

    case 3: // Popup menu. "Sensitize Identify in CW" in keybord mode
      if( Flag[GUEST_TRANSMIT_KEYBD] )
        gtk_widget_set_sensitive(
            Builder_Get_Object( olivia_gui.popup_menu_builder, "olivia_identify_in_cw"), TRUE );
      else
        gtk_widget_set_sensitive(
            Builder_Get_Object( olivia_gui.popup_menu_builder, "olivia_identify_in_cw"), FALSE );

      // Desensitize Mode menus when transmitting or receiving
      if( Flag[GUEST_TRANSMIT_KEYBD] || Flag[GUEST_RECEIVING] )
        Olivia_Menus_Activate( FALSE );
      else
        Olivia_Menus_Activate( TRUE );

      gtk_menu_popup_at_pointer( GTK_MENU(olivia_gui.popup_menu), NULL );
  } // switch( event->button )

  return( TRUE );
} // Olivia_Textview_Button_Press()

//------------------------------------------------------------------------

/* Olivia_Tx_Textview_Key_Press()
 *
 * Handles keyboard input during transmit
 */
  gboolean
Olivia_Tx_Textview_Key_Press( GdkEventKey *event )
{
  // Ignore Shift keys
  if( (event->keyval == GDK_KEY_Shift_L) ||
      (event->keyval == GDK_KEY_Shift_R) )
    return( FALSE );

  // Deal with Backspace key
  if( event->keyval == GDK_KEY_BackSpace )
  {
    // Roll back word wrap counter
    if( keybd_buf.char_count ) keybd_buf.char_count--;

    // Enter Back Space code to key buffer
    key_buf = event->keyval;
    if( Flag[GUEST_TRANSMIT_KEYBD] )
      Olivia_Transmit_Keybd();

    // Print Backspace on text view
    tx_print_chr.printchr = event->keyval;
    Queue_Character( &tx_print_chr );

    return( TRUE );
  } // if( event->keyval == GDK_KEY_BackSpace )

  // Print Return on Return key or on Word Wrap
  keybd_buf.char_count++;
  if( (event->keyval == GDK_KEY_Return) ||
      ((keybd_buf.char_count >= olivia_rc_data.word_wrap) &&
       (event->keyval == ' ')) )
  {
    // For word wrap case add a GDK_KEY_Return entry to buffer
    key_buf = GDK_KEY_Return;
    if( Flag[GUEST_TRANSMIT_KEYBD] )
      Olivia_Transmit_Keybd();

    // Print GDK_KEY_Return to text view
    tx_print_chr.printchr = GDK_KEY_Return;
    Queue_Character( &tx_print_chr );

    // Reset Word Wrap counter
    keybd_buf.char_count = 0;
    return( TRUE );
  }

  // Escape is used to stop Transmission
  if( event->keyval == GDK_KEY_Escape )
  {
    event->keyval &= 0xFF;
    key_buf = event->keyval;
    if( Flag[GUEST_TRANSMIT_KEYBD] )
      Olivia_Transmit_Keybd();

    return( TRUE );
  }

  // Don't process non-ASCII characters
  if( (event->keyval & 0xFF) > 0x7F )
    return( TRUE );

  // Capitalize letters if enabled
  if( Flag[GUEST_CAPITALIZE] )
    key_buf = (guint)toupper( event->keyval );
  else
    key_buf = event->keyval;

  // Print character
  tx_print_chr.printchr = key_buf;
  Queue_Character( &tx_print_chr );

  // Send character to Transmitter
  if( Flag[GUEST_TRANSMIT_KEYBD] )
    Olivia_Transmit_Keybd();

  return( TRUE );
} // Olivia_Tx_Textview_Key_Press()

//------------------------------------------------------------------------

/* Set_Olivia_Mode()
 *
 * Sets the Olivia parameters on mode selection
 */
  void
Set_Olivia_Mode( uint8_t mode_flag )
{
  uint8_t  bits = 3, mode = RX_MODE_8_250;
  uint16_t bwidth = 250;
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Clear all mode flags first
  Flag[OLIVIA_MODE_4_125]   = False;
  Flag[OLIVIA_MODE_4_250]   = False;
  Flag[OLIVIA_MODE_8_250]   = False;
  Flag[OLIVIA_MODE_8_500]   = False;
  Flag[OLIVIA_MODE_16_500]  = False;
  Flag[OLIVIA_MODE_16_1000] = False;
  Flag[OLIVIA_MODE_32_1000] = False;

  // Set active mode flag
  Flag[mode_flag] = True;

  // Set new Olivia parameters and Preset general
  // parameters and Transmitter prameters
  switch( mode_flag )
  {
    case OLIVIA_MODE_4_125:
      bits   = 2;
      bwidth = 125;
      mode   = RX_MODE_4_125;
      if( olivia_rc_data.mode_olivia )
        TRx->RSID_Mode = MODE_OLIVIA_4_125;
      else
        TRx->RSID_Mode = MODE_CONTESTIA_4_125;
      break;

    case OLIVIA_MODE_4_250:
      bits   = 2;
      bwidth = 250;
      mode   = RX_MODE_4_250;
      if( olivia_rc_data.mode_olivia )
        TRx->RSID_Mode = MODE_OLIVIA_4_250;
      else
        TRx->RSID_Mode = MODE_CONTESTIA_4_250;
      break;

    case OLIVIA_MODE_8_250:
      bits   = 3;
      bwidth = 250;
      mode   = RX_MODE_8_250;
      if( olivia_rc_data.mode_olivia )
        TRx->RSID_Mode = MODE_OLIVIA_8_250;
      else
        TRx->RSID_Mode = MODE_CONTESTIA_8_250;
      break;

    case OLIVIA_MODE_8_500:
      bits   = 3;
      bwidth = 500;
      mode   = RX_MODE_8_500;
      if( olivia_rc_data.mode_olivia )
        TRx->RSID_Mode = MODE_OLIVIA_8_500;
      else
        TRx->RSID_Mode = MODE_CONTESTIA_8_500;
      break;

    case OLIVIA_MODE_16_500:
      bits   = 4;
      bwidth = 500;
      mode   = RX_MODE_16_500;
      if( olivia_rc_data.mode_olivia )
        TRx->RSID_Mode = MODE_OLIVIA_16_500;
      else
        TRx->RSID_Mode = MODE_CONTESTIA_16_500;
      break;

    case OLIVIA_MODE_16_1000:
      bits   = 4;
      bwidth = 1000;
      mode   = RX_MODE_16_1000;
      if( olivia_rc_data.mode_olivia )
        TRx->RSID_Mode = MODE_OLIVIA_16_1000;
      else
        TRx->RSID_Mode = MODE_CONTESTIA_16_1000;
      break;

    case OLIVIA_MODE_32_1000:
      bits   = 5;
      bwidth = 1000;
      mode   = RX_MODE_32_1000;
      if( olivia_rc_data.mode_olivia )
        TRx->RSID_Mode = MODE_OLIVIA_32_1000;
      else
        TRx->RSID_Mode = MODE_CONTESTIA_32_1000;
      break;
  } // switch( mode_flag )

  // Set Receiver modulation mode and the common Guest modulation mode
  Transceiver[Indices.TRx_Index]->guest_modulation_mode = mode;
  Rx_Modulation_Mode_Changed( Indices.TRx_Index, mode );
  Flag[OLIVIA_LABELS_CB] = True;
  g_idle_add_full(
      G_PRIORITY_DEFAULT_IDLE,
      Olivia_Set_TxRx_Labels,
      NULL,
      Olivia_g_Idle_cb );

  Olivia_Parameters.BitsPerSymbol = bits;
  Olivia_Parameters.Bandwidth     = bwidth;

  /* Initialize fixed and permanent settings
   * in Parameters, Transmitter and Receiver */
  MFSK_Params_Preset( &Olivia_Parameters );

  // Free, initialize and preset/allocate Transmitter and Receiver objects
  Receiver.Free( &Receiver );
  Receiver.Preset( &Receiver, &Olivia_Parameters );
  Transmitter.Free( &Transmitter );
  Transmitter.Preset( &Transmitter, &Olivia_Parameters );

  // Puts the center freq of olivia's bandwidth in middle of waterfall
  uint16_t idx = (2 * olivia_ifft_data.data_len) / olivia_wfall.width;
  uint16_t freq =
    (uint16_t)Olivia_Parameters.LowerBandEdge +
    (uint16_t)Olivia_Parameters.Bandwidth / 2;
  if( idx ) olivia_ifft_data.ifft_stride = SND_DSP_RATE / freq / idx;

} // Set_Olivia_Mode()

//------------------------------------------------------------------------

