/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef BUFFER_H
#define BUFFER_H    1

#include "struc.h"
#include "lowpass3.h"
#include "../common/common.h"
#include <stdint.h>

//----------------------------------------------------------------------

// A simple FIFO buffer. Use Init() as constructor / Free() as destructor.
typedef struct _FIFO
{
  uint32_t Len;
  uint32_t ReadPtr;
  uint32_t WritePtr;
  uint8_t *Data;

  // Pointers to functions that were in class FIFO
  void     (*Init)(struct _FIFO *);
  void     (*Free)(struct _FIFO *);
  void     (*Reset)(struct _FIFO *);
  void     (*Clear)(struct _FIFO *);
  void     (*Preset)(struct _FIFO *, uint32_t);
  void     (*IncrPtr)(const struct _FIFO *, uint32_t *, uint32_t);
  BOOLEAN  (*Full)(struct _FIFO *);
  BOOLEAN  (*Empty)(const struct _FIFO *);
  uint32_t (*WriteReady)(struct _FIFO *);
  uint32_t (*ReadReady)(struct _FIFO *);
  BOOLEAN  (*Write)(struct _FIFO *, uint8_t);
  BOOLEAN  (*Read)(struct _FIFO *, uint8_t *);
  BOOLEAN  (*Lookup)(struct _FIFO *, uint8_t *, uint32_t);

} FIFO_t;

extern void FIFO_Initialize(FIFO_t *);

//-----------------------------------------------------------------------

// A circular buffer to store history of double type data. Data may
// come as single numbers or in batches of fixed size (-> Width )
// Use Init() as constructor and Free() as destructor.
typedef struct _CIRCULARBUFFER_FLOAT
{
  uint32_t Width;  // Input/output data width ( row width )
  uint32_t Len;    // Buffer length ( column height )
  uint32_t Size;   // Total size of the storage in the buffer
  uint32_t Ptr;    // Current pointer ( counts rows )
  double   *Data;   // Allocated storage

  // Pointers to functions that were in class CircularBuffer
  void    (*Init)(struct _CIRCULARBUFFER_FLOAT *);
  void    (*Free)(struct _CIRCULARBUFFER_FLOAT *);
  void    (*Preset)(struct _CIRCULARBUFFER_FLOAT *);
  void    (*Clear)(struct _CIRCULARBUFFER_FLOAT *);
  void    (*IncrPtr)(const struct _CIRCULARBUFFER_FLOAT *, uint32_t *, uint32_t);
  void    (*DecrPtr)(const struct _CIRCULARBUFFER_FLOAT *, uint32_t *, uint32_t);
  void    (*WrapPhase)(struct _CIRCULARBUFFER_FLOAT *, double *);
  void    (*WrapDiffPhase)(struct _CIRCULARBUFFER_FLOAT *, double *);
  double *(*OffsetPtr)(struct _CIRCULARBUFFER_FLOAT *, int);
  void    (*Reset)(struct _CIRCULARBUFFER_FLOAT *);
  void    (*PresetNew)(struct _CIRCULARBUFFER_FLOAT *, uint32_t, uint32_t);
  void    (*Set)(struct _CIRCULARBUFFER_FLOAT *, const double *);
  double *(*CurrPtr)(const struct _CIRCULARBUFFER_FLOAT *);

} CircularBuffer_Float_t;

extern void CircularBuffer_Float_Initialize(CircularBuffer_Float_t *);

//----------------------------------------------------------------------

#endif

