/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "buffer.h"
#include "struc.h"
#include "../common/utils.h"
#include <stdint.h>
#include <stdlib.h>

//----------------------------------------------------------------------

  static void
FIFO_Init( FIFO_t *Self )
{
  Self->Data = NULL;
  Self->Len  = 0;
}

//----------------------------------------------------------------------

  static void
FIFO_Free( FIFO_t *Self )
{
  Mem_Free( (void **) &Self->Data );
  Self->Len = 0;
}

//----------------------------------------------------------------------

  static void
FIFO_Reset( FIFO_t *Self )
{
  Self->ReadPtr  = 0;
  Self->WritePtr = 0;
}

//----------------------------------------------------------------------

  static void
FIFO_Clear( FIFO_t *Self )
{
  Self->ReadPtr = Self->WritePtr;
}

//----------------------------------------------------------------------

/* Combined overloaded Preset() functions of the C++ FIFO class.
 * Make argument NewLen = Self->Len if Self->Len is not to change.
 */
  static void
FIFO_Preset( FIFO_t *Self, uint32_t NewLen )
{
  Self->Len = NewLen;
  Mem_Realloc( (void **) &Self->Data, sizeof(uint8_t) * Self->Len );
  Self->Reset( Self );
}

//----------------------------------------------------------------------

/* Increment the pointer (with wrapping around). The Step argument is set
 * to default value of 1 and apparently in C++ version it may be omitted. */
  static void
FIFO_IncrPtr( const FIFO_t *Self, uint32_t *Ptr, uint32_t Step )
{
  *Ptr += Step;
  if( *Ptr >= Self->Len )
    *Ptr -= Self->Len;
}

//----------------------------------------------------------------------

// FIFO is full ?
  static BOOLEAN
FIFO_Full( FIFO_t *Self )
{
  uint32_t Ptr = Self->WritePtr;
  Self->IncrPtr( Self, &Ptr, 1 );
  return( Ptr == Self->ReadPtr );
}

//----------------------------------------------------------------------

// FIFO is empty ?
  static BOOLEAN
FIFO_Empty( const FIFO_t *Self )
{
  return( Self->ReadPtr == Self->WritePtr );
}

//----------------------------------------------------------------------

// How many elements we can write = space left in the FIFO
  static uint32_t
FIFO_WriteReady( FIFO_t *Self )
{
  int Ready = (int)Self->ReadPtr - (int)Self->WritePtr;
  if( Ready <= 0 ) Ready += Self->Len;
  return( (uint32_t)(Ready - 1) );
}

//----------------------------------------------------------------------

// How many elements we can read = space taken in the FIFO
  static uint32_t
FIFO_ReadReady( FIFO_t *Self )
{
  int Ready = (int)Self->WritePtr - (int)Self->ReadPtr;
  if( Ready < 0 ) Ready += Self->Len;
  return( (uint32_t)Ready );
}

//----------------------------------------------------------------------

// Write a new element
  static uint8_t
FIFO_Write( FIFO_t *Self, uint8_t NewData )
{
  uint32_t Ptr = Self->WritePtr;
  Self->IncrPtr( Self, &Ptr, 1 );
  if( Ptr == Self->ReadPtr ) return( 0 );
  Self->Data[Self->WritePtr] = NewData;
  Self->WritePtr = Ptr;
  return( 1 );
}

//----------------------------------------------------------------------

// Read the oldest element
  static BOOLEAN
FIFO_Read( FIFO_t *Self, uint8_t *OldData )
{
  if( Self->ReadPtr == Self->WritePtr ) return( False );
  *OldData = Self->Data[Self->ReadPtr];
  Self->IncrPtr( Self, &Self->ReadPtr, 1 );
  return( True );
}

//----------------------------------------------------------------------

// Lookup data in the FIFO but without taking them out
  static BOOLEAN
FIFO_Lookup( FIFO_t *Self, uint8_t *OldData, uint32_t Offset )
{
  uint32_t Ready = Self->ReadReady( Self );
  if( Offset >= Ready ) return( False );
  uint32_t Ptr = Self->ReadPtr;
  Self->IncrPtr( Self, &Ptr, Offset );
  *OldData = Self->Data[Ptr];
  return( True );
}

//----------------------------------------------------------------------

// My addition, initialize FIFO
  void
FIFO_Initialize( FIFO_t *Self )
{
  Self->Init       = FIFO_Init;
  Self->Free       = FIFO_Free;
  Self->Reset      = FIFO_Reset;
  Self->Clear      = FIFO_Clear;
  Self->Preset     = FIFO_Preset;
  Self->IncrPtr    = FIFO_IncrPtr;
  Self->Full       = FIFO_Full;
  Self->Empty      = FIFO_Empty;
  Self->WriteReady = FIFO_WriteReady;
  Self->ReadReady  = FIFO_ReadReady;
  Self->Write      = FIFO_Write;
  Self->Read       = FIFO_Read;
  Self->Lookup     = FIFO_Lookup;

  FIFO_Init( Self );
}

//----------------------------------------------------------------------

  static void
CircularBuffer_Init( CircularBuffer_Float_t *Self )
{
  Self->Data  = NULL;
  Self->Size  = 0;
  Self->Width = 1;
}

//----------------------------------------------------------------------

  static void
CircularBuffer_Free( CircularBuffer_Float_t *Self )
{
  Mem_Free( (void **) &Self->Data );
  Self->Size = 0;
}

//----------------------------------------------------------------------

// Reset: set pointer to the beginning of the buffer
  static void
CircularBuffer_Reset( CircularBuffer_Float_t *Self )
{
  Self->Ptr = 0;
}

//----------------------------------------------------------------------

// Preset for given length and width.
// Changed name to avoid stupid function overloads */
  static void
CircularBuffer_PresetNew(
    CircularBuffer_Float_t *Self,
    uint32_t NewLen,
    uint32_t NewWidth )
{
  Self->Len   = NewLen;
  Self->Width = NewWidth;
  Self->Preset( Self );
}

//----------------------------------------------------------------------

  static void
CircularBuffer_Preset( CircularBuffer_Float_t *Self )
{
  Self->Size = Self->Width * Self->Len;
  Mem_Realloc( (void **) &Self->Data, (size_t)Self->Size * sizeof(double) );
  Self->Reset( Self );
}

//----------------------------------------------------------------------

// Set all elements to given value
  static void
CircularBuffer_Set( CircularBuffer_Float_t *Self, const double *Value )
{
  uint32_t Idx;
  for( Idx = 0; Idx < Self->Size; Idx++ )
    Self->Data[Idx] = *Value;
}

//----------------------------------------------------------------------

// Set all elements to zero
  static void
CircularBuffer_Clear( CircularBuffer_Float_t *Self )
{
  double Zero;
  Zero = 0.0;
  Self->Set( Self, &Zero );
}

//----------------------------------------------------------------------

// Increment the pointer ( with wrapping around )
  static void
CircularBuffer_IncrPtr(
    const CircularBuffer_Float_t *Self,
    uint32_t *Ptr,
    uint32_t Step )
{
  *Ptr += Step;
  if( *Ptr >= Self->Len )
    *Ptr -= Self->Len;
}

//----------------------------------------------------------------------

// Decrement the pointer ( with wrapping around )
  static void
CircularBuffer_DecrPtr(
    const CircularBuffer_Float_t *Self,
    uint32_t *Ptr,
    uint32_t Step )
{
  if( *Ptr >= Step )
    *Ptr -= Step;
  else
    *Ptr += ( Self->Len - Step );
}

//----------------------------------------------------------------------

  static void
CircularBuffer_WrapPhase( CircularBuffer_Float_t *Self, double *Phase )
{
  if( *Phase < 0.0 )
    *Phase += (double)Self->Len;
  else if( *Phase >= (double)Self->Len )
    *Phase -= (double)Self->Len;
}

//----------------------------------------------------------------------

  static void
CircularBuffer_WrapDiffPhase( CircularBuffer_Float_t *Self, double *Phase )
{
  if( *Phase < (-(double)Self->Len / 2.0) )
    *Phase += (double)Self->Len;
  else if( *Phase >= ((double)Self->Len / 2.0) )
    *Phase -= (double)Self->Len;
}

//----------------------------------------------------------------------

// Get the current pipe pointer
  static double
*CircularBuffer_CurrPtr( const CircularBuffer_Float_t *Self )
{
  return( Self->Data + (Self->Ptr * Self->Width) );
}

//----------------------------------------------------------------------

// Get storage pointer relative to current pointer +/- offset
  static double
*CircularBuffer_OffsetPtr( CircularBuffer_Float_t *Self, int Offset )
{
  Offset += (int)Self->Ptr;
  Offset *= (int)Self->Width;
  if( Offset < 0 )
    Offset += (int)Self->Size;
  else if( Offset >= (int)Self->Size )
    Offset -= (int)Self->Size;
  return( Self->Data + Offset );
}

//----------------------------------------------------------------------

// My addition, initialize CircularBuffer
  void
CircularBuffer_Float_Initialize( CircularBuffer_Float_t *Self )
{
  Self->Init          = CircularBuffer_Init;
  Self->Free          = CircularBuffer_Free;
  Self->Reset         = CircularBuffer_Reset;
  Self->PresetNew     = CircularBuffer_PresetNew;
  Self->Preset        = CircularBuffer_Preset;
  Self->Set           = CircularBuffer_Set;
  Self->Clear         = CircularBuffer_Clear;
  Self->IncrPtr       = CircularBuffer_IncrPtr;
  Self->DecrPtr       = CircularBuffer_DecrPtr;
  Self->WrapPhase     = CircularBuffer_WrapPhase;
  Self->WrapDiffPhase = CircularBuffer_WrapDiffPhase;
  Self->CurrPtr       = CircularBuffer_CurrPtr;
  Self->OffsetPtr     = CircularBuffer_OffsetPtr;

  CircularBuffer_Init( Self );
}

//----------------------------------------------------------------------

