/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "morse_rc.h"
#include "detect.h"
#include "decode.h"
#include "interface.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/interface.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

//------------------------------------------------------------------

#define MORSE_CONFIG_FILE "/.hermes2/morse/morse.config"

#define MAX_SPEED   40  // Maximum allowed Morse speed wpm
#define MIN_SPEED   5   // Minimum allowed Morse speed wpm

//----------------------------------------------------------------------

/* Morse_Read_Config()
 *
 * Loads the morse.config configuration file
 */
  gboolean
Morse_Read_Config( gpointer data )
{
  // Buffer for Read_Line
  char line[READ_LINE_BUF_SIZE];

  // Config file pointer
  FILE *morserc;

  // Setup file path to morse.config
  Strlcpy(  morse_rc.rc_fpath, getenv("HOME"),
      sizeof( morse_rc.rc_fpath) );
  Strlcat(  morse_rc.rc_fpath, MORSE_CONFIG_FILE,
      sizeof( morse_rc.rc_fpath) );

  // Open morse.config file
  morserc = fopen(  morse_rc.rc_fpath, "r" );
  if( morserc == NULL )
  {
    perror(  morse_rc.rc_fpath );
    Error_Dialog(
        _("Failed to open morse.config file.\n"\
          "Quit hermes2 and correct"), HIDE_OK );
    return( FALSE );
  }

  // *** Read runtime configuration data ***
  // Read max WPM, abort if EOF
  if( Read_Line(line, morserc, _("Maximum WPM")) != SUCCESS )
    return( FALSE );
  morse_rc.max_wpm = (uint16_t)atoi( line );

  // Read min WPM, abort if EOF
  if( Read_Line(line, morserc, _("Minimum WPM")) != SUCCESS )
    return( FALSE );
  morse_rc.min_wpm = (uint16_t)atoi( line );

  // Check range of Morse speeds
  if( (morse_rc.min_wpm < MIN_SPEED) ||
      (morse_rc.max_wpm > MAX_SPEED) )
  {
    Close_File( &morserc );
    Error_Dialog(
        _("Morse code speed (WPM)\n"\
          "range is out of limits\n"\
          "Quit and correct morse.config"), HIDE_OK );
    return( FALSE );
  }

  // Set spinbutton range
  gtk_spin_button_set_range( morse_gui.speed_spinbtn,
      (gdouble)morse_rc.min_wpm, (gdouble)morse_rc.max_wpm );

  // Read initial WPM, abort if EOF
  if( Read_Line(line, morserc, _("Initial WPM")) != SUCCESS )
    return( FALSE );
  uint16_t wpm = (uint16_t)atoi( line );

  // Read main Window's position and place it
  if( Read_Line(line, morserc, _("Window Position") ) != SUCCESS )
    return( FALSE );
  gint x = (gint)atoi( line );
  gint y = (gint)atoi( &line[5] );
  gtk_window_move( GTK_WINDOW(morse_gui.window), x, y );

  // Check initial Morse speed
  if( (wpm > morse_rc.max_wpm) ||
      (wpm < morse_rc.min_wpm) )
  {
    Close_File( &morserc );
    Error_Dialog(
        _("Initial Morse code speed (WPM)\n"\
          "is out of specified range\n"\
          "Quit and correct morse.config"), HIDE_OK );
    return( FALSE );
  }

  // Default BFO Tone frequency
  morse_rc.tone_freq = 600;

  // Get sizes of waterfall display
  GtkAllocation alloc;
  gtk_widget_get_allocation( morse_wfall.canvas, &alloc );
  Morse_Waterfall_Configure_Event(
      (uint16_t)alloc.width, (uint16_t)alloc.height );
  morse_rc.center_line = morse_wfall.width / 2 + 1;

  // Get sizes of scope display
  gtk_widget_get_allocation( morse_gui.scope, &alloc );
  morse_gui.scope_width  = (uint16_t)alloc.width  - 2;
  morse_gui.scope_height = (uint16_t)alloc.height - 2;

  // Set speed spinbutton value
  gtk_spin_button_set_value( morse_gui.speed_spinbtn, (gdouble)wpm );
  morse_rc.unit_elem =
    (60 * morse_rc.tone_freq) / (50 * CYCLES_PER_FRAG * wpm);

  Flag[MORSE_RCCONFIG_SETUP] = True;
  if( Morse_Initialize_Detector() )
    Flag[MORSE_ENABLE_RECEIVE] = True;

  Close_File( &morserc );
  return( FALSE );
} // End of Morse_Read_Config()

//------------------------------------------------------------------

