/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "interface.h"
#include "detect.h"
#include "morse_rc.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/ifft.h"
#include "../common/hermes2_rc.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/interface.h"
#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdint.h>

//------------------------------------------------------------------

#define MORSE_GLADE_FILE "/.hermes2/morse/morse.glade"

#define MORSE_WINDOW_IDS \
  "morse_window", \
  "level_adjustment", \
  "ratio_adjustment", \
  "wpm_adjustment", \
  NULL

//----------------------------------------------------------------------

/* Create_Morse_Window()
 *
 * Creates the Morse Decoder main window
 */
  GtkWidget *
Create_Morse_Window( GtkBuilder **builder )
{
  gchar *object_ids[] = { MORSE_WINDOW_IDS };
  gchar morse_glade[FILE_NAME_SIZE];

  Strlcpy( morse_glade, getenv("HOME"), sizeof(morse_glade) );
  Strlcat( morse_glade, MORSE_GLADE_FILE, sizeof(morse_glade) );
  Gtk_Builder( builder, morse_glade, object_ids );
  morse_gui.window = Builder_Get_Object( morse_gui.window_builder, "morse_window" );

  // Get Rx text buffer and scroller
  morse_gui.text_view = GTK_TEXT_VIEW(
      Builder_Get_Object(morse_gui.window_builder, "morse_textview") );
  morse_gui.text_buffer = gtk_text_view_get_buffer( morse_gui.text_view );

  // Get widgets and spin buttons
  morse_gui.scope = Builder_Get_Object( morse_gui.window_builder, "morse_scope_drawingarea" );
  morse_wfall.canvas = Builder_Get_Object( morse_gui.window_builder, "morse_waterfall_drawingarea" );
  morse_gui.scope_label = Builder_Get_Object( morse_gui.window_builder, "morse_scope_label" );
  morse_gui.speed_spinbtn = GTK_SPIN_BUTTON( Builder_Get_Object(morse_gui.window_builder, "wpm_spinbutton") );
  morse_gui.squelch_spinbtn = GTK_SPIN_BUTTON( Builder_Get_Object(morse_gui.window_builder, "morse_squelch_spinbutton") );
  morse_gui.ratio_spinbtn = GTK_SPIN_BUTTON( Builder_Get_Object( morse_gui.window_builder, "ratio_spinbutton") );

  // Initialize some values
  morse_rc.det_squelch =
    (uint16_t)gtk_spin_button_get_value_as_int( morse_gui.squelch_spinbtn );
  morse_rc.det_ratio   =
    gtk_spin_button_get_value( morse_gui.ratio_spinbtn );
  morse_rc.speed_wpm   =
    (uint16_t)gtk_spin_button_get_value_as_int( morse_gui.speed_spinbtn );
  Flag[MORSE_SELECT_LEVEL]   = True;
  Flag[MORSE_DISPLAY_SIGNAL] = True;
  morse_rc.unit_elem = 15;

  // Set the position of Morse window
  Set_Window_Geometry(
      morse_gui.window, morse_gui.window_x, morse_gui.window_y, 0, 0 );
  gtk_widget_show( morse_gui.window );

  // Load runtime config file, abort on error
  g_idle_add( Morse_Read_Config, NULL );

  return( morse_gui.window );
} // Create_Morse_Window()

//------------------------------------------------------------------

/* Morse_Waterfall_Configure_Event()
 *
 * Handles the "configure" event on the waterfall
 */
  void
Morse_Waterfall_Configure_Event( uint16_t width, uint16_t height )
{
  uint16_t idx;

  // Destroy existing pixbuff
  if( morse_wfall.pixbuf != NULL )
  {
    g_object_unref( G_OBJECT(morse_wfall.pixbuf) );
    morse_wfall.pixbuf = NULL;
  }

  // Create waterfall pixbuf
  morse_wfall.pixbuf =
    gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8, width, height );
  if( morse_wfall.pixbuf == NULL )
  {
    Error_Dialog( _("Failed to create pixbuf for waterfall"), HIDE_OK );
    return;
  }

  morse_wfall.pixels = gdk_pixbuf_get_pixels( morse_wfall.pixbuf );
  morse_wfall.width  = (uint16_t)gdk_pixbuf_get_width( morse_wfall.pixbuf );
  morse_wfall.height = (uint16_t)gdk_pixbuf_get_height( morse_wfall.pixbuf );
  morse_wfall.rowstride  = (uint16_t)gdk_pixbuf_get_rowstride( morse_wfall.pixbuf );
  morse_wfall.n_channels = (uint8_t)gdk_pixbuf_get_n_channels( morse_wfall.pixbuf );
  gdk_pixbuf_fill( morse_wfall.pixbuf, 0 );

  /* Initialize ifft. The waterfall width has to be odd
   * to provide a center line and the FFT width has to
   * be a power of 2 so we compensate by adding 1 */
  morse_ifft_data.ifft_width = morse_wfall.width + 1;
  if( !Initialize_iFFT(&morse_ifft_data) )
    return;

  // Puts the tone freq in middle of waterfall
  idx = (2 * morse_ifft_data.data_len) / morse_wfall.width;

  /* The tone freq must be rounded so that the ifft_stride
   * is an integer otherwise the waterfall is not accurate */
  if( idx )
  {
    morse_ifft_data.ifft_stride = INPUT_SAMPLE_RATE / morse_rc.tone_freq / idx;
    morse_rc.tone_freq = INPUT_SAMPLE_RATE / morse_ifft_data.ifft_stride / idx;
  }

  // Calculate parameters that depend on above
  morse_rc.max_unit = (60 * morse_rc.tone_freq) /
    (50 * CYCLES_PER_FRAG * morse_rc.min_wpm);
  morse_rc.max_unit_x2 = morse_rc.max_unit * 2;
  morse_rc.min_unit = (60 * morse_rc.tone_freq) /
    (50 * CYCLES_PER_FRAG * morse_rc.max_wpm);

} // Morse_Waterfall_Configure_Event()

//------------------------------------------------------------------------

