/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "detect.h"
#include "decode.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/ifft.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../hpsdr/settings.h"
#include <gtk/gtk.h>
#include <cairo/cairo.h>
#include <stdint.h>
#include <string.h>

//------------------------------------------------------------------------

// Some colors used in plotting
#define WHITE_FOREGND       1.0, 1.0, 1.0
#define CAIRO_LINE_WIDTH    1.5

//----------------------------------------------------------------------

/* Morse_Display_Detector()
 *
 * Updates the Detector scope display
 */
  void
Morse_Display_Detector( cairo_t *cr, GdkPoint *points )
{
  // Draw scope background
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_rectangle(
      cr, 0.0, 0.0,
      (double)morse_gui.scope_width  + 2.0,
      (double)morse_gui.scope_height + 2.0 );
  cairo_fill( cr );

  // Plot signal graph
  cairo_set_source_rgb( cr, SCOPE_FOREGND );
  cairo_set_line_width( cr, CAIRO_LINE_WIDTH );
  cairo_move_to( cr, (double)points[0].x, (double)points[0].y );
  for( uint16_t idx = 1; idx < morse_gui.scope_width; idx++ )
    cairo_line_to( cr, (double)points[idx].x, (double)points[idx].y );

  // Stroke paths
  cairo_stroke( cr );

  // Draw Mark/Space threshold references
  cairo_set_source_rgb( cr, WHITE_FOREGND );
  uint16_t temp = morse_gui.scope_height - morse_rc.det_threshold;
  cairo_move_to( cr, 0.0, (double)temp );
  cairo_line_to( cr, (double)morse_gui.scope_width, (double)temp );

  // Stroke paths
  cairo_stroke( cr );

} // Morse_Display_Detector()

//------------------------------------------------------------------------

/* Morse_Display_Signal()
 *
 * Updates the Signal scope display
 */
  void
Morse_Display_Signal( cairo_t *cr, GdkPoint *points )
{
  // Draw scope background
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_rectangle(
      cr, 0.0, 0.0,
      (double)morse_gui.scope_width  + 2.0,
      (double)morse_gui.scope_height + 2.0 );
  cairo_fill( cr );

  // Plot signal graph
  cairo_set_source_rgb( cr, SCOPE_FOREGND );
  cairo_set_line_width( cr, CAIRO_LINE_WIDTH );
  cairo_move_to( cr, (double)points[0].x, (double)points[0].y );
  for( uint16_t idx = 1; idx < morse_gui.scope_width; idx++ )
    cairo_line_to( cr, (double)points[idx].x, (double)points[idx].y );

  // Stroke paths
  cairo_stroke( cr );

} // Morse_Display_Signal

//------------------------------------------------------------------------

/* Morse_Display_Waterfall()
 *
 * Displays audio spectrum as "waterfall"
 */
  void
Morse_Display_Waterfall( void )
{
  uint16_t
    idh, idv,   // Index to hor. and vert. position in warterfall
    idf,        // Index to ifft output array
    idb,        // Index to average bin values array
    temp;

  // Constant needed to draw center line in waterfall
  uint16_t center_line = (uint16_t)morse_wfall.width / 2;

  // Pointer to current pixel
  static guchar *pix;

  // Draw a vertical white line in waterfall at detector's frequency
  pix    = morse_wfall.pixels + morse_wfall.rowstride + morse_wfall.n_channels;
  pix   += morse_wfall.n_channels * center_line;
  pix[0] = pix[1] = pix[2] = 0xff;

  // Copy each line of waterfall to next one
  temp = (uint16_t)morse_wfall.height - 2;
  for( idv = temp; idv > 0; idv-- )
  {
    pix = morse_wfall.pixels + morse_wfall.rowstride * idv + morse_wfall.n_channels;

    for( idh = 0; idh < morse_wfall.width; idh++ )
    {
      pix[0] = pix[- morse_wfall.rowstride];
      pix[1] = pix[1 - morse_wfall.rowstride];
      pix[2] = pix[2 - morse_wfall.rowstride];
      pix   += morse_wfall.n_channels;
    }
  }

  // Go to top left +1 horirzontally +1 vertically of pixbuf
  pix = morse_wfall.pixels + morse_wfall.rowstride;

  // First column of display (DC) not used
  morse_ifft_data.bin_ave[0] = 0;

  // Do the iFFT on input array
  iFFT_Real( &morse_ifft_data );
  idb = 0;
  for( idf = 2; idf < morse_ifft_data.data_len; idf += 2 )
  {
    // Greyscale value of pixel derived from ifft o/p
    uint16_t pixel_val;

    // Calculate signal power at each frequency (bin)
    pixel_val = iFFT_Bin_Value( &morse_ifft_data, idf, False );

    // Calculate average bin values
    morse_ifft_data.bin_ave[idb++] = pixel_val;

    // Color code signal strength
    Colorize( pix, (uint8_t)pixel_val );
    pix += morse_wfall.n_channels;

  } // for( idf = 2; idf < ifft_data_length; idf += 2 )

  // Reset function
  iFFT_Bin_Value( &morse_ifft_data, 0, True );

  // At last draw waterfall
  Queue_Draw( morse_wfall.canvas );

} // Morse_Display_Waterfall()

//------------------------------------------------------------------------

/* _Morse_Print_Character()
 *
 * Idle callback to prints characters to text view port
 */
  static gboolean
_Morse_Print_Character( gpointer data )
{
  // Text buffer marker
  static GtkTextIter iter;

  // Print character
  gtk_text_buffer_get_iter_at_offset( morse_gui.text_buffer, &iter,
      gtk_text_buffer_get_char_count(morse_gui.text_buffer) );
  gtk_text_buffer_insert( morse_gui.text_buffer, &iter, (gchar *)data, -1 );

  // Scroll Text View to bottom on change of page size
  gtk_text_view_scroll_mark_onscreen( morse_gui.text_view,
      gtk_text_buffer_create_mark(morse_gui.text_buffer, "end", &iter, FALSE) );

  return( FALSE );
} // _Morse_Print_Character()

//------------------------------------------------------------------------

/* Morse_Print_Character()
 *
 * Prints decoded Morse characters using the idle callback above
 */
  void *
Morse_Print_Character( void *data )
{
  // Decoded Morse character
  static gchar dec_char[MAX_CHAR_LENGTH];

  // Character ready flag
  static BOOLEAN ready = False;

  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Reconfigure on BFO setting change
  if( morse_rc.tone_freq != TRx->rx_weaver_frequency )
  {
    morse_rc.tone_freq = (uint16_t)TRx->rx_weaver_frequency;

    // Puts the tone freq in middle of waterfall
    uint16_t idx = ( 2 * morse_ifft_data.data_len ) / (uint16_t)morse_wfall.width;

    /* The tone freq must be rounded so that the ifft_stride
     * is an integer otherwise the waterfall is not accurate */
    if( idx )
    {
      morse_ifft_data.ifft_stride = INPUT_SAMPLE_RATE / morse_rc.tone_freq / idx;
      morse_rc.tone_freq = INPUT_SAMPLE_RATE / morse_ifft_data.ifft_stride / idx;
    }

    // Calculate parameters that depend on above
    morse_rc.max_unit    = (60 * morse_rc.tone_freq) / (50 * CYCLES_PER_FRAG * morse_rc.min_wpm);
    morse_rc.max_unit_x2 = morse_rc.max_unit * 2;
    morse_rc.min_unit    = (60 * morse_rc.tone_freq) / (50 * CYCLES_PER_FRAG * morse_rc.max_wpm);
    morse_rc.unit_elem   = (60 * morse_rc.tone_freq) / (50 * CYCLES_PER_FRAG * morse_rc.speed_wpm);

    if( !Morse_Initialize_Detector() ) return( NULL );
  } // if( tone_freq != morse_rc.tone_freq )

  Flag[GUEST_RECEIVING] = True;
  // Loop and print decoded characters
  dec_char[0] = '\0';
  while( !Flag[GUEST_QUIT] && Morse_Get_Character(dec_char, &ready) )
  {
    // Return on 0 character length
    if( !ready ) continue;
    g_idle_add( _Morse_Print_Character, (gpointer)dec_char );
    while( g_main_context_iteration(NULL, FALSE) );
    ready = False;
  } // while( Morse_Get_Character(&dec_char) ... )

  // Destroy digimode semaphore
  Init_Semaphore( &digimode_semaphore, False );
  Flag[GUEST_RECEIVING] = False;
  return( NULL );

} // Morse_Print_Character()

//------------------------------------------------------------------------

