/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "detect.h"
#include "display.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/ifft.h"
#include "../common/utils.h"
#include <stdint.h>
#include <stdlib.h>
#include <semaphore.h>
#include <math.h>
#include <gtk/gtk.h>

//------------------------------------------------------------------------

#define MORSE_SIGNAL_SCALE   110     // Scale factor to fit signal in scope
#define SQUELCH_MULTIPLIER   150     // Multiplier for the squelch threshold
#define RATIO_MULTIPLIER     20      // Multiplier for the ratio threshold

//------------------------------------------------------------------------

// Buffers and data needed by tone detector
static struct
{
  uint16_t
    frag_len,         // Length of fragment in samples
    samples_buff_len; // Length of DSP samples buffer

  // Sample buffer index
  int16_t samples_buff_idx;

  // Circular signal samples buffer for Goertzel detector
  int16_t *samples_buff;

  // Variables for the Goertzel algorithm
  double coeff;

  // Circular signal level buffer and index
  uint16_t *sig_level_buff;
  uint16_t  sig_level_idx;

  uint8_t *state, state_idx;

} detector_data;

//------------------------------------------------------------------------

static uint16_t
  frag_level,   // Level of audio signal over a fragment
  det_ratio;    // Ratio of trailing edge / leading edge

//------------------------------------------------------------------------

// Sum of trailing edge+leading edge, used as squelch
static uint32_t level_sum;

//------------------------------------------------------------------------

// Points to plot on scope
static GdkPoint *points = NULL;

/* Morse_Get_Fragment()
 *
 * Detects the cw beat frequency signal o/p from the radio
 * receiver and determines the status, (mark or space) of a
 * 'fragment' (a small fraction, ~1/8) of a Morse code element.
 * Signal detection is done by using a Goertzel algorithm.
 */
  BOOLEAN
Morse_Get_Fragment( void )
{
  uint32_t
    lead_edge,  // Level of signal's leading edge integrator
    trail_edge; // Level of signal's trailing edge integrator

  uint16_t
    ref_ratio,  // Reference for above to determine Mark | Space
    block_size, // Block size (N) of the Goertzel algorithm
    state_cnt,  // Count of the detected signal state above
    plot = 0,   // Value to be plotted in xcope
    idx;

  int16_t ids;

  // Wait for sample buffers full
  static BOOLEAN wait = True;

  // Current state of signal
  static uint8_t state;

  // Variables for the Goertzel algorithm
  double q1, q2, level;

  // Counter for timing duration of fragment
  static uint16_t frag_timer = 0;

  // Index to digimode samples buffer currently in use
  static uint8_t digi_buf_cnt = 0;


  // Initialize on first call
  if( points == NULL )
  {
    Mem_Alloc( (void **) &points, (size_t)morse_gui.scope_width * sizeof(GdkPoint) );
    digi_buf_cnt = 0;
  }

  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return( False );

  // Goertzel block size depends on Morse speed
  block_size = detector_data.frag_len * morse_rc.unit_elem / 2;

  // Wait for digimode buffer to fill with audio samples from demodulator.
  if( wait )
  {
    sem_wait( &digimode_semaphore );
    wait = False;
  }

  // Buffer dsp samples of input signal for a fragment
  while( frag_timer < detector_data.frag_len )
  {
    static uint32_t digi_buf_idx = 0;

    // Fill the detector buffer from the audio samples buffer
    detector_data.samples_buff[detector_data.samples_buff_idx] =
      digimode_buffer[digi_buf_cnt][digi_buf_idx];

    // Advance/Reset circular buffers' index
    detector_data.samples_buff_idx++;
    if( detector_data.samples_buff_idx >= detector_data.samples_buff_len )
      detector_data.samples_buff_idx = 0;

    // Save the sample to iFFT buffer
    if( iFFT_Data(digimode_buffer[digi_buf_cnt][digi_buf_idx], &morse_ifft_data) )
      Morse_Display_Waterfall();
    digi_buf_idx++;

    // Wait for digimode buffer to be filled
    if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
    {
      digi_buf_idx = 0;

      // Increment current audio sample output buffer index
      digi_buf_cnt++;
      if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
        digi_buf_cnt = 0;

      /* If sync between ring buffer input and output indices is lost,
       * sem_wait is not enabled, to allow the input index to catch up */
      if( digi_buf_cnt == digi_buf_input )
      {
        wait  = True;
        return( True );
      }
    } // if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )

    frag_timer++;
  } // while( frag_timer < frag_len )
  frag_timer = 0;

  // Calculate signal fragment level over a block
  // Backstep buffer index for use of samples
  detector_data.samples_buff_idx -= block_size;
  if( detector_data.samples_buff_idx < 0 )
    detector_data.samples_buff_idx += detector_data.samples_buff_len;

  // Calculate fragment level using Goertzel algorithm
  q1 = q2 = 0.0;
  for( idx = 0; idx < block_size; idx++ )
  {
    double q0 = detector_data.coeff * q1 - q2 +
      (double)detector_data.samples_buff[detector_data.samples_buff_idx];
    q2 = q1;
    q1 = q0;

    // Advance/Reset circular buffers' index
    detector_data.samples_buff_idx++;
    if( detector_data.samples_buff_idx >= detector_data.samples_buff_len )
      detector_data.samples_buff_idx = 0;
  } // for( idx = 0; idx < block_size; idx++ )

  // Scalar magnitude of input signal scaled by block size
  q1 /= (double)block_size;
  q2 /= (double)block_size;
  level = q1 * q1 + q2 * q2 - q1 * q2 * detector_data.coeff;
  double sq = sqrt( level );
  frag_level = (uint16_t)sq;

  // Advance/Reset circular buffers' index
  detector_data.sig_level_idx++;
  if( detector_data.sig_level_idx >= morse_rc.max_unit_x2 )
    detector_data.sig_level_idx = 0;

  // Save signal power level to circular buffer
  detector_data.sig_level_buff[detector_data.sig_level_idx] = frag_level;

  // Summate the level of leading edge of signal
  ids = (int16_t)detector_data.sig_level_idx;
  lead_edge = 0;
  for( idx = 0; idx < morse_rc.unit_elem; idx++ )
  {
    lead_edge += detector_data.sig_level_buff[ids];
    ids--;
    if( ids < 0 ) ids += morse_rc.max_unit_x2;
  }

  // Summate the level of trailing edge of signal
  trail_edge = 0;
  for( idx = 0; idx < morse_rc.unit_elem; idx++ )
  {
    trail_edge += detector_data.sig_level_buff[ids];
    ids--;
    if( ids < 0 ) ids += morse_rc.max_unit_x2;
  }

  // Scale lead and trail sums by length of unit element
  if( morse_rc.unit_elem )
  {
    lead_edge  /= morse_rc.unit_elem;
    trail_edge /= morse_rc.unit_elem;
  }

  // Set signal tone detector status
  level_sum = lead_edge + trail_edge;
  ref_ratio = (uint16_t)( morse_rc.det_ratio * RATIO_MULTIPLIER );

  // Advance/Reset detector state buffer index
  detector_data.state_idx++;
  if( detector_data.state_idx >= morse_rc.max_unit )
    detector_data.state_idx = 0;

  /* Determine state of signal, Mark | Space, if sum is above
   * squelch level. A mark state is denoted by a value of 20,
   * as further below this effectively multiplies the sum of
   * mark states by 20 and simplifies thresholding further below */
  if( level_sum > morse_rc.det_squelch * SQUELCH_MULTIPLIER )
  {
    if( ((int)RATIO_MULTIPLIER) * lead_edge > ref_ratio * trail_edge )
      state = 20; // State is a Mark, e.g. a "Dit" or "Dah" tone
    else if( ((int)RATIO_MULTIPLIER) * trail_edge > ref_ratio * lead_edge )
      state = 0; // State is a Space, e.g. no tone
  } // if( (lead_edge + trail_edge) > LEVEL_THRESHOLD )

  // Save Mark | Space detector state
  detector_data.state[detector_data.state_idx] = state;

  /* Summate detector states for the
   * duration of a unit element (dit) */
  ids = detector_data.state_idx;
  state_cnt = 0;
  for( idx = 0; idx < morse_rc.unit_elem; idx++ )
  {
    state_cnt += detector_data.state[ids];
    ids--;
    if( ids < 0 ) ids += morse_rc.max_unit;
  }

  /* If state count is > 11/20 of unit element, call it a
   * Mark. state is already * 20 at detector further above.
   * If state count is < 9/20 of unit element, call it a
   * Space. This is sort of a thresholding to combat noise */
  if( (int)state_cnt > (11 * morse_rc.unit_elem) )
  {
    Flag[MORSE_MARK_TONE]  = True;
    Flag[MORSE_SPACE_TONE] = False;
  }
  else if( (int)state_cnt < (9 * morse_rc.unit_elem) )
  {
    Flag[MORSE_MARK_TONE]  = False;
    Flag[MORSE_SPACE_TONE] = True;
  }

  // Lead/Trail ratio for the Scope display
  det_ratio = 1;
  if( (lead_edge > trail_edge) && trail_edge )
  {
    det_ratio =
      (uint16_t)( (RATIO_MULTIPLIER * lead_edge) / trail_edge );
  }
  else if( lead_edge )
  {
    det_ratio =
      (uint16_t)( (RATIO_MULTIPLIER * trail_edge) / lead_edge );
  }

  // Display signal or detector graph (scope)
  if( Flag[MORSE_DISPLAY_RATIO] )
  {
    morse_rc.det_threshold = (uint16_t)( morse_rc.det_ratio * RATIO_MULTIPLIER );
    plot = det_ratio;
  }
  else if( Flag[MORSE_DISPLAY_LEVEL] )
  {
    morse_rc.det_threshold = morse_rc.det_squelch;
    plot = (uint16_t)( level_sum / SQUELCH_MULTIPLIER );
  }
  else if( Flag[MORSE_DISPLAY_SIGNAL] )
    plot = frag_level / MORSE_SIGNAL_SCALE;

  if( !Flag[MORSE_SCOPE_HOLD] )
  {
    // Index to points to plot on scope
    static uint16_t points_idx = 0;

    // Save values to be plotted
    points[points_idx].y = morse_gui.scope_height - plot - 1;
    if( points[points_idx].y <= 0 )
      points[points_idx].y = 1;
    if( points[points_idx].y >= morse_gui.scope_height )
      points[points_idx].y = morse_gui.scope_height - 1;
    points[points_idx].x = points_idx;

    // Recycle buffer idx when full and plot
    points_idx++;
    if( points_idx >= morse_gui.scope_width )
    {
      // Draw scope
      Flag[MORSE_SCOPE_READY] = True;
      Queue_Draw( morse_gui.scope );
      points_idx = 0;
    }
  } // if( !Flag[MORSE_SCOPE_HOLD] )

  return( True );
} // Morse_Get_Fragment()

//------------------------------------------------------------------------

/* Morse_Display_Scope()
 *
 * Displays graphs in the scope display on "draw" signal
 */
  void
Morse_Display_Scope( cairo_t *cr )
{
  if( Flag[MORSE_SCOPE_READY] )
  {
    // Display signal or detector graph (scope)
    if( Flag[MORSE_DISPLAY_RATIO] ||
        Flag[MORSE_DISPLAY_LEVEL] )
      Morse_Display_Detector( cr, points );
    else if( Flag[MORSE_DISPLAY_SIGNAL] )
      Morse_Display_Signal( cr, points );
    Flag[MORSE_SCOPE_READY] = False;
  }
  else // Clear scope
  {
    // Draw scope background
    cairo_set_source_rgb( cr, SCOPE_BACKGND );
    cairo_rectangle(
        cr, 0.0, 0.0,
        (double)morse_gui.scope_width  + 2.0,
        (double)morse_gui.scope_height + 2.0 );
    cairo_fill( cr );
  }

} // Morse_Display_Scope()

//------------------------------------------------------------------------

/* Morse_Initialize_Detector()
 *
 * Initializes variables and allocates buffers
 * for the Goertzel tone detector
 */
  BOOLEAN
Morse_Initialize_Detector( void )
{
  size_t alloc;

  // Goertzel detector coefficients
  double w = M_2PI * (double)morse_rc.tone_freq / INPUT_SAMPLE_RATE;
  detector_data.coeff = 2.0 * cos( w );

  // Clear buffer indices
  detector_data.samples_buff_idx = 0;
  detector_data.sig_level_idx = 0;
  detector_data.state_idx = 0;

  // Length of a signal 'fragment', a small part of one Morse element
  detector_data.frag_len = (uint16_t)( (INPUT_SAMPLE_RATE * CYCLES_PER_FRAG) / (int)morse_rc.tone_freq );

  // Length of DSP samples buffer for Goertzel detector
  detector_data.samples_buff_len = detector_data.frag_len * morse_rc.max_unit;

  // Allocate samples buffer
  alloc = (size_t)detector_data.samples_buff_len * sizeof(uint16_t);
  Mem_Alloc( (void **) &detector_data.samples_buff, alloc );

  // Clear buffer
  memset( detector_data.samples_buff, 0, alloc );

  // Allocate signal level (Goerstzel detector o/p) buffer
  alloc = (size_t)morse_rc.max_unit_x2 * sizeof(uint16_t);
  Mem_Alloc( (void **) &detector_data.sig_level_buff, alloc );

  // Clear buffer
  memset( detector_data.sig_level_buff, 0, alloc );

  // Allocate signal level (Goerstzel detector o/p) buffer
  alloc = (size_t)morse_rc.max_unit * sizeof(uint8_t);
  Mem_Alloc( (void **) &detector_data.state, alloc );

  // Clear buffer
  memset( detector_data.state, 0, alloc );

  return( True );
} // Morse_Initialize_Detector()

//------------------------------------------------------------------------

// Free resources
  void
Free_Morse_Detector( void )
{
  Mem_Free( (void **) &points );
  Mem_Free( (void **) &detector_data.samples_buff );
  Mem_Free( (void **) &detector_data.sig_level_buff );
  Mem_Free( (void **) &detector_data.state );
  Free_iFFT( &morse_ifft_data );
}

//------------------------------------------------------------------------

