/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "decode.h"
#include "detect.h"
#include "shared.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/interface.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include <gtk/gtk.h>
#include <stdint.h>

//------------------------------------------------------------------------

static uint16_t
  space_elem_cnt = 0, // Number of space elements processed
  space_frag_cnt = 0, // Number of space fragments processed
  mark_elem_cnt  = 0, // Number of mark elements processed
  mark_frag_cnt  = 0; // Number of mark fragments processed

//------------------------------------------------------------------------

// ASCII equivalents to Morse hex code. Last
// one (*) used for unrecognized characters.
#define MORSE_ASCII_CHAR \
  "A","B","C","D","E","F","G","H","I","J","K","L",  \
  "M","N","O","P","Q","R","S","T","U","V","W","X",  \
  "Y","Z","1","2","3","4","5","6","7","8","9","0",  \
  ".",",","?","\"","!","/","(",")","&",":",";","=", \
  "+","-","_","\"","$","@","<OK>","<GO>","<BK>","<SK>", \
  "\u00c4","\u00d6","\u00dc","\u00df","<!!>"," ","*"

// Hex equivalents to Morse code chars above except (*).
// Formed by starting with a 1 and following with a 0 for
// dash and 1 for dit e.g: B = dahdididit = 10111 = Hex 0x17
#define MORSE_HEX_CODE \
  0x06,0x17,0x15,0x0b,0x03,0x1d,0x09,0x1f,0x07,0x18,0x0a,0x1b, \
  0x04,0x05,0x08,0x19,0x12,0x0d,0x0f,0x02,0x0e,0x1e,0x0c,0x16, \
  0x14,0x13,0x30,0x38,0x3c,0x3e,0x3f,0x2f,0x27,0x23,0x21,0x20, \
  0x6a,0x4c,0x73,0x61,0x54,0x2d,0x29,0x52,0x37,0x47,0x55,0x2e, \
  0x35,0x5e,0x72,0x6d,0xf6,0x65,0x3d,0x2a,0xba,0x7a,0x1a,0x11, \
  0x1c,0xf3,0xff,0x01

//----------------------------------------------------------------------

// Definitions of contexts (stages) in Morse decoding process
#define NO_CONTEXT         0        // Context is not defined
#define CNT_MARK_SIGNAL    0x000001 // Count fragments of a mark element
#define CNT_ELEM_SPACE     0x000002 // Count frag. of inter-element space
#define MORSE_CHAR_SPACE   0x000004 // Count fragments of inter-char space
#define WAIT_WORD_SPACE    0x000008 // Wait for an inter-word space
#define CNT_WORD_SPACE     0x000010 // Count fragments of inter-word space
#define WAIT_FOR_MARK      0x000020 // Count fragments of no-signal space
#define WAIT_FOR_SPACE     0x000040 // Wait for a space after a long dash
#define LONG_SPACE         0x000100 // Long period of space (no tone)
#define ENTER_DOT          0x01     // Enter a dot element into Morse decode

//----------------------------------------------------------------------

// Idle callback to set the speed spin button
  static gboolean
Set_Spinbutton( gpointer data )
{
  gtk_spin_button_set_value(
      morse_gui.speed_spinbtn, (gdouble)(*((uint8_t *)data)) );
  return( FALSE );
}

/* Adapt_Decoder()
 *
 * Adjusts Morse speed from measurements on the incoming signal
 */
  static void
Adapt_Decoder( void )
{
  // Calculate Morse speed
  if( mark_elem_cnt  &&
      mark_frag_cnt  &&
      space_elem_cnt &&
      space_frag_cnt )
  {
    if( Flag[MORSE_ADAPT_SPEED] )
    {
      // Estimate Morse speed from space and mark counts
      int16_t ratio =
        (int16_t)( mark_frag_cnt + space_frag_cnt ) /
        (int16_t)( mark_elem_cnt + space_elem_cnt );

      int16_t speed_err = ratio - (int16_t)morse_rc.unit_elem;

      // Morse speed limits (60-6 wpm)
      if( (morse_rc.unit_elem > morse_rc.min_unit) && (speed_err < 0) )
        morse_rc.unit_elem--;
      if( (morse_rc.unit_elem < morse_rc.max_unit) && (speed_err > 0) )
        morse_rc.unit_elem++;

      // Display speed in wpm
      static uint8_t ispeed;
      ispeed = (uint8_t)
        ( (60 * morse_rc.tone_freq) / (50 * CYCLES_PER_FRAG * morse_rc.unit_elem) );
      Set_Spinbutton( (gpointer)&ispeed );
      while( g_main_context_iteration(NULL, FALSE) );

    } // if( Flag[ADAPT_SPEED] )

  } // if( mark_elem_cnt && space_elem_cnt && space_frag_cnt )

  // Clear counters
  space_elem_cnt = space_frag_cnt = 0;
  mark_elem_cnt  = mark_frag_cnt  = 0;

} // Adapt_Decoder()

//------------------------------------------------------------------------

/* Hex_to_Ascii()
 *
 * Converts the hex equivalent of
 * a Morse code character to ASCII
 */
  static const char *
Hex_to_Ascii( uint16_t *hex_code )
{
  // Table of ASCII characters available in Morse code
  static const char *
    morse_ascii_char[NUMBER_OF_CHAR + 1] = { MORSE_ASCII_CHAR };

  // Table of hex equivalent of Morse characters
  static const unsigned char
    morse_hex_char[NUMBER_OF_CHAR] = { MORSE_HEX_CODE };

  uint8_t idx; // Loop index

  // Look for a match in hex table
  for( idx = 0; idx < NUMBER_OF_CHAR; idx++ )
    if( *hex_code ==  morse_hex_char[idx] )
      break;

  // Clear hex code after conversion
  *hex_code = ENTER_DOT;

  // Return ascii equivalent of hex code
  return( morse_ascii_char[idx] );

} // Hex_to_Ascii()

//------------------------------------------------------------------------

/* Morse_Get_Character()
 *
 * Decodes a Morse code character from the
 * sequence of marks (dits and dahs) and spaces
 */
  BOOLEAN
Morse_Get_Character( char *chr, BOOLEAN *ready )
{
  static uint16_t
    mark_cnt  = 0,          // Count of Mark fragments detected
    space_cnt = 0,          // Count of Space fragments detected
    context   = NO_CONTEXT, // What context Morse decoding is in
    hex_code  = ENTER_DOT;  // Hex equivalent of Morse character

  uint16_t
    unit_d2  = morse_rc.unit_elem / 2,
    unit_x2  = morse_rc.unit_elem * 2,
    unit_x5  = morse_rc.unit_elem * 5,
    unit_x7  = morse_rc.unit_elem * 7,
    unit_x8  = morse_rc.unit_elem * 8,
    unit_x16 = morse_rc.unit_elem * 16;

  /* Hex equivalent of Morse code is formed by left-shifting 1 or 0
   * into hex_code. The ENTER_DOT (0x01) initial value marks the
   * beginning of the bit field, 1 being a dit and 0 a dash. */

  // Get the level of a fragment from tone detector. A fragment
  // is a small fraction of a morse code element, there are from
  // 10 to 30 frags/element depending on Morse speed (30-10 w.p.m)
  if( !Morse_Get_Fragment() ) return( False );

  // Increment mark or space count
  if( Flag[MORSE_MARK_TONE] )
    mark_cnt++;
  else
    space_cnt++;

  // If a mark element is too long, limit count
  if( mark_cnt > unit_x8 )
    mark_cnt = unit_x8;

  // If a space element is too long, limit count
  if( space_cnt > unit_x16 )
    space_cnt = unit_x16;

  // Process mark and space element counts to decode Morse
  switch( context )
  {
    case CNT_MARK_SIGNAL: // Process mark element
      // If mark element is too long, reset and wait for space
      if( mark_cnt >= unit_x8 )
      {
        // Clear space counter
        space_cnt = 0;

        // Clear hex character code
        hex_code = ENTER_DOT;

        // Wait for a space fragment
        context = WAIT_FOR_SPACE;
      } // if( Flag[MORSE_MARK_TONE] )
      else if( Flag[MORSE_SPACE_TONE] ) // If fragment is a space
      {
        // Clear space count to 1
        space_cnt = 1;

        // Switch to processing inter-element space
        context = CNT_ELEM_SPACE;
      }
      break;

    case CNT_ELEM_SPACE: // Process inter-element space
      /* If space duration reaches 1/2 units or a mark
       * tone arrives, we have an inter-element space */
      if( space_cnt >= unit_d2 || Flag[MORSE_MARK_TONE] )
      {
        // If mark is < 2 units its a dit else a dash
        if( mark_cnt < unit_x2 )
        {
          // Insert dit and increment mark frag and elem count
          hex_code = (uint16_t)(hex_code << 1) | ENTER_DOT;
          mark_frag_cnt += mark_cnt;
          mark_elem_cnt += 1; // A dit is 1 element long
        }
        else
        {
          // Insert dash and increment mark frag and elem count
          hex_code <<= 1;
          mark_frag_cnt += mark_cnt;
          mark_elem_cnt += 3; // A dash is 3 elements long
        } // if( mark_cnt < morse_rc.unit_elem * 2 )

        // Wait for inter-char space count
        if( Flag[MORSE_SPACE_TONE] )
        {
          mark_cnt = 0;
          context  = MORSE_CHAR_SPACE;
        }
        else
        {
          space_cnt = 0;
          mark_cnt  = 1;
          context = CNT_MARK_SIGNAL;
        }
      } // if( space_cnt >= (morse_rc.unit_elem / 2)... )
      break;

    case MORSE_CHAR_SPACE: // Wait for inter-char space
      // If fragment is space
      if( Flag[MORSE_SPACE_TONE] )
      {
        // If space reaches 2 units its inter-character
        if( space_cnt >= unit_x2 )
        {
          // Switch to waiting for inter-word space
          context = WAIT_WORD_SPACE;

          // Return decoded Morse char
          const char *str = Hex_to_Ascii( &hex_code );
          Strlcpy( chr, str, MAX_CHAR_LENGTH );
          *ready = True;
          return( True );
        }
      }  // if( Flag[SPACE_TONE] )
      else // Its the end of inter-char space
      {
        // Count up space frags and elements
        space_frag_cnt += space_cnt;
        space_elem_cnt++; // Inter-element space

        // Clear space cnt and process marks
        space_cnt = 0;
        mark_cnt  = 1;
        context = CNT_MARK_SIGNAL;
      }
      break;

    case WAIT_WORD_SPACE: // Wait for an inter-word space
      // If fragment is space
      if( Flag[MORSE_SPACE_TONE] )
      {
        // If space count reaches 5, its word space
        if( space_cnt >= unit_x5 )
          context = CNT_WORD_SPACE;

      }  // if( Flag[SPACE_TONE] )
      else // Its the end of inter-character space
      {
        // Adapt to incoming signal
        Adapt_Decoder();

        // Switch to processing mark signal
        space_cnt = 0;
        mark_cnt  = 1;
        context = CNT_MARK_SIGNAL;
      }
      break;

    case CNT_WORD_SPACE: // Process Inter-word space
      // If fragment is space
      if( Flag[MORSE_SPACE_TONE] )
      {
        if( space_cnt >= unit_x7 )
        {
          Strlcpy( chr, " ", MAX_CHAR_LENGTH );
          context = WAIT_FOR_MARK;
          *ready = True;
          return( True );
        }
      } // if( Flag[SPACE_TONE] )
      else
      {
        // Adapt to incoming signal
        Adapt_Decoder();

        // Switch to processing mark signal
        space_cnt = 0;
        mark_cnt  = 1;
        Strlcpy( chr, " ", MAX_CHAR_LENGTH );
        context = CNT_MARK_SIGNAL;
        *ready = True;
        return( True );
      }
      break;

    case WAIT_FOR_MARK: // Process no-signal space
      // If fragment is mark, switch to processing marks
      if( Flag[MORSE_MARK_TONE] )
      {
        space_cnt = 0;
        context = CNT_MARK_SIGNAL;
      }
      break;

    case WAIT_FOR_SPACE: // Wait for space after long dash
      // If fragment is space, switch to counting space
      if( Flag[MORSE_SPACE_TONE] )
      {
        space_cnt = 1;
        mark_cnt  = 0;
        context = WAIT_FOR_MARK;
      }
      break;

    default: // Set context if none
      if( Flag[MORSE_MARK_TONE] )
        context = CNT_MARK_SIGNAL;
      else
        context = WAIT_FOR_MARK;

  } // End of switch( context )

  return( True );
} // Morse_Get_Character()

//------------------------------------------------------------------------

