/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "detect.h"
#include "display.h"
#include "interface.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/interface.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <semaphore.h>
#include <stdint.h>
#include <stdlib.h>

//------------------------------------------------------------------------

// Clean up Morse decoder
  static void
Morse_Cleanup( void )
{
  // Reset all flags
  Flag[MORSE_ADAPT_SPEED]    = False;
  Flag[MORSE_MAN_THRESHOLD]  = False;
  Flag[MORSE_DISPLAY_SIGNAL] = False;
  Flag[MORSE_DISPLAY_RATIO]  = False;
  Flag[MORSE_DISPLAY_LEVEL]  = False;
  Flag[MORSE_SCOPE_READY]    = False;
  Flag[MORSE_SCOPE_HOLD]     = False;
  Flag[MORSE_SELECT_LEVEL]   = False;
  Flag[MORSE_ENABLE_RECEIVE] = False;
  Flag[MORSE_RCCONFIG_SETUP] = False;
  Flag[MORSE_MARK_TONE]      = False;
  Flag[MORSE_SPACE_TONE]     = False;
  Flag[GUEST_RECEIVING]      = False;
  morse_gui.window = NULL;
  hermes2_gui.guest_window = NULL;
  Free_Morse_Detector();
  Init_Semaphore( &digimode_semaphore, False );
}

//------------------------------------------------------------------------

// All callback functions of the Morse decoder
  void
on_morse_window_destroy(
    GObject *object,
    gpointer user_data)
{
  g_object_unref( morse_gui.window_builder );
  Morse_Cleanup();
}

//------------------------------------------------------------------------

  gboolean
on_morse_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data)
{
  Save_Window_Position( morse_gui.window, morse_rc.rc_fpath );
  Flag[GUEST_QUIT] = True;

  // Wait for Rx mode to exit
  if( Flag[GUEST_RECEIVING] )
    pthread_join( hermes2_rc.guest_rx_thread, NULL );

  Guest_Quit_Activate( morse_gui.window );
  return( FALSE );
}

//------------------------------------------------------------------------

  gboolean
on_morse_scope_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer  user_data)
{
  Morse_Display_Scope( cr );
  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_morse_waterfall_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data)
{
  if( morse_wfall.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, morse_wfall.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
  }
  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_morse_waterfall_drawingarea_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data)
{
  if( event->button == 1 )
  {
    Tune_to_Monitor(
        event->x, CLICK_TUNE_RANGE, morse_wfall.width,
        morse_rc.tone_freq, &morse_ifft_data );
  }
  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_wpm_spinbutton_value_changed(
    GtkSpinButton *spin_button,
    gpointer       user_data)
{
  uint16_t val = (uint16_t)gtk_spin_button_get_value_as_int( spin_button );
  morse_rc.unit_elem = ( 60 * morse_rc.tone_freq ) / ( 50 * val * CYCLES_PER_FRAG );
}

//------------------------------------------------------------------------

  void
on_morse_squelch_spinbutton_value_changed(
    GtkSpinButton *spin_button,
    gpointer       user_data)
{
  morse_rc.det_squelch =
    (uint16_t)gtk_spin_button_get_value_as_int( spin_button );
}

//------------------------------------------------------------------------

  void
on_ratio_spinbutton_value_changed(
    GtkSpinButton *spin_button,
    gpointer       user_data)
{
  morse_rc.det_ratio =
    gtk_spin_button_get_value( spin_button );
}

//------------------------------------------------------------------------

  void
on_morse_clear_button_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_text_buffer_set_text( morse_gui.text_buffer, "", -1 );
}

//------------------------------------------------------------------------

  void
on_morse_auto_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
    Flag[MORSE_ADAPT_SPEED] = True;
  else
    Flag[MORSE_ADAPT_SPEED] = False;
}

//------------------------------------------------------------------------

  void
on_morse_waterfall_drawingarea_configure(
    GtkWidget    *widget,
    const GdkEventConfigure *event,
    gpointer     user_data )
{
  Morse_Waterfall_Configure_Event(
      (uint16_t)event->width, (uint16_t)event->height );
}

//------------------------------------------------------------------------

  void
on_morse_receive_togglebutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  if( ddv->transmit_on            ||
      !Flag[MORSE_RCCONFIG_SETUP] ||
      !Transceiver[Indices.TRx_Index]->receive_active )
    return;

  if( gtk_toggle_button_get_active(togglebutton) )
  {
    // Detector initialiazed OK
    Flag[GUEST_QUIT] = False;
    if( Flag[MORSE_ENABLE_RECEIVE] )
    {
      // Initialize digimode semaphore
      if( !Init_Semaphore(&digimode_semaphore, True) )
        return;

      // Start receiving and decoding signals
      if( !Pthread_Create(
            &hermes2_rc.guest_rx_thread, NULL, Morse_Print_Character, NULL,
            "Failed to create the Morse Decoder thread") )
        return;
     }
    else
      Error_Dialog( _("Failed to initialize signal detector"), SHOW_OK );
  } // if( gtk_toggle_button_get_active(togglebutton)
  else
  {
    if( Flag[MORSE_ENABLE_RECEIVE] )
    {
      // This flag only stops the decoder when set here
      Flag[GUEST_QUIT] = True;
    }
  } // else of if( gtk_toggle_button_get_active(togglebutton)
}

//------------------------------------------------------------------------

  void
on_ratio_radiobutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    gtk_label_set_text(
        GTK_LABEL(morse_gui.scope_label), _("Lead/Trail Ratio") );
    Flag[MORSE_SELECT_LEVEL]   = False;
    Flag[MORSE_DISPLAY_LEVEL]  = False;
    Flag[MORSE_DISPLAY_SIGNAL] = False;
    Flag[MORSE_DISPLAY_RATIO]  = True;
  }
}

//------------------------------------------------------------------------

  void
on_level_radiobutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    gtk_label_set_text( GTK_LABEL(morse_gui.scope_label), _("Squelch Level") );
    Flag[MORSE_SELECT_LEVEL]   = True;
    Flag[MORSE_DISPLAY_LEVEL]  = True;
    Flag[MORSE_DISPLAY_RATIO]  = False;
    Flag[MORSE_DISPLAY_SIGNAL] = False;
  }
}

//------------------------------------------------------------------------

  void
on_stop_radiobutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    gtk_label_set_text(
        GTK_LABEL(morse_gui.scope_label), _("Hold Display") );
    Flag[MORSE_DISPLAY_SIGNAL] = False;
    Flag[MORSE_SELECT_LEVEL]   = False;
    Flag[MORSE_DISPLAY_LEVEL]  = False;
    Flag[MORSE_DISPLAY_RATIO]  = False;
    Flag[MORSE_SCOPE_HOLD]     = True;
  }
  else Flag[MORSE_SCOPE_HOLD] = False;
}

//------------------------------------------------------------------------

  void
on_signal_radiobutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data)
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    gtk_label_set_text( GTK_LABEL(morse_gui.scope_label), _("Signal Detector") );
    Flag[MORSE_SELECT_LEVEL]   = False;
    Flag[MORSE_DISPLAY_LEVEL]  = False;
    Flag[MORSE_DISPLAY_RATIO]  = False;
    Flag[MORSE_DISPLAY_SIGNAL] = True;
  }
}

//------------------------------------------------------------------------

