/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "settings.h"
#include "discovery.h"
#include "pc_to_hw.h"
#include "../common/common.h"
#include "../common/convert.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../Hermes2/display.h"
#include "../Hermes2/interface.h"
#include "../Hermes2/sound.h"
#include <gtk/gtk.h>
#include <math.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

//----------------------------------------------------------------------

/* Hpsdr_Auto_LNA_Gain()
 *
 * Automatically controls ADC LNA Gain on overload
 */
  gboolean
Hpsdr_Auto_LNA_Gain( gpointer data )
{
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  const BOOLEAN *adc_ol = (BOOLEAN *)data;

  if( ddv->lna_gain_auto )
  {
    if( *adc_ol )
    {
      // Decrease attenuator setting if > -12 dB
      if( ddv->lna_gain > -12 ) ddv->lna_gain--;
    }
    else if( ddv->lna_gain < 48 ) // Gain setting < 48 dB
    {
      static uint8_t cnt = 0;

      // Increase setting gradually
      if( cnt++ >= 30 )
      {
        cnt = 0;
        ddv->lna_gain++;
       }
    }

    // Update attenuator spin button
    gtk_spin_button_set_value(
        GTK_SPIN_BUTTON(hermes2_gui.lna_gain_spinbutton), (gdouble)ddv->lna_gain );
  } // if( ddv->lna_auto_status )

  return( False );
} // Hpsdr_Auto_LNA_Gain()

//----------------------------------------------------------------------

/* Limit_Tx_Frequency()
 *
 * Limits the Transmitter frequency to within the Bands above
 */
  static void
Limit_Tx_Frequency( uint32_t *frequency )
{
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Keep the given frequency within the Bands
  switch( TRx->tx_bands_idx )
  {
    case IDX_GEN: case IDX_160:
      if( *frequency < LO_160 ) *frequency = LO_160;
      if( *frequency > HI_160 ) *frequency = HI_160;
      break;

    case IDX_80:
      if( *frequency < LO_80 ) *frequency = LO_80;
      if( *frequency > HI_80 ) *frequency = HI_80;
      break;

    case IDX_60:
      if( *frequency < LO_60 ) *frequency = LO_60;
      if( *frequency > HI_60 ) *frequency = HI_60;
      break;

    case IDX_40:
      if( *frequency < LO_40 ) *frequency = LO_40;
      if( *frequency > HI_40 ) *frequency = HI_40;
      break;

    case IDX_30:
      if( *frequency < LO_30 ) *frequency = LO_30;
      if( *frequency > HI_30 ) *frequency = HI_30;
      break;

    case IDX_20:
      if( *frequency < LO_20 ) *frequency = LO_20;
      if( *frequency > HI_20 ) *frequency = HI_20;
      break;

    case IDX_17:
      if( *frequency < LO_17 ) *frequency = LO_17;
      if( *frequency > HI_17 ) *frequency = HI_17;
      break;

    case IDX_15:
      if( *frequency < LO_15 ) *frequency = LO_15;
      if( *frequency > HI_15 ) *frequency = HI_15;
      break;

    case IDX_12:
      if( *frequency < LO_12 ) *frequency = LO_12;
      if( *frequency > HI_12 ) *frequency = HI_12;
      break;

    case IDX_10:
      if( *frequency < LO_10 ) *frequency = LO_10;
      if( *frequency > HI_10 ) *frequency = HI_10;
      break;

  } // switch( hermes2_rc.tx_bands_idx )

} // Limit_Tx_Frequency()

//----------------------------------------------------------------------

/* Hermes2_Set_Center_Frequency()
 *
 * Sets the Center Frequency and Filters of the Hermes Lite 2 device
 */
  void
Hermes2_Set_Center_Frequency( Transceiver_t *TRx, BOOLEAN rx_flag )
{
  uint32_t center_freq;
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  double corrected_freq;
  BOOLEAN ok = True;


  // If Tx Lock, abort if not changing Receiver frequency
  if( TRx->tx_freq_lock && !rx_flag )
    return;

  /* Offset center freq for SSB operation to take into account the
   * Weaver fold frequency offset and apply frequency correction */
  if( rx_flag )
  {
    // Enter DDC frequency, to be converted to phase word
    corrected_freq  = (double)TRx->rx_frequency + (double)TRx->rx_weaver_offset;
    corrected_freq *= ( 1.0 + hermes2_rc.device_freq_error / 1.0E6 );
  }
  else
  {
    // Limit Tx frequency within Amateur bands
    Limit_Tx_Frequency( &TRx->tx_frequency );

    // Enter DUC frequency, to be converted to phase word
    corrected_freq  = (double)TRx->tx_frequency + (double)TRx->tx_weaver_offset;
    corrected_freq *= ( 1.0 + hermes2_rc.device_freq_error / 1.0E6 );
  }

  // Display center frequency
  Update_Spin_Dial( TRx, rx_flag );

  // Reduce center frequency to 1 kHz steps
  center_freq = (uint32_t)( round(corrected_freq) ) / 1000;
  if( center_freq > ddv->frequency_max / 1000 )
    ok = False;

  /* Setup Filters if not in wideband Bandscope mode.
   * Clear all filter bits except 3MHz High Pass filter */
  ddv->filter_settings = HP_FILTER_3MHZ;
  if( !TRx->spectrum_data.wideband_spectrum )
  {
    // Setup Rx Filter settings according to frequency range
    if( center_freq < 3000 ) // Bypass 3.0 MHz HPF
    {
      ddv->filter_settings &= ~HP_FILTER_3MHZ;
    }

    if( center_freq < 2000 ) // Activate Top Band LPF
    {
      ddv->filter_settings |= LP_FILTER_160M;
    }
    else if( center_freq < 4000 ) // Activate 80m LPF
    {
      ddv->filter_settings |= LP_FILTER_80M;
    }
    else if( center_freq < 7500 ) // Activate 60m & 40m LPF
    {
      ddv->filter_settings |= LP_FILTER_60_40M;
    }
    else if( center_freq < 15000 ) // Activate 30m & 20m LPF
    {
      ddv->filter_settings |= LP_FILTER_30_20M;
    }
    else if( center_freq < 22000 ) // Activate 17m & 15m LPF
    {
      ddv->filter_settings |= LP_FILTER_17_15M;
    }
    else if( center_freq < 30000 ) // Activate 12m & 10m LPF
    {
      ddv->filter_settings |= LP_FILTER_12_10M;
    }
    else if( center_freq < ddv->frequency_max )
    {
      // Clear all filter selections. This leaves the 3.0MHz HPF only
      ddv->filter_settings = HP_FILTER_3MHZ;
    }
    else ok = False;

  } // if( !TRx->spectrum_data.wideband_status )
  else // Wideband Bandscope selected, disable filters
  {
    // Disables 3.0MHz HP filter as well as all LP filters
    ddv->filter_settings = 0x00;
  }

  if( Flag[HERMES2_INITIALIZED] )
  {
    // Prepare C&C buffer for sending Filter settings to HL2
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][2] = ddv->filter_settings;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][2] = ddv->filter_settings;
    Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
    Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

    // Prepare C&C buffer for sending Rx/Tx Frequency to HL2
    if( rx_flag )
    {
      // Enter RX1 frequency to C1-C4
      uint8_t cc_idx = 0x02 + TRx->index; // Rx address + TRx index
      Uint32_to_String( (uint32_t)corrected_freq, &Cmnd_Ctrl_Audio_IQ_Data.Cn1[cc_idx][1] );
      Uint32_to_String( (uint32_t)corrected_freq, &Cmnd_Ctrl_Audio_IQ_Data.Cn2[cc_idx][1] );
      Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = cc_idx;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = cc_idx;
      Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
    }
    else
    {
      // Enter TX frequency to C1-C4
      Uint32_to_String( (uint32_t)corrected_freq, &Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x01][1] );
      Uint32_to_String( (uint32_t)corrected_freq, &Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x01][1] );
      Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x01;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x01;
      Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
    }

    // Display status of frequency change
    if( ok )
    {
      if( rx_flag )
        TRx->set_rx_freq_status = True;  // Success
      else
        TRx->set_tx_freq_status = True;  // Success
    }
    else
    {
      if( rx_flag )
        TRx->set_rx_freq_status = False;  // Failure
      else
        TRx->set_tx_freq_status = False;  // Failure
    }

    gdk_threads_add_idle( Hermes2_Frequency_Status, (gpointer)TRx );

  } // if( Flag[HERMES2_INITIALIZED] )

} // Hermes2_Set_Center_Frequency()

//----------------------------------------------------------------------

/* Hpsdr_Set_Sample_Rate()
 *
 * Sets the Hpsdr Sample Rate
 */
  void
Hpsdr_Set_Sample_Rate( Transceiver_t *TRx )
{
  // Set C1 according to sample rate
  switch( hermes2_rc.ddc_sample_rate )
  {
    case 48000:
      Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][1] = 0x00;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][1] = 0x00;
      break;
    case 96000:
      Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][1] = 0x01;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][1] = 0x01;
      break;
    case 192000:
      Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][1] = 0x02;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][1] = 0x02;
      break;
    case 384000:
      Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][1] = 0x03;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][1] = 0x03;
      break;
  }

  // Set C&C C0 byte for Sample Rate (addr=0, C1[1:0] 00=48 01=96 10=192 11=384)
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

  // Calculate sound_decimate to give 48000 S/s to sound card
  TRx->sound_decimate = (uint16_t)( hermes2_rc.ddc_sample_rate / SND_DSP_RATE );

  return;
} // Hpsdr_Set_Sample_Rate()

//----------------------------------------------------------------------

/* Adc_Change_Antenna()
 *
 * Changes the ADC antenna. Apparently not applicable
 * to the Hermes Lite 2 that I have been working with ??
 */
  void
Adc_Change_Antenna( uint8_t ant )
{
  hermes2_rc.rx_antenna = ant;

  // Set C&C C0 byte for Rx Antenna selection
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

  if( ant == 1 )
  {
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][3] = 0x00;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][3] = 0x00;
  }
  else if( ant == 2 )
  {
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][3] = 0x20;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][3] = 0x20;
  }

} // Adc_Change_Antenna()

//----------------------------------------------------------------------

/* Hpsdr_Set_LNA_Gain()
 *
 * Changes an ADC's LNA Gain setting
 */
  void
Hpsdr_Set_LNA_Gain( int8_t gain )
{
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  ddv->lna_gain = gain;

  // Set C&C C0 byte for Direct LNA Gain control
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x0A][4]  = BIT_6; // Set Direct LNA Gain control
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x0A][4]  = BIT_6; // Set Direct LNA Gain control

  // Convert gain range -12 to 48dB to uint8_t range 0-63
  uint8_t setting = (uint8_t)( ((gain + 12) * 63) / 60 );
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x0A][4] |= setting;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x0A][4] |= setting;
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0A;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0A;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
} // Hpsdr_Set_LNA_Gain()

//----------------------------------------------------------------------

