/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "pc_to_hw.h"
#include "hw_to_pc.h"
#include "discovery.h"
#include "hpsdr_init.h"
#include "settings.h"
#include "../common/common.h"
#include "../common/convert.h"
#include "../common/filters.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/interface.h"
#include "../Hermes2/sound.h"
#include <gtk/gtk.h>
#include <math.h>
#include <pthread.h>
#include <stdint.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

//----------------------------------------------------------------------

// Start/Stop command buffer size
#define START_STOP_BUF_SIZE   64

// Position in buffer for start/stop ID
#define START_STOP_ID_POSN    2

// Position in buffer for start/stop command
#define START_STOP_POSN       3

// Start/Stop commands
#define RADIO_STREAM_START    0x01
#define RADIO_STREAM_STOP     0xFE
#define SPEC_STREAM_START     0x01
#define SPEC_STREAM_STOP      0xFE
#define BANDSCOPE_START       0x03
#define BANDSCOPE_STOP        0xFD

//----------------------------------------------------------------------

/* Init_Cmnd_Ctrl_Audio_IQ_Data()
 *
 * Initializes Command Control and Audio IQ Data buffers
 */
  void
Init_Cmnd_Ctrl_Audio_IQ_Data( void )
{
  // Clear the HERMES2 Frame buffer
  memset( Cmnd_Ctrl_Audio_IQ_Data.buffer, 0, sizeof(Cmnd_Ctrl_Audio_IQ_Data.buffer) );

  // Packet Preamble
  Uint16_to_String( METIS_PACKET_LEADER, Cmnd_Ctrl_Audio_IQ_Data.buffer );
  Cmnd_Ctrl_Audio_IQ_Data.buffer[2] = 0x01;

  // End Point is 2
  Cmnd_Ctrl_Audio_IQ_Data.buffer[3] = 0x02;

  // Clear sequence number
  Cmnd_Ctrl_Audio_IQ_Data.sequence_num = 0;
  Uint32_to_String(
      Cmnd_Ctrl_Audio_IQ_Data.sequence_num,
      &Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_SEQ_NUM_POSN] );

  // Clear C&C address stack index
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx = 0;

  // HERMES2 Frame 1&2 sync preamble
  Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_FRAME1_POSN]     = 0x7F;
  Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_FRAME1_POSN + 1] = 0x7F;
  Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_FRAME1_POSN + 2] = 0x7F;
  Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_FRAME2_POSN]     = 0x7F;
  Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_FRAME2_POSN + 1] = 0x7F;
  Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_FRAME2_POSN + 2] = 0x7F;

  // Clear the Command & Control bytes
  memset( Cmnd_Ctrl_Audio_IQ_Data.Cn1, 0, CCD_NUM_ADRESSES * CCD_NUM_CC_BYTES );
  memset( Cmnd_Ctrl_Audio_IQ_Data.Cn2, 0, CCD_NUM_ADRESSES * CCD_NUM_CC_BYTES );

  // Preset the address field of C[0] for each of the addresses
  for( uint8_t idx = 0; idx < CCD_NUM_ADRESSES; idx++ )
  {
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[idx][0] = (uint8_t)( idx << 1 );
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[idx][0] = (uint8_t)( idx << 1 );
  }

} // Init_Cmnd_Ctrl_Audio_IQ_Data()

//----------------------------------------------------------------------

/* Send_Start_Stop_Packet()
 *
 * Sends the Start/Stop command packet to the HL2 device
 */
  static BOOLEAN
Send_Start_Stop_Packet( uint8_t Command )
{
  // Start/Stop Command buffer
  uint8_t buffer[START_STOP_BUF_SIZE];
  memset( (void *)buffer, 0, sizeof(buffer) );

  // Prepare buffer for start/stop command
  Uint16_to_String( METIS_PACKET_LEADER, buffer );
  buffer[START_STOP_ID_POSN] = 0x04;  // Start/Stop packet ID

  // Enter Start/Stop Command
  buffer[START_STOP_POSN] = Command;

  // Send the Start/Stop packet
  ssize_t len = sendto(
      data_socket_fd,
      buffer, sizeof(buffer), 0,
      (struct sockaddr *) &sock_addr.base_address,
      sock_addr.base_address_length );
  if( (len < 0) || (len != sizeof(buffer)) )
  {
    perror( _("hermes2: Device_Start(): sendto() failed") );
    return( False );
  }

  return( True );
} // Send_Start_Stop_Packet()

//----------------------------------------------------------------------

/* Device_Start_Stop()
 *
 * Starts or Stops the selected HERMES2 device
 */
  BOOLEAN
Device_Start_Stop( uint8_t command )
{
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Start/Stop Command byte
  static uint8_t Command = 0;

  // Flags to indicate whether Receiver, Spectrum or Bandscope running
  static BOOLEAN
    radio_stream = False,
    spec_stream  = False,
    rcve_thread  = False;

  static uint8_t bscp_stream = 0;

  switch( command )
  {
    // Start Radio stream (common to Spectrum display)
    case START_RADIO_STREAM:
      Command |= RADIO_STREAM_START;
      radio_stream = True;
      break;

      // Stop Radio stream if Spectrum not required
    case STOP_RADIO_STREAM:
      if( !spec_stream && !bscp_stream )
        Command &= RADIO_STREAM_STOP;
      radio_stream = False;
      break;

      // Start Spectrum stream
    case START_SPEC_STREAM:
      Command |= SPEC_STREAM_START;
      spec_stream = True;
      break;

      // Stop Spectrum stream if Radio not required
    case STOP_SPEC_STREAM:
      if( !radio_stream && !bscp_stream )
        Command &= SPEC_STREAM_STOP;
      spec_stream = False;
      break;

      // Start Radio and Bandscope stream
    case START_BANDSCOPE:
      Command |= BANDSCOPE_START;
      bscp_stream++;
      break;

      // Stop Bandscope stream and Radio/Spectrum if not required
    case STOP_BANDSCOPE:
      bscp_stream--;
      if( !bscp_stream ) Command &= BANDSCOPE_STOP;
      if( !spec_stream && !radio_stream && !bscp_stream )
        Command &= RADIO_STREAM_STOP;
      break;
  } // switch( command )

  // Do nothing if command does not change
  static uint8_t comm = 0;
  if( comm == Command ) return( True );
  comm = Command;

  // Determine if the Command requests Stopping HL2 Radio Stream
  if( ~Command & RADIO_STREAM_START ) // Stop request
  {
    // Abort if HERMES2 device is in use
    for( uint8_t idx = 0; idx < Indices.Num_of_TRXs; idx++ )
    {
      if( Transceiver[idx]->receive_active ||
          Transceiver[idx]->spectrum_data.spectrum_running ||
          Transceiver[idx]->timedec_active || ddv->transmit_on )
        return( True );
    }

    // Disable Running of device
    if( ddv->rcve_thread_run )
    {
      // Stop the Data Receive thread
      ddv->rcve_thread_run = False;

      /* Wait for data thread to terminate if
       * running, before sending Stop command */
      if( ddv->rcve_thread_running )
        pthread_join( ddv->data_receive_thread, NULL );
      rcve_thread = False;
      if( !Send_Start_Stop_Packet(Command) )
        return( False );
    }

    // Clear sequence number at Stop command
    Cmnd_Ctrl_Audio_IQ_Data.sequence_num = 0;
    Uint32_to_String(
        Cmnd_Ctrl_Audio_IQ_Data.sequence_num,
        &Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_SEQ_NUM_POSN] );
  } //if( Command & ~RADIO_STREAM_STOP )
  else // Start request
  {
    // Send the Start command
    if( !Send_Start_Stop_Packet(Command) )
      return( False );

    // Start Receive thread if not already started
    if( !rcve_thread )
    {
      // Start a Receive thread to collect Data packets
      ddv->rcve_thread_run = True;
      if( !Pthread_Create(
            &(ddv->data_receive_thread), NULL, Data_Receive_Thread, NULL,
            _("Failed to create HERMES2 Receive thread")) )
      {
        // Stop device
        ddv->rcve_thread_run = False;
        Command &= STOP_RADIO_STREAM | STOP_BANDSCOPE;
        Send_Start_Stop_Packet( Command );

        // Clear sequence number at Stop command
        Cmnd_Ctrl_Audio_IQ_Data.sequence_num = 0;
        Uint32_to_String(
            Cmnd_Ctrl_Audio_IQ_Data.sequence_num,
            &Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_SEQ_NUM_POSN] );

        Close_Data_IO_Socket();
        return( False );
      } // if( !Pthread_Create() )

      rcve_thread = True;
    } // if( !rcve_thread )

    Hermes2_Set_Center_Frequency( Transceiver[Indices.TRx_Index], RX_FLAG );
  } // else of if( Command & ~RADIO_STREAM_STOP )

  return( True );
} // Device_Start_Stop

//----------------------------------------------------------------------

/* DDC_Start()
 *
 * Starts IQ data transfer from a DDC
 */
  BOOLEAN
DDC_Start( Transceiver_t *TRx, uint8_t command )
{
  // Initialize SDR device if not already
  if( !Hpsdr_Initialize() ) return( False );

  // Calculate sound_decimate to give 48000 S/s to sound card
  TRx->sound_decimate = (uint16_t)( hermes2_rc.ddc_sample_rate / SND_DSP_RATE );

  // Set up main filter data and init filter
  if( hermes2_rc.ddc_sample_rate ) Init_Roofing_Filter( TRx );

  // Start the HERMES2 device (Run command)
  if( !Device_Start_Stop(command) )
    return( False );

  return( True );
} // DDC_Start()

//----------------------------------------------------------------------

/* Pack_DUC_Tx_Data()
 *
 * Sends 16-bit I/Q data to the DUC for transmission
 * and 16-bit L/R Received audio data for monitoring (??)
 */
  BOOLEAN
Pack_DUC_Tx_Data(
    const int16_t *data_i, const int16_t *data_q,
    const int16_t *left,   const int16_t *right )
{
  // DUC I/Q and Audio Packet buffer
  uint32_t packet_buff_idx;
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  uint8_t *buffer = Cmnd_Ctrl_Audio_IQ_Data.buffer;
  static BOOLEAN tx_overload_icon = False;

  // **** Writes IQ samples to file, for testing only ****
  /* {
    static FILE *fdi = NULL, *fdq = NULL;
    static size_t cnt = DUC_NUM_IQ_SAMPLES;
    if( fdi == NULL ) fdi = fopen( "i.s", "w" );
    if( fdq == NULL ) fdq = fopen( "q.s", "w" );
    fwrite( data_i, sizeof(int16_t), cnt, fdi );
    fwrite( data_q, sizeof(int16_t), cnt, fdq );
  //return( True );
  } */

  // Abort if SDR device is not running
  if( !ddv->rcve_thread_run ) return( True );

  // First data location
  packet_buff_idx = CCD_FRAME1_DATA_POSN;

  // Enter I/Q and Audio sample data to the first USB frame, if available
  uint8_t data_idx    = 0;
  BOOLEAN tx_overload = False;
  while( packet_buff_idx < CCD_BUFFER_SIZE / 2 )
  {
    /* Calculate the scalar magnitude of the IQ data. If it is greater
     * than (2^15 - 1) (32767) then we have a DUC overload situation.
     * Here though we reduce the threshold to 32700, for a small margin */
    double chk = hypot( data_i[data_idx], data_q[data_idx] );
    if( chk > 32700 ) tx_overload = True;

    Int16_to_String( left[data_idx],   &buffer[packet_buff_idx] );
    packet_buff_idx += 2;
    Int16_to_String( right[data_idx],  &buffer[packet_buff_idx] );
    packet_buff_idx += 2;
    Int16_to_String( data_i[data_idx], &buffer[packet_buff_idx] );
    packet_buff_idx += 2;
    Int16_to_String( data_q[data_idx], &buffer[packet_buff_idx] );
    packet_buff_idx += 2;
    data_idx++;
  }

  // Enter I/Q and Audio sample data to the second USB frame
  packet_buff_idx = CCD_FRAME1_DATA_POSN + CCD_USB_FRAME_SIZE;
  while( packet_buff_idx < CCD_BUFFER_SIZE )
  {
    /* Calculate the scalar magnitude of the IQ data. If it is greater
     * than (2^15 - 1) (32767) then we have a DUC overload situation.
     * Here though we reduce the threshold to 32700, for a small margin */
    double chk = hypot( data_i[data_idx], data_q[data_idx] );
    if( chk > 32700 ) tx_overload = True;

    Int16_to_String( (int16_t)left[data_idx],   &buffer[packet_buff_idx] );
    packet_buff_idx += 2;
    Int16_to_String( (int16_t)right[data_idx],  &buffer[packet_buff_idx] );
    packet_buff_idx += 2;
    Int16_to_String( (int16_t)data_i[data_idx], &buffer[packet_buff_idx] );
    packet_buff_idx += 2;
    Int16_to_String( (int16_t)data_q[data_idx], &buffer[packet_buff_idx] );
    packet_buff_idx += 2;
    data_idx++;
  }

  // Set the TX/PA overload icon. Delay Red O/L change to Green for a short while.
  static uint8_t delay = 0;
  if( tx_overload && !tx_overload_icon )
  {
    Set_Icon( GTK_IMAGE(hermes2_gui.tx_overload_icon), "gtk-no", GTK_ICON_SIZE_BUTTON );
    tx_overload_icon = True;
    delay = 0;
  }
  else if( !tx_overload && tx_overload_icon )
  {
    delay++;
    if( delay >= 100 )
    {
      Set_Icon( GTK_IMAGE(hermes2_gui.tx_overload_icon), "gtk-yes", GTK_ICON_SIZE_BUTTON );
      tx_overload_icon = False;
      delay = 0;
    }
  }

  return( True );
} // Send_DUC_Tx_Data()

//----------------------------------------------------------------------

/* Send_Cmnd_Ctrl_Audio_IQ_Data()
 *
 * Sends Command, Control, Received Audio L/R and Tx IQ Data to Metis
 */
  BOOLEAN
Send_Cmnd_Ctrl_Audio_IQ_Data( char *cmnt )
{
  // Temporary for debugging
  // puts( cmnt );

  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Limit value of C&C address index
  if( Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx > CCD_ADDR_STACK )
    Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx = CCD_ADDR_STACK;

  // printf("IDX %2d\n",Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx);

  // Count down C&C address index
  if( Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx )
    Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx--;

  // Current index to C&C address stack
  uint8_t addr_idx = Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx;

  // Current C&C address
  uint8_t index1 = Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[addr_idx];
  uint8_t index2 = Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[addr_idx];

  // printf( "%2d  %02x  %02x\n", addr_idx, index1, index2 );

  // Enter Sequence Number
  Uint32_to_String(
      Cmnd_Ctrl_Audio_IQ_Data.sequence_num,
      &Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_SEQ_NUM_POSN] );

  // Set or Clear the MOX bit of C&C (C0[0])
  if( ddv->transmit_on )
  {
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1][0] |=  BIT_0;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2][0] |=  BIT_0;
  }
  else
  {
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1][0] &= ~BIT_0;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2][0] &= ~BIT_0;
  }

  // Enter 5 Command and Control data to buffer
  memcpy(
      &Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_FRAME1_C_C_POSN],
       Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1], 5 );
  memcpy(
      &Cmnd_Ctrl_Audio_IQ_Data.buffer[CCD_FRAME2_C_C_POSN],
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2], 5 );

  // Temporary for debugging
  //if( index1 == 0x00 )
  /* {
    printf("A: CNT %2d IDX %02x MX %02x C0 %02x C1 %02x C2 %02x C3 %02x C4 %02x\n",
        addr_idx, index1,
        Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1][0] & 0x01,
        Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1][0] >> 1,
        Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1][1],
        Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1][2],
        Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1][3],
        Cmnd_Ctrl_Audio_IQ_Data.Cn1[index1][4] );
    printf("B: IDX %02x MX %02x C0 %02x C1 %02x C2 %02x C3 %02x C4 %02x\n",
        index2,
        Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2][0] & 0x01,
        Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2][0] >> 1,
        Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2][1],
        Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2][2],
        Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2][3],
        Cmnd_Ctrl_Audio_IQ_Data.Cn2[index2][4] );
  } */

  // Send the C&C packet
  ssize_t len = sendto(
      data_socket_fd,
      Cmnd_Ctrl_Audio_IQ_Data.buffer, sizeof(Cmnd_Ctrl_Audio_IQ_Data.buffer),
      0, (struct sockaddr *) &sock_addr.base_address, sock_addr.base_address_length );
  if( (len < 0) || (len != sizeof(Cmnd_Ctrl_Audio_IQ_Data.buffer)) )
  {
    perror( _("hermes2: Send_Cmnd_Ctrl_Data: sendto() failed") );
    return( False );
  }

  // Increment Sequence Number
  Cmnd_Ctrl_Audio_IQ_Data.sequence_num++;

  return( True );
} // Send_Cmnd_Ctrl_Audio_IQ_Data()

//----------------------------------------------------------------------

