/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "hw_to_pc.h"
#include "pc_to_hw.h"
#include "discovery.h"
#include "settings.h"
#include "../common/common.h"
#include "../olivia/operation.h"
#include "../common/convert.h"
#include "../common/filters.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/interface.h"
#include "../Hermes2/process.h"
#include "../Hermes2/modulate.h"
#include "../Hermes2/sound.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/types.h>
#include <sys/socket.h>

//----------------------------------------------------------------------

// Number of buffers in Net ring buffer
#define NET_BUF_RING_SIZE    4

// Size of Net ring buffer for receiving data from sockets
#define NET_BUFFER_SIZE     1032  // Specific to Metis Protocol 1

// Network I/O data buffer
static uint8_t net_buffer[NET_BUF_RING_SIZE][NET_BUFFER_SIZE];
static uint8_t net_buf_index = 0;

//----------------------------------------------------------------------

/* MOX_Activation()
 *
 * Activates MOX when needed by PTT On or CW Keyer On
 */
  static void
MOX_Activation( uint8_t dot_dash_ptt )
{
  // Mox On or Off status
  static BOOLEAN mox_status = False;

  // If PTT ON (CN4 ring grounded)
  if( dot_dash_ptt == PTT_REPORT )
  {
    // Enable MOX
    if( !mox_status )
    {
      mox_status = True;
      MOX_Button_Toggled( mox_status );
    }
  }
  else if( mox_status ) // Disable MOX
  {
    mox_status = False;
    MOX_Button_Toggled( mox_status );
  }

} // MOX_Activation()

//----------------------------------------------------------------------

static double fwd_power_ave, vswr_ave;

/* Count of High Priority packets before PA power and
 * vswr is displayed. This gives a display rate of ~10/sec
 * since on Transmit this packet arrives every ~2 mSec */
#define PA_PWR_DISPLAY_CNT  50

/* Display_PA_Power()
 *
 * Displays PA forward power and VSWR
 */
  gboolean
Display_PA_Power( gpointer data )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Display in Transceiver window
  gtk_level_bar_set_value( GTK_LEVEL_BAR(TRx->power_levelbar), fwd_power_ave );
  gtk_level_bar_set_value( GTK_LEVEL_BAR(TRx->vswr_levelbar),  vswr_ave - 1.0 );

  return( FALSE );
} // Display_PA_Power()

//----------------------------------------------------------------------

// Maximum VSWR before reducing power
#define MAX_LOAD_VSWR   2.0

// Sliding window averaging factors
#define AVE_WINDOW       5.0
#define AVE_WINDOW_1     4.0

/* Count of calls to function before action is taken, because there is a
 * latency in the changes of PA Power and VSWR after C&C bytes are sent */
#define CONTROL_COUNT   10

/* PA_Power_SWR()
 *
 * Uses the Forward and Reverse Voltage data from the High Priority
 * function and periodically displays Forward power and VSWR
 */
  static void
PA_Power_SWR( BOOLEAN transmit_on, double fwd_power, double rev_power )
{
  static uint8_t
    control_count  = 0,  // Reduce rate of PA Power/SWR control actions
    display_count  = 0;  // Used to reduce rate of PWR/SWR display

  // During transmit only
  if( transmit_on )
  {
    static double rev_power_ave = 0.0;

    // Calculate sliding window averaged PA fwd and rev power
    fwd_power_ave *= AVE_WINDOW_1;
    fwd_power_ave += fwd_power;
    fwd_power_ave /= AVE_WINDOW;
    rev_power_ave *= AVE_WINDOW_1;
    rev_power_ave += rev_power;
    rev_power_ave /= AVE_WINDOW;

    // Calculate VSWR if fwd power report is > 0.2w
    double gamma = 0.0;
    if( fwd_power_ave > 0.2 ) gamma = sqrt( rev_power_ave / fwd_power_ave );
    if( gamma < 1.0 )
      vswr_ave = ( 1 + gamma ) / ( 1 - gamma );
    else
      vswr_ave = 1.0;

    // Control PA Power O/P according to setting of PA Power slider and VSWR
    if( control_count >= CONTROL_COUNT )
    {
      // PA drive level to send to HL2
      static int8_t pa_drive_level = 0;

      // Saves the current Tx power limit request
      static double power_limit = 0.0;
      double vswr_pwr_limit;

      discovered_device_t *ddv = &Device[hermes2_rc.device_index];


      // Reset on MOX ON
      if( ddv->pa_power_up == RESET )
      {
        power_limit      = 0.0;
        ddv->pa_power_up = False;
      }

      // Reduce PA Power if VSWR is above limit (2/1)
      if( vswr_ave > MAX_LOAD_VSWR )
      {
        // Reduce resolution of vswr_pwr_limit's value to 0.2
        vswr_pwr_limit = 5.0 * hermes2_rc.tx_power_limit * MAX_LOAD_VSWR / vswr_ave;
        vswr_pwr_limit = round( vswr_pwr_limit ) / 5.0;
      }
      else
        vswr_pwr_limit = hermes2_rc.tx_power_limit;

      // If Tx power O/P request is increased by user
      if( power_limit < vswr_pwr_limit )
      {
        ddv->pa_power_up = True;
      }
      power_limit = vswr_pwr_limit;

      // Reduce PA Drive level under these conditions
      if( (fwd_power > vswr_pwr_limit) && (pa_drive_level > 0) )
      {
        // Reduce drive level if not zero
        pa_drive_level--;
        ddv->pa_power_up = False;

        // Signal the sending of PA Drive C&C bytes
        Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x09][1] = (uint8_t)( pa_drive_level << 4 );
        Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x09][1] = (uint8_t)( pa_drive_level << 4 );
        Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x09;
        Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x09;
        Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
      }
      else if( ddv->pa_power_up && (pa_drive_level < 0x0F) ) // Max drive level
      {
        pa_drive_level++;

        // Signal the sending of PA Drive C&C bytes
        Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x09][1] = (uint8_t)( pa_drive_level << 4 );
        Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x09][1] = (uint8_t)( pa_drive_level << 4 );
        Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x09;
        Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x09;
        Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
      }

      control_count = 0;
    } // if( control_count >= CONTROL_COUNT )
    control_count++;

    // Reduce display rate to ~10/sec
    if( display_count++ > PA_PWR_DISPLAY_CNT )
    {
      display_count = 1;
      g_idle_add( Display_PA_Power, NULL );
    }
  } // if( tx_status )
  else if( display_count ) // Clear displays
  {
    vswr_ave      = 1.0;
    fwd_power_ave = 0.0;
    display_count = 0;
    control_count = 0;
    g_idle_add( Display_PA_Power, NULL );
  }

} // PA_Power_SWR()

//----------------------------------------------------------------------

// Request or Acknowledge bit (REQ | ACK)
#define REQ_ACK_BIT     0x80

// Count down before action is taken on low priority C&C data
#define LOW_PRIORITY_COUNT  200

/* Process_High_Priority_Data()
 *
 * Processes High Priority and other data from the Hardware
 */
  static void
Process_High_Priority_Data( const uint8_t *buffer )
{
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Process Command and Control data
  uint8_t cmnd0 = buffer[CC_C0];          // C&C C0
  uint8_t raddr = ( cmnd0 >> 3 ) & 0x0F;  // C&C RADDR (return address)

  // Carry out actions depending on RADDR in C0
  switch( raddr )
  {
    case 0x00: // Tx Inhibit, RF ADC O/L, U/Overflow Recovery, TX IQ FIFO Count
      {
        // Slow down the update since its not urgent
        static uint8_t cnt = 0;
        if( cnt++ >= LOW_PRIORITY_COUNT )
        {
          // *** ADC Overload flag ***
          static BOOLEAN adc_ol;
          adc_ol = (buffer[CC_C1] & BIT_0) == BIT_0;

          // Auto adjust RF LNA Gain
          if( ddv->lna_gain_auto )
            g_idle_add( Hpsdr_Auto_LNA_Gain, (gpointer)&adc_ol );

          // Set ADC O/L LED indicator
          static BOOLEAN adc_ol_yes = False;
          if( adc_ol && adc_ol_yes )
          {
            Set_Icon( GTK_IMAGE(hermes2_gui.adc_ol_icon), "gtk-no", GTK_ICON_SIZE_BUTTON );
            adc_ol_yes = False;
          }
          if( !adc_ol && !adc_ol_yes )
          {
            Set_Icon( GTK_IMAGE(hermes2_gui.adc_ol_icon), "gtk-yes", GTK_ICON_SIZE_BUTTON );
            adc_ol_yes = True;
          }

          // Set Tx Inhibited LED indicator (active low)
          static BOOLEAN tx_inh_yes = False;
          if( (buffer[CC_C1] & BIT_1) && (ddv->tx_pa_enable) && !tx_inh_yes )
          {
            Set_Icon( GTK_IMAGE(hermes2_gui.tx_enable_icon), "gtk-yes", GTK_ICON_SIZE_BUTTON );
            tx_inh_yes = True;
          }
          if( (!(buffer[CC_C1] & BIT_1) || !(ddv->tx_pa_enable)) && tx_inh_yes )
          {
            Set_Icon( GTK_IMAGE(hermes2_gui.tx_enable_icon), "gtk-no", GTK_ICON_SIZE_BUTTON );
            tx_inh_yes = False;
          }

          // Set Overflow / Underflow LED indicators
          if( ddv->transmit_on )
          {
            static BOOLEAN overflow_yes = False, underflow_yes = False;
            if( buffer[CC_C3] & BIT_7 ) // Recovery
            {
              if( !overflow_yes )
              {
                Set_Icon( GTK_IMAGE(hermes2_gui.overflow_icon),  "gtk-yes", GTK_ICON_SIZE_BUTTON );
                overflow_yes = True;
              }
              if( !underflow_yes )
              {
                Set_Icon( GTK_IMAGE(hermes2_gui.underflow_icon), "gtk-yes", GTK_ICON_SIZE_BUTTON );
                underflow_yes = True;
              }
            }
            else // Buffer Error?
            {
              if( (buffer[CC_C3] & BIT_6) && overflow_yes ) // Overflow
              {
                Set_Icon( GTK_IMAGE(hermes2_gui.overflow_icon), "gtk-no",  GTK_ICON_SIZE_BUTTON );
                overflow_yes = False;
              }
              if( !(buffer[CC_C3] & BIT_6) && !overflow_yes )
              {
                Set_Icon( GTK_IMAGE(hermes2_gui.overflow_icon), "gtk-yes", GTK_ICON_SIZE_BUTTON );
                overflow_yes = True;
              }
              if( (buffer[CC_C3] == 0) && underflow_yes ) // Underflow
              {
                Set_Icon( GTK_IMAGE(hermes2_gui.underflow_icon), "gtk-no",  GTK_ICON_SIZE_BUTTON );
                underflow_yes = False;
              }
              if( !(buffer[CC_C3] == 0) && !underflow_yes )
              {
                Set_Icon( GTK_IMAGE(hermes2_gui.underflow_icon), "gtk-yes", GTK_ICON_SIZE_BUTTON );
                underflow_yes = False;
              }
            } // if( buffer[CC_C3] & BIT_7  )

            // Show TX IQ FIFO count
            gchar txt[6];
            snprintf( txt, sizeof(txt), "%d", buffer[CC_C3] & ~BIT_7 );
            Set_Label_Text( GTK_LABEL(hermes2_gui.tx_fifo_label), txt );

          } //if( ddv->transmit_on )

          cnt = 0;
        } // if( cnt++ >= LOW_PRIORITY_COUNT )

        break;
      } // case 0x00:

    case 0x01: // Temperature and Tx Forward Power
      {
        // Convert the HermesLite fwd power code to watts forward.
        ddv->fwd_power  = (double)( (buffer[CC_C3] << 8) | buffer[CC_C4] ); // Binary Code
        ddv->fwd_power  = REF_3_3_VOLT * ddv->fwd_power / ADC_SCALE;        // Fwd Voltage
        ddv->fwd_power *= 0.9034;  // Scale factor
        ddv->fwd_power  = ddv->fwd_power * ddv->fwd_power; // Forward Power watts

        // Slow down the update since its not urgent
        static uint8_t cnt = 0;
        if( cnt++ >= LOW_PRIORITY_COUNT )
        {
          /* Rig Temperature. 3.30 is nominal Reference voltage.
           * 4096 steps in ADC. Other factors are unknown. */
          ddv->temperature = (buffer[CC_C1] << 8) | buffer[CC_C2];
          ddv->temperature = ( REF_3_3_VOLT * (ddv->temperature / ADC_SCALE) - 0.50 ) * 100.0;
          gchar txt[10];
          snprintf( txt, sizeof(txt), "%4.1f C", ddv->temperature );
          Set_Label_Text( GTK_LABEL(hermes2_gui.temp_label), txt );
          cnt = 0;
        }

        break;
      } // case 0x01:

    case 0x02: // Tx Reverse Power, Tx Current
      {
        // Convert the HermesLite rev power code to watts reverse power
        ddv->rev_power  = (double)( (buffer[CC_C1] << 8) | buffer[CC_C2] ); // Binary Code
        ddv->rev_power  = REF_3_3_VOLT * ddv->rev_power / ADC_SCALE;        // Rev Voltage
        ddv->rev_power *= 0.9034;  // Scale factor
        ddv->rev_power  = ddv->rev_power * ddv->rev_power;  // Reverse Power watts

        // Slow down display of PA current since its not urgent
        static uint8_t cnt = 0;
        if( cnt++ >= LOW_PRIORITY_COUNT )
        {
          /* Sense amp gain = 50. Sense resistor = 0.04 Ohm.
           * Scale by resistor voltage divider 1000/(1000+270) at input of slow ADC */
          ddv->pa_current = (buffer[CC_C3] << 8) | buffer[CC_C4];
          ddv->pa_current = ( (REF_3_3_VOLT * (ddv->pa_current / ADC_SCALE)) / 50.0 ) * 25.0;
          ddv->pa_current = ddv->pa_current * 1270.0; // In mA
          gchar txt[8];
          snprintf( txt, sizeof(txt), "%3.0f mA", ddv->pa_current );
          Set_Label_Text( GTK_LABEL(hermes2_gui.pa_cur_label), txt );
          cnt = 0;
        }

        break;
      } // case 0x02:
  } // switch( raddr )

  // Status of PTT, Dot and Dash Report
  ddv->dot_dash_ptt = cmnd0 & ( DOT_REPORT | DASH_REPORT | PTT_REPORT );

  // Activate MOX if PTT or CW Keyer is active
  if( !TRx->tx_cw_modes ) MOX_Activation( ddv->dot_dash_ptt );

  // *** Control PA power O/P and handle VSWR protection ***
  if( (ddv->fwd_power != 0) && (ddv->rev_power != 0) )
  {
    PA_Power_SWR( ddv->transmit_on, ddv->fwd_power, ddv->rev_power );
    ddv->fwd_power = 0;
    ddv->rev_power = 0;
  }

} // Process_High_Priority_Data()

//----------------------------------------------------------------------

// Mic filter constants
#define TX_MIC_FILTER_POLES    6
#define TX_MIC_FILTER_RIPPLE   1.0

// Mic compressor constants
#define COMPRESS_DECAY   0.99999
#define COMPRESS_MIN     0.001

//----------------------------------------------------------------------

/* Process_Microphone_Data()
 *
 * Processes Microphone/Line data from the Hardware
 */
  static BOOLEAN
Process_Microphone_Data( double mic_sample, double *mic_samples )
{
  // Index to input packet buffer and output buffer
  static uint8_t samples_cnt     = 0;
  const Transceiver_t *TRx       = Transceiver[Indices.TRx_Index];
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];


  // Speech modes (SSB, AM, FM). Process microphone samples, if Transmitting
  if( (TRx->tx_modulation_mode <= TX_MODE_NBFM2) && ddv->transmit_on )
  {
    // Two-tone test signal, for testing only. Uncomment block when needed.
/*
    static double dW1 = M_2PI * 1200 / 48000;   // Delta Omega of first signal of 300Hz
    static double dW2 = M_2PI * 2800 / 48000;  // Delta Omega of second signal of 3800Hz
    static double  A1 = 30000;                 // Amplitude of first signal
    static double  A2 = 0000;                 // Amplitude of second signal
    static double  f1 = 0, f2 = 0;

    mic_sample = A1 * sin(f1) + A2 * sin(f2);
    f1 += dW1; CLAMP_2PI(f1);
    f2 += dW2; CLAMP_2PI(f2);
*/

    /* The "standard" Hermes Lite 2 has no Microphone input so the client
     * application must use microphone samples from the computer's Sound Card */
    short sample_val;
    if( !Signal_Sample(&sample_val) ) return( False );
    mic_sample = (double)sample_val;

    // *** Do simple compression of mic samples if enabled ***
    if( TRx->mic_compress )
    {
      // New value of a mic sample
      static double compress;
      double level;

      // mic_sample increased by 1 to avoid divide by zero
      level = ( fabs(mic_sample) + 1 ) / MIC_MAX_VALUE;
      if( compress < level )
      {
        compress = level;            // This is the Compressor "attack" function
      }
      else if( compress > COMPRESS_MIN )
      {
        compress *= COMPRESS_DECAY;  // This the Compressor "decay" function
      }

      // Scale microphone samples as needed
      mic_sample /= compress;
    } // if( TRx->mic_compress )

  } // if( (TRx->tx_modulation_mode <= TX_MODE_NBFM2) && ddv->transmit_on )

  // Apply Microphone level control from Slider
  mic_sample *= (double)TRx->mic_level / 100.0;

  /* Save MIC_NUM_OF_SAMPLES of incoming Microphone samples. Since the DDC
   * sampling rate (usually 192kS/s) may be higher than the mic sampling rate
   * (48kS/s), duplicate mic samples must be discarded to maintain timing. */
  mic_samples[samples_cnt] = mic_sample;
  samples_cnt++;

  // Continue saving Microphone samples
  if( samples_cnt < MIC_NUM_OF_SAMPLES ) return( FALSE );
  samples_cnt = 0;

  // Low pass filter microphone stream if enabled and in Speech modes (SSB, AM, FM).
  if( (TRx->tx_modulation_mode <= TX_MODE_NBFM2) && ddv->transmit_on && TRx->mic_filter )
  {
    // Microphone Filter data
    static filter_data_t mic_filter_data;

    // Filter cut off frequency
    static uint16_t cutoff = 0;

    // Initialize filter on parameters change or first call
    if( cutoff != hermes2_rc.mic_lpf_cutoff )
    {
      cutoff = hermes2_rc.mic_lpf_cutoff;
      mic_filter_data.cutoff   = (double)cutoff / MIC_SAMPLE_RATE;
      mic_filter_data.ripple   = TX_MIC_FILTER_RIPPLE;
      mic_filter_data.npoles   = TX_MIC_FILTER_POLES;
      mic_filter_data.type     = FILTER_LOWPASS;
      mic_filter_data.ring_idx = 0;
      mic_filter_data.samples_buffer     = mic_samples;
      mic_filter_data.samples_buffer_len = MIC_NUM_OF_SAMPLES;
      mic_filter_data.init_filter = True;
      //return( False );
    }

    // Low-pass filter Microphone samples buffer
    DSP_Filter( &mic_filter_data );

  } // if( (TRx->tx_modulation_mode <= TX_MODE_NBFM2) && ddv->transmit_on && ... )

  return( True );
} // Process_Microphone_Data()

//----------------------------------------------------------------------

/* Transmit_Control()
 *
 * Controls the various functions that are ascociated with Transmitting
 */
  static BOOLEAN
Transmit_Control( double *mic_samples )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  const Transceiver_t       *TRx = Transceiver[Indices.TRx_Index];

  /* If in Transmit Mode. Modulator() is a function pointer to
   * the selected modulator function (SSB, AM, FM) in this case */
  if( ddv->transmit_on && (Modulator != NULL) )
  {
    // If TX Tune is selected
    if( ddv->tx_pa_tune )
    {
      // All done in Tune_Button_Toggled() in callback_func.c
    }
    else if( Flag[TRANSMIT_MORSE_MESG] ) // Send Morse code message
    {
      Modulator( hermes2_rc.morse_mesg );
    }
    else if( (TRx->tx_modulation_mode == TX_MODE_CW_EXTERN) ) // Send CW samples to DUC (CW Mode)
    {
      Modulator( NULL );
    }
    else if( (TRx->tx_modulation_mode <= TX_MODE_NBFM2) ) // Speech modes (SSB, AM, FM)
    {
      Modulator( mic_samples );
    }
    else if( Flag[HERMES2_SEND_DUC_PACKET] )  // Send samples packets to DUC
    {
      Modulator( NULL );
    }
  }  // if( ddv->transmit_on && (Modulator != NULL) )

  // Send the DUC I/Q and Audio data packet to HL2
  if( Send_Cmnd_Ctrl_Audio_IQ_Data("Transmit_Control") )
    return( True );
  else
    return( False );

} // Transmit_Control()

//----------------------------------------------------------------------

// Position of sequence number in net buffer
#define SEQ_NUMBER_POSN    4

// Start of HERMES2 USB frames in EP6 buffer (sould have 7F7F7F sync bytes)
#define HERMES2_FRAME1_START    8
#define HERMES2_FRAME2_START    520

// Start of IQ / MIC data of HERMES2 USB frames in EP6 buffer
#define HERMES2_FRAME1_IQ    16
#define HERMES2_FRAME2_IQ    528

// Number of data bytes available in each 512 byte USB frame
#define FRAME_DATA_LENGTH    504

// Local data buffer size (two frames as above)
#define LOCAL_BUF_SIZE   1008


/* Process_DDC_IQ_Data()
 *
 * Processes ADC I/Q data from a DDC Receiver Slice
 */
  static void
Process_DDC_IQ_Data( void )
{
  static uint32_t
    current_buf  = 0,
    data_buf_cnt = 0,
    data_buf_idx = 0,
    data_buf_len = 0;

  // Converted incoming IQ data ready (buffer full)
  static BOOLEAN data_ready = False;

  // Local buffer used to accumulate IQ / Mic data from the 2 USB frames
  uint8_t local_buf[LOCAL_BUF_SIZE];  // 2 * (frame 512 - 3 sync - 5 C&C)

  // Microphone samples for the modulators
  static double mic_samples[MIC_NUM_OF_SAMPLES];

  // Response data buffer's index
  uint16_t local_buf_idx;

  // Number of IQ/Mic sample blocks in buffer
  uint16_t num_blocks;

  // Count of duplicate copies of Microphone samples
  static uint8_t dup_cnt = 0;

  // Current receiver slice index
  uint8_t rx_idx;

  // Initialize on first call or change of params
  if( data_buf_len != hermes2_rc.ddc_buffer_length )
  {
    data_buf_cnt = 0;
    data_buf_idx = 0;
    data_buf_len = hermes2_rc.ddc_buffer_length;
  }

  /* Length of IQ / Microphone data blocks in bytes. This is
   * 3 I's + 3 Q's (6) per receiver slice plus 2 Microphone samples */
  uint8_t block_len = 6 * Indices.Num_of_TRXs + 2;

  // Number of IQ / Mic sample blocks in the EP6 512 byte frame
  num_blocks = FRAME_DATA_LENGTH / block_len;

  // Accumulate IQ / Mic data into local buffer, after sync & comm
  size_t siz = num_blocks * block_len;
  memcpy( local_buf,       &net_buffer[net_buf_index][HERMES2_FRAME1_IQ], siz );
  memcpy( &local_buf[siz], &net_buffer[net_buf_index][HERMES2_FRAME2_IQ], siz );

  ///// Reads IQ samples from file, for testing only.
  ///// NOTE: Both Receive and FFT Waterfall must be disabled first and a
  ///// recording of the samples must be made before lhpsdr will receive
  /* int16_t buf_i[DUC_NUM_IQ_SAMPLES];
  int16_t buf_q[DUC_NUM_IQ_SAMPLES];
  {
    static FILE *fdi = NULL, *fdq = NULL;
    static size_t cnt = DUC_NUM_IQ_SAMPLES;
    if( fdi == NULL ) fdi = fopen( "i.s", "r" );
    if( fdq == NULL ) fdq = fopen( "q.s", "r" );
    if( fdq == NULL || fdi == NULL ) return;
    fread( buf_i, sizeof(int16_t), cnt, fdi );
    fread( buf_q, sizeof(int16_t), cnt, fdq );
  } */

  // Store IQ samples to local buffers
  // Now there are two USB frames in local buffer
  uint16_t blocks = 2 * num_blocks;
  local_buf_idx   = 0;
  for( uint16_t idx = 0; idx < blocks; idx++ )
  {
    // Save Receiver "slice" I/Q Data to each active Transceiver instance
    for( rx_idx = 0; rx_idx < Indices.Num_of_TRXs; rx_idx++ )
    {
      // Sample value converted to int from buffer
      int32_t sample_i, sample_q;
      Transceiver_t *TRx = Transceiver[rx_idx];

      // Convert buffer to 32-bit signed integer values
      sample_i = String_to_Sample( &local_buf[local_buf_idx] );
      local_buf_idx += 3;  // For 24-bit (3-byte) format
      sample_q = String_to_Sample( &local_buf[local_buf_idx] );
      local_buf_idx += 3;  // For 24-bit (3-byte) format + 2 Mic data

      ///// If using the commented block above to read I/Q from file,
      ///// uncomment this block. The commented extensions are white noise
     /* {
        sample_i = (int32_t)buf_i[idx]; //+ (int32_t)(DUC_SAMPLE_MAX*(0.5-(double)(rand())/RAND_MAX));
        sample_q = (int32_t)buf_q[idx]; //+ (int32_t)(DUC_SAMPLE_MAX*(0.5-(double)(rand())/RAND_MAX));
      } */

      // During a "beep", suppress output from demodulators
      if( Flag[BEEP_ACTIVE] )
      {
        sample_i = 1;
        sample_q = 1;
      }

      // Copy I/Q sample data to local buffers
      TRx->data_buf_i[data_buf_cnt][data_buf_idx] = (double)sample_i;
      TRx->data_buf_q[data_buf_cnt][data_buf_idx] = (double)sample_q;

    } // for( rx_idx = 0; rx_idx < Indices.Num_of_TRXs; rx_idx++ )

    // Signal data ready when local buffer full
    data_buf_idx++;  // Transceiver buffers index
    if( data_buf_idx >= data_buf_len )
    {
      data_buf_idx = 0;
      data_ready   = True;

      // Change ring buffer
      current_buf = data_buf_cnt;
      data_buf_cnt++;
      if( data_buf_cnt >= NUM_DATA_BUFFERS )
        data_buf_cnt = 0;
    } // if( data_buf_idx >= data_buf_len )

    // Signal to process IQ data
    if( data_ready )
    {
      data_ready = False;

      // Copy local buffers to roofing filter buffers
      size_t len = data_buf_len * sizeof( double );
      for( rx_idx = 0; rx_idx < Indices.Num_of_TRXs; rx_idx++ )
      {
        Transceiver_t *TRx = Transceiver[rx_idx];
        memcpy( TRx->roof_filter_i.samples_buffer, TRx->data_buf_i[current_buf], len );
        memcpy( TRx->roof_filter_q.samples_buffer, TRx->data_buf_q[current_buf], len );
        Hermes2_Run( TRx );
      }
    } // if( data_ready )

    /* The sampling rate of the Microphone and the TX DUC is fixed at 48kS/s.
     * The sampling rate of the DDC can be 48, 96, 192 or 384 kS/s, and if it
     * is greater than 48kS/s the Microphone samples are duplicated: For a
     * DDC rate of 192kS/s, the Microphone data is duplicated over 192/48 = 4
     * Metis samples packets. So Microphone samples are collected every 4 Metis
     * packets and this is also used as the timer for sending Metis DUC packets */
    if( dup_cnt == 0 )
    {
      double mic_sample = (double)( String_to_Int16( &local_buf[local_buf_idx]) );
      BOOLEAN ret = Process_Microphone_Data( mic_sample, mic_samples );

      /* Function above returns True when all Microphone samples have been
       * processed. This is used as the timer for sending DUC samples to Metis */
      if( ret ) Transmit_Control( mic_samples );
    } // if( dup_cnt == 0 )

    // Bypass Microphone data
    local_buf_idx += 2;

  } // for( uint16_t idx = 0; idx < blocks; idx++ )

  // Count the duplicated Microphone samples
  dup_cnt++;
  if( dup_cnt >= hermes2_rc.ddc_mic_rate_ratio )
    dup_cnt = 0;

} // Process_DDC_IQ_Data()

//----------------------------------------------------------------------

// Length of wideband samples buffer
#define WIDEBAND_BUFFER_LENGTH  1032

// Buffer index where wideband samples begin
#define WIDEBAND_DATA_START  8

/* Process_Wideband_Data()
 *
 * Processes Wideband ADC data from the Hardware
 */
  static void
Process_Wideband_Data( uint8_t *data_in )
{
  uint32_t sequence_num;
  static uint16_t data_in_idx, local_buf_idx;
  static int16_t *local_buf = NULL;
  static BOOLEAN sync;


  // Initialize at first call if FFT initialized
  if( local_buf == NULL )
  {
    data_in_idx   = WIDEBAND_DATA_START;
    local_buf_idx = 0;
    sync          = False;
    Mem_Alloc( (void **)&local_buf, sizeof(int16_t) * REAL_FFT_INPUT_LENGTH );
  }

  // Sequence number check
  sequence_num = String_to_Uint32( &data_in[SEQ_NUMBER_POSN] );

  /* Synchronise samples collection with start of ADC samples block.
   * W/B samples are sent in 4 frames of 512x16 bit raw samples.
   * fft_in_count == 0 indicates that the FFT function is ready. */
  if( (sequence_num & 0x03) == 0 )
  {
    sync = True;
    for( uint8_t rx_idx = 0; rx_idx < Indices.Num_of_TRXs; rx_idx++ )
    {
      const Transceiver_t *TRx = Transceiver[rx_idx];
      if( (TRx->spectrum_data.fft_in_count != 0) &&
           TRx->spectrum_data.wideband_spectrum )
        sync = False;
    }
  }

  // Accumulate fft_in_length Wideband (Bandscope) raw ADC samples
  while( sync &&
      (data_in_idx   < WIDEBAND_BUFFER_LENGTH) &&
      (local_buf_idx < REAL_FFT_INPUT_LENGTH) )
  {
    local_buf[local_buf_idx] = String_to_Int16(&data_in[data_in_idx]);
    data_in_idx += 2;
    local_buf_idx++;
  }

  // Continue accumulating Wideband samples till buffer full
  data_in_idx = WIDEBAND_DATA_START;
  if( local_buf_idx < REAL_FFT_INPUT_LENGTH ) return;
  local_buf_idx = 0;

  // Pass on local buffer to Transceiver's FFT samples buffer
  for( uint8_t rx_idx = 0; rx_idx < Indices.Num_of_TRXs; rx_idx++ )
  {
    // Enter real samples to I and zero Q buffer
    Transceiver_t *TRx = Transceiver[rx_idx];

    // TRx in Bandscope mode
    if( TRx->spectrum_data.wideband_spectrum && TRx->spectrum_data.spectrum_init )
    {
      for( uint16_t fft_in_idx = 0; fft_in_idx < REAL_FFT_INPUT_LENGTH; fft_in_idx++ )
      {
        TRx->spectrum_data.fft_in_i[fft_in_idx] = (double)local_buf[fft_in_idx];
        TRx->spectrum_data.fft_in_q[fft_in_idx] = 0.0;
        TRx->spectrum_data.fft_in_count++;
      }
    }
  }

  sync = False;
  return;
} // Process_Wideband_Data()

//----------------------------------------------------------------------

// Position of End Point number in net buffer
#define END_POINT_POSN    3
#define END_POINT_4       4
#define END_POINT_6       6

/* Frame_Check()
 *
 * Checks the USB Frame Sequence number and Sync train
 */
  static void
Frame_Check( uint8_t *net_buf, uint8_t end_point )
{
  uint32_t sequence;

  // Sequence number check for given End Point
  sequence = String_to_Uint32( &net_buf[SEQ_NUMBER_POSN] );
  if( end_point == END_POINT_6 )
  {
    static uint32_t ep6_seq_num = 0;
    if( Flag[HERMES2_ERROR_MESSAGES] && (sequence != ep6_seq_num) )
    {
      fprintf( stderr,
          _("hermes2: Frame_Check: Sequence Number Error at End Point:%u SEQ #%u\n"),
          end_point, sequence );
    }
    ep6_seq_num = sequence + 1;
  }
  else if( end_point == END_POINT_4 )
  {
    static uint32_t ep4_seq_num = 0;
    if( Flag[HERMES2_ERROR_MESSAGES] && (sequence != ep4_seq_num) )
    {
      fprintf( stderr,
          _("hermes2: Frame_Check: Sequence Number Error at End Point:%u SEQ #%u\n"),
          end_point, sequence );
    }
    ep4_seq_num = sequence + 1;
  }

  // Check the USB frame sync bytes
  if( end_point == END_POINT_6 )
  {
    uint32_t frame_sync1, frame_sync2;
    frame_sync1   = net_buf[HERMES2_FRAME1_START];
    frame_sync1 <<= 8;
    frame_sync1  |= net_buf[HERMES2_FRAME1_START+1];
    frame_sync1 <<= 8;
    frame_sync1  |= net_buf[HERMES2_FRAME1_START+2];
    frame_sync2   = net_buf[HERMES2_FRAME2_START];
    frame_sync2 <<= 8;
    frame_sync2  |= net_buf[HERMES2_FRAME2_START+1];
    frame_sync2 <<= 8;
    frame_sync2  |= net_buf[HERMES2_FRAME2_START+2];
    if( Flag[HERMES2_ERROR_MESSAGES] &&
        ((frame_sync1 != 0x007f7f7f) || (frame_sync2 != 0x007f7f7f)) )
    {
      fprintf( stderr,
          _("hermes2: Frame_Check(): Frame Sync Error at SEQ #%u\n"),
          sequence );
    }
  }

} // Frame_Check()

//----------------------------------------------------------------------

// Command and Control data positions
#define FRAME1_COMM_CTRL    11
#define FRAME2_COMM_CTRL    523

/* Data_Receive_Thread()
 *
 * Thread to receive and decode DDC IQ Data Packets
 */
  void *
Data_Receive_Thread( void *arg )
{
  struct sockaddr_in socket_addr;  // Socket address for Data
  socklen_t sock_len;              // Length of above
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  uint8_t end_point;


  // Keep trying to read data from Data socket
  sock_len = sizeof( socket_addr );
  ddv->rcve_thread_running = True;
  while( ddv->rcve_thread_run && data_socket_fd )
  {
    // Receive from current Data socket
    int bytes_read = (int)recvfrom(
        data_socket_fd, net_buffer[net_buf_index], NET_BUFFER_SIZE,
        0, (struct sockaddr *) &socket_addr, &sock_len );

    // Abort if no response from Data socket
    if( (bytes_read < 0) || (bytes_read != NET_BUFFER_SIZE) )
    {
      perror( _("hermes2: Data_Receive_Thread(): recvfrom() failed") );
      sleep( 1 );
      continue;
    }

    // Suspend data IO during some operations (malloc)
    if( Flag[HERMES2_SUSPEND_DATA_IO] ) continue;

    // End Point in data buffer header
    end_point = net_buffer[net_buf_index][END_POINT_POSN];

    // Carry out a USB Frame check for Sequence number and Sync train (for EP6)
    Frame_Check( net_buffer[net_buf_index], end_point );

    // Direct data to appropriate processing function
    switch( end_point )
    {
      case END_POINT_4: // Wideband IQ data (Bandscope)
        Process_Wideband_Data( net_buffer[net_buf_index] );
        break;

      case END_POINT_6: // Rx IQ data and Microphone data
        {
          // Acknowledge flag (ACK)
          BOOLEAN ack;

          // Process regular High Priority data (ACK = 0)
          ack = ( net_buffer[net_buf_index][FRAME1_COMM_CTRL] & REQ_ACK_BIT ) == REQ_ACK_BIT;
          if( !ack )
            Process_High_Priority_Data( &net_buffer[net_buf_index][FRAME1_COMM_CTRL] );
          ack = ( net_buffer[net_buf_index][FRAME2_COMM_CTRL] & REQ_ACK_BIT ) == REQ_ACK_BIT;
          if( !ack )
            Process_High_Priority_Data( &net_buffer[net_buf_index][FRAME2_COMM_CTRL] );

          // Process ADC/Receiver Slice I/Q samples for Demodulation and Rx Spectrum
          Process_DDC_IQ_Data();
        }
    }

    // Increment and cycle Net ring buffer index
    net_buf_index++;
    if( net_buf_index >= NET_BUF_RING_SIZE )
      net_buf_index = 0;

  } // while( ddv->Running && data_socket_fd )
  ddv->rcve_thread_running = False;

  return( NULL );
} // Data_Receive_Thread()

//----------------------------------------------------------------------

