/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "hpsdr_init.h"
#include "discovery.h"
#include "pc_to_hw.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/interface.h"
#include <gtk/gtk.h>
#include <pthread.h>
#include <sys/socket.h>
#include <stdio.h>
#include <stddef.h>
#include <stdint.h>
#include <string.h>

//----------------------------------------------------------------------

/* Close_Data_IO_Socket()
 *
 * Closes the Data I/O (discovery) socket
 */
  void
Close_Data_IO_Socket( void )
{
  // Close data receive socket fd, if open
  if( data_socket_fd )
  {
    Device[hermes2_rc.device_index].rcve_thread_run = False;
    close( data_socket_fd );
    data_socket_fd = 0;
  }
} // Close_Data_IO_Socket()

//----------------------------------------------------------------------

/* Hermes2_New_Buffer()
 *
 * Makes changes needed when hermes2 samples buffer changes
 */
  static void
Hermes2_New_Buffer( gint index )
{
  // Allocate ring buffers on buffer change
  audio_ring_buff.buff_in   = 0;
  audio_ring_buff.buff_out  = audio_ring_buff.buff_size / 2;
  audio_ring_buff.buff_size = 3 * hermes2_rc.ddc_buffer_length;
  size_t mreq = (size_t)audio_ring_buff.buff_size * sizeof( int16_t );
  Mem_Realloc( (void **) &audio_ring_buff.buff_left,  mreq );
  Mem_Realloc( (void **) &audio_ring_buff.buff_right, mreq );
  memset( audio_ring_buff.buff_left,  0, mreq );
  memset( audio_ring_buff.buff_right, 0, mreq );

  // Reset sound buffer indices
  demod_ring_buf_idx = NUM_SOUND_RING_BUFFS / 2;
  sound_ring_buf_idx = NUM_SOUND_RING_BUFFS / 2;

} // Hermes2_New_Buffer()

//----------------------------------------------------------------------

// Free resources
  void
Free_Hermes2_Buffer( void )
{
  Mem_Free( (void **) &audio_ring_buff.buff_left );
  Mem_Free( (void **) &audio_ring_buff.buff_right );
}

//----------------------------------------------------------------------

/* Hpsdr_Initialize()
 *
 * Initialize HERMES2 device status
 */
  BOOLEAN
Hpsdr_Initialize( void )
{
  // Abort if already initialized
  if( Flag[HERMES2_INITIALIZED] ) return( True );

  // Check function return values
  int rc, on;
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Create the Data I/O socket fd
  data_socket_fd = socket( PF_INET, SOCK_DGRAM, IPPROTO_UDP );
  if( data_socket_fd < 0 )
  {
    perror( _("!!hermes2: Hpsdr_Initialize(): socket() failed") );
    Error_Dialog( _("Failed to create data I/O socket"), SHOW_OK );
    return( False );
  }

  // Set the Data I/O socket fd options
  on = True;
  rc = setsockopt( data_socket_fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on) );
  if( rc != 0 )
  {
    perror( _("!!hermes2: Hpsdr_Initialize(): cannot set SO_REUSEADDR") );
    return( False );
  }
  on = True;
  rc = setsockopt( data_socket_fd, SOL_SOCKET, SO_REUSEPORT, &on, sizeof(on) );
  if( rc != 0 )
  {
    perror( _("!!hermes2: Hpsdr_Initialize(): cannot set SO_REUSEPORT") );
    return( False );
  }

  // Set Data I/O socket time out to 200 msec
  struct timeval tv; // Data response time out
  tv.tv_sec  = 0;
  tv.tv_usec = 200000;

  // Set Data socket options
  if( setsockopt(
        data_socket_fd, SOL_SOCKET, SO_RCVTIMEO,
        (char *) &tv, sizeof(struct timeval)) )
  {
    perror( _("!!hermes2: Hpsdr_Initialize(): setsockopt() failed") );
    return( False );
  }

  // Bind to the discovered device interface
  rc = bind( data_socket_fd,
      (struct sockaddr *) &(ddv->interface_address),
      ddv->interface_address_length );
  if( rc < 0 )
  {
    perror( _("!!hermes2: Hpsdr_Initialize(): bind() failed") );
    return( False );
  }
  if( Flag[HERMES2_VERBOSE_MESSAGES] )
  {
    printf( _("hermes2: Hpsdr_Initialize():"
          " Data socket bound to Device Interface\n") );
  }

  // Create the various addresses needed to Send/Receive packets
  memcpy( &sock_addr.base_address,
      &(ddv->network_address), ddv->network_address_length );
  sock_addr.base_address_length   = ddv->network_address_length;
  sock_addr.base_address.sin_port = htons(DISCOVERY_PORT);

  // Set the HERMES2 buffer size
  Hermes2_New_Buffer( (gint)hermes2_rc.ddc_buffer_idx );

  Flag[HERMES2_INITIALIZED] = True;
  return( True );
} // Hpsdr_Initialize()

//----------------------------------------------------------------------

