/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "discovery.h"
#include "hw_to_pc.h"
#include "../common/common.h"
#include "../common/convert.h"
#include "../common/hermes2_rc.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/interface.h"
#include <gtk/gtk.h>
#include <ifaddrs.h>
#include <net/if.h>
#include <sys/errno.h>
#include <pthread.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <semaphore.h>
#include <time.h>

//----------------------------------------------------------------------

// Device (SDR hardware) types
#define DEVICE_ATLAS        0
#define DEVICE_HERMES       1
#define DEVICE_HERMES2      2
#define DEVICE_ANGELIA      3
#define DEVICE_ORION        4
#define DEVICE_ORION2       5
#define DEVICE_HERMES_LITE  6

//----------------------------------------------------------------------

// State of HERMES2 hardware
#define STATE_AVAILABLE     2
#define STATE_SENDING       3

//----------------------------------------------------------------------

// Buffers in discovery reply packet

// Address  Bits    Description
// 0x00     [7:0]   0xEF    PACKET_LEADER
// 0x01     [7:0]   0xFE    PACKET_LEADER
// 0x02     [7:0]   0x02 if not sending and 0x03 if sending data
// 0X03     [7:0]   U of MAC U:V:W:X:Y:Z
// 0x04     [7:0]   V of MAC U:V:W:X:Y:Z
// 0x05     [7:0]   W of MAC U:V:W:X:Y:Z
// 0x06     [7:0]   X of MAC U:V:W:X:Y:Z
// 0x07     [7:0]   Y of MAC U:V:W:X:Y:Z
// 0x08     [7:0]   Z of MAC U:V:W:X:Y:Z
// 0x09     [7:0]   Gateware Major Version
// 0x0A     [7:0]   Board ID, 0x06 for HL2, 0x01 for Hermes Emulation
// 0x0B     [7:0]   MCP4662 0x06 Config Bits
// 0x0C     [7:0]   MCP4662 0x07 Reserved Config Bits
// 0x0D     [7:0]   MCP4662 0x08 Fixed IP
// 0x0E     [7:0]   MCP4662 0x09 Fixed IP
// 0x0F     [7:0]   MCP4662 0x0A Fixed IP
// 0x10     [7:0]   MCP4662 0x0B Fixed IP
// 0x11     [7:0]   MCP4662 0x0C MAC
// 0x12     [7:0]   MCP4662 0x0D MAC
// 0x13     [7:0]   Number of Hardware Receivers
// 0x14     [7:6]   00 wide band data is 12-bit sign extended two's complement
//                  01 wide band data is 16-bit two's complement
//          [5:0]   Board ID: 5, 3 or 2 for build
// 0x15     [7:0]   Gateware Minor Version/Patch
// 0x16     [5:0]   Reserved
// 0x17     [7:0]   Response Data [31:24]
// 0x18     [7:0]   Response Data [23:16]
// 0x19     [7:0]   Response Data [15:8]
// 0x1A     [7:0]   Response Data [7:0]
// 0x1B     [7]     External CW Key
//          [6]     PTT (TX is on)
//          [1:0]   ADC clip count
// 0x1C     [3:0]   Temperature msb
// 0x1D     [7:0]   Temperature lsb
// 0x1E     [3:0]   Forward power msb
// 0x1F     [7:0]   Forward power lsb
// 0x20     [3:0]   Reverse power msb
// 0x21     [7:0]   Reverse power lsb
// 0x22     [3:0]   Bias current msb
// 0x23     [7:0]   Bias current lsb
// 0x24     [7]     Under/overflow recovery flag
// 0x24     [6:0]   TX IQ FIFO count most significant bits
// 0x26-0x3B        Reserved

#define STATUS_BUFF         0x02
#define MAC_ADDR_BUFF       0X03
#define GATEWARE_MAJOR_VER  0x09
#define BOARD_ID            0x0A
#define HARDWARE_RECEIVERS  0x13
#define WIDE_BAND_DATA_ID   0x14
#define GATEWARE_MINOR_VER  0x15
#define RESPONSE_DATA       0x17
#define CWKEY_TXON_ADCCLIP  0x1B
#define TEMPERATURE         0x1C
#define FORWARD_POWER       0x1E
#define REVERSE_POWER       0x20
#define BIAS_CURRENT        0x22
#define UO_FLOW_TX_IQ_FIFO  0x24

//----------------------------------------------------------------------

// Reply to DISCOVER command
#define DISCOVER_REPLY_SIZE      60

//----------------------------------------------------------------------

// Struct to be used as the argument to the discovery thread
typedef struct _DISCOVERY_THREAD_ARG
{
  char    *iface_name;
  struct   sockaddr_in *iface_addr;
  int      socket_fd;
} discovery_thread_arg_t;

// File variables
static pthread_t discovery_thread;

//----------------------------------------------------------------------

/* Discovery_Receive_Thread()
 *
 * Thread to receive and decode the Discovery Response Packet
 */
  static void *
Discovery_Receive_Thread( void *arg )
{
  // Response data buffer
  uint8_t buffer[DISCOVER_REPLY_SIZE];

  // Socket address for discovery
  struct   sockaddr_in socket_addr;
  uint32_t len;         // Length of above
  struct   timeval tv;  // Discovery response time out
  uint32_t idx, try = 0;
  discovery_thread_arg_t *ptr = (discovery_thread_arg_t *)arg;
  gchar mesg[MESG_STRING_SIZE];
  int ret;

  // Set Discovery response time out to 2 sec
  tv.tv_sec  = 2;
  tv.tv_usec = 0;

  // Set discovery socket options
  ret = setsockopt(
      ptr->socket_fd, SOL_SOCKET, SO_RCVTIMEO,
      (char *) &tv, (socklen_t)(sizeof(struct timeval)) );
  if( ret != 0 )
  {
    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"red\">Discovery_Receive_Thread():"
          " setsockopt() failed\n</span>") );
    Discovery_Dialog_Text( mesg );
    perror( _("hermes2: Discovery_Receive_Thread(): setsockopt() failed") );
    pthread_exit( NULL );
  }

  // Keep trying to read response data from discovery socket
  len = sizeof( socket_addr );
  while( True )
  {
    // Number of bytes read by recvfrom()
    ssize_t  bytes_read;

    if( Flag[HERMES2_VERBOSE_MESSAGES] )
      printf( _("hermes2: Discovery_Receive_Thread(): Try #%u\n"), try );

    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg), _("Discovery_Receive_Thread(): Try #%u\n"), try );
    Discovery_Dialog_Text( mesg );
    try++;

    // Receive from current discovery socket
    bytes_read = recvfrom(
        ptr->socket_fd, buffer, sizeof(buffer), 0,
        (struct sockaddr *) &socket_addr, (socklen_t *) &len );

    // Abort if no response from discovery socket
    if( (bytes_read < 0) || (bytes_read != sizeof(buffer)) )
    {
      // Report on Discovery dialog
      snprintf( mesg, sizeof(mesg),
          _("<span foreground=\"orange\">Discovery_Receive_Thread():"
            " recvfrom() failed\n</span>") );
      Discovery_Dialog_Text( mesg );

      // Report on Console if enabled
      if( Flag[HERMES2_ERROR_MESSAGES] )
        perror( _("hermes2: Discovery_Receive_Thread(): recvfrom() failed\n") );
      break;
    }

    // Report on Console if enabled
    if( Flag[HERMES2_VERBOSE_MESSAGES] )
      printf( _("hermes2: Discovery_Receive_Thread():"
            " Received %ld bytes: Found Device #%u\n\n"),
          bytes_read, hermes2_rc.number_of_devices );

    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"green\">Discovery_Receive_Thread():"
          " Received %ld bytes: Found Device #%u\n\n</span>"),
        bytes_read, hermes2_rc.number_of_devices );
    Discovery_Dialog_Text( mesg );

    /* If sequence number of Discovery response packet is correct,
     * process response packet and count up number of devices */
    uint16_t leader = String_to_Uint16( buffer ); /// NOTE!!!
    if( (leader == METIS_PACKET_LEADER) &&
        (hermes2_rc.number_of_devices < MAX_DEVICES) )
    {
      discovered_device_t *ddv = &Device[hermes2_rc.number_of_devices];

      /* Device status: 2 = Running and available
       * 3 = Running but connected to other host */
      ddv->device_status = buffer[STATUS_BUFF];
      if( (ddv->device_status == STATE_AVAILABLE) ||
          (ddv->device_status == STATE_SENDING) )
      {
        // Read the device MAC address
        for( idx = 0; idx < 6; idx++ )
          ddv->mac_address[idx] = buffer[MAC_ADDR_BUFF + idx];

        // Read and set the device's board type
        ddv->device_type = buffer[BOARD_ID];
        switch( ddv->device_type )
        {
          case DEVICE_ATLAS:
            snprintf( ddv->device_name, sizeof(ddv->device_name),
                "Atlas" );
            ddv->num_of_adcs   = 1;
            ddv->frequency_min = 0;
            ddv->frequency_max = 61440000;
            break;

          case DEVICE_HERMES:
            snprintf( ddv->device_name, sizeof(ddv->device_name),
                "Hermes" );
            ddv->num_of_adcs   = 1;
            ddv->frequency_min = 0;
            ddv->frequency_max = 61440000;
            break;

          case DEVICE_HERMES2:
            snprintf( ddv->device_name, sizeof(ddv->device_name),
                "Hermes-2" );
            ddv->num_of_adcs   = 2;
            ddv->frequency_min = 0;
            ddv->frequency_max = 61440000;
            break;

          case DEVICE_ANGELIA:
            snprintf( ddv->device_name, sizeof(ddv->device_name),
                "Angelia" );
            ddv->num_of_adcs   = 2;
            ddv->frequency_min = 0;
            ddv->frequency_max = 61440000;
            break;

          case DEVICE_ORION:
            snprintf( ddv->device_name, sizeof(ddv->device_name),
                "Orion" );
            ddv->num_of_adcs   = 2;
            ddv->frequency_min = 0;
            ddv->frequency_max = 61440000;
            break;

          case DEVICE_ORION2:
            snprintf( ddv->device_name, sizeof(ddv->device_name),
                "Orion-II" );
            ddv->num_of_adcs   = 2;
            ddv->frequency_min = 0;
            ddv->frequency_max = 61440000;
            break;

          case DEVICE_HERMES_LITE:
            snprintf( ddv->device_name, sizeof(ddv->device_name),
                "Hermes Lite" );
            ddv->num_of_adcs   = 1;
            ddv->frequency_min = 0;
            ddv->frequency_max = 38400000;
            break;

          default:
            Strlcpy( ddv->device_name, _("Unknown"), sizeof(ddv->device_name) );
            break;
        } // switch( ddv->device_type )

        // Gateware major version number
        ddv->gateware_ver[0] = buffer[GATEWARE_MAJOR_VER];

        // Gateware minor version number
        ddv->gateware_ver[1] = buffer[GATEWARE_MINOR_VER];

        // Number of DDC's implemented by device
        ddv->num_of_ddcs = buffer[HARDWARE_RECEIVERS];

        // Wideband Data Format and Board Build ID
        ddv->wideband_data  = buffer[WIDE_BAND_DATA_ID] & 0xC0;
        ddv->board_build_id = buffer[WIDE_BAND_DATA_ID] & 0x1F;

        // External CW key, PTT (Tx On) status and ADC clip count
        ddv->external_cw_key  = (buffer[CWKEY_TXON_ADCCLIP] & BIT_7) >> 7;
        ddv->ptt_tx_on_status = (buffer[CWKEY_TXON_ADCCLIP] & BIT_6) >> 6;
        ddv->adc_clip_count   =  buffer[CWKEY_TXON_ADCCLIP] & 0x2F;

        // Under/Over-flow recovery flag
        ddv->UO_flow_recovery = (buffer[UO_FLOW_TX_IQ_FIFO] & BIT_7) >> 7;

        // TX IQ FIFO count
        ddv->tx_iq_fifo_count =  buffer[UO_FLOW_TX_IQ_FIFO] & 0x4F;

        // Temperature, Tx forward and reverse power, PA bias current
        ddv->temperature =
          (double)( (buffer[TEMPERATURE] << 8) | buffer[TEMPERATURE + 1] );
        ddv->fwd_power   =
          (double)( (buffer[FORWARD_POWER] << 8) | buffer[FORWARD_POWER + 1] );
        ddv->rev_power   =
          (double)( (buffer[REVERSE_POWER] << 8) | buffer[REVERSE_POWER + 1] );
        ddv->pa_current  =
          (double)( (buffer[BIAS_CURRENT]  << 8) | buffer[BIAS_CURRENT  + 1] );

        /* Convert above data to actual values.
         * These formulae were taken from Quisk SDR client source code */

        // 3.30 is nominal Reference voltage. 4096 steps in ADC. Other factors unknown.
        ddv->temperature = ( REF_3_3_VOLT * (ddv->temperature / ADC_SCALE) - 0.50 ) / 0.01;

        // Sense amp gain = 50. Sense resistor = 0.04 Ohm.
        ddv->pa_current = ( (REF_3_3_VOLT * (ddv->pa_current / ADC_SCALE)) / 50.0 ) / 0.04;

        // Scale by resistor voltage divider 1000/(1000+270) at input of slow ADC
        ddv->pa_current = ddv->pa_current / (1000.0 / 1270.0);

        // Convert the HermesLite fwd/rev power code to watts forward and reverse.
        ddv->fwd_power  = REF_3_3_VOLT * ddv->fwd_power / ADC_SCALE; // Fwd Voltage
        ddv->fwd_power *= 0.9034;  // Scale factor
        ddv->fwd_power  = ddv->fwd_power * ddv->fwd_power; // Forward Power watts
        ddv->rev_power  = REF_3_3_VOLT * ddv->rev_power / ADC_SCALE; // Rev Voltage
        ddv->rev_power *= 0.9034;  // Scale factor
        ddv->rev_power  = ddv->rev_power * ddv->rev_power;  // Reverse Power watts

        // Device socket IP address
        memcpy(
            (void *) &(ddv->network_address),
            (void *) &socket_addr, sizeof(socket_addr) );
        ddv->network_address_length = (socklen_t)( sizeof(socket_addr) );

        // Interface socket IP address
        memcpy(
            (void *) &(ddv->interface_address),
            (void *) ptr->iface_addr,
            sizeof(ddv->interface_address) );
        ddv->interface_address_length = (socklen_t)( sizeof(ddv->interface_address) );

        // Interface socket name
        Strlcpy( ddv->interface_name, ptr->iface_name, sizeof(ddv->interface_name) );

        // Count devices found
        hermes2_rc.number_of_devices++;
        break;

      } // if( (ddv->device_status == STATE_AVAILABLE) || ...
    } // if( (snr == PACKET_LEADER) && ...

  } // while( True )

  if( Flag[HERMES2_VERBOSE_MESSAGES] )
    fprintf( stderr, _("hermes2: Discovery_Receive_Thread(): exiting\n") );

  return( NULL );
} // Discovery_Receive_Thread()

//----------------------------------------------------------------------

// Discover request packet size
#define DISCOVER_REQUEST_SIZE    63
#define COMMAND_DISCOVER         0x02
#define COMMAND_POSITION         2

/* Send_Discovery_Packet()
 *
 * Send a discovery packet to an interface and create receive thread
 */
  static BOOLEAN
Send_Discovery_Packet( struct ifaddrs *iface )
{
  int rc, on;
  struct sockaddr_in *sa;
  struct sockaddr_in *mask;
  char iface_name[64], addr[16], net_mask[16];
  discovery_thread_arg_t thread_arg;
  struct sockaddr_in iface_addr = { 0 };
  gchar mesg[MESG_STRING_SIZE];

  // Take interface name and report
  Strlcpy( iface_name, iface->ifa_name, sizeof(iface_name) );
  if( Flag[HERMES2_VERBOSE_MESSAGES] )
    printf( _("hermes2: Send_Discovery_Packet():"
          " Looking for HPSDR devices on %s\n"), iface_name );

  // Report on Discovery dialog
  snprintf( mesg, sizeof(mesg),
      _("Looking for HPSDR devices on %s\n"), iface_name );
  Discovery_Dialog_Text( mesg );

  // Create the discovery socket
  thread_arg.socket_fd = socket( PF_INET, SOCK_DGRAM, IPPROTO_UDP );
  if( thread_arg.socket_fd < 0 )
  {
    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"red\">Send_Discovery_Packet():"
          " socket() failed\n</span>") );
    Discovery_Dialog_Text( mesg );
    perror( _("hermes2: Send_Discovery_Packet(): socket() failed") );
    return( False );
  }

  // Set socket options, reuse addr and reuse port
  on = True;
  rc = setsockopt( thread_arg.socket_fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on) );
  if( rc != 0 )
  {
    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"red\">Send_Discovery_Packet():"
          " cannot set SO_REUSEADDR\n</span>") );
    Discovery_Dialog_Text( mesg );
    perror( _("hermes2: Send_Discovery_Packet():"
          " cannot set SO_REUSEADDR") );
    close( thread_arg.socket_fd );
    return( False );
  }

  rc = setsockopt( thread_arg.socket_fd, SOL_SOCKET, SO_REUSEPORT, &on, sizeof(on) );
  if( rc != 0 )
  {
    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"red\">Send_Discovery_Packet():"
          " cannot set SO_REUSEPORT\n</span>") );
    Discovery_Dialog_Text( mesg );
    perror( _("hermes2: Send_Discovery_Packet():"
          " cannot set SO_REUSEPORT") );
    close( thread_arg.socket_fd );
    return( False );
  }

  // Take the interface socket address
  sa   = (struct sockaddr_in *)iface->ifa_addr;
  mask = (struct sockaddr_in *)iface->ifa_netmask;

  // Bind to this interface and the discovery port
  iface_addr.sin_family      = AF_INET;
  iface_addr.sin_addr.s_addr = sa->sin_addr.s_addr;
  iface_addr.sin_port        = htons( 0 );
  rc = bind(
      thread_arg.socket_fd,
      (struct sockaddr *) &iface_addr,
      sizeof(iface_addr) );
  if( rc < 0 )
  {
    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"red\">Send_Discovery_Packet():"
          " bind() failed\n</span>") );
    Discovery_Dialog_Text( mesg );
    perror( _("hermes2: Send_Discovery_Packet(): bind() failed") );
    close( thread_arg.socket_fd );
    return( False );
  }

  Strlcpy( addr, inet_ntoa(sa->sin_addr), sizeof(addr) );
  Strlcpy( net_mask, inet_ntoa(mask->sin_addr), sizeof(net_mask) );
  if( Flag[HERMES2_VERBOSE_MESSAGES] )
  {
    printf( _("hermes2: Send_Discovery_Packet():"
          " Bound to %s Addr %s Net-Mask %s\n"), iface_name, addr, net_mask );
  }

  // Report on Discovery dialog
  snprintf( mesg, sizeof(mesg),
      _("<span foreground=\"green\">Send_Discovery_Packet():\n"
        "Bound to %s Addr %s Net-Mask %s\n</span>"), iface_name, addr, net_mask );
  Discovery_Dialog_Text( mesg );

  // Allow broadcast on the socket
  on = True;
  rc = setsockopt( thread_arg.socket_fd, SOL_SOCKET,
      SO_BROADCAST, &on, (socklen_t)(sizeof(on)) );
  if( rc != 0 )
  {
    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"red\">Send_Discovery_Packet():"
          " cannot set SO_BROADCAST\n</span>") );
    Discovery_Dialog_Text( mesg );
    perror( _("hermes2: Send_Discovery_Packet():"
          " cannot set SO_BROADCAST") );
    close( thread_arg.socket_fd );
    return( False );
  }

  // Setup to address
  struct sockaddr_in to_addr = { 0 };
  to_addr.sin_family         = AF_INET;
  to_addr.sin_port           = htons( DISCOVERY_PORT );
  to_addr.sin_addr.s_addr    = htonl( INADDR_BROADCAST );

  // Prepare the discovery packet
  uint8_t buffer[DISCOVER_REQUEST_SIZE];
  memset( (void *)buffer, 0, sizeof(buffer) );
  Uint16_to_String( METIS_PACKET_LEADER, buffer );
  buffer[COMMAND_POSITION] = COMMAND_DISCOVER;

  // Start a receive thread to collect discovery response packets
  thread_arg.iface_name = iface_name;
  thread_arg.iface_addr = &iface_addr;
  if( !Pthread_Create(
      &discovery_thread, NULL, Discovery_Receive_Thread, (void *)&thread_arg,
      _("Send_Discovery_Packet(): failed to create thread")) )
  {
    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"red\">Send_Discovery_Packet():"
          " Pthread_Create() failed</span>") );
    Discovery_Dialog_Text( mesg );
    close( thread_arg.socket_fd );
    return( False );
  }

  // Send the discovery packet
  ssize_t len = sendto(
      thread_arg.socket_fd, buffer, sizeof(buffer), 0,
      (struct sockaddr *) &to_addr, sizeof(to_addr) );
  if( (len < 0) || (len != sizeof(buffer)) )
  {
    // Report on Discovery dialog
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"red\">Send_Discovery_Packet():"
          " sendto() failed\n</span>") );
    Discovery_Dialog_Text( mesg );
    perror( _("hermes2: Send_Discovery_Packet(): sendto() failed") );
    if( errno != EHOSTUNREACH ) return( False );
  }

  // Wait for Discovery receive thread to terminate
  pthread_join( discovery_thread, NULL );
  if( Flag[HERMES2_VERBOSE_MESSAGES] )
    printf( _("hermes2: Send_Discovery_Packet(): exiting discovery for %s\n\n"),
        iface->ifa_name );

  // Report on Discovery dialog
  snprintf( mesg, sizeof(mesg),
      _("Send_Discovery_Packet(): exiting discovery for %s\n\n"), iface->ifa_name );
  Discovery_Dialog_Text( mesg );

  return( True );
} // Send_Discovery_Packet()

//----------------------------------------------------------------------

/* Discover_Devices()
 *
 * Looks for HPSDR devices on all Ethernet interfaces
 */
  static void
Discover_Devices( void )
{
  struct ifaddrs *addrs, *ifa;

  // Clear device index and number discovered
  hermes2_rc.device_index = 0;
  hermes2_rc.number_of_devices = 0;

  // Get available Ethernet interfaces
  getifaddrs( &addrs );
  ifa = addrs;

  // Work through real interfaces looking for a HPSR device
  while( ifa )
  {
    if( (ifa->ifa_addr) && (ifa->ifa_addr->sa_family == AF_INET) )
    {
      if( ((ifa->ifa_flags & IFF_UP)       == IFF_UP) &&
          ((ifa->ifa_flags & IFF_RUNNING)  == IFF_RUNNING) &&
          ((ifa->ifa_flags & IFF_LOOPBACK) != IFF_LOOPBACK) )
      {
        if( !Send_Discovery_Packet(ifa) ) return;
        // Comment out for more than one device
        // if( hermes2_rc.number_of_devices ) break;
      }
    }

    ifa = ifa->ifa_next;
  } // while( ifa )
  freeifaddrs( addrs );

  // Print results of discovery
  uint32_t idx;
  for( idx = 0; idx < hermes2_rc.number_of_devices; idx++ )
  {
    char dev_addr[16];
    char ifa_addr[16];

    // Make local copies of addresses
    Strlcpy( dev_addr, inet_ntoa(Device[idx].network_address.sin_addr), sizeof(dev_addr) );
    Strlcpy( ifa_addr, inet_ntoa(Device[idx].interface_address.sin_addr), sizeof(ifa_addr) );

    // Prepare report on discovered device
    gchar mesg[600];
    snprintf( mesg, sizeof(mesg),
        _("Hermes2: Discover_IF():\n"
          "Interface Name   = %s\n"
          "Interface Addr   = %s\n"
          "Device Address   = %s\n"
          "Device Name      = %s\n"
          "Device Number    = %u\n"
          "Device Status    = %u\n"
          "MAC Address      = %02X:%02X:%02X:%02X:%02X:%02X\n"
          "Gateware Version = %up%u\n"
          "Number of DDCs   = %u\n"
          "Number of ADCs   = %u\n"
          "Board Build ID   = %u\n"
          "External CW Key  = %u\n"
          "PTT Tx On Status = %u\n"
          "ADC Clip Count   = %u\n"
          "Under/Overflow   = %u\n"
          "Tx IQ FIFO Count = %u\n"
          "Rig Temperature  = %4.1f\n"
          "Tx Forward Power = %4.2f\n"
          "Tx Reverse Power = %4.2f\n"
          "PA Bias Current  = %4.2f\n" ),
        Device[idx].interface_name,
        ifa_addr,
        dev_addr,
        Device[idx].device_name,
        idx,
        Device[idx].device_status,
        Device[idx].mac_address[0],
        Device[idx].mac_address[1],
        Device[idx].mac_address[2],
        Device[idx].mac_address[3],
        Device[idx].mac_address[4],
        Device[idx].mac_address[5],
        Device[idx].gateware_ver[0],
        Device[idx].gateware_ver[1],
        Device[idx].num_of_ddcs,
        Device[idx].num_of_adcs,
        Device[idx].board_build_id,
        Device[idx].external_cw_key >> 7,
        Device[idx].ptt_tx_on_status >> 6,
        Device[idx].adc_clip_count,
        Device[idx].UO_flow_recovery,
        Device[idx].tx_iq_fifo_count,
        Device[idx].temperature,
        Device[idx].fwd_power,
        Device[idx].rev_power,
        Device[idx].pa_current );

        // Show Wideband IQ data format
        size_t len = strlen( mesg );
        if( Device[idx].wideband_data == 0x00 )
        {
          snprintf( &mesg[len], sizeof(mesg),
              _("Bandscope Format = 12-bit signed extended two's complement\n") );
        }
        else if( Device[idx].wideband_data == 0x40 )
        {
          snprintf( &mesg[len], sizeof(mesg),
              _("Bandscope Format = 16-bit two's complement\n") );
        }
        else  // Should not happen
        {
          snprintf( &mesg[len], sizeof(mesg),
              _("Bandscope Format = Unknown!\n") );
        }

        // Print to console if verbose
        if( Flag[HERMES2_VERBOSE_MESSAGES] )
          puts( mesg );

        // Show report in Discovery dialog in monospace
        gchar markup[650];
        snprintf( markup, sizeof(markup),
            _("<span font_weight=\"bold\">Hermes2 Discovery: %s\n</span>"),
            Device[idx].interface_name );
        Discovery_Dialog_Text( markup );

        snprintf( markup, sizeof(markup),
            _("<span font=\"mono\">%s</span>"), &mesg[23] );
        Discovery_Dialog_Text( markup );

        // Limit max number of DDCs to max number of Transceivers
        if( Device[idx].num_of_ddcs > MAX_RECEIVERS )
          Device[idx].num_of_ddcs = MAX_RECEIVERS;

        // Default config parameters
        Device[idx].transmit_on         = False;
        Device[idx].tx_pa_enable        = False;
        Device[idx].tx_pa_tune          = False;
        Device[idx].duplex              = True;
        Device[idx].lna_gain_auto       = True;
        Device[idx].lna_gain_hw         = True;
        Device[idx].lna_gain            = 20;
        Device[idx].rcve_thread_run     = False;
        Device[idx].rcve_thread_running = False;

  } // for( idx = 0; idx < hermes2_rc.number_of_devices; idx++ )

  return;
} // Discover_Devices()

//----------------------------------------------------------------------

/* Do_Discovery()
 *
 * Open Discovery dialog and discover devices
 */
  gboolean
Do_Discovery( gpointer data )
{
  /* Open discovery dialog and make button box insensitive */
  Discovery_Dialog( HIDE_BUTTON_BOX );

  // Try to discover an HPSDR device
  Discover_Devices();
  if( !hermes2_rc.number_of_devices )
  {
    Discovery_Dialog_Text(
        _("<span foreground=\"red\" font_weight=\"bold\">"
          "Failed to Discover HPSDR Device(s)\n</span>") );

    // Make close button insensitive (only show button box)
    Discovery_Dialog( SHOW_BUTTON_BOX );
    return( FALSE );
  }

  // Show all buttons
  Discovery_Dialog( SHOW_BUTTON_BOX | SHOW_CLOSE_BUTTON );

  // Do the run time config actions if not done
  if( !Flag[HERMES2_RCCONFIG_SETUP] )
    g_idle_add( Verify_Config, NULL );

  return( FALSE );
} // Do_Discovery()

//----------------------------------------------------------------------

