/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "codec.h"
#include "detect.h"
#include "display.h"
#include "operation.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/sound.h"
#include "../rsid/rsid_modes.h"
#include <gtk/gtk.h>
#include <semaphore.h>
#include <stdio.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

//------------------------------------------------------------------------

/* Baud rate definitions. They are slightly different
 * from nominal to make integer arithmetic more accurate */
#define BAUD122     122.5
#define BAUD105     105.0
#define BAUD61       61.25
#define BAUD30       30.625
#define BAUD15       15.3125
#define BAUD7         7.65625
#define BAUD14       14.0
#define BAUD2         2.45
#define BAUD1         1.225
#define BAUD06        0.6125
#define BAUD_RATES    10

// Default normal and low-res (FM105) font file
#define FONT_NRMRES "FeldFatEn.bdf"
#define FONT_LOWRES "FMFatLoEn.bdf"

//------------------------------------------------------------------------

/* Hell_Read_Config()
 *
 * Loads the hellrc configuration file
 */
  gboolean
Hell_Read_Config( gpointer data )
{
  // Set Receive window height
  uint16_t height;

  // Buffer for Read_Line
  char line[READ_LINE_BUF_SIZE];

  // Config file pointer
  FILE *hellrc;


  // Setup file path to hellrc
  Strlcpy( hell_rc_data.rc_fpath, getenv("HOME"),
      sizeof(hell_rc_data.rc_fpath) );
  Strlcat( hell_rc_data.rc_fpath, "/.hermes2/hell/hell.config",
      sizeof(hell_rc_data.rc_fpath) );

  // Open hellrc file
  hellrc = fopen( hell_rc_data.rc_fpath, "r" );
  if( hellrc == NULL )
  {
    perror( hell_rc_data.rc_fpath );
    Error_Dialog(
        _("Failed to open hell.config file\n"\
          "Quit Hell Shcreiber and correct"), HIDE_OK );
    return( FALSE );
  }

  // *** Read runtime configuration data ***

  // Read CW message to send at tag '*', abort if EOF
  if( Read_Line(line, hellrc, _("CW Message") ) != SUCCESS )
    return( FALSE );
  Strlcpy( hell_rc_data.cw_mesg, line, sizeof(hell_rc_data.cw_mesg) );

  // Read word wrap column, abort if EOF
  if( Read_Line(line, hellrc, _("Word Wrap Column") ) != SUCCESS )
    return( FALSE );
  hell_rc_data.word_wrap = (uint8_t)( atoi(line) );

  // *** Setup default main menu items ***
  // Read Record QSO enable flag, abort if EOF
  if( Read_Line(line, hellrc, _("Record QSOs Enable") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(hell_gui.popup_menu_builder, "hell_record_qsos")), TRUE );
    Flag[GUEST_RECORD_QSO] = True;
  }
  else if( strcmp(line, "no") == 0 )
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(hell_gui.popup_menu_builder, "hell_record_qsos")), FALSE );
  else
  {
    Close_File( &hellrc );
    Error_Dialog(
        _("Error reading hell.config\n"\
          "Unrecognized Record QSO option\n"\
          "Quit hell and correct"), HIDE_OK );
    return( FALSE );
  }

  // Disable callbacks to menu handlers to prevent messups
  Flag[GUEST_NO_CALLBACKS] = True;

  // Read Hell Schreiber mode
  if( Read_Line(line, hellrc, _("Hell Schreiber Mode")) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "FeldHell") == 0 )
  {
    // Set appropriate radio button active
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(hell_gui.popup_menu_builder, "hell_feldhell")), TRUE);
    Flag[HELL_MODE_FELDHELL] = True;
    Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_FELDHELL;
  }
  else if( strcmp(line, "FMHell") == 0 )
  {
    // Set appropriate radio button active
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(hell_gui.popup_menu_builder, "hell_fmhell")), TRUE);
    Flag[HELL_MODE_FELDHELL] = False;
    Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_FSKH105;
  }
  else
  {
    Close_File( &hellrc );
    Error_Dialog(
        _("Error reading hell.config\n"\
          "Unrecognized Hell Mode option\n"\
          "Quit hell and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Hell Schreiber baud rate
  if( Read_Line(line, hellrc, _("Hell Schreiber Baud Rate")) != SUCCESS )
    return( FALSE );
  hell_rc_data.baud_rate = Atof( line );

  // Read dot size
  if( Read_Line(line, hellrc, _("Read Dot Size")) != SUCCESS )
    return( FALSE );
  hell_rc_data.dot_size = (uint8_t)( atoi(line) );
  if( (hell_rc_data.dot_size > 4) || (hell_rc_data.dot_size < 2) )
  {
    Close_File( &hellrc );
    Error_Dialog(
        _("Error reading hell.config\n"\
          "Dot size incorrect\n"\
          "Quit hell and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Capitalize enable flag, abort if EOF
  if( Read_Line(line, hellrc, _("Capitalize Enable") ) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "yes") == 0 )
  {
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(hell_gui.popup_menu_builder, "hell_capitalize_letters")), TRUE );
    Flag[GUEST_CAPITALIZE] = True;
  }
  else if( strcmp(line, "no") == 0 )
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
          Builder_Get_Object(hell_gui.popup_menu_builder, "hell_capitalize_letters")), FALSE );
  else
  {
    Close_File( &hellrc );
    Error_Dialog(
        _("Error reading hell.config\n"\
          "Unrecognized Capitalize option\n"\
          "Quit hell and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Receive window half size setting
  if( Read_Line(line, hellrc, _("Receive Window Size")) != SUCCESS )
    return( FALSE );
  if( strcmp(line, "no") == 0 ) height = 0;
  else
    if( strcmp(line, "yes") == 0 ) height = 1;
    else
    {
      Close_File( &hellrc );
      Error_Dialog(
          _("Error reading hell.config\n"\
            "Unrecognized Receive Window Size option\n"\
            "Quit hell and correct"), HIDE_OK );
      return( FALSE );
    }

  // Read Station info and Macros
  if( !Read_Station_Info( hellrc, hell_macro,
        "hell_label", hell_gui.window_builder) )
    return( FALSE );

  // Read main Window's position and place it
  if( Read_Line(line, hellrc, _("Window Position") ) != SUCCESS )
    return( FALSE );
  gint x = (gint)atoi( line );
  gint y = (gint)atoi( &line[5] );
  gtk_window_move( GTK_WINDOW(hell_gui.window), x, y );

  // *** Do the global config variables ***
  // Set appropriate baud rate menu item
  {
    char btn[16];
    GtkWidget *item;

    snprintf( btn, sizeof(btn),
        "hell_br%d", (int)(hell_rc_data.baud_rate * 10.0 + 0.001) );
    btn[15] = '\0';
    item = Builder_Get_Object(hell_gui.popup_menu_builder, btn);
    if( !item )
    {
      Close_File( &hellrc );
      Error_Dialog(
          _("Error reading hell.config\n"\
            "Invalid Baud rate\n"\
            "Quit hell and correct"), HIDE_OK );
      return( FALSE );
    }
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(item), TRUE );
  }

  // Set apropriate dot height menu item
  switch( hell_rc_data.dot_size )
  {
    case 2:
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(hell_gui.popup_menu_builder, "hell_md2x2")), TRUE);
      break;

    case 3:
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(hell_gui.popup_menu_builder, "hell_md3x3")), TRUE);
      break;

    case 4:
      gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(
            Builder_Get_Object(hell_gui.popup_menu_builder, "hell_md4x4")), TRUE);
      break;

    default:
      Close_File( &hellrc );
      Error_Dialog(
          _("Error reading hell.config\n"\
            "Invalid dot height\n"\
            "Quit hell and correct"), HIDE_OK );
      return( FALSE );
  }

  // Set Receive window height check button
  GtkToggleButton *toggle = GTK_TOGGLE_BUTTON(
      Builder_Get_Object(hell_gui.window_builder, "hell_height_checkbutton"));
  if( height )
  {
    gtk_toggle_button_set_active( toggle, TRUE );
    gtk_widget_set_size_request(
        hell_gui.drawingarea, hell_rc_data.rxarea_width, SHORT_HEIGHT );
  }
  else
  {
    gtk_toggle_button_set_active( toggle, FALSE );
    gtk_widget_set_size_request(
        hell_gui.drawingarea, hell_rc_data.rxarea_width, NORMAL_HEIGHT );
  }

  // Close rc file
  Close_File( &hellrc );

  // Enable menu handlers
  Flag[GUEST_NO_CALLBACKS] = False;

  // Clear Font file name and set params
  hell_rc_data.font_file[0] = '\0';
  New_Parameters();

  return( FALSE );
} // End of Hell_Read_Config()

//------------------------------------------------------------------

/* Load_Fonts()
 *
 * Loads a .bdf font file into a font buffer
 */
  static BOOLEAN
Load_Fonts( const char *font_file )
{
  FILE *font_fp = NULL;

  int16_t
    bmx,
    bmy,
    old_fbb_h;

  // Buffer for Read_Line
  char line[READ_LINE_BUF_SIZE];


  // Open font file, abort on error
  font_fp = fopen( font_file, "r" );
  if( font_fp == NULL )
  {
    perror( font_file );
    Error_Dialog( _("Failed to open Font file"), SHOW_OK );
    return( False );
  }

  // Find font bounding box
  if( !Find_Line(line, "FONTBOUNDINGBOX ", font_fp) )
  {
    Error_Dialog(
        _("Error reading Font file\n"\
          "FONTBOUNDINGBOX entry not found"), SHOW_OK );
    return( False );
  }

  uint16_t str = 14;
  Find_String( line, &str );
  hell_rc_data.font_data.fbb_w = (int8_t)( atoi(&line[str]) );
  Find_String( line, &str );
  old_fbb_h = hell_rc_data.font_data.fbb_h;
  hell_rc_data.font_data.fbb_h = (int8_t)( atoi(&line[str]) );
  Find_String( line, &str );
  hell_rc_data.font_data.fbb_x = (int8_t)( atoi(&line[str]) );
  Find_String( line, &str );
  hell_rc_data.font_data.fbb_y = (int8_t)( atoi(&line[str]) );

  // If fbbx height changes, clear Rx window
  if( old_fbb_h != hell_rc_data.font_data.fbb_h )
    Flag[GUEST_CLEAR_RX_WINDOW] = True;

  // Sanitize font bounding box size and position
  if( (hell_rc_data.font_data.fbb_w > 30) ||
      (hell_rc_data.font_data.fbb_w <  5) ||
      (hell_rc_data.font_data.fbb_h > 14) ||
      (hell_rc_data.font_data.fbb_x != 0) ||
      (hell_rc_data.font_data.fbb_y < -4) ||
      (hell_rc_data.font_data.fbb_y > 14) )
  {
    Error_Dialog(
        _("Error reading Font file\n"\
          "FONTBOUNDINGBOX data incorrect"), SHOW_OK );
    return( False );
  }

  /* bitmap_height & bitmap_width are the height and width in pixels
   * of bitmap of Hell characters. This is the product of dot size in
   * pixels * size in dots of Hell characters. Basic Hell Schreiber
   * characters are 7x7 dots but vertically dots are two pixels
   * high so the Tx font height is 2x the character size in dots.
   */
  hell_rc_data.bitmap_height =
    (uint16_t)( hell_rc_data.dot_size * hell_rc_data.font_data.fbb_h );

  // Read font name
  if( !Find_Line(line, "FONT_NAME ", font_fp) )
  {
    Error_Dialog(
        _("Error reading Font file\n"\
          "FONT_NAME entry not found"), SHOW_OK );
    return( False );
  }

  // Remove spaces
  str = 8;
  Find_String( line, &str );
  Strlcpy( hell_rc_data.font_data.font_name, &line[str],
      sizeof( hell_rc_data.font_data.font_name ) );

  // Read number of characters
  if( !Find_Line(line, "CHARS ", font_fp) )
  {
    Error_Dialog(
        _("Error reading Font file\n"\
          "CHARS entry not found"), SHOW_OK );
    return( False );
  }
  hell_rc_data.font_data.num_glyphs = (uint16_t)( atoi(&line[6]) );

  // Sanitize for Coverity Scan
  if( (hell_rc_data.font_data.num_glyphs < 1) ||
      (hell_rc_data.font_data.num_glyphs > 120) )
    return( False );

  // Allocate memory to glyph buffer
  hell_rc_data.font_data.glyph_data = NULL;
  Mem_Alloc( (void **) &hell_rc_data.font_data.glyph_data,
      (size_t)hell_rc_data.font_data.num_glyphs * sizeof(glyph_data_t) );

  // Read all glyph data
  for( uint16_t glyph_idx = 0; glyph_idx < hell_rc_data.font_data.num_glyphs; glyph_idx++ )
  {
    // Read ENCODING
    if( !Find_Line(line, "STARTCHAR ", font_fp) )
    {
      Error_Dialog(
          _("Error reading Font file\n"\
            "STARTCHAR entry not found"), SHOW_OK );
      return( False );
    }

    if( !Find_Line(line, "ENCODING ", font_fp) )
    {
      Error_Dialog(
          _("Error reading Font file\n"\
            "ENCODING entry not found"), SHOW_OK );
      return( False );
    }
    hell_rc_data.font_data.glyph_data[glyph_idx].encoding = (uint8_t)( atoi(&line[9]) );

    // Read DWIDTH
    if( !Find_Line(line, "DWIDTH ", font_fp) )
    {
      Error_Dialog(
          _("Error reading Font file\n"\
            "DWIDTH entry not found"), SHOW_OK );
      return( False );
    }

    str = 5;
    Find_String( line, &str );
    hell_rc_data.font_data.glyph_data[glyph_idx].dwidth_x = (uint8_t)( atoi(&line[str])+1) ;
    Find_String( line, &str );
    hell_rc_data.font_data.glyph_data[glyph_idx].dwidth_y = (uint8_t)( atoi( &line[str]) );

    // Read BBX
    if( !Find_Line(line, "BBX ", font_fp) )
    {
      Error_Dialog(
          _("Error reading Font file\n"\
            "BBX entry not found"), SHOW_OK );
      return( False );
    }

    str = 2;
    Find_String( line, &str );
    hell_rc_data.font_data.glyph_data[glyph_idx].bbx_w = (int8_t)( atoi(&line[str]) );
    Find_String( line, &str );
    hell_rc_data.font_data.glyph_data[glyph_idx].bbx_h = (int8_t)( atoi(&line[str]) );
    Find_String( line, &str );
    hell_rc_data.font_data.glyph_data[glyph_idx].bbx_x = (int8_t)( atoi(&line[str]) );
    Find_String( line, &str );
    hell_rc_data.font_data.glyph_data[glyph_idx].bbx_y = (int8_t)( atoi(&line[str]) );

    // Sanitize bounding box size and position
    if( (hell_rc_data.font_data.glyph_data[glyph_idx].bbx_w > 30) ||
        (hell_rc_data.font_data.glyph_data[glyph_idx].bbx_w <  0) ||
        (hell_rc_data.font_data.glyph_data[glyph_idx].bbx_h > 14) ||
        (hell_rc_data.font_data.glyph_data[glyph_idx].bbx_h <  0) ||
        (hell_rc_data.font_data.glyph_data[glyph_idx].bbx_x != 0) ||
        (hell_rc_data.font_data.glyph_data[glyph_idx].bbx_y < -4) ||
        (hell_rc_data.font_data.glyph_data[glyph_idx].bbx_y > 14) )
    {
      Error_Dialog(
          _("Error reading Font file\n"\
            "BBX data incorrect"), SHOW_OK );
      return( False );
    }

    /* Check DWIDTH and BBX w are same. DWIDTH
     * was incremented by 1 to leave char space */
    if( (hell_rc_data.font_data.glyph_data[glyph_idx].bbx_w !=
          hell_rc_data.font_data.glyph_data[glyph_idx].dwidth_x - 1) &&
        (hell_rc_data.font_data.glyph_data[glyph_idx].encoding != 0x20) )
      fprintf( stderr, _("Encoding %2x: DWIDTH != BBX width\n"),
          hell_rc_data.font_data.glyph_data[glyph_idx].encoding );

    // Read and convert bitmap entries
    int bmap_idx =
      ( hell_rc_data.font_data.fbb_h + hell_rc_data.font_data.fbb_y ) -
      ( hell_rc_data.font_data.glyph_data[glyph_idx].bbx_h +
        hell_rc_data.font_data.glyph_data[glyph_idx].bbx_y );

    // Allocate bitmap buffer
    hell_rc_data.font_data.glyph_data[glyph_idx].bitmap = NULL;
    Mem_Alloc( (void **) &hell_rc_data.font_data.glyph_data[glyph_idx].bitmap,
        sizeof(uint32_t) * (size_t)hell_rc_data.font_data.fbb_h );

    // Clear unused part of bitmap
    for( uint8_t idx = 0; idx < bmap_idx; idx++ )
      hell_rc_data.font_data.glyph_data[glyph_idx].bitmap[idx] = 0;

    if( !Find_Line(line, "BITMAP", font_fp) )
    {
      Error_Dialog(
          _("Error reading Font file\n"\
            "BITMAP entry not found"), SHOW_OK );
      return( False );
    }

    // Read hex bitmap and convert to binary bitmap
    for( uint8_t idx = 0; idx < hell_rc_data.font_data.glyph_data[glyph_idx].bbx_h; idx++ )
    {
      if( Read_Line(line, font_fp, "Font hex bitmap" ) != SUCCESS )
        return( False );
      hell_rc_data.font_data.glyph_data[glyph_idx].bitmap[bmap_idx++] = Hex2Bitmap( line );
    }

    // Clear unused part of bitmap
    for( int idx = bmap_idx; idx < hell_rc_data.font_data.fbb_h; idx++ )
      hell_rc_data.font_data.glyph_data[glyph_idx].bitmap[idx] = 0;

    // Check that there are no lone 1 or 0 bits in glyph columns
    uint8_t   dot_cnt = 1;
    uint32_t last_dot = 0, one = 1; // To silence -sanitize = undefined ??
    for( bmx = 0; bmx < hell_rc_data.font_data.glyph_data[glyph_idx].dwidth_x; bmx++ )
    {
      // Go up in bitmap buffer row by row
      for( bmy = hell_rc_data.font_data.fbb_h-1; bmy >= 0; bmy-- )
      {
        dot_cnt++;
        uint32_t new_dot =
          hell_rc_data.font_data.glyph_data[glyph_idx].bitmap[bmy] & (one << (31-bmx));
        if( new_dot != last_dot )
        {
          // Check for lone 1|0 bits
          if( dot_cnt < 2 )
            fprintf( stderr,
                _("Encoding %2x: Single half-pixel in column %d\n"),
                hell_rc_data.font_data.glyph_data[glyph_idx].encoding, bmx );

          // Check for odd num. of half-pixels in Lo-res fonts
          if( (dot_cnt & 1) && (strstr(hell_rc_data.font_data.font_name, "Lo") != NULL) )
          {
            fprintf( stderr,
                _("Encoding %2x: Odd number of half-pixels in column %d\n"),
                hell_rc_data.font_data.glyph_data[glyph_idx].encoding, bmx );
          }

          dot_cnt = 0;
        }
        last_dot = new_dot;

      } // for( bmy = hell_rc_data.font_data.fbb_h-1; bmy >= 0; bmy-- )
    } // for( bmx = 0; bmx < ..... )
  } // for( glyph_idx = 0; glyph_idx < ..... )

  // Encoding number of first & last charracter in file
  hell_rc_data.font_data.first_glyph = hell_rc_data.font_data.glyph_data[0].encoding;
  hell_rc_data.font_data.last_glyph  =
    hell_rc_data.font_data.first_glyph + hell_rc_data.font_data.num_glyphs;

  Close_File( &font_fp );

  // Calculate samples buffer length
  hell_rc_data.tx_buff_len =
    (uint32_t)hell_rc_data.font_data.fbb_h * hell_rc_data.tx_samp_per_dot;

  return( True );
} // Load_Fonts()

//------------------------------------------------------------------------

/* Hex2Bitmap()
 *
 * Convert a hex number to bitmap
 */
  uint32_t
Hex2Bitmap( const char *hex_str )
{
  uint32_t bitmap;

  const char hex[] =
  {
    '0','1','2','3','4','5','6','7',
    '8','9','A','B','C','D','E','F'
  };

  uint8_t
    str_idx,
    hex_idx;

  // Convert characters of hex string to binary bitmap
  str_idx = bitmap = 0;
  while( (str_idx < 8) && (hex_str[str_idx] != '\0') )
  {
    bitmap <<= 4;
    for( hex_idx = 0; hex_idx < 16; hex_idx++ )
      if( hex[hex_idx] == hex_str[str_idx] ) break;
    str_idx++;

    bitmap |= (uint32_t)hex_idx;
  }

  // Shift bitmap to uppermost bit
  while( str_idx++ < 8 ) bitmap <<= 4;

  return( bitmap );
} // Hex2Bitmap( char *hex_str )

//------------------------------------------------------------------------

/* Render_Font()
 *
 * Renders a font buffer on the Rx screen
 */
  BOOLEAN
Render_Font( const font_data_t *font, int glx )
{
  uint8_t
    dot_val, // Dot value (black|white)
    bmx,     // Bitmap x index
    bmy;     // Bitmap y index

  /* Character column */
  static uint8_t *column = NULL;

  // Allocate memory to column
  uint8_t fbb_h = (uint8_t)( hell_rc_data.dot_size * hell_rc_data.font_data.fbb_h );
  Mem_Realloc( (void **) &column, sizeof(uint8_t) * (size_t)fbb_h );

  // Use columns according to DWIDTH y of font
  uint32_t one = 1; // To silence -sanitize = undefined ??
  for( bmx = 0; bmx < font->glyph_data[glx].dwidth_x; bmx++ )
  {
    // Pointer to current pixel in column
    int fpix = hell_rc_data.dot_size * hell_rc_data.font_data.fbb_h - 1;

    // Go down in bitmap buffer row by row
    for( bmy = 0; bmy < hell_rc_data.font_data.fbb_h; bmy++ )
    {
      // Set black or white according to bit value
      if( font->glyph_data[glx].bitmap[bmy] & (one << (31-bmx)) )
        dot_val = 0xff; // Black dot (intensity is inverted)
      else dot_val = 0x00; // White dot

      // Fill in all pixels of each dot
      for( uint8_t idx = 0; idx < hell_rc_data.dot_size; idx++ )
        column[fpix--] = dot_val;
    }

    // Draw the Hellschreiber glyph column
    g_idle_add( Hell_Draw_Column, (gpointer)column );
    while( g_main_context_iteration(NULL, FALSE) );

  } // for( bmx = font->glyph_data[glx].dwidth_y; bmx >= 0; bmx-- )

  Mem_Free( (void **) &column );
  return( True );
} // Render_Font()

//------------------------------------------------------------------------

/* Hell_Save_Pixbuf()
 *
 * Saves the Receive pixbuf as a jpeg image
 */
  gboolean
Hell_Save_Pixbuf( void )
{
  // For saving pixbuf
  char pixbuf_fpath[FILE_NAME_SIZE];
  GError *error = NULL;
  gboolean err;
  uint8_t len;

  // Variables for reading time (UTC)
  time_t tp;
  struct tm utc;

  if( !Flag[HELL_SAVE_PIXBUF] ) return( TRUE );

  // Make a file path for pixbuf file
  snprintf( pixbuf_fpath, sizeof(pixbuf_fpath),
      "%s/.hermes2/hell/pixbufs/", getenv("HOME") );

  // Prepare a file name as UTC date-time.
  // Default paths are images/ and record/
  time( &tp );
  utc = *gmtime( &tp );
  len = (uint8_t)strlen( pixbuf_fpath );
  strftime( &pixbuf_fpath[len], 21, "%d%b%Y-%H%M%S.jpg", &utc );

  // Save Receive pixbuf
  err = gdk_pixbuf_save( hell_receive_pixbuf.pixbuf, pixbuf_fpath,
      "jpeg", &error, "quality", "75", NULL );
  if( error != NULL )
  {
    fprintf( stderr, "%s\n", error->message );
    Error_Dialog( _("Failed to save Receive pixbuf\n"),SHOW_OK );
  }

  return( err );
} // Hell_Save_Pixbuf()

//------------------------------------------------------------------------

/* New_Parameters()
 *
 * Sets up parameters in the program after one changes
 */
  void
New_Parameters( void )
{
  char btn[16], font_file[FILE_NAME_SIZE];
  const double baud_rate[BAUD_RATES] =
  { BAUD122, BAUD105, BAUD61, BAUD30, BAUD15, BAUD7, BAUD14, BAUD2, BAUD1, BAUD06 };

  if( Flag[GUEST_NO_CALLBACKS] ) return;

  // Find active baud rate menu item
  uint8_t idx;
  for( idx = 0; idx < BAUD_RATES; idx++ )
  {
    snprintf( btn, sizeof(btn), "hell_br%02d", (int)(baud_rate[idx] * 10.0 + 0.001) );
    btn[15] = '\0';
    if( gtk_check_menu_item_get_active(
          GTK_CHECK_MENU_ITEM( Builder_Get_Object(hell_gui.popup_menu_builder, btn))) )
      break;
  }
  if( idx == BAUD_RATES ) idx = 0; // BAUD122
  hell_rc_data.baud_rate = baud_rate[idx];

  // Prepare default Font file path if needed
  if( strlen(hell_rc_data.font_file) )
    Strlcpy( font_file, hell_rc_data.font_file, sizeof(font_file) );
  else
  {
    // Prepare font path
    Strlcpy( font_file, getenv("HOME"), sizeof(font_file) );
    Strlcat( font_file, "/.hermes2/hell/fonts/", sizeof(font_file) );
    if( hell_rc_data.baud_rate == BAUD105 )
    {
      Strlcat( font_file, "12pt/", sizeof(font_file) );
      if( !Flag[HELL_MODE_FELDHELL] && (baud_rate[idx] == BAUD105) )
        Strlcat( font_file, FONT_LOWRES, sizeof(font_file) );
      else
        Strlcat( font_file, FONT_NRMRES, sizeof(font_file) );
    }
    else
    {
      Strlcat( font_file, "14pt/", sizeof(font_file) );
      Strlcat( font_file, FONT_NRMRES, sizeof(font_file) );
    }
  }
  font_file[FILE_NAME_SIZE - 1] = '\0';

  // Free font buffers before loading new fonts
  if( hell_rc_data.font_data.glyph_data != NULL )
  {
    for( idx = 0; idx < hell_rc_data.font_data.num_glyphs; idx++ )
      Mem_Free( (void **) &hell_rc_data.font_data.glyph_data[idx].bitmap );
    Mem_Free( (void **) &hell_rc_data.font_data.glyph_data );
  }
  Load_Fonts( font_file );

  // Duration of various elements in Rx Audio samples
  hell_rc_data.rx_samp_per_dot =
    (uint32_t)( SND_DSP_RATE / hell_rc_data.baud_rate / 2.0 + 0.5 );
  hell_rc_data.rx_samp_per_cycle = SND_DSP_RATE / HELL_AUDIO_FREQUENCY;

  // Duration of various elements in Tx DUC samples
  hell_rc_data.tx_samp_per_dot =
    (uint32_t)( DUC_SAMPLE_RATE / hell_rc_data.baud_rate / 2.0 + 0.5 );
  hell_rc_data.tx_samp_per_cycle = DUC_SAMPLE_RATE / HELL_AUDIO_FREQUENCY;

  // Puts the tone freq in middle of waterfall
  int ix = ( 2 * hell_ifft_data.data_len ) / hell_wfall.width;
  if( ix  )
    hell_ifft_data.ifft_stride = (uint16_t)( SND_DSP_RATE / HELL_AUDIO_FREQUENCY / ix );

  // Height and width in pixels of bitmap of Hell characters
  hell_rc_data.bitmap_height =
    (uint16_t)( hell_rc_data.dot_size * hell_rc_data.font_data.fbb_h );
  hell_rc_data.bitmap_width = (uint16_t)( hell_rc_data.bitmap_height / 2 );

  // Make wavetables for FeldHell
  Make_Cos_Wavetable();

  Flag[HELL_NEW_BAUD_RATE]    = True;
  Flag[GUEST_CLEAR_RX_WINDOW] = True;
  g_idle_add( Hell_Set_TxRx_Labels, NULL );

  // Calculate samples buffer length
  hell_rc_data.tx_buff_len =
    (uint32_t)hell_rc_data.font_data.fbb_h * hell_rc_data.tx_samp_per_dot;

  return;
} // New_Parameters()

//------------------------------------------------------------------------

/* Hell_Cleanup()
 *
 * Cleans up before quitting
 */
  void
Hell_Cleanup( void )
{
  // Save outstanding records
  if( Flag[GUEST_SAVE_RECORD] )
  {
    Save_QSO_Record( &qso_record );
    Hell_Save_Pixbuf();
  }

  // Close open files
  Close_File( &(qso_record.qso_record_fp) );
  Close_File( &(qso_record.adif_log_fp) );
  Close_File( &(qso_record.station_log_fp) );

  // Free Digimode semaphores
  Init_Semaphore( &digimode_semaphore, False );
  Init_Semaphore( &duc_send_semaphore, False );

  // Free receive display and waterfall pixbuf
  if( hell_receive_pixbuf.pixbuf )
    g_object_unref( hell_receive_pixbuf.pixbuf );
  hell_receive_pixbuf.pixbuf = NULL;
  if( hell_wfall.pixbuf )
    g_object_unref( hell_wfall.pixbuf );
  hell_wfall.pixbuf = NULL;

  // Close GUI objects
  g_object_unref( hell_gui.window_builder );
  hell_gui.window_builder = NULL;
  hell_gui.window = NULL;
  hermes2_gui.guest_window = NULL;

  if( hell_gui.popup_menu_builder != NULL )
  {
    g_object_unref( hell_gui.popup_menu_builder );
    hell_gui.popup_menu_builder = NULL;
  }

  /* Free allocations */
  Hell_Free_Codec();
  Hell_Free_Points();
  Hell_Free_Display();
  Hell_Free_Ops();
  Hell_Free_Utils();
  Free_iFFT( &hell_ifft_data );

  // Clear General flags
  Flag[GUEST_RECORD_QSO]        = False;
  Flag[GUEST_RECEIVE_MODE]      = False;
  Flag[GUEST_TRANSMIT_MODE]     = False;
  Flag[GUEST_TRANSMIT_TAG]      = False;
  Flag[GUEST_TRANSMIT_MACRO]    = False;
  Flag[GUEST_TRANSMIT_KEYBD]    = False;
  Flag[GUEST_KEYBD_BUSY]        = False;
  Flag[GUEST_CAPITALIZE]        = False;
  Flag[GUEST_SAVE_RECORD]       = False;
  Flag[GUEST_QUIT]              = False;
  Flag[HERMES2_SEND_DUC_PACKET] = False;
  Flag[TRANSMIT_MORSE_MESG]     = False;
  Flag[GUEST_RECEIVING]         = False;
  Flag[GUEST_TRANSMITTING]      = False;

  // Clear Hellschreiber flags
  Flag[HELL_AUTO_DESKEW]   = False;
  Flag[HELL_SAVE_PIXBUF]   = False;
  Flag[HELL_MODE_FELDHELL] = False;
  Flag[HELL_NEW_BAUD_RATE] = False;
  Flag[HELL_DAREA_CONFIG]  = False;

  // Disable the common Guest modulation mode
  if( Transceiver[Indices.TRx_Index] != NULL )
    Transceiver[Indices.TRx_Index]->guest_modulation_mode = RX_MODE_ITEMS;

} // Hell_Cleanup( void )

//------------------------------------------------------------------------

// Free buffers allocated in this file
  void
Hell_Free_Utils( void )
{
  for( uint8_t idx = 0; idx < NUM_OF_LABELS; idx++ )
    Mem_Free( (void **) &hell_macro[idx] );
  for( uint8_t idx = 0; idx < hell_rc_data.font_data.num_glyphs; idx++ )
    Mem_Free( (void **) &hell_rc_data.font_data.glyph_data[idx].bitmap );
  Mem_Free( (void **) &hell_rc_data.font_data.glyph_data );
  Free_Xmit_Buffers();

} // Hell_Free_Utils();

//------------------------------------------------------------------------

