/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "operation.h"
#include "codec.h"
#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/guest_utils.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/modulate.h"
#include <ctype.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

// Idle character
#define IDLE_CHAR   0x1f

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

//------------------------------------------------------------------------

// Index to macros
uint8_t hell_macro_idx = 0;

/* Hell_Initialize_Transmit()
 *
 * Initializes the Transmit mode
 */
  BOOLEAN
Hell_Initialize_Transmit( void )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Open qso record file
  if( !Open_Record_File("hell") )
  {
    Flag[HERMES2_SEND_DUC_PACKET] = False;
    Flag[GUEST_TRANSMIT_KEYBD]    = False;
    Flag[GUEST_TRANSMIT_MACRO]    = False;
    Flag[GUEST_TRANSMIT_MODE]     = False;
    Flag[GUEST_KEYBD_BUSY]        = False;
    g_idle_add( Hell_Set_TxRx_Labels, NULL );
    MOX_Control( MOX_OFF );
    return( False );
  }

  // Allocate transmit buffers
  uint32_t siz =
    (uint32_t)hell_rc_data.font_data.fbb_h * hell_rc_data.tx_samp_per_dot;
  Alloc_Xmit_Buffers( siz );

  // Allocate loopback buffer
  siz = (uint32_t)hell_rc_data.font_data.fbb_h;
  Mem_Alloc( (void **) &loopback_buf, (size_t)siz * sizeof(uint8_t) );

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(hell_gui.window_builder, "hell_tx_textview") );

  // Set transmit mode
  Flag[GUEST_TRANSMIT_MODE] = True;
  Flag[RSIDTX_ACTIVE]       = False;
  g_idle_add( Hell_Set_TxRx_Labels, NULL );
  while( g_main_context_iteration(NULL, FALSE) );

  // This is just a default since the RSID modulator produces signal samples
  // centered on the Tx weaver frequency and hence centered on the dial frequency
  TRx->tx_weaver_frequency = TRx->rx_weaver_frequency;

  // These are needed by the SSB modulator
  TRx->tx_modulation_mode  = TX_MODE_USBN;
  TRx->tx_bandwidth        = 250;

  // Enable Transmit if needed
  MOX_Control( MOX_ON );
  Flag[HERMES2_SEND_DUC_PACKET] = True;

  // Send RSID tones or Hell signals
  if( Transceiver[Indices.TRx_Index]->tx_rsid_enable )
    Next_Modulator = DUC_Buffer_Transmit;
  else
    Modulator = DUC_Buffer_Transmit;

  return( True );
} // Hell_Initialize_Transmit()

//------------------------------------------------------------------------

/* Hell_Transmit_Keybd()
 *
 * Transmits characters typed on  the keyboard
 */
  void *
Hell_Transmit_Keybd( void *data )
{
  Flag[GUEST_TRANSMITTING] = True;

  // May happen if characters typed in Receive mode
  if( keybd_buf.space_count > 0 ) Flag[GUEST_KEYBD_BUSY] = True;

  // Print a Line Feed
  tx_print_chr.printchr = LF;
  Queue_Character( (gpointer) &tx_print_chr );

  // *** Cancel Transmit if Tx flag cleared ***
  while( !Flag[GUEST_QUIT] && (Flag[GUEST_TRANSMIT_KEYBD] || Flag[GUEST_KEYBD_BUSY]) )
  {
    // Transmit key buffer
    if( keybd_buf.space_count > 0 )
    {
      // Xmit "smiley" etc glyphs with (shift) F1-F12
      if( (keybd_buf.key_buf[keybd_buf.char_pointer] >= GDK_KEY_F1) &&
          (keybd_buf.key_buf[keybd_buf.char_pointer] <= GDK_KEY_F12) )
      {
        uint16_t glx = (uint16_t)( 16 + keybd_buf.key_buf[keybd_buf.char_pointer] - GDK_KEY_F1 );
        if( !Transmit_Character(&hell_rc_data.font_data, glx) )
          break;
      }
      else
      {
        // Illegal chars are ignored
        if( ((int)keybd_buf.key_buf[keybd_buf.char_pointer] >=
              hell_rc_data.font_data.first_glyph) &&
            ((int)keybd_buf.key_buf[keybd_buf.char_pointer] <=
             hell_rc_data.font_data.last_glyph) )
        {
          uint16_t glx = (uint16_t)keybd_buf.key_buf[keybd_buf.char_pointer];
          if( !Transmit_Character(&hell_rc_data.font_data, glx) )
            break;
        }
      } // else of if( (keybd_buff[keybd_buf.char_pointer] >= GDK_KEY_F1) ...

      // Count down spaces
      if( (keybd_buf.key_buf[keybd_buf.char_pointer] == ' ') ||
          (keybd_buf.key_buf[keybd_buf.char_pointer] == '\0') )
        keybd_buf.space_count--;

      /* Reset key buffer pointer. It must go back one count
       * to align with the keystroke buffer pointer due
       * to the extra space inserted after each character */
      if( keybd_buf.space_count == 0 )
      {
        keybd_buf.char_pointer--;
        Flag[GUEST_KEYBD_BUSY] = False;
        if( Flag[GUEST_RECEIVING] ) break;
      }

      keybd_buf.char_pointer++;
      if( keybd_buf.char_pointer >= KEY_BUF_SIZE )
        keybd_buf.char_pointer = 0;
    } // if( keybd_buf.space_count > 0 )
    else  // Transmit idle character
    {
      if( !Transmit_Character(&hell_rc_data.font_data, IDLE_CHAR) )
          break;

      if( Flag[GUEST_TRANSMIT_MACRO] )
        Flag[GUEST_TRANSMIT_KEYBD] = False;

      // Transmit Morse code identity and go to Receive mode
      if( Flag[GUEST_TRANSMIT_ID] && !Flag[GUEST_TRANSMIT_MACRO] )
      {
        // Send Morse Identity message
        Clear_DUC();
        Strlcpy( hermes2_rc.morse_mesg, hell_rc_data.cw_mesg, sizeof(hermes2_rc.morse_mesg) );
        Flag[HERMES2_SEND_DUC_PACKET] = False;
        Flag[TRANSMIT_MORSE_MESG]     = True;
        Modulator = Morse_Transmit;
        if( Flag[GUEST_TRANSMITTING] ) sem_wait( &duc_send_semaphore );
        Flag[HERMES2_SEND_DUC_PACKET] = True;
        break;
      }
    }
  } // while( !Flag[GUEST_QUIT] && ...

  // Abort on user quit
  if( Flag[GUEST_QUIT] )
  {
    MOX_Control( MOX_OFF );
    Flag[GUEST_KEYBD_BUSY]    = False;
    Flag[GUEST_TRANSMIT_MODE] = False;
    Flag[GUEST_TRANSMITTING]  = False;
    g_idle_add( Hell_Set_TxRx_Labels, NULL );
    return( NULL );
  }

  // Exit transmit mode if no macro enabled
  if( !Flag[GUEST_TRANSMIT_MACRO] )
  {
    // Flush Transmit buffers
    if( !Flush_Tx_Buffer((uint32_t)hell_rc_data.tx_buff_len) )
    {
      Flag[GUEST_TRANSMIT_MODE] = False;
      Flag[GUEST_TRANSMITTING]  = False;
      return( NULL );
    }

    Flag[GUEST_TRANSMIT_KEYBD] = False;
    Flag[GUEST_TRANSMIT_ID]    = False;
    Flag[GUEST_KEYBD_BUSY]     = False;
    Flag[GUEST_RECEIVE_MODE]   = True;
    Flag[GUEST_TRANSMIT_MODE]  = False;
    Flag[GUEST_TRANSMITTING]   = False;
    MOX_Control( MOX_OFF );

    if( !Pthread_Create(
          &hermes2_rc.guest_rx_thread, NULL, Hell_Receive_Mode,
          NULL, _("Failed to create Receive thread")) )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      return( NULL );
    }

    g_idle_add( Hell_Set_TxRx_Labels, NULL );
  } // if( !Flag[GUEST_TRANSMIT_MACRO] )
  else // Create a thread to Transmit Macros
  {
    if( !Pthread_Create(
          &hermes2_rc.guest_tx_thread, NULL, Hell_Transmit_Macro, NULL,
          _("Failed to create Transmit_Macro thread")) )
    {
      Flag[GUEST_TRANSMIT_MACRO] = False;
      Flag[GUEST_TRANSMIT_MODE]  = False;
      Flag[GUEST_TRANSMITTING]   = False;
      return( NULL );
    }
  }

  Flag[GUEST_TRANSMITTING] = False;
  return( NULL );
} // End of Hell_Transmit_Keybd()

//------------------------------------------------------------------------

/* Hell_Transmit_Macro()
 *
 * Transmits a pre-recorded Macro
 */
  void *
Hell_Transmit_Macro( void *data )
{
  static uint8_t
    macro_cnt = 0,  // Count macro characters tx'd
    tag_idx   = 0;  // Index for transmitting tags

  // Tags for entering fields in macros
  const char *tags[NUM_OF_TAGS] = TAG_TABLE;

  // String within a tag <>
  char tag_str[TAG_STRING_SIZE];

  // Tag replacement string
  static char tag_rep[TAG_REPLACE_SIZE];


  Flag[GUEST_TRANSMITTING] = True;

  // Start with an idle character
  if( !Transmit_Character(&hell_rc_data.font_data, IDLE_CHAR) )
  {
    Flag[GUEST_TRANSMIT_MODE] = False;
    Flag[GUEST_TRANSMITTING]  = False;
    return( NULL );
  }

  // Abort macro if user disabled, remain in Tx mode
  // Stop macro on '~' character, remain in Tx mode
  while( !Flag[GUEST_QUIT] && Flag[GUEST_TRANSMIT_MACRO] && 
        (hell_macro[hell_macro_idx][macro_cnt] != '~') )
  {
    // Stop macro at string terminator
    // or CW id char (*), go to Rx mode
    if( (hell_macro[hell_macro_idx][macro_cnt] == '\0') ||
        (hell_macro[hell_macro_idx][macro_cnt] == '*') )
    {
      // Flush qso record file
      if( Flag[GUEST_RECORD_QSO] )
        fflush( qso_record.qso_record_fp );

      // Transmit Morse ID if '*'
      if( hell_macro[hell_macro_idx][macro_cnt] == '*' )
      {
        Strlcpy( hermes2_rc.morse_mesg,
            hell_rc_data.cw_mesg, sizeof(hermes2_rc.morse_mesg) );
        Flag[HERMES2_SEND_DUC_PACKET] = False;
        Flag[TRANSMIT_MORSE_MESG]     = True;
        Modulator = Morse_Transmit;
        if( Flag[GUEST_TRANSMIT_MODE] ) sem_wait( &duc_send_semaphore );
        Flag[HERMES2_SEND_DUC_PACKET] = True;
      }

      // Enable receive mode at end of macro or CW ID
      // Flush Transmit buffers
      if( !Flush_Tx_Buffer((uint32_t)hell_rc_data.tx_buff_len) )
      {
        Flag[GUEST_TRANSMIT_MODE]    = False;
        Flag[GUEST_TRANSMITTING] = False;
        return( NULL );
      }

      macro_cnt = tag_idx = 0;
      Flag[GUEST_TRANSMIT_MACRO] = False;
      Flag[GUEST_TRANSMIT_KEYBD] = False;
      Flag[GUEST_KEYBD_BUSY]     = False;
      Flag[GUEST_TRANSMIT_TAG]   = False;
      Flag[GUEST_TRANSMIT_MODE]  = False;
      Flag[GUEST_TRANSMITTING]   = False;
      MOX_Control( MOX_OFF );

      // Go to receive mode if no user quit
      if( !Flag[GUEST_QUIT] )
      {
        Flag[GUEST_RECEIVE_MODE] = True;
        if( !Pthread_Create(
              &hermes2_rc.guest_rx_thread, NULL, Hell_Receive_Mode, NULL,
              _("Failed to create Receive thread")) )
        {
          Flag[GUEST_RECEIVE_MODE]  = False;
          Flag[GUEST_TRANSMIT_MODE] = False;
          Flag[GUEST_TRANSMITTING]  = False;
          return( NULL );
        }
      }

      Flag[GUEST_TRANSMIT_MODE] = False;
      Flag[GUEST_TRANSMITTING]  = False;
      return( NULL );
    } // if( (hell_macro[hell_macro_idx][macro_cnt] == '\0') || ...

    // *** Look for tags and substitude accordingly ***
    if( (hell_macro[hell_macro_idx][macro_cnt] == '<') && !Flag[GUEST_TRANSMIT_TAG] )
    {
      uint16_t idx;

      // *** Copy tag string to buffer and identify ***
      macro_cnt++; // Move into tag field
      Flag[GUEST_TRANSMIT_TAG] = True;

      // Copy line to tag, max 12 chars or EOS
      idx = 0;
      uint16_t siz = (uint16_t)sizeof( tag_str ) - 1;
      while( (idx < siz) && (hell_macro[hell_macro_idx][macro_cnt] != '>') )
      {
        tag_str[idx] = hell_macro[hell_macro_idx][macro_cnt];
        idx++;
        macro_cnt++;
      }

      // Terminate tag buffer and advance macro buffer
      tag_str[idx] = '\0';
      macro_cnt++;

      // Identify tag and substitude accordingly
      idx = 0;
      while( idx < NUM_OF_TAGS )
      {
        if( strcmp(tag_str, tags[idx]) == 0 )
          break;
        idx++;
      }

      // Abort if tag unidentified
      if( idx == NUM_OF_TAGS )
      {
        // If tag is not a number
        if( (tag_str[0] < 0x30) || (tag_str[0] > 0x39) )
        {
          Error_Dialog(
              _("Error reading hell.config file\n"\
                "A tag in a macro is invalid\n"\
                "Quit hell and correct"), HIDE_OK );
          MOX_Control( MOX_OFF );
          Flag[GUEST_TRANSMIT_MODE] = False;
          Flag[GUEST_TRANSMITTING]  = False;
          return( NULL );
        }
        else
        {
          // Take number as glyph encoding and xmit
          if( !Transmit_Character(&hell_rc_data.font_data, (uint16_t)atoi(tag_str)) )
          {
            Flag[GUEST_TRANSMIT_MODE] = False;
            Flag[GUEST_TRANSMITTING]  = False;
            return( NULL );
          }

          Flag[GUEST_TRANSMIT_MODE] = False;
          Flag[GUEST_TRANSMIT_TAG]  = False;
          continue;
        }
      }

      // Replace tags
      siz = sizeof( tag_rep );
      switch( idx )
      {
        case 0: // own-call
          Strlcpy( tag_rep, op_data.call, siz );
          break;

        case 1: // own-name
          Strlcpy( tag_rep, op_data.name, siz );
          break;

        case 2: // own-qth
          Strlcpy( tag_rep, op_data.qth, siz );
          break;

        case 3: // own-loc
          Strlcpy( tag_rep, op_data.loc, siz );
          break;

        case 4: // own-rst
          Strlcpy( tag_rep, qso_record.my_rst, siz );
          break;

        case 5: // rem-call
          Strlcpy( tag_rep, qso_record.dx_call, siz );
          break;

        case 6: // rem-name
          Strlcpy( tag_rep, qso_record.dx_name, siz );
          break;

        case 7: // rem-qth
          Strlcpy( tag_rep, qso_record.dx_qth, siz );
          break;

        case 8: // rem-loc
          Strlcpy( tag_rep, qso_record.dx_loc, siz );
          break;

        case 9: // rem-rst
          Strlcpy( tag_rep, qso_record.dx_rst, siz );
          break;

        case 10: // date-time
          Strlcpy( tag_rep, qso_record.date, siz );
          Strlcat ( tag_rep, " ", siz );
          Strlcat( tag_rep, qso_record.time, siz );
          break;

        case 11: // op-freq, strip trailing spaces
          {
            Strlcpy( tag_rep, qso_record.freq, siz );
            int8_t i = (int8_t)strlen( tag_rep ) - 1;
            for( ; i >= 0; i-- )
              if( tag_rep[i] == ' ' ) tag_rep[i] = '\0';
          }
          break;

        case 12: // app-version
          snprintf( tag_rep, siz, "%s", PACKAGE_STRING );

      } // switch( idx )
    } // if( (hell_macro[macro_idx][macro_cnt] == '<') && )

    // *** Transmit tag replacement ***
    if( Flag[GUEST_TRANSMIT_TAG] )
    {
      if( tag_rep[tag_idx] != '\0' )
      {
        // Capitalize letters if enabled
        if( Flag[GUEST_CAPITALIZE] )
          tag_rep[tag_idx] = (char)toupper( tag_rep[tag_idx] );
        tx_print_chr.printchr = (guint)tag_rep[tag_idx];
        Queue_Character( &tx_print_chr );

        if( !Transmit_Character(
              &hell_rc_data.font_data, (uint16_t)tag_rep[tag_idx]) )
        {
          Flag[GUEST_TRANSMIT_MODE] = False;
          Flag[GUEST_TRANSMITTING]  = False;
          return( NULL );
        }
        tag_idx++;
      } // if( (tag_rep[tag_idx] != '\0') )
      else
      {
        Flag[GUEST_TRANSMIT_TAG] = False;
        tag_idx = 0;
      }

      continue;
    } // if( Flag[TRANSMIT_TAG] )

    // Capitalize letters if enabled
    if( Flag[GUEST_CAPITALIZE] )
      hell_macro[hell_macro_idx][macro_cnt] =
        (char)toupper( hell_macro[hell_macro_idx][macro_cnt] );
    tx_print_chr.printchr = (guint)hell_macro[hell_macro_idx][macro_cnt];
    Queue_Character( &tx_print_chr );

    // Transmit a char from Macro
    if( (hell_macro[hell_macro_idx][macro_cnt] == CR) ||
        (hell_macro[hell_macro_idx][macro_cnt] == LF) )
    {
      if( !Transmit_Character(&hell_rc_data.font_data, ' ') )
      {
        Flag[GUEST_TRANSMIT_MODE] = False;
        Flag[GUEST_TRANSMITTING]  = False;
        return( NULL );
      }
    }
    else
    {
      uint16_t glx = (uint16_t)hell_macro[hell_macro_idx][macro_cnt];
      if( !Transmit_Character(&hell_rc_data.font_data, glx) )
      {
        Flag[GUEST_TRANSMIT_MODE] = False;
        Flag[GUEST_TRANSMITTING]  = False;
        return( NULL );
      }
    }
    macro_cnt++;
  } // while( !Flag[GUEST_QUIT] && ...

  // Clean up some
  if( Flag[GUEST_RECORD_QSO] ) fflush( qso_record.qso_record_fp );
  macro_cnt = tag_idx = 0;
  Flag[GUEST_TRANSMIT_MACRO] = False;
  Flag[GUEST_TRANSMIT_TAG]   = False;

  // Abort on user quit
  if( Flag[GUEST_QUIT] )
  {
    MOX_Control( MOX_OFF );
    Flag[GUEST_TRANSMIT_MODE] = False;
    Flag[GUEST_TRANSMITTING]  = False;
    g_idle_add( Hell_Set_TxRx_Labels, NULL );
    return( NULL );
  }

  /* Switch to Transmit from Keyboard mode
   * if no Receive mode selected */
  if( !Flag[GUEST_RECEIVE_MODE] )
  {
    // Create a thread to run keyboard transmit mode
    Flag[GUEST_TRANSMIT_KEYBD] = True;
    if( !Pthread_Create(
          &hermes2_rc.guest_tx_thread, NULL, Hell_Transmit_Keybd, NULL,
          _("Failed to create Transmit Keyboard thread")) )
    {
      Flag[GUEST_TRANSMIT_KEYBD] = False;
      Flag[GUEST_KEYBD_BUSY]     = False;
      Flag[GUEST_TRANSMIT_MODE]  = False;
      Flag[GUEST_TRANSMITTING]   = False;
      MOX_Control( MOX_OFF );
    }

    Flag[GUEST_TRANSMIT_MODE] = False;
    Flag[GUEST_TRANSMITTING]  = False;
    return( NULL );
  } //if( !Flag[GUEST_RECEIVE_MODE] )
  else
  {
    // Flush Transmit buffers
    if( !Flush_Tx_Buffer((uint32_t)hell_rc_data.tx_buff_len) )
    {
      Flag[GUEST_TRANSMIT_MODE] = False;
      Flag[GUEST_TRANSMITTING]  = False;
      return( NULL );
    }

    Flag[GUEST_TRANSMIT_MODE] = False;
    Flag[GUEST_KEYBD_BUSY]    = False;
    Flag[GUEST_RECEIVE_MODE]  = True;
    Flag[GUEST_TRANSMITTING]  = False;
    MOX_Control( MOX_OFF );

    if( !Pthread_Create(
          &hermes2_rc.guest_rx_thread, NULL, Hell_Receive_Mode, NULL,
          _("Failed to create Receive thread")) )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      return( NULL );
    }
    g_idle_add( Hell_Set_TxRx_Labels, NULL );
  }

  Flag[GUEST_TRANSMITTING] = False;
  return( NULL );
} // Hell_Transmit_Macro()

//------------------------------------------------------------------------

// Character Bitmap height
static uint16_t bm_height = 0;

// Character column data ("element" values)
static uint8_t *column = NULL;

/* Hell_Receive_Mode()
 *
 * Controls receiving and decoding Hell signals
 */
  void *
Hell_Receive_Mode( void *data )
{
  // First call of function flag
  static BOOLEAN first_call = True;

  Flag[GUEST_RECEIVING] = True;

  // *** Initialize ***
  if( first_call )
  {
    // Abort if QSO record file fails to open
    if( !Open_Record_File("hell") )
    {
      g_idle_add( Hell_Set_TxRx_Labels, NULL );
      first_call = True;
      Flag[GUEST_RECEIVE_MODE] = False;
      Flag[GUEST_RECEIVING]    = False;
      return( NULL );
    }

    g_idle_add( Hell_Set_TxRx_Labels, NULL );
    first_call = False;
  } // if( first_call )

  // Cancel Receive if Rx flag cleared
  while( !Flag[GUEST_QUIT] && Flag[GUEST_RECEIVE_MODE] )
  {
    // Allocate memory to column if needed (dot resolution changed)
    if( bm_height != hell_rc_data.bitmap_height )
    {
      bm_height = hell_rc_data.bitmap_height;
      Mem_Realloc( (void **) &column, sizeof(uint8_t) * (size_t)bm_height );
    }

    // *** Draw characters to Receive window ***
    // Make and draw a column, abort on dsp error
    if( !Assemble_Column(column) )
    {
      g_idle_add( Hell_Set_TxRx_Labels, NULL );
      Flag[GUEST_RECEIVE_MODE] = False;
      Flag[GUEST_RECEIVING]    = False;
      return( NULL );
    }

    // Draw the Hellschreiber glyph column
    g_idle_add( Hell_Draw_Column, (gpointer)column );
    while( g_main_context_iteration(NULL, FALSE) );

  } // while( !Flag[GUEST_QUIT] && Flag[GUEST_RECEIVE_MODE] )

  first_call = True;
  Flag[GUEST_RECEIVING] = False;
  g_idle_add( Hell_Set_TxRx_Labels, NULL );

  return( NULL );
} // End of Hell_Receive_Mode()

//------------------------------------------------------------------------

// Free resources
  void
Hell_Free_Ops( void )
{
  Mem_Free( (void **) &loopback_buf );
  Mem_Free( (void **) &column );
  bm_height = 0;
}

//------------------------------------------------------------------------

