/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "interface.h"
#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include <gtk/gtk.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>

//------------------------------------------------------------------

#define HELL_POPUP_MENU_IDS \
  "hell_popup_menu", \
  NULL

#define HELL_WINDOW_IDS \
  "hell_window", \
  "hell_contrast_adjustment", \
  "hell_scroller_adjustment", \
  "hell_power_adjustment", \
  NULL

#define FILECHOOSER_IDS \
  "hell_filechooser", \
  NULL

//------------------------------------------------------------------

/* Create_Hell_Window()
 *
 * Creates the main Hell window
 */
  GtkWidget *
Create_Hell_Window( GtkBuilder **builder )
{
  char *hell_window_ids[] = { HELL_WINDOW_IDS };

  // Initialize Digimode and DUC Send semaphore
  if( !Init_Semaphore(&digimode_semaphore, True) )
    return( NULL );
  if( !Init_Semaphore(&duc_send_semaphore, True) )
    return( NULL );
  xmit_buffer.xmit_buf_i = NULL;
  xmit_buffer.xmit_buf_q = NULL;

  // Create the hell glade file name
  Strlcpy( hell_rc_data.hell_glade,
      getenv("HOME"), sizeof(hell_rc_data.hell_glade) );
  Strlcat( hell_rc_data.hell_glade,
      "/.hermes2/hell/hell.glade", sizeof(hell_rc_data.hell_glade) );

  // Create the main window
  Gtk_Builder( builder, hell_rc_data.hell_glade, hell_window_ids );
  hell_gui.window = Builder_Get_Object( *builder, "hell_window" );

  // Create pop-up menu
  hell_gui.popup_menu = Hell_Popup_Menu( &(hell_gui.popup_menu_builder) );

  // Get drawing area widget
  hell_gui.drawingarea =
    Builder_Get_Object( hell_gui.window_builder, "hell_rx_drawingarea" );

  // Get Tx text buffer and scroller
  hell_gui.tx_text_view =
    GTK_TEXT_VIEW( Builder_Get_Object(hell_gui.window_builder, "hell_tx_textview") );
  hell_gui.tx_text_buffer = gtk_text_view_get_buffer( hell_gui.tx_text_view );

  // Setup Tx text buffer
  tx_print_chr.text_view      = hell_gui.tx_text_view;
  tx_print_chr.text_buffer    = hell_gui.tx_text_buffer;
  tx_print_chr.iter           = &hell_gui.tx_text_buffer_iter;
  tx_print_chr.rx_mode        = False;
  gtk_text_buffer_get_iter_at_offset(
      hell_gui.tx_text_buffer, &hell_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(hell_gui.tx_text_buffer) );

  // Get waterfall widget
  hell_wfall.canvas =
    Builder_Get_Object( hell_gui.window_builder, "hell_wfall_drawingarea" );

  // Get scope widget
  hell_gui.hell_scope =
    Builder_Get_Object( hell_gui.window_builder, "hell_scope_drawingarea" );

  // Get and enter contrast value
  GtkRange *range =
    GTK_RANGE( Builder_Get_Object(hell_gui.window_builder, "hell_contrast") );
  GtkAdjustment *adjust =
    GTK_ADJUSTMENT( gtk_range_get_adjustment(range) );
  uint8_t val = (uint8_t)( gtk_adjustment_get_value(adjust) );
  hell_rc_data.contrast = ELEMENT_MAX - (ELEMENT_MAX / 10) * val;

  // Set labels to Xmit and Rcve buttons
  GtkWidget *button;

  button = Builder_Get_Object( hell_gui.window_builder, "hell_transmit" );
  hell_gui.xmit_status = gtk_label_new( _("TRANSMIT") );
  gtk_widget_show( hell_gui.xmit_status );
  gtk_container_add( GTK_CONTAINER(button), hell_gui.xmit_status );
  g_object_set_data_full( G_OBJECT(hell_gui.window), "hell_transmit_status",
      g_object_ref(hell_gui.xmit_status), (GDestroyNotify)g_object_unref );

  button = Builder_Get_Object( hell_gui.window_builder, "hell_receive" );
  hell_gui.rcve_status = gtk_label_new( _("RECEIVE") );
  gtk_widget_show( hell_gui.rcve_status );
  gtk_container_add( GTK_CONTAINER(button), hell_gui.rcve_status );
  g_object_set_data_full( G_OBJECT(hell_gui.window), "hell_receive_status",
      g_object_ref( hell_gui.rcve_status ), (GDestroyNotify)g_object_unref );

  // Set default flags
  Flag[GUEST_RECORD_QSO]   = True;
  Flag[HELL_SAVE_PIXBUF]   = True;
  Flag[GUEST_TRANSMIT_TAG] = False;

  // Initialize some variables
  hell_rc_data.num_rows       = 2;
  Flag[HELL_REVERSE_VIDEO]    = False;
  Flag[HELL_LOOPBACK]         = True;
  qso_record.adif_log_fp    = NULL;
  qso_record.station_log_fp = NULL;
  hell_rc_data.bitmap_height  = 14;

  // Get sizes of displays
  GtkAllocation alloc;
  gtk_widget_get_allocation( hell_gui.drawingarea, &alloc );
  hell_rc_data.rxarea_width  = (uint16_t)alloc.width;
  hell_rc_data.rxarea_height = (uint16_t)alloc.height;
  if( !Hell_Create_Rx_Pixbuf() )
    Error_Dialog( _("Failed to create pixbuf for Rx window"), HIDE_OK );
  Flag[GUEST_CLEAR_RX_WINDOW] = True;

  // Waterfall drawingarea allocation
  gtk_widget_get_allocation( hell_wfall.canvas, &alloc );
  if( !Create_Wfall_Pixbuf(&hell_wfall, &hell_ifft_data,
        (uint16_t)alloc.width, (uint16_t)alloc.height) )
    Error_Dialog( _("Failed to create pixbuf for waterfall"), HIDE_OK );

  // Scope drawingarea allocation
  gtk_widget_get_allocation( hell_gui.hell_scope, &alloc );
  hell_gui.hell_scope_height = (uint16_t)alloc.height;
  hell_gui.hell_scope_width  = (uint16_t)alloc.width;

  // Load runtime config file, abort on error
  g_idle_add( Hell_Read_Config, hell_gui.window );
  while( g_main_context_iteration(NULL, FALSE) );

  return( hell_gui.window );
} // Create_Hell_Window()

//------------------------------------------------------------------

/* Hell_Popup_Menu()
 *
 * Creates the Hell pop-up menu
 */
  GtkWidget *
Hell_Popup_Menu( GtkBuilder **builder )
{
  char *popup_menu_ids[] = { HELL_POPUP_MENU_IDS };
  Gtk_Builder( builder, hell_rc_data.hell_glade, popup_menu_ids );
  GtkWidget *dialog = Builder_Get_Object( *builder, "hell_popup_menu" );
  return( dialog );
} // Hell_Popup_Menu()

//------------------------------------------------------------------

/* Create_Filechooser()
 *
 * Creates the file chooser dialog
 */
  GtkWidget *
Create_Filechooser( GtkBuilder **builder )
{
  char *filechooser_ids[] = { FILECHOOSER_IDS };
  Gtk_Builder( builder, hell_rc_data.hell_glade, filechooser_ids );
  GtkWidget *dialog = Builder_Get_Object( *builder, "hell_filechooser" );
  return( dialog );
} // Create_Filechooser()

//------------------------------------------------------------------

