/*
 *  This program is free software; you can redistribute it and/or
*  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "shared.h"
#include "utils.h"
#include "../common/common.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>


// Receive and Transmit status indicators
#define RECV_SBY      _("<span background=\"orange\"> RECEIVE (Standby) </span>")
#define HELL_RECV_ON  _("<span background=\"green\" foreground=\"white\"> RECEIVE (%s) </span>")

//------------------------------------------------------------------------

/* Hell_Clear_Rx_Pixbuf()
 *
 * Initializes pixbuf of Receive window
 */
  void
Hell_Clear_Pixbuf( pixbuffer_t *pixbuf, guint32 fg )
{
  guchar *p1, *p2;

  gdk_pixbuf_fill( pixbuf->pixbuf, fg );

  // Draw a box around pixbuf
  p1 = pixbuf->pixels;
  p2 = pixbuf->pixels +
    (pixbuf->height-1) * pixbuf->rowstride;
  for( uint16_t idx = 0; idx < pixbuf->width; idx++ )
  {
    p1[0] = p1[1] = p1[2] = 0;
    p1 += pixbuf->n_channels;
    p2[0] = p2[1] = p2[2] = 0;
    p2 += pixbuf->n_channels;
  }

  p1 = pixbuf->pixels + pixbuf->rowstride;
  p2 = pixbuf->pixels + pixbuf->rowstride +
    (pixbuf->width-1) * pixbuf->n_channels;
  for( uint16_t idx = 0; idx < pixbuf->height-1; idx++ )
  {
    p1[0] = p1[1] = p1[2] = 0;
    p1 += pixbuf->rowstride;
    p2[0] = p2[1] = p2[2] = 0;
    p2 += pixbuf->rowstride;
  }

  Queue_Draw( hell_gui.drawingarea );

} // Hell_Clear_Rx_Pixbuf()

//------------------------------------------------------------------------

static uint8_t *last_column = NULL;
static uint16_t bm_height   = 0;

/* Hell_Draw_Column()
 *
 * Draws a character's column in the Rx drawing area
 */
  gboolean
Hell_Draw_Column( gpointer data )
{
  // Index to current pixel in pixbuf
  static guchar *chr_pix = NULL;

  // Char row height, drawing area column and row
  static uint16_t
    scrn_fill,    // How much of Rx screen is filled
    row_height,   // Total height of rows on screen
    chr_col = 0,  // Current column when painting characters
    chr_row = 0;  // Current row when painting characters

  uint8_t pix_idx; // Index to pixel in column
  const uint8_t *column = (uint8_t *)data;

  // Allocate/clear memory for column buffer
  if( bm_height != hell_rc_data.bitmap_height )
  {
    Mem_Realloc( (void **) &last_column,
        sizeof(uint8_t) * (size_t)hell_rc_data.bitmap_height );
    memset( last_column, 100,
        sizeof(uint8_t) * (size_t)hell_rc_data.bitmap_height );
    bm_height = hell_rc_data.bitmap_height;
  }

  // Clear Receive window if drawing area configured
  if( !Flag[HELL_DAREA_CONFIG] ) return( FALSE );
  if( (chr_pix == NULL) || Flag[GUEST_CLEAR_RX_WINDOW] )
  {
    Flag[GUEST_CLEAR_RX_WINDOW] = False;
    row_height = hell_rc_data.num_rows * hell_rc_data.bitmap_height;
    chr_col   = 2;
    scrn_fill = 0;

    // Put cursor and char line at bottom of display
    chr_row = hell_rc_data.rxarea_height - 2 - row_height;
    chr_pix =
      hell_receive_pixbuf.pixels +
      ( hell_rc_data.rxarea_height - 2 - row_height ) * hell_receive_pixbuf.rowstride +
      2 * hell_receive_pixbuf.n_channels;

  } // if( !chr_pix || Flag[CLEAR_RX_WINDOW] )

  // *** Draw a character column ***
  // Number of columns depends on dot size:
  // 2 for 2x2, 3 for 3x3 and 4 for 4x4 pixels
  for( uint8_t idx = 0; idx < hell_rc_data.dot_size; idx++ )
  {
    // Draw character dot pixels to column pixbuf
    for( pix_idx = 0; pix_idx < hell_rc_data.bitmap_height; pix_idx++ )
    {
      if( Flag[GUEST_CLEAR_RX_WINDOW] ) return( FALSE );
      uint32_t ip =
        hell_receive_pixbuf.rowstride * (hell_rc_data.bitmap_height - pix_idx - 1);
      uint8_t grlev = 0xFF - column[pix_idx];
      chr_pix[ip]     = grlev;
      chr_pix[ip + 1] = grlev;
      chr_pix[ip + 2] = grlev;
    }

    // Draw 2nd raw if traditional 2-row mode is in use
    if( !Flag[HELL_AUTO_DESKEW] )
    {
      for( pix_idx = 0; pix_idx < hell_rc_data.bitmap_height; pix_idx++ )
      {
        if( Flag[GUEST_CLEAR_RX_WINDOW] ) return( FALSE );
        uint32_t ip = hell_receive_pixbuf.rowstride * (row_height - pix_idx - 1);
        uint8_t grlev = 0xFF - last_column[pix_idx];
        chr_pix[ip]     = grlev;
        chr_pix[ip + 1] = grlev;
        chr_pix[ip + 2] = grlev;
      }
    }

    // Draw new column
    gtk_widget_queue_draw_area(
        hell_gui.drawingarea, chr_col, chr_row, 1, row_height );

    chr_col++;
    chr_pix += hell_receive_pixbuf.n_channels;

    // Move back to beginning of row when reaching end of row
    if( chr_col >= hell_receive_pixbuf.width - 2 )
    {
      chr_col = hell_rc_data.bitmap_width + 2;
      chr_pix =
        hell_receive_pixbuf.pixels +
        chr_col * hell_receive_pixbuf.n_channels +
        chr_row * hell_receive_pixbuf.rowstride;

      // Save pixbuf when screen filled
      scrn_fill += row_height;
      if( scrn_fill >= hell_rc_data.rxarea_height - 4 )
      {
        Hell_Save_Pixbuf();
        scrn_fill = 0;
      }

      // Scroll up display one row
      gdk_pixbuf_copy_area(
          hell_receive_pixbuf.pixbuf,
          2, row_height + 2,
          hell_rc_data.rxarea_width  - 4,
          hell_rc_data.rxarea_height - 4 - row_height,
          hell_receive_pixbuf.pixbuf, 2, 2 );

      // Redraw last block to beginning of new line
      gdk_pixbuf_copy_area(
          hell_receive_pixbuf.pixbuf,
          hell_receive_pixbuf.width-2 - hell_rc_data.bitmap_width,
          chr_row,
          hell_rc_data.bitmap_width,
          row_height,
          hell_receive_pixbuf.pixbuf,
          2, chr_row );

      // Clear new row
      for( pix_idx = 0; pix_idx < row_height; pix_idx++ )
      {
        uint32_t ip =
          ( hell_rc_data.rxarea_height - 3 - pix_idx ) * hell_receive_pixbuf.rowstride;
        ip += chr_col * hell_receive_pixbuf.n_channels;

        uint32_t temp = hell_rc_data.rxarea_width - 3;
        for( uint32_t jp = chr_col; jp < temp; jp++ )
        {
          ip += hell_receive_pixbuf.n_channels;
          hell_receive_pixbuf.pixels[ip]     = 0xFF;
          hell_receive_pixbuf.pixels[ip + 1] = 0xFF;
          hell_receive_pixbuf.pixels[ip + 2] = 0xFF;
        }
      }

      Queue_Draw( hell_gui.drawingarea );

    } // if( chr_col >= width-2 )
  } // for( idx = 0; idx < hell_rc_data.dot_wdt; idx++ )

  // Save current column
  memcpy( last_column, column, (size_t)hell_rc_data.bitmap_height );

  return( FALSE );
} // Hell_Draw_Column()

//------------------------------------------------------------------------

/* Hell_Create_Rx_Pixbuff()
 *
 * Creates a GDK pixbuf for the receive window
 */
  BOOLEAN
Hell_Create_Rx_Pixbuf( void )
{
  // Destroy existing pixbuff
  if( hell_receive_pixbuf.pixbuf != NULL )
  {
    g_object_unref( G_OBJECT(hell_receive_pixbuf.pixbuf) );
    hell_receive_pixbuf.pixbuf = NULL;
  }

  // Create pixbuf
  hell_receive_pixbuf.pixbuf = gdk_pixbuf_new(
      GDK_COLORSPACE_RGB, FALSE, 8,
      hell_rc_data.rxarea_width, hell_rc_data.rxarea_height );
  if( hell_receive_pixbuf.pixbuf == NULL ) return( False );

  hell_receive_pixbuf.pixels =
    gdk_pixbuf_get_pixels( hell_receive_pixbuf.pixbuf );
  hell_receive_pixbuf.width =
    (uint16_t)gdk_pixbuf_get_width ( hell_receive_pixbuf.pixbuf ) - 1;
  hell_receive_pixbuf.height =
    (uint16_t)gdk_pixbuf_get_height( hell_receive_pixbuf.pixbuf );
  hell_receive_pixbuf.rowstride =
    (uint16_t)gdk_pixbuf_get_rowstride( hell_receive_pixbuf.pixbuf );
  hell_receive_pixbuf.n_channels =
    (uint16_t)gdk_pixbuf_get_n_channels( hell_receive_pixbuf.pixbuf );

  Hell_Clear_Pixbuf( &hell_receive_pixbuf, 0xd0d0d0ff );
  return( True );

} // Hell_Create_Rx_Pixbuff()

//------------------------------------------------------------------------

/* Hell_Set_TxRx_Labels()
 *
 * Sets up the labels in the Tx/Rx frames
 */
  gboolean
Hell_Set_TxRx_Labels( gpointer data )
{
  char lbl[80];

  // Set Tx button label
  if( Flag[GUEST_TRANSMIT_MODE] || Flag[GUEST_TRANSMITTING] )
    gtk_label_set_markup( GTK_LABEL(hell_gui.xmit_status), XMIT_ON );
  else
    gtk_label_set_markup( GTK_LABEL(hell_gui.xmit_status), XMIT_OFF );

  // Set Tx fonts label
  gtk_entry_set_text(
      GTK_ENTRY(Builder_Get_Object(hell_gui.window_builder, "hell_font_entry")),
      hell_rc_data.font_data.font_name );

  // Set Tx/Rx mode and baud rate
  if( Flag[HELL_MODE_FELDHELL] )
    snprintf( lbl, sizeof(lbl), "FeldHell @ %1.2fBd", hell_rc_data.baud_rate );
  else
    snprintf( lbl, sizeof(lbl), "FMHell @ %1.2fBd", hell_rc_data.baud_rate );
  gtk_entry_set_text(
      GTK_ENTRY(Builder_Get_Object(hell_gui.window_builder, "hell_mode_entry")), lbl );

  // Set Rx button label
  if( Flag[GUEST_RECEIVE_MODE] )
  {
    if( Flag[GUEST_KEYBD_BUSY] )
      gtk_label_set_markup( GTK_LABEL(hell_gui.rcve_status), RECV_SBY );
    else
    {
      if( Flag[HELL_AUTO_DESKEW] )
        snprintf( lbl, sizeof(lbl), HELL_RECV_ON, _("Deskew") );
      else
        snprintf( lbl, sizeof(lbl), HELL_RECV_ON, _("Normal") );

      gtk_label_set_markup( GTK_LABEL(hell_gui.rcve_status), lbl );
    }
  }
  else gtk_label_set_markup( GTK_LABEL(hell_gui.rcve_status), RECV_OFF );

  return( FALSE );
} // Hell_Set_TxRx_Labels()

//------------------------------------------------------------------------

// Free resources
  void
Hell_Free_Display( void )
{
  Mem_Free( (void **)&last_column );
  bm_height = 0;
}

//------------------------------------------------------------------------

