/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "detect.h"
#include "codec.h"
#include "shared.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/sound.h"
#include <cairo/cairo.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <math.h>
#include <stdlib.h>

// Scale factors to keep signal level < 255
#define FELDHL_SCALE_FACTOR      950.0
#define FMHELL_SCALE_FACTOR     1300.0

// Points to plot
static GdkPoint *points = NULL;

// Local samples buffer
static int16_t *local_buf[2] = { NULL, NULL };

//------------------------------------------------------------------------

/* Get_Element()
 *
 * Gets the signal level over a part of a character's dot.
 */
  BOOLEAN
Get_Element( uint8_t *elem_lev )
{
  uint8_t ret;

  // Initialize on first call
  if( points == NULL )
  {
    size_t mreq = sizeof(GdkPoint) * (size_t)hell_gui.hell_scope_width;
    Mem_Alloc((void **) &points, mreq);
    memset( points, 0, mreq );
  }

  // Get the signal level according to mode
  if( Flag[HELL_MODE_FELDHELL] )
  {
    ret = Feld_Level( elem_lev );
    if( ret == ABORT ) return( False );
  }
  else
  {
    ret = FMHell_Level( elem_lev );
    if( ret == ABORT ) return( False );
  }

  // Display (scaled) signal level graph
  static uint16_t points_idx = 0;

  // Save values to be plotted (y is scaled)
  points[points_idx].y = hell_gui.hell_scope_height - *elem_lev - 1;
  points[points_idx].x = points_idx;

  // Recycle buffer idx when full and plot
  points_idx++;
  if( points_idx >= hell_gui.hell_scope_width )
  {
    points_idx = 0;
    Queue_Draw( hell_gui.hell_scope );
  }

  return( True );
} // End of Get_Element()

//------------------------------------------------------------------------

/* Feld_Level()
 *
 * Detects level of a FeldHell signal using Goertzel's algorithm.
 */
  uint8_t
Feld_Level( uint8_t *elem_lev )
{
  static BOOLEAN first_call = True;

  static uint8_t
    loc_buf_idx  = 0,  // Index to local samples buffer in use
    digi_buf_cnt = 0;  // Index to digimode samples buffer currently in use

  // Font Bounding Box Height
  static int8_t fbb_h = 0;

  // Integration period of Goertzel detector
  uint16_t det_period = 0;

  // Variables for the Goertzel algorithm
  static double coeff, scale;
  double q1 = 0.0, q2 = 0.0;

  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return( ABORT );

  // Initialize on change of parameters
  if( first_call || Flag[HELL_NEW_BAUD_RATE] ||
      (fbb_h != hell_rc_data.font_data.fbb_h) )
  {
    first_call = False;
    Flag[HELL_NEW_BAUD_RATE] = False;
    digi_buf_cnt = 0;

    // Allocate local buffers
    fbb_h = hell_rc_data.font_data.fbb_h;
    size_t siz = (size_t)( hell_rc_data.rx_samp_per_dot + (uint32_t)fbb_h ) * sizeof(int16_t);
    Mem_Alloc( (void **) &local_buf[0], siz );
    Mem_Alloc( (void **) &local_buf[1], siz );

    // Calculate omega for Goertzel detector
    double w = M_2PI / (double)hell_rc_data.rx_samp_per_cycle;
    coeff = 2.0 * cos( w );
  } // if( first_call || Flag[NEW_BAUD_RATE] || ... )

  // Use samples from local buffer for a font dot pixel, include deskew if selected
  det_period = (uint16_t)( (int16_t)hell_rc_data.rx_samp_per_dot + hell_rc_data.deskew );
  for( uint16_t idx = 0; idx < det_period; idx++ )
  {
    // *** Calculate signal element level ***
    // Calculate element level using Goertzel algorithm
    int16_t signal_sample = local_buf[loc_buf_idx][idx];
    double q0 = coeff * q1 - q2 + (double)signal_sample;
    q2 = q1;
    q1 = q0;
  } // for( uint16_t idx = 0; idx < det_period; idx++ )

  // Increment local buffer index and keep to 0 or 1 value
  loc_buf_idx = (loc_buf_idx + 1) & 0x01;

  // Use samples from digimodes buffer for a font dot pixel,
  // include deskew if selected. Save samples to local buffer
  for( uint16_t idx = 0; idx < det_period; idx++ )
  {
    // Digimode buffer index
    static uint32_t digi_buf_idx = 0;

    // Get next signal sample from digimodes buffer
    int16_t signal_sample = digimode_buffer[digi_buf_cnt][digi_buf_idx];
    local_buf[loc_buf_idx][idx] = signal_sample;

    // Display waterfall when input buffer full
    if( iFFT_Data(signal_sample, &hell_ifft_data) )
      Display_Waterfall( &hell_wfall, &hell_ifft_data );

    // *** Calculate signal element level ***
    // Calculate element level using Goertzel algorithm
    double q0 = coeff * q1 - q2 + (double)signal_sample;
    q2 = q1;
    q1 = q0;

    // Wait for digimode buffer to be filled when used up
    digi_buf_idx++;
    if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
    {
      digi_buf_idx = 0;

      // Increment current audio sample output buffer index
      digi_buf_cnt++;
      if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
        digi_buf_cnt = 0;

      if( digi_buf_cnt == digi_buf_input )
        sem_wait( &digimode_semaphore );
    } // if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
  } // for( uint16_t idx = 0; idx < det_period; idx++ )

  // To reduce signal level to usable value
  scale = 2.0 * det_period * FELDHL_SCALE_FACTOR;

  // Level of input signal scaled down and clamped
  q1 /= scale;
  q2 /= scale;
  uint16_t tmp = (uint16_t)( q1 * q1 + q2 * q2 - q1 * q2 * coeff );
  if( tmp > 254 ) tmp = 254;
  else if( tmp < 1 ) tmp = 1;
  *elem_lev = (uint8_t)tmp;

  return( READY );
} // Feld_Level()

//------------------------------------------------------------------------

/* FMHell_Level()
 *
 * Detects frequency shifts of FMHell carrier
 */
  uint8_t
FMHell_Level( uint8_t *elem_lev )
{
  static BOOLEAN first_call = True;

   // Integration period of Goertzel detector
  static uint16_t det_period = 0;

  int16_t
    blk_lev,  // Level of the black FeldHell signal 'element'
    wht_lev;  // Level of the white FeldHell signal 'element'

  // Variables for the Goertzel algorithm
  static double blk_coeff, wht_coeff, scale;
  static double blk_q1 = 0.0, blk_q2 = 0.0;
  static double wht_q1 = 0.0, wht_q2 = 0.0;

  static uint8_t
    loc_buf_idx  = 0,  // Index to local samples buffer in use
    digi_buf_cnt = 0;  // Index to digimode samples buffer currently in use

  // Font Bounding Box Height
  static int8_t fbb_h = 0;

  // Abort if Receive stopped
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return( ABORT );

  // Initialize on change of parameters
  if( first_call || Flag[HELL_NEW_BAUD_RATE] ||
      (fbb_h != hell_rc_data.font_data.fbb_h) )
  {
    first_call = False;
    Flag[HELL_NEW_BAUD_RATE] = False;
    digi_buf_cnt = 0;

    // Allocate local buffers
    fbb_h = hell_rc_data.font_data.fbb_h;
    size_t siz = (size_t)( hell_rc_data.rx_samp_per_dot + (uint32_t)fbb_h ) * sizeof(int16_t);
    Mem_Alloc( (void **) &local_buf[0], siz );
    Mem_Alloc( (void **) &local_buf[1], siz );

    // Frequency shift is +-baud_rate/2 for Minimum Shift Keying
    // Omega for the white frequency
    double w = M_2PI / SND_DSP_RATE * ( HELL_AUDIO_FREQUENCY + hell_rc_data.baud_rate / 2.0 );
    double wht_cosw = cos( w );
    wht_coeff = 2.0 * wht_cosw;

    // Omega for the black frequency
    w = M_2PI / SND_DSP_RATE * ( HELL_AUDIO_FREQUENCY - hell_rc_data.baud_rate / 2.0 );
    double blk_cosw = cos( w );
    blk_coeff = 2.0 * blk_cosw;

  } // if( first_call... )

  // Use samples from local buffer for a font dot pixel, include deskew if selected
  det_period = (uint16_t)( (int16_t)hell_rc_data.rx_samp_per_dot + hell_rc_data.deskew );
  for( uint16_t idx = 0; idx < det_period; idx++ )
  {
    // *** Calculate signal element level ***
    // Calculate element level using Goertzel algorithm
    int16_t signal_sample = local_buf[loc_buf_idx][idx];

    /* Calculate signal level of black and white
     * tone frequencies using Goertzel algorithm */
    double blk_q0 = blk_coeff * blk_q1 - blk_q2 + (double)signal_sample;
    blk_q2 = blk_q1;
    blk_q1 = blk_q0;

    double wht_q0 = wht_coeff * wht_q1 - wht_q2 + (double)signal_sample;
    wht_q2 = wht_q1;
    wht_q1 = wht_q0;
  } // for( uint16_t idx = 0; idx < det_period; idx++ )

  // Increment local buffer index and keep to 0 or 1 value
  loc_buf_idx = (loc_buf_idx + 1) & 0x01;

  // Use samples from digimodes buffer for a font dot pixel,
  // include deskew if selected. Save samples to local buffer
  for( uint16_t idx = 0; idx < det_period; idx++ )
  {
    // Digimode buffer index
    static uint32_t digi_buf_idx = 0;

    // Get next signal sample from digimodes buffer
    int16_t signal_sample = digimode_buffer[digi_buf_cnt][digi_buf_idx];
    local_buf[loc_buf_idx][idx] = signal_sample;

    // Display waterfall when input buffer full
    if( iFFT_Data(signal_sample, &hell_ifft_data) )
      Display_Waterfall( &hell_wfall, &hell_ifft_data );

    /* Calculate signal level of black and white
     * tone frequencies using Goertzel algorithm */
    double blk_q0 = blk_coeff * blk_q1 - blk_q2 + (double)signal_sample;
    blk_q2 = blk_q1;
    blk_q1 = blk_q0;

    double wht_q0 = wht_coeff * wht_q1 - wht_q2 + (double)signal_sample;
    wht_q2 = wht_q1;
    wht_q1 = wht_q0;

    // Wait for digimode buffer to be filled when used up
    digi_buf_idx++;
    if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
    {
      digi_buf_idx = 0;

      // Increment current audio sample output buffer index
      digi_buf_cnt++;
      if( digi_buf_cnt >= NUM_DIGIMODE_BUFFERS )
        digi_buf_cnt = 0;

      if( digi_buf_cnt == digi_buf_input )
        sem_wait( &digimode_semaphore );
    } // if( digi_buf_idx >= DIGIMODE_BUFFER_SIZE )
  } // for( uint16_t idx = 0; idx < det_period; idx++ )

  // To reduce signal level to usable value
  scale = 2.0 * det_period * FMHELL_SCALE_FACTOR;

  // Magnitude of black tone scaled by dot size and tone freq
  blk_q1 /= scale;
  blk_q2 /= scale;
  blk_lev = (int16_t)
    ( blk_q1 * blk_q1 + blk_q2 * blk_q2 - blk_q1 * blk_q2 * blk_coeff );

  // Magnitude of white tone scaled by dot size and tone freq
  wht_q1 /= scale;
  wht_q2 /= scale;
  wht_lev = (int16_t)
    ( wht_q1 * wht_q1 + wht_q2 * wht_q2 - wht_q1 * wht_q2 * wht_coeff );

  /* The detector o/p is the level diff
   * between white and black signals */
  int16_t tmp = 127 + blk_lev - wht_lev;
  if( tmp > 254 ) tmp = 254;
  if( tmp < 1 )   tmp = 1;
  *elem_lev = (uint8_t)tmp;

  // Reset for next detection
  wht_q1 = wht_q2 = 0.0;
  blk_q1 = blk_q2 = 0.0;

  return( READY );
} // End of FMHell_Level()

//------------------------------------------------------------------------

/* Hell_Display_Signal()
 *
 * Updates the signal scope display
 */
  void
Hell_Display_Signal( cairo_t *cr )
{
  // Draw scope background
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_rectangle(
      cr, 0.0, 0.0,
      (double)hell_gui.hell_scope_width,
      (double)hell_gui.hell_scope_height );
  cairo_fill( cr );

  // Plot signal graph
  if( points == NULL ) return;
  cairo_set_source_rgb( cr, SCOPE_FOREGND );
  cairo_move_to( cr,
      (double)points[0].x,
      (double)points[0].y );
  for( uint16_t idx = 1; idx < hell_gui.hell_scope_width; idx++ )
    cairo_line_to( cr,
        (double)points[idx].x,
        (double)points[idx].y );

  // Stroke paths
  cairo_stroke( cr );

} // Hell_Display_Signal( void )

//------------------------------------------------------------------------

// Free resources
  void
Hell_Free_Points( void )
{
  Mem_Free( (void **) &points );
  Mem_Free( (void **) &local_buf );
}

//------------------------------------------------------------------------

