/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "codec.h"
#include "display.h"
#include "detect.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include <gtk/gtk.h>
#include <math.h>
#include <semaphore.h>
#include <stdint.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#define MAXSEG_AVE_WIN      4 // Max seg ave window len (samples)
#define ROW_AVE_WIN         3 // Length of row averaging window

// Tone control flags
#define SILENT_TONE     0x01
#define RAISE_TONE      0x02
#define LOWER_TONE      0x03
#define KEEP_TONE       0x04

// cosine lookup tables
static int16_t *cosine_table = NULL;

// Element dot values for a column
static uint8_t *dots = NULL;

// Column buffer for drawing chars
static uint8_t *column = NULL;

// Average element value per row
static uint16_t *row_ave = NULL;

// Font bbx height flag
static int8_t fbb_h = 0;

// Font bbx height flag / 2
static int8_t fbb_h2 = 0;

//------------------------------------------------------------------------

/* Assemble_Column()
 *
 * Assembles a character column ( 14 or 12 dots) from incoming
 * Hell Schreiber pulse train. The column is stored as an array
 * of signal intensity values for the elements that comprise
 * each dot, depending on dot size. For a 2x4 pixel/dot font size,
 * there would be 4 elements/dot e.g. 1 elements/pixel vertically.
 */
  BOOLEAN
Assemble_Column( uint8_t *colmn )
{
  uint8_t
    row_min,    // Minimum value of row averages
    row_max,    // Maximum value of row averages
    min_idx,    // Index to row with minimum value
    max_idx,    // Index to row with maximum value
    elem_lev,   // Signal level during a element
    elem_max;   // Max value of column's elements

  // Average value of element max
  static uint16_t
    col_ave,       // Average value of columns
    ave_max = 0;   // Average of element max values


  // Allocate memory to row_average
  if( (fbb_h != hell_rc_data.font_data.fbb_h) || !fbb_h || !fbb_h2 )
  {
    fbb_h  = hell_rc_data.font_data.fbb_h;
    fbb_h2 = fbb_h / 2;
    Mem_Realloc( (void **) &row_ave, sizeof(int) * (size_t)fbb_h );
    memset( row_ave, 0, sizeof(uint16_t) * (size_t)fbb_h );

    // Allocate the element dots buffer
    Mem_Realloc( (void **) &dots, sizeof(uint8_t) * (size_t)fbb_h );
  } // if( fbb_h != hell_rc_data.font_data.fbb_h )

  // Assemble a column's dots. The entries in the elem
  // array represent "elements" of dots. There is one
  // element/pixel in a dot, according to its height.
  elem_max = min_idx = max_idx = row_max = col_ave = 0;
  row_min = 255;
  for( uint8_t idx = 0; idx < fbb_h; idx++ )
  {
    // Get the signal level of an Element dot
    if( !Get_Element(&elem_lev) ) return( False );
    dots[idx] = elem_lev;

    /* Find average value of each row of a character.
     * Needed for automatic deskewing of the display. */
    if( Flag[HELL_AUTO_DESKEW] )
    {
      row_ave[idx] *= ( ROW_AVE_WIN - 1 );
      row_ave[idx] += elem_lev;
      row_ave[idx] /= ROW_AVE_WIN;
      col_ave += row_ave[idx];

      // Find row with the lowest value
      if( row_ave[idx] < row_min )
      {
        row_min = (uint8_t)row_ave[idx];
        min_idx = idx;
      }

      // Find row with the highest value
      if( row_ave[idx] > row_max )
      {
        row_max = (uint8_t)row_ave[idx];
        max_idx = idx;
      }
    } // if( Flag[AUTO_DESKEW] )

    // Find max of element levels
    if( elem_lev > elem_max ) elem_max = elem_lev;
  } // for( idx = 0; idx < fbb_h; idx++ )

  // Try to correct skew by adjusting sampling freq
  if( Flag[HELL_AUTO_DESKEW] )
  {
    int16_t
      deskew_idx  = 0,  // Index used to calc resampling
      deskew_intg = 0;  // Integral of deskew factor

    /* Select either index to min level row (usually
     * when characters are being sent) or index to max
     * level row (usually when idle dots are being sent)
     * On reverse video the test has to be reversed */
    col_ave /= fbb_h;
    BOOLEAN test = row_max < (3 * col_ave);
    if( Flag[HELL_REVERSE_VIDEO] ) test = !test;

    if( test )  // Receiving characters (normal video)
      deskew_idx = min_idx;
    else        // Receiving idle dots (normal video)
      deskew_idx = max_idx;

    /* Change the direction of skew
     * correction around mid column */
    if( deskew_idx > fbb_h2 )
      hell_rc_data.deskew = -deskew_idx;
    else
      hell_rc_data.deskew = deskew_idx;

    // Integrate error if in limits
    if( (hell_rc_data.deskew > 0) && (deskew_intg < fbb_h) )
      deskew_intg++;
    if( (hell_rc_data.deskew < 0) && (deskew_intg > -fbb_h) )
      deskew_intg--;

    // Make it proportional + integral control
    hell_rc_data.deskew += deskew_intg;
    if( hell_rc_data.deskew > fbb_h )  hell_rc_data.deskew = fbb_h;
    if( hell_rc_data.deskew < -fbb_h ) hell_rc_data.deskew = -fbb_h;

  } // if( Flag[AUTO_DESKEW] )
  else hell_rc_data.deskew = 0;

  // Sliding window average of element max
  ave_max *= ( MAXSEG_AVE_WIN - 1 );
  ave_max += elem_max;
  ave_max /= MAXSEG_AVE_WIN;

  // *** Scale element values to a max of 255 ***
  // Limits up-scaling of noise
  uint16_t temp = hell_rc_data.contrast * MAXSEG_AVE_WIN;
  if( ave_max < temp ) ave_max = temp;
  if( !ave_max ) ave_max = 1;

  // Scale element values to a max of 255
  uint8_t idc = 0;
  for( uint8_t idx = 0; idx < fbb_h; idx++ )
  {
    min_idx++;
    if( min_idx >= fbb_h ) min_idx = 0;
    temp = 255 * MAXSEG_AVE_WIN * dots[idx] / ave_max;
    if( temp > 255 ) temp = 255;
    elem_lev = (uint8_t)temp;

    // Fill in pixels of element dot
    for( uint8_t idd = 0; idd < hell_rc_data.dot_size; idd++ )
      colmn[idc++] = elem_lev;
  }

  return( True );
} // Assemble_Column()

//------------------------------------------------------------------------

// Bitmap Height
static uint16_t bm_height = 0;

/* Hell_Loopback()
 *
 * Loops back the transmitted Hellscreiber
 * signal to the Receive window for monitoring
 */
  static void
Hell_Loopback( void )
{
  static uint16_t ave_max = 0;

  uint8_t
    elem_lev,   // Signal level during a element
    elem_max;   // Max value of column's elements

  // Loopback buffer index
  uint16_t buf_idx = 0;

  // Allocate memory to dots buffer
  if( (fbb_h != hell_rc_data.font_data.fbb_h) || !fbb_h )
  {
    fbb_h  = hell_rc_data.font_data.fbb_h;
    Mem_Realloc( (void **) &dots, sizeof(uint8_t) * (size_t)fbb_h );
  } // if( fbb_h != hell_rc_data.font_data.fbb_h )

  // Assemble a column's dots. The entries in the elem
  // array represent "elements" of dots. There is one
  // element/pixel in a dot, according to its height.
  elem_max = 0;
  for( uint8_t idx = 0; idx < fbb_h; idx++ )
  {
    // Get the signal level of an Element dot
    elem_lev = dots[idx] = loopback_buf[buf_idx++];

    // Find max of element levels
    if( elem_lev > elem_max ) elem_max = elem_lev;
  } // for( idx = 0; idx < fbb_h; idx++ )

  // Sliding window average of element max
  ave_max *= ( MAXSEG_AVE_WIN - 1 );
  ave_max += elem_max;
  ave_max /= MAXSEG_AVE_WIN;

  // *** Scale element values to a max of 255 ***
  // Limits up-scaling of noise
  uint8_t temp = hell_rc_data.contrast * MAXSEG_AVE_WIN;
  if( ave_max < temp ) ave_max = temp;
  if( !ave_max ) ave_max = 1;

  // Scale element values to a max of 255
  uint8_t idc = 0;
  for( uint8_t idx = 0; idx < fbb_h; idx++ )
  {
    uint16_t tmp = 255 * MAXSEG_AVE_WIN * dots[idx] / ave_max;
    if( tmp > 255 ) tmp = 255;
    elem_lev = (uint8_t)tmp;

    // Fill in pixels of element dot
    for( uint8_t idd = 0; idd < hell_rc_data.dot_size; idd++ )
      column[idc++] = elem_lev;
  }

  // Draw the Hellschreiber glyph column
  g_idle_add( Hell_Draw_Column, (gpointer)column );
  while( g_main_context_iteration(NULL, FALSE) );

} // Hell_Loopback

//------------------------------------------------------------------------

/* Tx_Feld_Character()
 *
 * Transmits one Feld Hell character pointed by "glx"
 */
  static BOOLEAN
Tx_Feld_Character( font_data_t *font_data, uint16_t glx )
{
  int8_t
    bmx,  // Bitmap x index
    bmy;  // Bitmap y index

  // Tone raise/lower control indx
  static uint8_t tone_ctrl = 0;

  uint32_t
    half_dot, // Duration in DSP samples of half a dot
    last_dot, // State of last dot transmitted
    new_dot;  // State of new dot to transmit


  // Adjust glyph index
  glx -= font_data->first_glyph;

  // Transmit columns
  uint32_t one = 1; // To silence -sanitize = undefined ??
  half_dot = hell_rc_data.tx_samp_per_dot / 2;
  for( bmx = 0; bmx < font_data->glyph_data[glx].dwidth_x; bmx++ )
  {
    // Initialize buffer idx
    uint32_t
      loop_buf_idx = 0, // Index to loopback levels buffer
      tx_buf_idx   = 0; // Index to transmit samples buffer

    // Reverse video if activated
    if( Flag[HELL_REVERSE_VIDEO] )
      last_dot = 1;
    else
      last_dot = 0;

    // Go up in bitmap buffer row by row
    for( bmy = font_data->fbb_h - 1; bmy >= 0; bmy-- )
    {
      // Set tone control according to bitmap value.
      new_dot = font_data->glyph_data[glx].bitmap[bmy] & ( one << (31 - bmx) );

      // Reverse video if activated
      if( Flag[HELL_REVERSE_VIDEO] ) new_dot = !new_dot;

      // If two consecutive 1's raise tone, else drop
      if( !last_dot && new_dot )
        tone_ctrl = RAISE_TONE;
      else if( last_dot && !new_dot )
        tone_ctrl = LOWER_TONE;
      else if( last_dot && new_dot )
        tone_ctrl = KEEP_TONE;
      else
        tone_ctrl = SILENT_TONE;
      last_dot = new_dot;

      // Raise/Lower audio tone as requested
      switch( tone_ctrl )
      {
        case SILENT_TONE: // *** No tone (carrier down) ***
                          // Fill Tx buffer with zero level samples
          for( uint32_t idx = 0; idx < hell_rc_data.tx_samp_per_dot; idx++ )
          {
            xmit_buffer.xmit_buf_i[tx_buf_idx] = 0;
            xmit_buffer.xmit_buf_q[tx_buf_idx] = 0;
            tx_buf_idx++;
          }

          // Fill loopback buffer with zero level samples
          loopback_buf[loop_buf_idx] = 0;
          loop_buf_idx++;
          break;

        case RAISE_TONE: // *** Raise tone (carrier up) ***
          // Fill buffer with cosine shaped edge for half dot
          for( uint32_t cos_idx = 0; cos_idx < half_dot; cos_idx++ )
          {
            xmit_buffer.xmit_buf_i[tx_buf_idx] = cosine_table[cos_idx];
            xmit_buffer.xmit_buf_q[tx_buf_idx] = 0;
            tx_buf_idx++;
          }

          // Fill buffer with steady tone for half dot
          for( uint32_t idx = 0; idx < half_dot; idx++ )
          {
            xmit_buffer.xmit_buf_i[tx_buf_idx] = DUC_SAMPLE_MAX;
            xmit_buffer.xmit_buf_q[tx_buf_idx] = 0;
            tx_buf_idx++;
          }

          // Fill loopback buffer with maximum level samples
          loopback_buf[loop_buf_idx] = 255;
          loop_buf_idx++;
          break;

        case LOWER_TONE: // *** Lower tone (Carrier down) ***
          // Fill buffer with cosine falling edge for half dot
          for( int32_t cos_idx = (int32_t)half_dot - 1; cos_idx >= 0; cos_idx-- )
          {
            xmit_buffer.xmit_buf_i[tx_buf_idx] = cosine_table[cos_idx];
            xmit_buffer.xmit_buf_q[tx_buf_idx] = 0;
            tx_buf_idx++;
          }

          // Fill buffer with zero level samples for half dot
          for( uint32_t idx = 0; idx < half_dot; idx++ )
          {
            xmit_buffer.xmit_buf_i[tx_buf_idx] = 0;
            xmit_buffer.xmit_buf_q[tx_buf_idx] = 0;
            tx_buf_idx++;
          }

          // Fill loopback buffer with zero level samples
          loopback_buf[loop_buf_idx] = 0;
          loop_buf_idx++;
          break;

        case KEEP_TONE: // *** Keep tone raised (carrier up) ***
          // Fill buffer with steady tone
          for( uint32_t idx = 0; idx < hell_rc_data.tx_samp_per_dot; idx++ )
          {
            xmit_buffer.xmit_buf_i[tx_buf_idx] = DUC_SAMPLE_MAX;
            xmit_buffer.xmit_buf_q[tx_buf_idx] = 0;
            tx_buf_idx++;
          }

          // Fill loopback buffer with maximum level samples
          loopback_buf[loop_buf_idx] = 255;
          loop_buf_idx++;
          break;

      } // switch( tone_ctrl )
    } // for( bmy = font_data->fbb_h-1; bmy >= 0; bmy-- )

    // Loop back transmitted Hell signal to Rx window
    if( Flag[HELL_LOOPBACK] ) Hell_Loopback();

    // *** Write samples buffer to DUC ***
    xmit_buffer.xmit_buf_len = (uint32_t)hell_rc_data.tx_buff_len;
    sem_wait( &duc_send_semaphore );
    if( !xmit_buffer.status || Flag[GUEST_QUIT] )
    {
      MOX_Control( MOX_OFF );
      return( False );
    }
  } // for( bmx = 0; bmx < font_data->glyph_data[glx].dwidth_x; bmx++ )

  return( True );
} // Tx_Feld_Character()

//------------------------------------------------------------------------

/* Tx_FMHell_Character()
 *
 * Transmits one FM Hell character pointed by "glx"
 */
  static BOOLEAN
Tx_FMHell_Character( font_data_t *font_data, uint16_t glx )
{
  int8_t
    bmx,  // Bitmap x index
    bmy;  // Bitmap y index

  uint32_t dot; // State of dot to transmit

  static double
    tone_phase,     // Phase of audio tone
    white_phinc,    // Phase increment for white tone
    black_phinc,    // Phase increment for black tone
    baud_rate = 0.0;


  // Initialize on new parameters or first call
  if( (baud_rate < hell_rc_data.baud_rate) ||
      (baud_rate > hell_rc_data.baud_rate) )
  {
    baud_rate   = hell_rc_data.baud_rate;
    white_phinc = M_2PI / DUC_SAMPLE_RATE;
    black_phinc = white_phinc;

    // Frequency shift is +-baud_rate/2 for Minimum Shift Keying
    white_phinc *= +baud_rate / 2.0;
    black_phinc *= -baud_rate / 2.0;
    tone_phase = 0.0;
  }

  // Adjust glyph index
  glx -= font_data->first_glyph;

  // Transmit columns
  uint32_t one = 1; // To silence -sanitize = undefined ??
  for( bmx = 0; bmx < font_data->glyph_data[glx].dwidth_x; bmx++ )
  {
    // Initialize buffer idx according to stereo/mono mode
    uint32_t
      loop_buf_idx = 0, // Index to loopback levels buffer
      tx_buf_idx   = 0; // Index to transmit samples buffer

    // Go up in bitmap buffer row by row
    for( bmy = font_data->fbb_h - 1; bmy >= 0; bmy-- )
    {
      // Set tone control according to bitmap (dot) value
      dot = font_data->glyph_data[glx].bitmap[bmy] & ( one << (31 - bmx) );

      // Reverse video if selected
      if( Flag[HELL_REVERSE_VIDEO] ) dot = !dot;

      // Black dot, send black tone
      if( dot )
      {
        // Fill buffer with samples at black tone rate
        for( uint32_t idx = 0; idx < hell_rc_data.tx_samp_per_dot; idx++ )
        {
          xmit_buffer.xmit_buf_i[tx_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(tone_phase) );
          xmit_buffer.xmit_buf_q[tx_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(tone_phase) );
          tx_buf_idx++;
          tone_phase += black_phinc;
          CLAMP_N2PI( tone_phase );
        }

        // Fill loopback buffer with maximum level samples
        loopback_buf[loop_buf_idx] = 255;
        loop_buf_idx++;
      } // if( dot )
      else
      {
        // Blank dot, fill buffer with samples at white tone rate
        for( uint32_t idx = 0; idx < hell_rc_data.tx_samp_per_dot; idx++ )
        {
          xmit_buffer.xmit_buf_i[tx_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * sin(tone_phase) );
          xmit_buffer.xmit_buf_q[tx_buf_idx] = (int16_t)( DUC_SAMPLE_MAX * cos(tone_phase) );
          tx_buf_idx++;
          tone_phase += white_phinc;
          CLAMP_N2PI( tone_phase );
        }

        // Fill loopback buffer with zero level samples
        loopback_buf[loop_buf_idx] = 0;
        loop_buf_idx++;
      }
    } // for( bmy = temp; bmy >= 0; bmy-- )

    // Loop back transmitted Hell signal to Rx window
    if( Flag[HELL_LOOPBACK] ) Hell_Loopback();

    // *** Write samples buffer to DUC ***
    xmit_buffer.xmit_buf_len = (uint32_t)hell_rc_data.tx_buff_len;
    sem_wait( &duc_send_semaphore );
    if( !xmit_buffer.status || Flag[GUEST_QUIT] )
    {
      MOX_Control( MOX_OFF );
      return( False );
    }
  } // for( bmx = 0; bmx < font_data->glyph_data[glx].dwidth_x; bmx++ )

  return( True );
} // Tx_FMHell_Character()

//------------------------------------------------------------------------

/* Transmit_Character()
 *
 * Transmits one Hell character pointed by "glx"
 */
  BOOLEAN
Transmit_Character( font_data_t *font_data, uint16_t glx )
{
  BOOLEAN ret;

  // Allocate memory to column. bitmap_height is in pixels
  if( (bm_height != hell_rc_data.bitmap_height) || (column == NULL) )
  {
    bm_height = hell_rc_data.bitmap_height;
    Mem_Realloc( (void **) &column, sizeof(unsigned char) * (size_t)bm_height );
  }

  if( Flag[HELL_MODE_FELDHELL] )
    ret = Tx_Feld_Character(font_data, glx);
  else
    ret = Tx_FMHell_Character(font_data, glx);

  return( ret );
} // Transmit_Character()

//------------------------------------------------------------------------

/* Make_Cos_Wavetable()
 *
 * Makes a cosine look-up wavetable 0-pi/2
 */
  void
Make_Cos_Wavetable( void )
{
  double angle;
  uint32_t half_dot, idx;

  // Allocate buffer
  half_dot = hell_rc_data.tx_samp_per_dot / 2;
  Mem_Realloc( (void **) &cosine_table, sizeof(int16_t) * (size_t)half_dot );

  // Make a raised cos shape
  angle = M_PI_2 / (double)half_dot;
  for( idx = 0; idx < half_dot; idx++ )
  {
    double cosine = sin( angle * (double)idx );
    cosine_table[idx] = (int16_t)( DUC_SAMPLE_MAX * cosine );
  }

} // Make_Cos_Wavetable()

//------------------------------------------------------------------------

// Free resources
  void
Hell_Free_Codec( void )
{
  Mem_Free( (void **) &row_ave );
  Mem_Free( (void **) &dots );
  Mem_Free( (void **) &column );
  Mem_Free( (void **) &cosine_table );
  fbb_h  = 0;
  fbb_h2 = 0;
  bm_height = 0;
}

//------------------------------------------------------------------------

